require('game/tools/Tool');

Tools.Hammer = class BaseHammer extends Tools.BaseTool {
  static get spriteName() {
    return 'metal_tools[4]_iron';
  }

  static get spriteIndex() {
    return 1;
  }

  static get level() {
    return 0;
  }

  static get toolName() {
    return 'hammer';
  }

  static get energyRequired() {
    return 5;
  }

  static giveRock(amount, x, y) {
    if (!Managers.Items.tryPickItemId('rock', amount)) {
      Managers.FarmObjects.createItemAt($gameMap._mapId, x, y, 'rock', amount);
    }
  }

  static checkStoneStatus(stone) {
    const amount = stone.modelName == FarmObjectState.STONE ? 4 : 1;

    if (stone.hp <= 1) {
      stone.hp = 1;

      $gameTemp.setTimeout(() => {
        const customClear = Managers.Content.onClearObject(stone);

        if (customClear === false) {
          return;
        }

        const x = stone.x;
        const y = stone.y;

        stone.clear();
        stone.updateEvents();

        if (customClear === true) {
          return;
        }

        $gameTemp.setTimeout(() => {
          this.giveRock(amount, x, y);
        }, 12);
      }, 3);
    }
  }

  static doMiningRockTooLEffect(rock) {
    Managers.Player.miningExp++;
    Managers.Sound.playHammer();

    if (rock.broken) {
      rock.clear();
    } else {
      rock.broken = true;
      $gameTemp.setTimeout(() => { //jshint ignore:line
        this.getMinedItem(rock);
      }, 12);
    }

    this.affectHealth(0);
  }

  static doStoneToolEffect(stone, toolLevel) {
    if (stone.hp === undefined) {
      stone.hp = 5;
    }

    stone.hp--;
    Managers.Player.miningExp++;
    this.checkStoneStatus(stone);

    let newLevel = toolLevel;
    let newHp = stone.hp;
    let waitTime = 3;
    this.affectHealth(0);
    Managers.Sound.playHammer();

    while ((newLevel > 0 || newHp < 3) && newHp > 1) {
      newHp--;
      newLevel--;
      if (newHp >= 3) {
        this.affectHealth(0);
      }
      Managers.Player.miningExp++;

      if (newHp > 1) {
        $gameTemp.setTimeout(() => { // jshint ignore:line
          stone.hp--;
          stone.updateEvents();
        }, waitTime);
        waitTime += 3;
      } else {
        $gameTemp.setTimeout(() => { //jshint ignore:line
          stone.hp = 1;
          newLevel = 0;
          this.checkStoneStatus(stone);
        }, waitTime);
      }
    }
  }

  static doBoulderToolEffect(boulder, toolLevel) {
    if (boulder.hp === undefined) {
      boulder.hp = 20;
    }

    if (toolLevel >= 3) {
      boulder.hp--;
      this.affectHealth(0);
      Managers.Player.miningExp++;
      Managers.Sound.playHammer();

      if (boulder.hp < 1) {
        boulder.modelName = FarmObjectState.STONE;
        boulder.hp = 5;
        boulder._forceEventSprite = true;
      }
    }
  }

  static doNewRockToolEffect(rock, toolLevel, tile) {
    Managers.Sound.playHammer();
    let customClear;
    switch(rock.modelName) {
      case 'rock-1':
        customClear = Managers.Content.onClearObject(rock);
        if (customClear === false) {
          return;
        }

        rock.clear();
        rock.updateEvents();

        if (customClear !== true) {
          $gameTemp.setTimeout(() => {
            this.giveRock(1, tile.x, tile.y);
          }, 12);

          this.affectHealth(0);
        }
        break;
      case 'rock-2':
        if (toolLevel >= 1) {
          customClear = Managers.Content.onClearObject(rock);
          if (customClear === false) {
            return;
          }

          rock.clear();
          rock.updateEvents();

          if (customClear !== true) {
            $gameTemp.setTimeout(() => {
              this.giveRock(4, tile.x, tile.y);
            }, 12);

            this.affectHealth(1);
          }
        }
        break;
      case 'rock-3':
        if (toolLevel >= 2) {
          const usedLevel = (toolLevel == 2 || rock.broken) ? 2 : 3;

          if (toolLevel == 2 && !rock.broken) {
            rock.broken = true;
          } else {
            customClear = Managers.Content.onClearObject(rock);
            if (customClear === false) {
              return;
            }

            rock.clear();

            if (customClear !== true) {
              $gameTemp.setTimeout(() => {
                this.giveRock(8, tile.x, tile.y);
              }, 12);
            }
          }

          rock.updateEvents();

          if (customClear !== true) {
            this.affectHealth(usedLevel);
          }
        }

        break;
      case 'rock-4':
        if (toolLevel >= 3) {
          if (!rock.broken) {
            rock.broken = true;
          } else if (!rock.brokenTwice) {
            rock.brokenTwice = true;
          } else {
            customClear = Managers.Content.onClearObject(rock);
            if (customClear === false) {
              return;
            }

            rock.clear();

            if (customClear !== true) {
              $gameTemp.setTimeout(() => {
                this.giveRock(24, tile.x, tile.y);
              }, 12);
            }
          }

          rock.updateEvents();

          if (customClear !== true) {
            this.affectHealth(3);
          }
        }

        break;
    }
  
  }

  static doToolEffect(level, targetTile = null) {
    const tiles = this.getAffectedTiles(level, false, TouchInput.wasMouseUsed(), targetTile);

    for (let i = 0; i < tiles.length; i++) {
      const tile = tiles[i];
      const farmObjects = Managers.FarmObjects.getFarmObjectsXy($gameMap._mapId, tile.x, tile.y);

      for (let j = 0; j < farmObjects.length; j++) {
        const farmObject = farmObjects[j];
        if (farmObject === undefined) {
          continue;
        }

        const state = farmObject.state;

        if (state == FarmObjectState.TILLED_SOIL) {
          farmObject.clear();
        } else if (state == FarmObjectState.MINING_ROCK) {
          this.doMiningRockTooLEffect(farmObject);
        } else if (farmObject.modelName == FarmObjectState.STONE) {
          this.doStoneToolEffect(farmObject, level);
        } else if (farmObject.modelName == FarmObjectState.BOULDER) {
          this.doBoulderToolEffect(farmObject, level);
        } else if (farmObject.modelName.startsWith('rock-')) {
          this.doNewRockToolEffect(farmObject, level, tile);
        } else {
          continue;
        }

        farmObject.updateEvents();
        break;
      }
    }

    Managers.FarmObjects.clearEmptyItems();
  }

  static getMinedItem(farmObject) {
    const type = farmObject.getFarmObjectType();
    if (!type || !type.itemChance) {
      console.log(farmObject._rockType, type);
      console.error('Invalid Rock Type');
      return;
    }

    let chance = type.itemChance(farmObject);
    let rockType = farmObject._rockType;
    let isBlueRock = rockType.indexOf('blue-') === 0;
    const maxLevel = Managers.Tools.getMaxToolLevel();

    const contentManagerChance = Managers.Content.getOreDropChange(type, chance, maxLevel);
    if (!isNaN(Number(contentManagerChance))) {
      chance = contentManagerChance;
    }

    rockType = rockType.replace('blue-', '');
    farmObject.clear();

    chance += (maxLevel * 10);

    if (Math.randomInt(100) < chance) {
      let gotBlueDrop = false;

      if (isBlueRock) {
        $gameTemp.setTimeout(() => {
          switch(rockType) {
            case 'copper-ore':
              if (Variables.blueCrystalDropRate <= 0) {
                //Start with a boost
                Variables.blueCrystalDropRate = 20;
              }
              gotBlueDrop = Math.randomInt(100) < Variables.blueCrystalDropRate;

              if (gotBlueDrop && !Switches.gotAnyCrystalToday) {
                if (Math.randomInt(100) >= (95 - maxLevel * 10)) {
                  Managers.Items.pickItemId('white-crystal');
                } else {
                  Managers.Items.pickItemId('blue-crystal');
                }

                Variables.blueCrystalsMined = Variables.blueCrystalsMined + 1;
                Variables.blueCrystalDropRate = 5;
                Switches.gotAnyCrystalToday = true;
              } else {
                Variables.blueCrystalDropRate += Math.floor((1 + maxLevel) / 2);
              }

              break;
            case 'silver-ore':
              if (Variables.bigBlueCrystalDropRate <= 0) {
                //Start with a boost
                Variables.bigBlueCrystalDropRate = 30;
              }
              gotBlueDrop = Math.randomInt(200) < Variables.bigBlueCrystalDropRate;

              if (gotBlueDrop && !Switches.gotAnyCrystalToday) {
                if (Math.randomInt(100) >= (95 - maxLevel * 10)) {
                  Managers.Items.pickItemId('big-white-crystal');
                } else {
                  Managers.Items.pickItemId('big-blue-crystal');
                }

                Variables.bigBlueCrystalDropRate = 5;
                Switches.gotAnyCrystalToday = true;
              } else {
                Variables.bigBlueCrystalDropRate += 1 + maxLevel;
              }
              break;
          }

          if (!gotBlueDrop) {
            Managers.Items.pickItemId(rockType);
          }
        }, 6);

      } else {
        $gameTemp.setTimeout(() => {
          Managers.Items.pickItemId(rockType);
        }, 6);
      }
    } else {
      if (!isBlueRock) {
        if (Math.randomInt(100) < 50) {
          Managers.Items.pickItemId('clay');
        } else {
          Managers.Items.pickItemId('rock');
        }
      }
    }

    farmObject.updateEvents();
  }

  static use(level, targetTile = null) {
    const oldStaminaLevel = Managers.Health.staminaLevel;
    level = Managers.Tools.getMaxToolLevel();

    this.doAnimation();
    this.affectHealth(0);
    setTimeout(() => {
      Managers.Sound.playSwing();
      this.doToolEffect(level, targetTile);
      Managers.Health.oldStaminaLevel = oldStaminaLevel;
    }, this.effectDelay);
  }

  static filterInvalidTiles(basicTiles) {
    const validTiles = [];
    const heightLevel = $gamePlayer.getHeightLevel();

    for (let i = 0; i < basicTiles.length; i++) {
      const tile = basicTiles[i];
      tile.valid = false;

      if (Managers.Map.getHeightLevel(tile.x, tile.y) != heightLevel) continue;

      const farmObjects = Managers.FarmObjects.getFarmObjectsXy($gameMap._mapId, tile.x, tile.y);
      for (let j = 0; j < farmObjects.length; j++) {
        const farmObject = farmObjects[j];

        if (farmObject === undefined) continue;

        const state = farmObject.state;

        const validStates = [FarmObjectState.STONE, FarmObjectState.MINING_ROCK, FarmObjectState.TILLED_SOIL, FarmObjectState.BOULDER, 'rock-1', 'rock-2', 'rock-3', 'rock-4'];
        if (validStates.indexOf(state) < 0) {
          continue;
        }

        if (state == FarmObjectState.BOULDER && this.level < 3) {
          tile.valid = false;
        } else if (state == 'rock-4' && this.level < 3) {
          this.valid = false;
        } else if (state == 'rock-3' && this.level < 2) {
          this.valid = false;
        } else if (state == 'rock-2' && this.level < 1) {
          this.valid = false;
        } else {
          tile.valid = true;
        }

        tile.farmObject = farmObject;
        // tile.valid = true;
        validTiles.push(tile);
        break;
      }
    }

    return validTiles;
  }

  static getAffectedTiles(level, includeInvalid, useCurrentMousePosition, targetTile = null) {
    if (level === undefined) level = this.level;

    const initialTile = targetTile || this.targetTile(useCurrentMousePosition);
    const allTiles = this.getBasicAffectedTiles(initialTile, level);

    const validTiles = this.filterInvalidTiles(allTiles);

    return validTiles;
  }

  static isTileBetterThanDefault(x, y) {
    return this.checkIfTileIsBetterThanDefault(x, y);
  }

  static shouldBeUsedOnEvent(event) {
    const farmObjectData = event._farmObjectData;
    const modelName = farmObjectData.modelName;

    if (modelName == FarmObjectState.STONE) {
      return true;
    }

    if (farmObjectData.rockType)  {
      return true;
    }

    if (modelName == FarmObjectState.BOULDER) {
      return Managers.Tools.toolLevel >= 3;
    }
    if (modelName == 'rock-1') {
      return true;
    }
    if (modelName == 'rock-2') {
      return Managers.Tools.toolLevel >= 1;
    }
    if (modelName == 'rock-3') {
      return Managers.Tools.toolLevel >= 2;
    }
    if (modelName == 'rock-4') {
      return Managers.Tools.toolLevel >= 3;
    }

    const state = farmObjectData.state;

    if (state == FarmObjectState.TILLED_SOIL) {
      return true;
    }

    return false;
  }
};

Managers.Tools.registerTool(Tools.Hammer);