require('game/tools/Tool');

Tools.WateringCan = class BaseWateringCan extends Tools.BaseTool {
  static get spriteName() {
    return 'watering_can[4]';
  }

  static get spriteIndex() {
    return 0;
  }

  static get level() {
    return 0;
  }

  static get toolName() {
    return 'watering-can';
  }

  static get energyRequired() {
    return 8; // Should be able to water 125 tiles
  }

  static get fatigueRequired() {
    return 0.3;
  }

  static doToolEffect(level, targetTile = null) {
    const tiles = this.getAffectedTiles(level, false, false, targetTile);

    if (tiles.length === 1) {
      if (tiles[0].isWater()) {
        this.doLoadWaterEffect();
        return;
      }
    }

    if (Managers.Tools.water > 0) {
      let decreasedWater = false;
      for (let i = 0; i < tiles.length; i++) {
        const tile = tiles[i];
        let farmObject = Managers.FarmObjects.getFarmObject($gameMap._mapId, tile.x, tile.y);
        if (farmObject === undefined) {
          farmObject = Managers.FarmObjects.createFarmObjectAt($gameMap._mapId, tile.x, tile.y);
        }

        farmObject.watered = true;
        Managers.Player.farmingExp++;
        Managers.Player.waterExp++;
        farmObject.updateEvents();

        this.decreaseWater();
        decreasedWater = true;
        this.affectHealth(0);

        if (Managers.Tools.water === 0) break;
      }

      if (!decreasedWater) {
        this.decreaseWater();
      }
    } else {
      this.doNoWaterEffect();
    }
  }

  static decreaseWater() {
    if (Managers.Tools.water > 0) {
      Managers.Tools.water--;
      Managers.Sound.playWateringCan();
    }

    Managers.Items.requestHudRefresh();
  }

  static doNoWaterEffect() {

  }

  static doLoadWaterEffect() {
    Managers.Tools.water = Managers.Tools.maxWater;
    Managers.Sound.playWateringCanFill();

    Managers.Items.requestHudRefresh();
  }

  static useLoadWater(/*level*/) {
    const oldStaminaLevel = Managers.Health.staminaLevel;
    $gamePlayer.setToolState('_load');
    this.doCustomAnimation(() => {
      $gamePlayer.doToolAnimation(this.toolName, undefined, undefined, '_empty');
    });

    this.affectHealth(0);
    setTimeout(() => {
      this.doLoadWaterEffect();
      Managers.Health.oldStaminaLevel = oldStaminaLevel;
    }, this.effectDelay);
  }

  static useNormal(level, targetTile = null) {
    const oldStaminaLevel = Managers.Health.staminaLevel;
    this.doAnimation();

    setTimeout(() => {
      this.doToolEffect(level, targetTile);
      Managers.Health.oldStaminaLevel = oldStaminaLevel;
    }, this.effectDelay);
  }

  static getSpriteIndex(toolState) {
    if (toolState) {
      return this.spriteIndex + 4;
    }

    return this.spriteIndex;
  }

  static useNoWater(level) {
    const oldStaminaLevel = Managers.Health.staminaLevel;
    this.doCustomAnimation(() => {
      $gamePlayer.doToolAnimation(this.toolName, undefined, undefined, '_empty');
    });

    this.affectHealth(0);
    setTimeout(() => {
      this.doNoWaterEffect(level);
      Managers.Health.oldStaminaLevel = oldStaminaLevel;
    }, this.effectDelay);
  }

  static isTileBetterThanDefault(x, y) {
    const tile = new TileHelper.Tile(x, y);
    if (!tile) return false;

    if (Managers.Tools.water < Managers.Tools.maxWater) {
      if (tile.isWater()) {
        const regionId = $gameMap.regionId(tile.x, tile.y);
        if (regionId === Region.BRIDGE) {
          return false;
        }

        return true;
      }
    }

    if (Managers.Tools.water > 0) {
      const allTiles = this.getBasicAffectedTiles.call(this, tile, 0);
      const validTiles = this.filterInvalidTiles(allTiles);

      if (validTiles.length > 0) {
        return true;
      }
    }

    return false;
  }

  static use(level, targetTile = null) {
    if (Managers.Tools.water < Managers.Tools.maxWater) {
      const target = targetTile || this.targetTile(false);
      if (!!target && target.isWater()) {
        $gamePlayer.turnTowardPosition(target.x, target.y);

        this.useLoadWater(level);
        return;
      }
    }

    if (Managers.Tools.water > 0) {
      this.useNormal(level, targetTile);
    } else {
      this.useNoWater(level);
    }
  }

  static getBasicAffectedTiles(initialTile, level) {
    if (level === undefined) level = this.level;
    if (level > this.level) level = this.level;

    const tiles = [];

    if (initialTile) {
      tiles.push(initialTile);

      if (level > 0) {
        const direction = $gamePlayer.directionToPosition(initialTile.x, initialTile.y);
        const rightDirection = DirectionHelper.rightSide(direction);
        const leftDirection = DirectionHelper.leftSide(direction);


        const rightTile1 = initialTile.tileAtDirection(rightDirection);
        const leftTile1 = initialTile.tileAtDirection(leftDirection);

        tiles.push(rightTile1);
        tiles.push(leftTile1);

        if (level > 1) {
          const nextTile = initialTile.tileAtDirection(direction);
          if (nextTile) {
            tiles.push(nextTile);

            const rightTile2 = nextTile.tileAtDirection(rightDirection);
            const leftTile2 = nextTile.tileAtDirection(leftDirection);

            tiles.push(rightTile2);
            tiles.push(leftTile2);

            if (nextTile !== undefined) {
              let rightTile1Right;
              let rightTile2Right;
              let leftTile1Left;
              let leftTile2Left;

              if (rightTile2 !== undefined) {
                rightTile2Right = rightTile2.tileAtDirection(rightDirection);
                tiles.push(rightTile2Right);
              }

              if (leftTile2 !== undefined) {
                leftTile2Left = leftTile2.tileAtDirection(leftDirection);
                tiles.push(leftTile2Left);
              }

              if (rightTile1 !== undefined) {
                rightTile1Right = rightTile1.tileAtDirection(rightDirection);
                tiles.push(rightTile1Right);
              }

              if (leftTile1 !== undefined) {
                leftTile1Left = leftTile1.tileAtDirection(leftDirection);
                tiles.push(leftTile1Left);
              }

              if (level > 2) {
                let rightTile2RightRight;
                let leftTile2LeftLeft;

                if (rightTile2Right !== undefined) {
                  rightTile2RightRight = rightTile2Right.tileAtDirection(rightDirection);

                  if (rightTile2RightRight) {
                    tiles.push(rightTile2RightRight);
                    tiles.push(rightTile2RightRight.tileAtDirection(direction));
                    tiles.push(rightTile2RightRight.tileAtDirection(direction, 2));
                  }

                  tiles.push(rightTile2Right.tileAtDirection(direction));
                  tiles.push(rightTile2Right.tileAtDirection(direction, 2));
                }

                if (leftTile2Left !== undefined) {
                  leftTile2LeftLeft = leftTile2Left.tileAtDirection(leftDirection);
                  if (leftTile2LeftLeft) {
                    tiles.push(leftTile2LeftLeft);
                    tiles.push(leftTile2LeftLeft.tileAtDirection(direction));
                    tiles.push(leftTile2LeftLeft.tileAtDirection(direction, 2));
                  }

                  tiles.push(leftTile2Left.tileAtDirection(direction));
                  tiles.push(leftTile2Left.tileAtDirection(direction, 2));
                }

                if (leftTile1Left !== undefined) {
                  tiles.push(leftTile1Left.tileAtDirection(leftDirection));
                }
                if (rightTile1Right !== undefined) {
                  tiles.push(rightTile1Right.tileAtDirection(rightDirection));
                }

                if (rightTile2 !== undefined) {
                  tiles.push(rightTile2.tileAtDirection(direction));
                  tiles.push(rightTile2.tileAtDirection(direction, 2));
                }

                if (leftTile2 !== undefined) {
                  tiles.push(leftTile2.tileAtDirection(direction));
                  tiles.push(leftTile2.tileAtDirection(direction, 2));
                }

                tiles.push(nextTile.tileAtDirection(direction));
                tiles.push(nextTile.tileAtDirection(direction, 2));
              }
            }

          }
        }
      }
    }

    const basicTiles = [];
    for (let i = 0; i < tiles.length; i++) {
      if (tiles[i] !== undefined) {
        basicTiles.push(tiles[i]);
      }
    }

    return basicTiles;
  }

  static filterInvalidTiles(basicTiles, includeInvalid) {
    const validTiles = [];
    const heightLevel = $gamePlayer.getHeightLevel();

    for (let i = 0; i < basicTiles.length; i++) {
      const tile = basicTiles[i];

      const regionId = $gameMap.regionId(tile.x, tile.y);
      if (regionId === Region.BRIDGE) {
        continue;
      }

      if (Managers.Map.getHeightLevel(tile.x, tile.y) != heightLevel) continue;

      const anyCropToWater = Managers.FarmObjects.quickFarmObjectsCheck($gameMap._mapId, tile.x, tile.y, {
        postFilter: (farmObject) => {
          if (!farmObject.tilled) return false;
          if (farmObject.watered) return false;
          if (farmObject.state == Constants.FarmObjectState.NONE) return false;
          if (farmObject.isFlower()) return false;
          if (farmObject.isTree()) return false;

          return true;
        }
      }, true);

      if (!anyCropToWater) {
        continue;
      }

      tile.highlightType = 'crop';
      tile.valid = true;
      validTiles.push(tile);
    }

    return validTiles;
  }

  static getAffectedTiles(level, includeInvalid, useCurrentMousePosition, targetTile) {
    if (level === undefined) level = this.level;
    targetTile = targetTile || this.targetTile(useCurrentMousePosition);

    if (Managers.Tools.water < Managers.Tools.maxWater) {
      if (!!targetTile && targetTile.isWater()) {

        const regionId = $gameMap.regionId(targetTile.x, targetTile.y);
        if (regionId === Region.BRIDGE) {
          return [];
        }

        targetTile.valid = true;
        targetTile.highlightType = 'water';
        return [targetTile];
      }
    }

    if (Managers.Tools.water > 0) {
      const allTiles = this.getBasicAffectedTiles.call(this, targetTile, level);
      return this.filterInvalidTiles(allTiles);
    }

    return [];
  }

  static shouldBeUsedOnEvent(event) {
    const farmObjectData = event._farmObjectData;
    const modelName = farmObjectData.modelName;

    if (!modelName && farmObjectData.tilled) {
      return true;
    }

    if (modelName.indexOf('-crop') >= 0 || modelName == 'soil') {
      if (farmObjectData.isFlower()) {
        return false;
      }

      if (farmObjectData.isTree()) {
        return false;
      }

      const model = farmObjectData.getFarmObjectType();
      if (model.isModel(Models.BushCrop)) {
        return false;
      }

      if (!farmObjectData.watered) {
        return true;
      }
    }

    return false;
  }

  static isTileValidForMouseDown(tile) {
    if (Managers.Tools.water == Managers.Tools.maxWater) {
      return false;
    }

    return tile.isWater();
  }
};

Managers.Tools.registerTool(Tools.WateringCan);