//-----------------------------------------------------------------------------
// Sprite_AffectedTile
//
// The sprite for displaying the tiles that will be affected if the player uses the equipped tool

function Sprite_AffectedTile() {
  this.initialize.apply(this, arguments);
}

Sprite_AffectedTile.prototype = Object.create(Sprite.prototype);
Sprite_AffectedTile.prototype.constructor = Sprite_AffectedTile;

Sprite_AffectedTile.prototype.initialize = function() {
  Sprite.prototype.initialize.call(this);
  this._frameCount = 0;
  this.oldMap = '';
};

Sprite_AffectedTile.prototype.getEmptyArea = function() {
  return {
    area : new Rectangle(0, 0, 0, 0),
    map : '',
    valid : false,
    tiles : []
  };
};

Sprite_AffectedTile.prototype.shouldShowAffectedArea = function() {
  if ($gamePlayer._doingToolAnimation === true) {
    return false;
  }

  if ($gamePlayer.isRiding()) {
    return false;
  }

  // if (!Managers.Config.showAreaOfEffect) {
  //   return false;
  // }

  if (Switches.finishedTutorial || !Switches.startedTutorial) {
    //The tutorial counts as a cutscene, but the affected tiles should be visible during it.
    if ($gameSystem.isPlayingCutscene()) {
      return false;
    }
  }

  return true;
};

Sprite_AffectedTile.prototype.getAffectedArea = function() {
  if (!this.shouldShowAffectedArea()) {
    return this.getEmptyArea();
  }

  var tiles = Managers.Tools.getAffectedTiles(true, true);
  var minX = 999;
  var maxX = -1;
  var minY = 999;
  var maxY = -1;
  var tile;

  for (var i = 0; i < tiles.length; i++) {
    tile = tiles[i];

    minX = Math.min(minX, tile.x);
    maxX = Math.max(maxX, tile.x);
    minY = Math.min(minY, tile.y);
    maxY = Math.max(maxY, tile.y);
  }

  var tileWidth = $gameMap.tileWidth();
  var tileHeight = $gameMap.tileHeight();

  var left = minX * tileWidth;
  var right = (maxX + 1) * tileWidth;
  var top = minY * tileHeight;
  var bottom = (maxY + 1) * tileHeight;
  var width = right - left;
  var height = bottom - top;

  if (width < 0 || height < 0) {
    left = 0; top = 0; width = 0; height = 0;
  }

  var result = {
    area : new Rectangle(left, top, width, height),
    valid : true,
    map : '',
    tiles : []
  };

  for (i = 0; i < tiles.length; i++) {
    tile = tiles[i];
    result.tiles.push({
      area : new Rectangle(tile.x * tileWidth - left, tile.y * tileHeight - top, tileWidth, tileHeight),
      valid : tile.valid,
      farmObject : tile.farmObject,
      highlightType : tile.highlightType
    });
  }

  for (let x = minX; x <= maxX; x++) {
    for (let y = minY; y <= maxY; y++) {
      let symbol = 'X';

      for (i = 0; i < tiles.length; i++) {
        tile = tiles[i];
        if (!tile || !tile.valid) continue;

        if (tile.x != x) continue;
        if (tile.y != y) continue;

        symbol = 'O';
        break;
      }

      result.map += symbol;
    }
  }

  return result;
};

Sprite_AffectedTile.prototype.update = function() {
  Sprite.prototype.update.call(this);
  if (Switches.drawAffectedTiles && Managers.Tools.effectiveToolId !== undefined) {
    this.updatePosition();
    this.updateAnimation();
    this.visible = true;
  } else {
    Managers.FarmObjects.clearOutlinedTiles();
    this._frameCount = 0;
    this.visible = false;
  }
};

Sprite_AffectedTile.prototype.getColor = function(valid) {
  if (!valid) return '#FF1518';

  switch (Managers.Tools.toolLevel)  {
    case 1 :
      return '#986D4D';
    case 2 :
      return '#C4C5C8';
    case 3 :
      return '#FFCD35';
    case 4 :
      return '#6FB1D4';
    default :
      return '#62523d';
  }
};

Sprite_AffectedTile.prototype.fillRects = function(data) {
  this.bitmap.clear();

  let baseBitmap;
  let drawType = 'rect';

  if (Managers.Tools.isToolTypeEquipped('watering-can')) {
    let isCrop = false;

    for (const tile of data.tiles) {
      if (!tile) continue;
      if (tile.highlightType == 'crop') {
        isCrop = true;
        break;
      }
    }

    if (isCrop) {
      drawType = 'bitmap';
      baseBitmap = Managers.Images.loadCharacter('-affected-tile-watering-can');
    } else {
      drawType = 'circle';
    }
  } else if (Managers.Tools.isToolTypeEquipped('fishing-rod')) {
    drawType = 'bitmap';
    baseBitmap = Managers.Images.loadCharacter('-affected-tile-fishing-rod');
  } else if (Managers.Tools.isToolTypeEquipped('hoe')) {
    drawType = 'bitmap';
    baseBitmap = Managers.Images.loadCharacter('-affected-tile-hoe');
  } else if (Managers.Tools.isToolTypeEquipped('axe')) {
    drawType = 'circle';
  } else if (Managers.Tools.isToolTypeEquipped('hammer')) {
    drawType = 'circle';
  } else if (Managers.Tools.isToolTypeEquipped('sickle')) {
    drawType = 'circle';
  }

  if (drawType == 'bitmap' && !baseBitmap.isReady()) {
    baseBitmap.addLoadListener(() => {
      this.fillRects(data);
    });
    return;
  }

  for (var i = 0; i < data.tiles.length; i++) {
    //Don't draw the light area if there's an outlined object
    if (Graphics.isWebGL()) {
      if (data.tiles[i].farmObject) continue;
    }

    var tileRect = data.tiles[i].area;
    var color = this.getColor(data.tiles[i].valid);

    switch (drawType) {
      case 'bitmap':
        this.bitmap.blt(baseBitmap, 0, 0, baseBitmap.width, baseBitmap.height, tileRect.x, tileRect.y);
        break;
      case 'rect':
        this.bitmap.fillRect(tileRect.x, tileRect.y, tileRect.width, tileRect.height, color);
        break;
      default:
        this.bitmap.drawCircle(tileRect.x + (tileRect.width * 0.5), tileRect.y + (tileRect.height * 0.65), tileRect.width / 3, color);
        break;
    }
  }
};

var uniqueAffectedTilesBitmap = false;
Sprite_AffectedTile.prototype.createBitmap = function(data) {
  const w = Math.max(data.area.width, 1);
  const h = Math.max(data.area.height, 1);

  if (uniqueAffectedTilesBitmap && !this.bitmap) {
    this.bitmap = uniqueAffectedTilesBitmap;
  }

  if (this.bitmap) {
    if (this.bitmap.width != w || this.bitmap.height != h) {
      this.bitmap.resize(w, h);
    }
  } else {
    this.bitmap = new Bitmap(w, h);
    uniqueAffectedTilesBitmap = this.bitmap;
  }

  this.width = w;
  this.height = h;

  this.updateBitmap(data);
};

Sprite_AffectedTile.prototype.updateBitmap = function(data) {
  this.fillRects(data);
  this.anchor.x = 0;
  this.anchor.y = 0;
  this.blendMode = Graphics.BLEND_ADD;
};

Sprite_AffectedTile.prototype.updatePosition = function() {
  var data;

  var same = true;
  if ($gameTemp.shouldRefreshAffectedTiles()) same = false;
  if (same && this.oldPlayerX !== $gamePlayer.x) same = false;
  if (same && this.oldPlayerY !== $gamePlayer.y) same = false;
  if (same && this.oldPlayerDirection !== $gamePlayer._direction) same = false;
  if (same && this.oldPlayerAnimating !== $gamePlayer._doingToolAnimation) same = false;
  if (same && this.oldToolLevel !== Managers.Tools.toolLevel) same = false;
  if (same && this.oldTool !== Managers.Tools.effectiveToolId) same = false;
  if (same && this.oldTargetTile !== Managers.Tools.targetTile) same = false;

  if (TouchInput.wasMouseUsed()) {
    if (same && !this.oldMouseUsed) same = false;
    if (same && this.oldMouseX !== TouchInput.mouseX) same = false;
    if (same && this.oldMouseY !== TouchInput.mouseY) same = false;
  }

  if (same) {
    data = this.oldData;
  }

  if (!data) {
    data = this.getAffectedArea();
  } else {
    if (!this.shouldShowAffectedArea()) {
      data = this.getEmptyArea();
    }
  }

  Managers.FarmObjects.setOutlinedTiles(data.tiles);

  // Use 1 for invalid values, as bitmaps will never have 0 width/height
  var w = data.area.width || 1;
  var h = data.area.height || 1;

  if (!this.bitmap || this.bitmap.width != w || this.bitmap.height != h || this.oldMap != data.map) {
    this.createBitmap(data);
  }

  this.oldMap = data.map;
  this.oldPlayerX = $gamePlayer.x;
  this.oldPlayerY = $gamePlayer.y;
  this.oldPlayerDirection = $gamePlayer._direction;
  this.oldPlayerAnimating = $gamePlayer._doingToolAnimation;
  this.oldTool = Managers.Tools.effectiveToolId;
  this.oldData = data;
  this.oldToolLevel = Managers.Tools.toolLevel;
  this.oldTargetTile = Managers.Tools.targetTile;
  this.oldMouseX = TouchInput.mouseX;
  this.oldMouseY = TouchInput.mouseY;
  this.oldMouseUsed = TouchInput.wasMouseUsed();

  this.x = data.area.x - ($gameMap.displayX() * $gameMap.tileWidth());
  this.y = data.area.y - ($gameMap.displayY() * $gameMap.tileHeight());

  this.zIndex = this.z;
  this.zOrder = -this.y;


  $gameTemp.markAffectedTilesAsRefreshed();
};

Sprite_AffectedTile.prototype.updateAnimation = function() {
  this._frameCount++;
  this._frameCount %= 30;
  this.opacity = 155 - (this._frameCount * 3);
};