const Scene_MenuBase = require('engine/scenes/Scene_MenuBase');
const Scene_Settings = require('../diary/Settings');

const Window_TitleOptions = require('../../windows/title/TitleOptions');
const Window_PatchNotes = require('../../windows/title/PatchNotes');
const Window_Credits = require('../../windows/title/Credits');

class Scene_Title extends Scene_MenuBase {
  initialize() {
    super.initialize(...arguments);

    this._zoomLevel = Graphics.windowZoomLevel;
    this._scale = new PIXI.Point(this._zoomLevel, this._zoomLevel);
  }

  create() {
    super.create(...arguments);
    this.createBackgroundContent();
    this.createPatchNotesWindow();
    this.createCreditsWindow();
  }

  start() {
    super.start();
    this.playTitleMusic();
  }

  bgImageName() {
    if (Graphics.width == 1360) {
      return 'title680';
    }

    if (Graphics.width == 1366) {
      return 'title683';
    }

    return 'title';
  }

  loadingBgBitmap() {
    return Managers.Images.loadBackground(this.bgImageName());
  }

  backgroundOpacity() {
    return 255;
  }

  createBackground() {
    super.createBackground();
    this._backgroundSprite.scale = this._scale;
  }

  createPatchNotesWindow() {
    this._patchNotesWindow = new Window_PatchNotes();
    this._patchNotesWindow.hide();
    this._patchNotesWindow.deactivate();
    this._patchNotesWindow.setCancelHandler(this.closePopupWindows.bind(this));
    this.addWindow(this._patchNotesWindow);

    const button = new Sprite_Button();
    button.bitmap = Managers.Images.loadSystem('ButtonSet');

    const buttonIndex = 5;
    const x = 48 * buttonIndex;
    const w = 48;
    const h = 48;

    button.setColdFrame(x, 0, w, h);
    button.setHotFrame(x, h, w, h);
    
    this._patchNotesWindow.addChild(button);
    this._patchNotesCloseButton = button;

    button.setClickHandler(this.closePopupWindows.bind(this));

    button.x = this._patchNotesWindow.width - w  + 2 * Graphics.windowZoomLevel;
    button.y = 0;
  }

  createCreditsWindow() {
    this._creditsWindow = new Window_Credits();
    this._creditsWindow.hide();
    this._creditsWindow.deactivate();
    this._creditsWindow.setCancelHandler(this.closePopupWindows.bind(this));
    this.addWindow(this._creditsWindow);

    const button = new Sprite_Button();
    button.bitmap = Managers.Images.loadSystem('ButtonSet');

    const buttonIndex = 5;
    const x = 48 * buttonIndex;
    const w = 48;
    const h = 48;

    button.setColdFrame(x, 0, w, h);
    button.setHotFrame(x, h, w, h);
    
    this._creditsWindow.addChild(button);
    this._creditsCloseButton = button;

    button.setClickHandler(this.closePopupWindows.bind(this));

    button.x = this._creditsWindow.width - w + 2 * Graphics.windowZoomLevel;
    button.y = 0;
  }

  createBackgroundContent() {
    this._bgContentWindow = new Window_TitleOptions(0, 0, Graphics.width, Graphics.height);

    this._bgContentWindow.setHandler('newGame', this.commandNewGame.bind(this));
    this._bgContentWindow.setHandler('load', this.commandLoad.bind(this));
    this._bgContentWindow.setHandler('options', this.commandOptions.bind(this));
    this._bgContentWindow.setHandler('updates', this.commandPatchNotes.bind(this));
    this._bgContentWindow.setHandler('exit', this.commandExit.bind(this));

    this.addWindow(this._bgContentWindow);
    this._bgContentWindow.refresh();
  }

  update() {
    super.update();

    if (Managers.Scenes.isSceneChanging()) return;

    if (this._bgContentWindow.canCheckInput()) {
      if (Engine.Input.isTriggered('map')) {
        this.commandCredits();
        return;
      }
    }
  }

  isCancelTriggered() {
    if ($gameTemp.shouldSkipSelectableWindowHandling()) return false;
    if (Engine.Input.isRepeated('cancel')) return true;

    if (Engine.Input.isRepeated('alternate')) {
      return true;
    }

    return false;
  }

  commandExit() {
    var gui = require('nw.gui');
    var win = gui.Window.get();
    win.close();
  }

  commandNewGame() {
    Engine.Data.autoNewGame();
    this.fadeOutAll();
  }

  commandLoad() {
    if (Engine.Data.anySaveExists()) {
      Managers.Scenes.push(GameScenes.LoadGame);
      this.startFadeOut(this.slowFadeSpeed());
    } else {
      Managers.Sound.playBuzzer();
    }
  }

  commandOptions() {
    this._bgContentWindow.close();
    Managers.Scenes.push(Scene_Settings);
  }

  commandPatchNotes() {
    this._bgContentWindow.openChangeLog();
    this._patchNotesWindow.show();
    this._patchNotesWindow.opacity = 255;
    this._patchNotesWindow.activate();
    this._patchNotesWindow.refresh();
  }

  commandCredits() {
    this._bgContentWindow.openChangeLog();
    this._creditsWindow.show();
    this._creditsWindow.opacity = 255;
    this._creditsWindow.activate();
    this._creditsWindow.refresh();
  }

  closePopupWindows() {
    this._patchNotesWindow.hide();
    this._patchNotesWindow.deactivate();

    this._creditsWindow.hide();
    this._creditsWindow.deactivate();

    this._bgContentWindow.closeChangeLog();
    Managers.Sound.playCancel();
  }

  playTitleMusic() {
    Engine.Audio.stopBgs();
    Engine.Audio.stopMe();
    Engine.Audio.playBgmName('title');
  }

  mustFadeOutBgm() {
    return false;
  }
}

GameScenes.Title = Scene_Title;
module.exports = Scene_Title;