const Scene_MenuBase = require('engine/scenes/Scene_MenuBase');

const Window_ShopItemInfo = require('../../windows/shop/ShopItemInfo');
const Window_ShopItemList = require('../../windows/shop/ShopItemList');
const Window_AnimatedBackground = require('../../windows/diary/AnimatedBackground');
const ShopNumberSubWindow = require('../../windows/shop/ShopNumberSubWindow');

class Scene_ShopItemList extends Scene_MenuBase {
  initialize() {
    super.initialize();
    this._container = new Engine.ItemContainer();
    this._container.clear();
    this._container.setMaxItemTypes(1);
    this._delay = Utils.getFrameCount(20);
    this.updateContainer();

    Switches.bagWasFullWhileBuying = false;
  }

  addItem(itemId, amount) {
    this._container.gainItemId(itemId, amount || Infinity);
  }

  selectItem(item) {
    this._listWindow.prepareToBuyItem(item);
    this._infoWindow.setItem(item);
    this._listWindow.deactivate();

    const price = Managers.Items.getDiscountedBuyPrice(item);
    const ingredients = Managers.Items.getItemIdIngredients(item.id);
    const maxBuy = Managers.Shop.maxBuyNumberForItem(item, price, ingredients);

    this._numberWindow.setup(item, maxBuy, price, ingredients);
    this._numberWindow.show();
    this._numberWindow.activate();
  }

  create() {
    super.create();
    this._inputDelay = Utils.getFrameCount(8);

    this.createBackgroundWindow();
    this.createListWindow();
    this.createInfoWindow();
    this.createCloseButton();
    this.createNumberWindow();

    this._listWindow.show();
    this._listWindow.activate();
    this._listWindow.refresh();
    this._listWindow.select(0);
  }

  createBackground() {
    const bitmap = Managers.Scenes.backgroundBitmap();
    if (bitmap) {
      this._backgroundSprite = new Sprite(bitmap);
      this.addChild(this._backgroundSprite);
      this.setBackgroundOpacity(this.backgroundOpacity());
    }
  }

  createCloseButton() {
    super.createCloseButton();
    this._closeButtonWindow.x = this._listWindow.x + this._listWindow.width - this._closeButtonWindow.width - (9 * Graphics.windowZoomLevel);
    this._closeButtonWindow.y = this._listWindow.y + (24 * Graphics.windowZoomLevel);
  }

  createBackgroundWindow() {
    if (!this._backgroundSprite) {
      this._bgAnimationWindow = new Window_AnimatedBackground(0, 0, Graphics.width, Graphics.height);
      this.addWindow(this._bgAnimationWindow);
      this._bgAnimationWindow.refresh();
    }
  }

  makeItemList() {
  }

  updateContainer() {
    this.makeItemList();
    this._container.setMaxItemTypes(this._container.itemCount());
  }

  update() {
    super.update();
    if (this._delay > 0) {
      this._delay--;
    }
  }

  createInfoWindow() {
    const x = this._listWindow.x + this._listWindow.width;
    const y = (Graphics.height - (164 * Graphics.windowZoomLevel)) / 2;

    this._infoWindow = new Window_ShopItemInfo(x, y);
    this.addWindow(this._infoWindow);
    this._infoWindow.setDiscountInformationEnabled(true);
    this._listWindow.setInfoWindow(this._infoWindow);
  }
  
  createListWindow() {
    this._listWindow = new Window_ShopItemList(this._container);
    this._listWindow.hide();
    this._listWindow.setHandler('ok', this.onItemOk.bind(this));
    this._listWindow.setHandler('cancel', this.onItemCancel.bind(this));
    this.addWindow(this._listWindow);
  }

  createNumberWindow() {
    const x = this._listWindow.x + 58 * Graphics.windowZoomLevel;
    const y = this._listWindow.y + 115 * Graphics.windowZoomLevel;

    this._numberWindow = new ShopNumberSubWindow(x, y);
    this._numberWindow.hide();
    this._numberWindow.setHandler('ok', this.onNumberOk.bind(this));
    this._numberWindow.setHandler('cancel', this.onNumberCancel.bind(this));
    this.addWindow(this._numberWindow);
  }

  endPurchase() {
  }

  onNumberOk() {
    const item = this._numberWindow.item();
    const number = this._numberWindow.totalNumber();
    const price = Managers.Items.getDiscountedBuyPrice(item);
    const ingredients = Managers.Items.getItemIdIngredients(item.id);
    const maxBuy = Managers.Shop.maxBuyNumberForItem(item, price, ingredients);

    if (maxBuy >= number && Managers.Items.buyItem(item.id, number, price)){
      Managers.Sound.playShop();
      this._numberWindow.hide();
      this._listWindow.activate();
      this._listWindow.endPurchase();

      this.endPurchase();
      return;
    }

    Managers.Sound.playBuzzer();
    this._numberWindow.activate();
  }

  onNumberCancel() {
    this._numberWindow.hide();
    this._numberWindow.deactivate();
    this._listWindow.activate();
    this._listWindow.endPurchase();
  }
  
  onItemOk() {
    if (this._delay > 0) {
      this._listWindow.activate();
      return;
    }

    const item = this._listWindow.item();
    if (!item) {
      this._listWindow.activate();
      return;
    }

    this.selectItem(item);
  }

  hideAllWindows() {
    this._infoWindow.hide();
    this._listWindow.hide();
  }

  onItemCancel() {
    if (this._delay > 0) {
      this._listWindow.activate();
      return;
    }
    this.onClickClose();
  }

  onClickClose() {
    Managers.Scenes.pop();
  }
}

GameScenes.ShopItemList = Scene_ShopItemList;
module.exports = Scene_ShopItemList;