const Scene_MenuBase = require('engine/scenes/Scene_MenuBase');
const Window_ShopCommand = require('../../windows/shop/ShopCommand');
const Window_AnimalsForSale = require('../../windows/shop/AnimalsForSale');

//-----------------------------------------------------------------------------
// Scene_AnimalsForSale
//
// The scene class of the list of animals for sale
class Scene_AnimalsForSale extends Scene_MenuBase {
  prepare(animalTypes) {
    this._animalTypes = animalTypes;

    Managers.Images.loadBackground('animal_shop_layout');
    Managers.Images.loadPicture('shop/animal_shop_buy_slot');
  }

  terminate() {
    Managers.Images.releaseBackground('animal_shop_layout');
    Managers.Images.releasePicture('shop/animal_shop_buy_slot');
    super.terminate();
  }

  create() {
    super.create();
    this.createAnimalListWindow();
    this.createCloseButton();
    this.createCommandWindow();
  }

  animal() {
    return this._animalListWindow.animal();
  }

  createCloseButton() {
    super.createCloseButton();
    this._closeButtonWindow.x = this._animalListWindow.x + this._animalListWindow.width - this._closeButtonWindow.width - (5 * Graphics.windowZoomLevel);
    this._closeButtonWindow.y = this._animalListWindow.y + (22 * Graphics.windowZoomLevel);
  }

  createBackground() {
    const bitmap = Managers.Scenes.backgroundBitmap();
    if (bitmap == null) {
      this._backgroundSprite = new Sprite(this.loadingBgBitmap());
      this._backgroundSprite.scale.x = this._backgroundSprite.scale.y = this.screenRatio;
      this._backgroundSprite.x = this.backgroundX();
    } else {
      this._backgroundSprite = new Sprite(bitmap);
    }

    this.addChild(this._backgroundSprite);
    this.setBackgroundOpacity(this.backgroundOpacity());
  }

  createAnimalListWindow() {
    this._animalListWindow = new Window_AnimalsForSale(this._animalTypes);
    this._animalListWindow.setHandler('ok', this.onAnimalOk.bind(this));
    this._animalListWindow.setHandler('cancel', this.onAnimalCancel.bind(this));
    this.addWindow(this._animalListWindow);
    this._animalListWindow.select(0);
    this.activateAnimalListWindow();
  }

  onAnimalCancel() {
    Managers.Scenes.pop();
  }

  activateAnimalListWindow() {
    this._animalListWindow.refresh();
    this._animalListWindow.activate();
  }

  createCommandWindow() {
    this._commandWindow = new Window_ShopCommand();
    this._commandWindow.setHandler('buy', this.onBuyOk.bind(this));
    this._commandWindow.setHandler('cancel', this.onBuyCancel.bind(this));
    this._commandWindow.hide();
    this._commandWindow.deactivate();
    this.addWindow(this._commandWindow);
  }

  closeCommandWindow() {
    this._commandWindow.hide();
    this._commandWindow.deactivate();
    this.activateAnimalListWindow();
  }

  onBuyOk() {
    const creature = this.animal();
    if (!creature) {
      Managers.Sound.playBuzzer();
      this.closeCommandWindow();
      return;
    }

    if (Managers.Creatures.prepareCreature(creature.creatureName, true)) {
      Managers.Sound.playOk();
      Managers.Creatures.inputNameForAnimal('buy_animal_complete');
      return;
    }

    this.popScene();

    if (Managers.Creatures.isCoopCreature(creature.creatureName)) {
      Managers.CommonEvent.playEvent('buy_coop_animal_failed');
    } else {
      Managers.CommonEvent.playEvent('buy_barn_animal_failed');
    }
  }

  onBuyCancel() {
    this.closeCommandWindow();
  }

  onAnimalOk() {
    const creature = this.animal();
    if (!creature) {
      Managers.Sound.playBuzzer();
      this.activateAnimalListWindow();
      return;
    }

    const enoughMoney = Managers.Items.gold >= creature._price;
    const enoughRoom = Managers.Creatures.hasRoomForAnimal(creature.creatureName);
    if (!enoughMoney || !enoughRoom) {
      Managers.Sound.playBuzzer();
      this.activateAnimalListWindow();
      return;
    }

    this._animalListWindow.deactivate();
    this._commandWindow.show();
    this._commandWindow.activate();
    this._commandWindow.select(1);
  }
}

GameScenes.AnimalsForSale = Scene_AnimalsForSale;
module.exports = Scene_AnimalsForSale;
