const Scene_MenuBase = require('engine/scenes/Scene_MenuBase');
const Window_BackgroundContent = require('../../windows/diary/BackgroundContent');
const Window_OverlayContent = require('../../windows/diary/OverlayContent');
const Window_MainContent = require('../../windows/diary/MainContent');
const Window_BestFriends = require('../../windows/diary/BestFriends');
const Window_BestAnimals = require('../../windows/diary/BestAnimals');
const Window_Stamps = require('../../windows/diary/Stamps');
const Window_StampContent = require('../../windows/diary/StampContent');
const Window_ItemsContent = require('../../windows/diary/ItemsContent');
const Window_ItemsPage2Content = require('../../windows/diary/ItemsPage2Content');
const Window_Backpack = require('../../windows/diary/Backpack');
const Window_DiaryItemHelp = require('../../windows/diary/DiaryItemHelp');
const Window_DiaryItemInfo = require('../../windows/diary/DiaryItemInfo');
const Window_DiaryItemCommand = require('../../windows/diary/DiaryItemCommand');
const Window_ToolsContent = require('../../windows/diary/ToolsContent');
const Window_ToolList = require('../../windows/diary/ToolList');
const Window_DiaryToolHelp = require('../../windows/diary/DiaryToolHelp');
const Window_DataContent = require('../../windows/diary/DataContent');
const Window_CalendarContent = require('../../windows/diary/CalendarContent');
const Window_TasksContent = require('../../windows/diary/TasksContent');
const Window_DataDetailedContent = require('../../windows/diary/DataDetailedContent');
const Window_SettingsContent = require('../../windows/diary/SettingsContent');
const Window_SettingsCategoryList = require('../../windows/diary/SettingsCategoryList');
const Window_SettingsList = require('../../windows/diary/SettingsList');
const Window_Warning = require('../../windows/core/Warning');
const Window_GameEnd = require('../../windows/Window_GameEnd');
const Window_MapDisplay = require('../../windows/diary/MapDisplay');
const Window_DiaryFriends = require('../../windows/diary/FriendList');
const Window_FriendDetails = require('../../windows/diary/FriendDetails');
const Window_FriendsContent = require('../../windows/diary/FriendsContent');

const Window_AnimalList = require('../../windows/diary/AnimalList');
const Window_AnimalDetails = require('../../windows/diary/AnimalDetails');
const Window_AnimalsContent = require('../../windows/diary/AnimalsContent');

const Window_TaskList = require('../../windows/diary/submenus/TaskList');
const Window_AreaList = require('../../windows/diary/submenus/AreaList');
const Window_ProgressList = require('../../windows/diary/submenus/ProgressList');
const Window_ShippedItemsList = require('../../windows/diary/submenus/ShippedItemsList');

const layoutWidth = 577;
const layoutHeight = 328;
const pageWidth = 216;
const pageMargin = 49;
const pageTopMargin = 36;
const pageHeight = 259;
const pageSeparator = 51;
const headerExtraHeight = 12;
const headerExtraWidth = -3;

class Scene_Diary extends Scene_MenuBase {
  initialize() {
    super.initialize(...arguments);
    this._currentLayout = 'main';

    this._zoomLevel = Graphics.windowZoomLevel;
    this._scale = new PIXI.Point(this._zoomLevel, this._zoomLevel);
    this._contentWidth = layoutWidth * this._zoomLevel;
    this._contentHeight = layoutHeight * this._zoomLevel;
    this._changingPage = false;
    this._delay = 0;

    const extraHeight = headerExtraHeight * this._zoomLevel;
    const extraWidth = headerExtraWidth * this._zoomLevel;

    this._xOffset = Math.floor((Graphics.width - this._contentWidth + extraWidth) / 2);
    this._yOffset = Math.floor((Graphics.height - this._contentHeight + extraHeight) / 2);

    this._pageWidth = pageWidth * this._zoomLevel;
    this._pageHeight = pageHeight * this._zoomLevel;
    this._pageMargin = pageMargin * this._zoomLevel;
    this._pageTopMargin = pageTopMargin * this._zoomLevel;
    this._pageSeparator = pageSeparator * this._zoomLevel;

    this._tabsX = this._xOffset + this._pageMargin + this._pageWidth * 2 + this._pageSeparator + (16 * this._zoomLevel);
  }

  prepare(initialLayout = 'main') {
    this._currentLayout = initialLayout || 'main';
  }

  create() {
    super.create(...arguments);
    this.preloadImages();
    this.createBackgroundContent();
    this.changeDiaryPage(this.currentTab());

    this.createOverlayContent();
  }

  preloadImages() {
    this.loadImagesForPage('main');
    this.loadImagesForPage('data');
    this.loadImagesForPage('friends');
    this.loadImagesForPage('animals');
    this.loadImagesForPage('calendar');
    this.loadImagesForPage('settings');
  }

  releaseImages() {
    const releasePictureFn = Managers.Images.releasePicture.bind(Managers.Images);
    const releaseBackgroundFn = Managers.Images.releaseBackground.bind(Managers.Images);

    releaseBackgroundFn('items_layout3');
    releaseBackgroundFn('items_layout2');
    releaseBackgroundFn('items_layout');
    releaseBackgroundFn('tools_layout');
    releaseBackgroundFn('friends_layout');
    releaseBackgroundFn('calendar_layout');
    releaseBackgroundFn('data_layout');
    releaseBackgroundFn('options_layout');
    releaseBackgroundFn('main_layout');

    releasePictureFn('menu/stamps');
    releasePictureFn('menu/stamp_font');
    releasePictureFn('menu/save_option_selected');
    releasePictureFn('menu/save_option_normal');
    releasePictureFn('menu/slot_1');
    releasePictureFn('menu/slot_2');
    releasePictureFn('menu/slot_3');
    releasePictureFn('menu/choices/choice_type_1');
    releasePictureFn('menu/choices/ok');
    releasePictureFn('menu/choices/nop');
    releasePictureFn('menu/stamp_font');
    releasePictureFn('menu/task_picture');
    releasePictureFn('menu/task_single_icon');
    releasePictureFn('menu/task_double_icon');
    releasePictureFn('menu/task_triple_icon');
    releasePictureFn('menu/task_description');
    releasePictureFn('menu/day_today');
    releasePictureFn('menu/day_holiday');
    releasePictureFn('menu/option_1');
    releasePictureFn('menu/option_1_selected');
    releasePictureFn('menu/option_2');
    releasePictureFn('menu/new_option');
    releasePictureFn('menu/arrows');
  }

  createBackground() {
    const bitmap = Managers.Scenes.backgroundBitmap();
    if (bitmap == null) {
      this._backgroundSprite = new Sprite(this.loadingBgBitmap());
      this._backgroundSprite.scale.x = this._backgroundSprite.scale.y = this.screenRatio;
      this._backgroundSprite.x = this.backgroundX();
    } else {
      this._backgroundSprite = new Sprite(bitmap);
    }

    this.addChild(this._backgroundSprite);
    this.setBackgroundOpacity(this.backgroundOpacity());
    this.createLayoutBackground();
  }

  getLayoutImage(layoutName) {
    layoutName = layoutName || this.currentTab();

    switch(layoutName) {
      case 'items':
        if (Switches.gotSecondBag) {
          return 'items_layout3';
        } else if (Switches.gotFirstBag) {
          return 'items_layout2';
        } else {
          return 'items_layout';
        }
      case 'tools':
        return 'tools_layout';
      case 'friends':
        return 'friends_layout';
      case 'animals':
        return 'friends_layout';
      case 'calendar':
        return 'calendar_layout';
      case 'data':
        return 'data_layout';
      case 'settings':
        return 'options_layout';
      default:
        return 'main_layout';
    }
  }

  getCurrentLayoutImage() {
    const layoutName = this.getLayoutImage();
    const bitmap = Managers.Images.loadBackground(layoutName);
    return bitmap;
  }

  createLayoutBackground() {
    this._layoutSprite = new Sprite(this.getCurrentLayoutImage());
    this._layoutSprite.scale = this._scale;
    this._layoutSprite.x = this._xOffset;
    this._layoutSprite.y = this._yOffset;
    this.addChild(this._layoutSprite);
  }

  updateDiarySprite() {
    this._layoutSprite.bitmap = this.getCurrentLayoutImage();
  }

  hideAllWindows() {
    this._windowLayer.children.forEach((diaryWindow) => {
      if (diaryWindow) {
        // Skip the alert and warning windows.
        if (diaryWindow === this._alertWindow || diaryWindow === this._warningWindow) {
          return;
        }

        diaryWindow.hide();
        diaryWindow.deactivate && diaryWindow.deactivate();
      }
    });
  }

  deactivateAllWindows() {
    this._windowLayer.children.forEach((diaryWindow) => {
      if (diaryWindow) {
        diaryWindow.deactivate && diaryWindow.deactivate();
      }
    });
  }

  loadLayoutWindows(layout) {
    switch (layout) {
      case 'items':
        this.loadItemWindows();
        break;
      case 'tools':
        this.loadToolWindows();
        break;
      case 'friends':
        this.loadFriendsWindows();
        break;
      case 'animals':
        this.loadAnimalWindows();
        break;
      case 'calendar':
        this.loadCalendarWindows();
        break;
      case 'data':
        this.loadDataWindows();
        break;
      case 'settings':
        this.loadSettingsWindows();
        break;
      default:
        this.loadMainWindows();
        break;
    }
  }

  showCurrentLayoutWindows(suppressRefreshes = false) {
    if (!this._bgContentWindow) {
      this.createBackgroundContent();
    } else {
      this._bgContentWindow.show();
    }

    if (!this._gameEndWindow) {
      this.createGameEndWindow();
    } else {
      this._gameEndWindow.hide();
    }

    switch (this._currentLayout) {
      case 'items':
        this.showItemWindows(suppressRefreshes);
        break;
      case 'tools':
        this.showToolWindows(suppressRefreshes);
        break;
      case 'friends':
        this.showFriendsWindows(suppressRefreshes);
        break;
      case 'animals':
        this.showAnimalWindows(suppressRefreshes);
        break;
      case 'calendar':
        this.showCalendarWindows(suppressRefreshes);
        break;
      case 'data':
        this.showDataWindows(suppressRefreshes);
        break;
      case 'settings':
        this.showSettingsWindows(suppressRefreshes);
        break;
      default:
        this.showMainWindows(suppressRefreshes);
        break;
    }
  }

  loadMainWindows(suppressRefreshes = false) {
    if (!this._mainContentWindow) {
      this.createMainContentWindow();
    }

    if (!suppressRefreshes) {
      this._mainContentWindow.refresh();
    }

    if (!this._bestFriendsWindow) {
      this.createBestFriendsWindow();
    }

    if (!suppressRefreshes) {
      this._bestFriendsWindow.refresh();
    }

    if (!this._bestAnimalsWindow) {
      this.createBestAnimalsWindow();
    }

    if (!suppressRefreshes) {
      this._bestAnimalsWindow.refresh();
    }

    if (!this._stampsWindow) {
      this.createStampsWindow();
    }

    if (!suppressRefreshes) {
      this._stampsWindow.refresh();
    }
  }

  getMainWindows() {
    return [
      this._mainContentWindow,
      this._bestFriendsWindow,
      this._bestAnimalsWindow,
      this._stampsWindow,
      this._stampsContentWindow
    ];
  }

  showMainWindows(suppressRefreshes = false) {
    this.loadMainWindows(suppressRefreshes);

    this._mainContentWindow.show();
    this._bestFriendsWindow.show();
    this._bestAnimalsWindow.show();

    this._stampsWindow.show();
    this._stampsContentWindow.show();

    this._stampsWindow.select(0);
    this._stampsWindow.activate();
  }

  getItemWindows() {
    return [
      this._itemsContentWindow,
      this._itemsPage2ContentWindow,
      this._backpackWindow,
      this._itemHelpWindow,
      this._itemInfoWindow
    ];
  }

  loadItemWindows(suppressRefreshes = false) {
    if (!this._itemsContentWindow) {
      this.createItemsContentWindow();
    }

    if (!this._backpackWindow) {
      this.createBackpackWindow();

      this._backpackWindow.setPage2Window(this._itemsPage2ContentWindow);
    }

    if (!this._itemHelpWindow) {
      this.createItemHelpWindow();

      this._backpackWindow.setHelpWindow(this._itemHelpWindow);
    }

    if (!this._itemInfoWindow) {
      this.createItemInfoWindow();

      this._backpackWindow.setInfoWindow(this._itemInfoWindow);
    }

    if (!suppressRefreshes) {
      this._backpackWindow.refresh();
      this._itemsContentWindow.refresh();
      this._itemsPage2ContentWindow.refresh();

      this._backpackWindow.select(0);
    }
  }

  showItemWindows(suppressRefreshes = false) {
    this.loadItemWindows(suppressRefreshes);

    this._itemsContentWindow.show();
    this._itemsPage2ContentWindow.show();
    this._backpackWindow.show();
    this._itemHelpWindow.show();
    this._itemInfoWindow.show();

    this._backpackWindow.activate();
  }

  loadToolWindows(suppressRefreshes = false) {
    if (!this._toolsContentWindow) {
      this.createToolsContentWindow();
    }

    if (!this._toolListWindow) {
      this.createToolListWindow();
    }

    this._toolsContentWindow.setToolListWindow(this._toolListWindow);
    this._toolListWindow.setContentWindow(this._toolsContentWindow);

    if (!suppressRefreshes) {
      this._toolListWindow.refresh();
      this._toolListWindow.select(0);
      this._toolsContentWindow.refresh();
    }

    if (!this._toolHelpWindow) {
      this.createToolHelpWindow();
      this._toolListWindow.setHelpWindow(this._toolHelpWindow);
    }
  }

  getToolWindows() {
    return [
      this._toolsContentWindow,
      this._toolListWindow,
      this._toolHelpWindow
    ];
  }

  showToolWindows(suppressRefreshes = false) {
    this.loadToolWindows(suppressRefreshes);

    this._toolsContentWindow.show();

    this._toolListWindow.show();
    this._toolHelpWindow.show();

    this._toolListWindow.activate();
  }

  loadCalendarWindows(suppressRefreshes = false) {
    if (!this._calendarContentWindow) {
      this.createCalendarContentWindow();
    }

    if (!suppressRefreshes) {
      this._tasksContentWindow.refresh();
      this._calendarContentWindow.refresh();
    }
  }

  getCalendarWindows() {
    return [
      this._calendarContentWindow,
      this._tasksContentWindow
    ];
  }

  showCalendarWindows(suppressRefreshes = false) {
    this.loadCalendarWindows(suppressRefreshes);

    this._calendarContentWindow.show();
    this._tasksContentWindow.show();
  }

  loadDataWindows(suppressRefreshes = false) {
    if (!this._dataContentWindow) {
      this.createDataContentWindow();
    }

    if (!suppressRefreshes) {
      this._dataContentWindow.resetCursor();
      this._dataContentWindow.refresh();
    }
  }

  getDataWindows() {
    return [
      this._dataContentWindow,
      this._dataDetailedContentWindow
    ];
  }

  showDataWindows(suppressRefreshes = false) {
    this.loadDataWindows(suppressRefreshes);

    this._dataContentWindow.show();
    this._dataDetailedContentWindow.show();
  }

  getFriendsWindows() {
    return [
      this._friendsContentWindow,
      this._friendListWindow,
      this._friendDetailsWindow,
    ];
  }

  loadFriendsWindows(suppressRefreshes = false) {
    if (!this._friendsContentWindow) {
      this.createFriendsContentWindow();
    }

    if (!this._friendListWindow) {
      this.createFriendsWindow();
    }

    if (!this._friendDetailsWindow) {
      this.createFriendDetailsWindow();
    }

    this._friendListWindow.setDetailsWindow(this._friendDetailsWindow);

    if (!suppressRefreshes) {
      this._friendsContentWindow.refresh();
    }
  }

  showFriendsWindows(suppressRefreshes = false) {
    this.loadFriendsWindows(suppressRefreshes);

    this._friendsContentWindow.show();

    this._friendListWindow.show();
    this._friendListWindow.activate();

    this._friendDetailsWindow.show();

    if (this._friendListWindow.index() < 0) {
      this._friendListWindow.select(0);
    }
  }

  getAnimalWindows() {
    return [
      this._animalContentWindow,
      this._animalListWindow,
      this._animalDetailsWindow,
    ];
  }

  loadAnimalWindows(suppressRefreshes = false) {
    if (!this._animalContentWindow) {
      this.createAnimalsContentWindow();
    }

    if (!this._animalListWindow) {
      this.createAnimalsWindow();
    }

    if (!this._animalDetailsWindow) {
      this.createAnimalDetailsWindow();
    }

    this._animalListWindow.setDetailsWindow(this._animalDetailsWindow);

    if (!suppressRefreshes) {
      this._animalContentWindow.refresh();
    }
  }

  showAnimalWindows(suppressRefreshes = false) {
    this.loadAnimalWindows(suppressRefreshes);

    this._animalContentWindow.show();

    this._animalListWindow.show();
    this._animalListWindow.activate();

    this._animalDetailsWindow.show();

    if (this._animalListWindow.index() < 0) {
      this._animalListWindow.select(0);
    }
  }

  loadSettingsWindows(suppressRefreshes = false) {
    if (!this._settingsContentWindow) {
      this.createSettingsContentWindow();
    }

    if (!this._settingsCategoryWindow) {
      this.createSettingsCategoryWindow();

      this._settingsCategoryWindow.setHandler('ok', this.onSettingsCategoryOk.bind(this));
    }

    if (!this._settingsWindow) {
      this.createSettingsWindow();

      this._settingsWindow.setCategoryWindow(this._settingsCategoryWindow);
      this._settingsCategoryWindow.setListWindow(this._settingsWindow);
      this._settingsWindow.setHandler('cancel', this.onSettingsCancel.bind(this));
    }

    if (!suppressRefreshes) {
      this._settingsContentWindow.refresh();
      this._settingsCategoryWindow.select(0);
      this._settingsCategoryWindow.refresh();
    }
  }

  getSettingsWindow() {
    return [
      this._settingsContentWindow,
      this._settingsCategoryWindow,
      this._settingsWindow
    ];
  }

  showSettingsWindows(suppressRefreshes = false) {
    this.loadSettingsWindows(suppressRefreshes);

    this._settingsContentWindow.show();
    this._settingsCategoryWindow.show();

    this._settingsWindow.show();
    this._settingsCategoryWindow.activate();

    this._settingsWindow.deactivate();
    this._settingsWindow._allowTouchWhenInactive = true;
    this._settingsCategoryWindow._allowTouchWhenInactive = true;
  }

  createMapDisplayWindow() {
    this._mapDisplayWindow = new Window_MapDisplay(0, 0, Graphics.width, Graphics.height);
    this.addWindow(this._mapDisplayWindow);

    this._mapDisplayWindow.refresh();
  }

  createBackgroundContent() {
    const y = this._yOffset - (headerExtraHeight * this._zoomLevel) * 2;

    this._bgContentWindow = new Window_BackgroundContent(this._xOffset, y, Graphics.width - this._xOffset, Graphics.height - y);
    this.addWindow(this._bgContentWindow);

    this._bgContentWindow.refresh();
  }

  createGameEndWindow() {
    this._gameEndWindow = new Window_GameEnd();
    this._gameEndWindow.setHandler('toTitle', this.commandToTitle.bind(this));
    this._gameEndWindow.setHandler('shutdown', this.commandExit.bind(this));
    this._gameEndWindow.setHandler('cancel', this.commandCancelQuit.bind(this));
    this.addWindow(this._gameEndWindow);

    this._gameEndWindow.hide();
    this._gameEndWindow.deactivate();
  }

  createOverlayContent() {
    this._overlayContentWindow = new Window_OverlayContent(0, 0, Graphics.width, Graphics.height);
    this.addWindow(this._overlayContentWindow);
  }

  createMainContentWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 36 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._mainContentWindow = new Window_MainContent(x, y, this._pageWidth, height);
    this._mainContentWindow.hide();
    this.addWindow(this._mainContentWindow);
  }

  createBestFriendsWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 113 * this._zoomLevel + this._yOffset;

    const height = 73 * this._zoomLevel;

    this._bestFriendsWindow = new Window_BestFriends(x, y, this._pageWidth, height);
    this._bestFriendsWindow.hide();
    this.addWindow(this._bestFriendsWindow);
  }

  createBestAnimalsWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 197 * this._zoomLevel + this._yOffset;

    const height = 53 * this._zoomLevel;

    this._bestAnimalsWindow = new Window_BestAnimals(x, y, this._pageWidth, height);
    this._bestAnimalsWindow.hide();
    this.addWindow(this._bestAnimalsWindow);
  }

  createStampsWindow() {
    const x = this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator;
    const y = 36 * this._zoomLevel + this._yOffset;
    const h = 220 * this._zoomLevel;

    const contentY = y + h;
    const contentHeight = 33 * this._zoomLevel;

    this._stampsWindow = new Window_Stamps(x, y, this._pageWidth, h);
    this._stampsWindow.hide();
    this.addWindow(this._stampsWindow);

    this._stampsContentWindow = new Window_StampContent(x, contentY, this._pageWidth, contentHeight);
    this._stampsContentWindow.hide();
    this.addWindow(this._stampsContentWindow);

    this._stampsWindow.setInfoWindow(this._stampsContentWindow);
    this._stampsContentWindow.setStampListWindow(this._stampsWindow);

    this._stampsWindow.setHandler('ok', this.onStampOk.bind(this));
    this._stampsWindow.setHandler('cancel', this.onStampCancel.bind(this));

  }

  createItemsContentWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 36 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._itemsContentWindow = new Window_ItemsContent(x, y, this._pageWidth, height);
    this._itemsContentWindow.hide();
    this.addWindow(this._itemsContentWindow);

    const page2X = x + this._pageWidth + this._pageSeparator;
    this._itemsPage2ContentWindow = new Window_ItemsPage2Content(page2X, y, this._pageWidth, height);
    this._itemsPage2ContentWindow.hide();
    this.addWindow(this._itemsPage2ContentWindow);
  }

  createBackpackWindow() {
    const x = this._pageMargin + this._xOffset + (40 * this._zoomLevel);
    const y = 100 * this._zoomLevel + this._yOffset;

    const height = 165 * this._zoomLevel;
    const width = 137 * this._zoomLevel;

    this._backpackWindow = new Window_Backpack(x, y, width, height);
    this._backpackWindow.hide();
    this.addWindow(this._backpackWindow);

    this._backpackWindow.setHandler('ok', this.onBackpackOk.bind(this));
    this._backpackWindow.setHandler('cancel', this.onBackpackCancel.bind(this));
  }

  createItemHelpWindow() {
    const x = this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator + (16 * this._zoomLevel);
    const y = 207 * this._zoomLevel + this._yOffset;
    const w = 186 * this._zoomLevel;
    const h = 61 * this._zoomLevel;

    this._itemHelpWindow = new Window_DiaryItemHelp(x, y, w, h);
    this._itemHelpWindow.hide();
    this.addWindow(this._itemHelpWindow);
  }

  createItemInfoWindow() {
    const x = this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator + (19 * this._zoomLevel);
    const y = 74 * this._zoomLevel + this._yOffset;
    const w = 181 * this._zoomLevel;
    const h = 102 * this._zoomLevel;

    this._itemInfoWindow = new Window_DiaryItemInfo(x, y, w, h);
    this._itemInfoWindow.hide();
    this.addWindow(this._itemInfoWindow);
  }

  createBackpackCommandWindow() {
    const x = this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator + (39 * this._zoomLevel);
    const y = (74 + 142) * this._zoomLevel + this._yOffset;
    const w = 142 * this._zoomLevel;

    this._backpackCommandWindow = new Window_DiaryItemCommand(x, y, w);
    this._backpackCommandWindow.hide();
    this.addWindow(this._backpackCommandWindow);

    this._backpackCommandWindow.setHandler('eat', this.onBackpackCommandEat.bind(this));
    // this._backpackCommandWindow.setHandler('sell', this.onBackpackCommandSell.bind(this));
    this._backpackCommandWindow.setHandler('drop', this.onBackpackCommandDrop.bind(this));
    this._backpackCommandWindow.setHandler('move', this.onBackpackCommandMove.bind(this));
    this._backpackCommandWindow.setHandler('cancel', this.onBackpackCommandCancel.bind(this));
  }

  createToolsContentWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 36 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._toolsContentWindow = new Window_ToolsContent(x, y, this._pageWidth, height);
    this._toolsContentWindow.hide();
    this.addWindow(this._toolsContentWindow);
  }

  createToolListWindow() {
    const x = this._pageMargin + this._xOffset + (40 * this._zoomLevel);
    const y = 104 * this._zoomLevel + this._yOffset;

    const height = 57 * this._zoomLevel;
    const width = 136 * this._zoomLevel;

    this._toolListWindow = new Window_ToolList(x, y, width, height);
    this._toolListWindow.hide();
    this.addWindow(this._toolListWindow);
  }

  createToolHelpWindow() {
    const x = this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator + (16 * this._zoomLevel);
    const y = 178 * this._zoomLevel + this._yOffset;
    const w = 186 * this._zoomLevel;
    const h = 61 * this._zoomLevel;

    this._toolHelpWindow = new Window_DiaryToolHelp(x, y, w, h);
    this._toolHelpWindow.hide();
    this.addWindow(this._toolHelpWindow);
  }

  createCalendarContentWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 36 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._tasksContentWindow = new Window_TasksContent(x, y, this._pageWidth, height);
    this._tasksContentWindow.hide();
    this.addWindow(this._tasksContentWindow);

    const calendarX = x + this._pageWidth + this._pageSeparator;
    this._calendarContentWindow = new Window_CalendarContent(calendarX, y, this._pageWidth, height);
    this._calendarContentWindow.hide();
    this.addWindow(this._calendarContentWindow);
  }

  createDataContentWindow() {
    const x = this._xOffset + 32 * this._zoomLevel;
    const y = 36 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._dataContentWindow = new Window_DataContent(x, y, this._pageWidth + 43 * this._zoomLevel, height);
    this._dataContentWindow.hide();
    this.addWindow(this._dataContentWindow);

    this._dataContentWindow.setEmptyListAction('saveMode');

    const page2X = x + (17 * this._zoomLevel) + this._pageWidth + this._pageSeparator;
    this._dataDetailedContentWindow = new Window_DataDetailedContent(page2X, y, this._pageWidth, height);
    this._dataDetailedContentWindow.hide();
    this.addWindow(this._dataDetailedContentWindow);
  }

  createFriendsWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 80 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight - 44 * this._zoomLevel;
    this._friendListWindow = new Window_DiaryFriends(x, y, this._pageWidth, height);
    this._friendListWindow.hide();
    this.addWindow(this._friendListWindow);
  }

  createFriendsContentWindow() {
    const x = this._xOffset;
    const y = this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._friendsContentWindow = new Window_FriendsContent(x, y, this._pageWidth + this._pageMargin * 2, height);
    this._friendsContentWindow.hide();
    this.addWindow(this._friendsContentWindow);
  }

  createFriendDetailsWindow() {
    const x = this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator;
    const y = this._yOffset + 36 * this._zoomLevel;
    const w = this._pageWidth;
    const h = this._pageHeight;

    this._friendDetailsWindow = new Window_FriendDetails(x, y, w, h);
    this._friendDetailsWindow.hide();
    this.addWindow(this._friendDetailsWindow);
  }

  createAnimalsWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 80 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight - 44 * this._zoomLevel;
    this._animalListWindow = new Window_AnimalList(x, y, this._pageWidth, height);
    this._animalListWindow.hide();
    this.addWindow(this._animalListWindow);
  }

  createAnimalsContentWindow() {
    const x = this._xOffset;
    const y = this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._animalContentWindow = new Window_AnimalsContent(x, y, this._pageWidth + this._pageMargin * 2, height);
    this._animalContentWindow.hide();
    this.addWindow(this._animalContentWindow);
  }

  createAnimalDetailsWindow() {
    const x = this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator;
    const y = this._yOffset + 36 * this._zoomLevel;
    const w = this._pageWidth;
    const h = this._pageHeight;

    this._animalDetailsWindow = new Window_AnimalDetails(x, y, w, h);
    this._animalDetailsWindow.hide();
    this.addWindow(this._animalDetailsWindow);
  }

  createSettingsContentWindow() {
    const x = this._pageMargin + this._xOffset;
    const y = 36 * this._zoomLevel + this._yOffset;

    const height = this._pageHeight + 14 * this._zoomLevel;

    this._settingsContentWindow = new Window_SettingsContent(x, y, this._pageWidth, height);
    this._settingsContentWindow.hide();
    this.addWindow(this._settingsContentWindow);
  }

  createSettingsCategoryWindow() {
    const x = this._pageMargin + this._xOffset + (17 * this._zoomLevel);
    const y = 66 * this._zoomLevel + this._pageTopMargin + this._yOffset;

    const height = 165 * this._zoomLevel;
    const width = 186 * this._zoomLevel;

    this._settingsCategoryWindow = new Window_SettingsCategoryList(x, y, width, height);
    this._settingsCategoryWindow.hide();
    this.addWindow(this._settingsCategoryWindow);
  }

  createSettingsWindow() {
    const x = this._pageMargin + this._xOffset + this._pageWidth + this._pageSeparator + 13 * this._zoomLevel;
    const y = 12 * this._zoomLevel + this._pageTopMargin + this._yOffset;

    const height = this._pageHeight - 24 * this._zoomLevel;
    const width = this._pageWidth;

    this._settingsWindow = new Window_SettingsList(x, y, width, height);
    this._settingsWindow.hide();
    this.addWindow(this._settingsWindow);
  }

  updateVisibleWindows(suppressRefreshes = false) {
    this.hideAllWindows();
    this.showCurrentLayoutWindows(suppressRefreshes);
  }

  loadImagesForPage(pageName) {
    const bitmaps = [];
    const loadPictureFn = Managers.Images.loadPicture.bind(Managers.Images);

    bitmaps.push(Managers.Images.loadBackground(this.getLayoutImage(pageName)));

    switch (pageName) {
      case 'main':
        bitmaps.push(loadPictureFn('menu/stamps'));
        bitmaps.push(loadPictureFn('menu/stamp_font'));
        break;
      case 'items':
        break;
      case 'tools':
        bitmaps.push(loadPictureFn('menu/selected_slot'));
        break;
      case 'friends':
        bitmaps.push(loadPictureFn('menu/friend_layout_dark'));
        bitmaps.push(loadPictureFn('menu/half_red_heart_dark'));
        bitmaps.push(loadPictureFn('menu/red_heart_dark'));
        bitmaps.push(loadPictureFn('menu/arrow_up'));
        bitmaps.push(loadPictureFn('menu/arrow_down'));

        // Load the faces of every friend
        if (!this._loadedFriendsFaces) {
          this._loadedFriendsFaces = true;
          const friends = Managers.Relationship.getUnsortedKnownVillagers(true);
          for (let friend of friends) {
            bitmaps.push(loadPictureFn('menu/villagers/' + friend.toLowerCase()));
          }
        }

        break;
      case 'animals':
        bitmaps.push(loadPictureFn('menu/friend_layout_dark'));
        bitmaps.push(loadPictureFn('menu/half_red_heart_dark'));
        bitmaps.push(loadPictureFn('menu/red_heart_dark'));
        bitmaps.push(loadPictureFn('menu/arrow_up'));
        bitmaps.push(loadPictureFn('menu/arrow_down'));

        if (!this._loadedAnimalFaces) {
          this._loadedAnimalFaces = true;
          const animals = Managers.Creatures.getAnimalList((data) => !!data._creatureName);
          for (let animal of animals) {
            bitmaps.push(loadPictureFn('menu/animals/' + animal.type));
          }
        }

        break;
      case 'data':
        bitmaps.push(loadPictureFn('menu/save_option_selected'));
        bitmaps.push(loadPictureFn('menu/save_option_normal'));
        bitmaps.push(loadPictureFn('menu/slot_1'));
        bitmaps.push(loadPictureFn('menu/slot_2'));
        bitmaps.push(loadPictureFn('menu/slot_3'));
        bitmaps.push(loadPictureFn('menu/choices/choice_type_1'));
        bitmaps.push(loadPictureFn('menu/choices/ok'));
        bitmaps.push(loadPictureFn('menu/choices/nop'));
        bitmaps.push(loadPictureFn('menu/stamp_font'));
        break;
      case 'calendar':
        bitmaps.push(loadPictureFn('menu/task_picture'));
        bitmaps.push(loadPictureFn('menu/task_single_icon'));
        bitmaps.push(loadPictureFn('menu/task_double_icon'));
        bitmaps.push(loadPictureFn('menu/task_triple_icon'));
        bitmaps.push(loadPictureFn('menu/task_description'));
        bitmaps.push(loadPictureFn('menu/day_today'));
        bitmaps.push(loadPictureFn('menu/day_holiday'));
        break;
      case 'settings':
        bitmaps.push(loadPictureFn('menu/option_1'));
        bitmaps.push(loadPictureFn('menu/option_1_selected'));
        bitmaps.push(loadPictureFn('menu/option_2'));
        bitmaps.push(loadPictureFn('menu/new_option'));
        bitmaps.push(loadPictureFn('menu/arrows'));
        break;
    }

    if (bitmaps.length > 0) {
      for (let i in bitmaps) {
        if (!bitmaps[i].isReady()) {
          return false;
        }
      }
    }

    return true;
  }

  tryPrepareNewPage(timestamp, newPage = 'main') {
    if (this.preloadPage(newPage)) {
      this.changeDiaryPage(newPage, true);
    } else {
      this._changingPage = true;

      // Do not lock the diary for too long
      if ((new Date() - timestamp) > 1000) {
        this.changeDiaryPage(newPage);
        return;
      }

      setTimeout(() => {
        this.tryPrepareNewPage(timestamp, newPage);
      }, 100);
    }
  }

  prepareNewPage(newPage = 'main') {
    const timestamp = new Date();
    this.tryPrepareNewPage(timestamp, newPage);
  }

  changeDiaryPage(newPage = 'main', suppressRefreshes = false) {
    this._currentLayout = newPage;
    this._changingPage = false;

    this.updateDiarySprite();
    this.updateVisibleWindows(suppressRefreshes);

    this.moveOverlayToTop();
  }

  preloadPage(newPage = 'main') {
    if (!this.loadImagesForPage(newPage)){
      return false;
    }

    this.loadLayoutWindows(newPage);
    return !this.anyRefreshExpected(newPage);
  }

  pageList() {
    return [
      'main',
      'items',
      'tools',
      'friends',
      'animals',
      'calendar',
      'data',
      'settings'
    ];
  }

  tabList() {
    return [
      'main',
      'items',
      'friends',
      'calendar',
      'data',
      'settings'
    ];
  }

  currentTab() {
    return this._currentLayout || 'main';
  }

  nextPageName() {
    const pages = this.pageList();
    let idx = pages.indexOf(this.currentTab()) + 1;
    if (idx >= pages.length) {
      idx = 0;
    }

    return pages[idx];
  }

  previousPageName() {
    const pages = this.pageList();
    let idx = pages.indexOf(this.currentTab()) - 1;
    if (idx < 0) {
      idx = pages.length - 1;
    }

    return pages[idx];
  }

  goToNextPage() {
    this.prepareNewPage(this.nextPageName());
  }

  goToPreviousPage() {
    this.prepareNewPage(this.previousPageName());
  }

  goToTabIndex(index) {
    const tabs = this.tabList();
    const newPageName = tabs[index % tabs.length];

    this.prepareNewPage(newPageName);
  }

  moveOverlayToTop() {
    if (!this._overlayContentWindow) return;

    this.addWindow(this._overlayContentWindow);
  }

  updateTouchCheckTabIndexes(x, y) {
    if (x >= this._tabsX) {
      const pageY = (y - this._yOffset) / this._zoomLevel;

      const tabPositions = [34, 78, 122, 166, 210, 254];
      let tabIndex = -1;
      for (let i = tabPositions.length -1; i >= 0; i--) {
        if (pageY > tabPositions[i]) {
          if (pageY < tabPositions[i] + 35) {
            tabIndex = i;
          }
          break;
        }
      }

      if (tabIndex >= 0) {
        const tabNames = this.tabList();
        if (tabIndex >= tabNames.length) return;

        this.prepareNewPage(tabNames[tabIndex]);
      }
    }
  }

  commandClose() {
    this.popScene();
  }

  commandEndGame() {
    this.addWindow(this._gameEndWindow);
    this._gameEndWindow.show();
    this._gameEndWindow.activate();
  }

  updateTouch() {
    if (!TouchInput.isTriggered()) {
      return;
    }
    if (Managers.Scenes.isSceneChanging()) {
      return;
    }

    if (this._bgContentWindow) {
      if (this._bgContentWindow.isMouseOverQuitButton()) {
        this.commandEndGame();
        return;
      }
      if (this._bgContentWindow.isMouseOverMapButton()) {
        this.commandShowMap();
        return;
      }
    }

    if (this.checkCloseButtonClick()) {
      this.commandClose();
      return;
    }

    const { x, y } = TouchInput;

    if (x < this._xOffset || y < this._yOffset || x > (Graphics.width - this._xOffset) || (y > (Graphics.height - this._yOffset))) {
      return;
    }

    this.updateTouchCheckTabIndexes(x, y);

    if (x < this._pageMargin + this._xOffset) return;
    if (y < this._pageTopMargin + this._yOffset) return;

    const pageIndex = x >= (this._xOffset + this._pageMargin + this._pageWidth + this._pageSeparator) ? 2 : 1;

    let pageX = x - this._xOffset - this._pageMargin;
    if (pageIndex > 1) {
      pageX = pageX - this._pageWidth - this._pageSeparator;
    }
    const pageY = y - this._yOffset - this._pageTopMargin;

    this.updateTouchCheckPages(pageIndex, pageX, pageY);
  }

  updateTouchCheckPages(pageIndex, pageX, pageY) {
    switch(this._currentLayout) {
      case 'settings':
        if (pageIndex === 2 && pageX >= 13 * this._zoomLevel && pageY >= 12 * this._zoomLevel) {
          this._settingsWindow.checkPageTouch(pageX - 13 * this._zoomLevel, pageY - 12 * this._zoomLevel);
        }
        break;
      case 'friends':
        if (this._friendsContentWindow.isMouseOverAnimals()) {
          this.changeDiaryPage('animals');
          return;
        }

        this._friendListWindow.updateTouch();
        break;
      case 'animals':
        if (this._animalContentWindow.isMouseOverFriends()) {
          this.changeDiaryPage('friends');
          return;
        }
        this._animalListWindow.updateTouch();
        break;
      case 'items':
        if (this._itemsContentWindow.isMouseOverTools()) {
          this.changeDiaryPage('tools');
        }
        break;
      case 'tools':
        if (this._toolsContentWindow.isMouseOverItems()) {
          this.changeDiaryPage('items');
        }
        break;
    }
  }

  activateBackpackWindow() {
    if (this._backpackCommandWindow) {
      this._backpackCommandWindow.hide();
      this._backpackCommandWindow.deactivate();
    }

    this.showItemWindows();
  }

  activateBackpackCommandWindow() {
    const item = this._backpackWindow.item();
    if (!item) {
      this.activateBackpackWindow();
      return;
    }

    if (!this._backpackCommandWindow) {
      this.createBackpackCommandWindow();
    }

    if (this._backpackCommandWindow.makeVisible(item)) {
      this._backpackCommandWindow.select(0);
      this._backpackCommandWindow.activate();
    } else {
      this.activateBackpackWindow();
    }
  }

  onBackpackCommandCancel() {
    this.activateBackpackWindow();
  }

  onBackpackCommandMove() {
    this._action = 'move';
    this._backpackWindow._movingItemIndex = this._backpackWindow.index();
    this.activateBackpackWindow();
  }

  onBackpackCommandDrop() {
    const item = this._backpackWindow.item();
    if (item) {
      if (Managers.Items.loseItem(item, item.amount || 1)) {
        this.activateBackpackCommandWindow();
      }
    }

    this.activateBackpackWindow();
  }

  onBackpackCommandEat() {
    Managers.Items.eat(this._backpackWindow.item().id);

    this.activateBackpackWindow();
  }

  onBackpackCommandSell() {
    Managers.Items.container.moveItemId(this._backpackWindow.item(), Managers.ShippingBin.container, 1);
    this.activateBackpackWindow();
  }

  onBackpackOk() {
    if (this._action == 'move') {
      this._backpackWindow._container.swapIndexes(this._backpackWindow._movingItemIndex, this._backpackWindow.index());
      this.cancelItemMove();
      return;
    }

    Managers.Items.setLastItem(this._backpackWindow.item());

    this._backpackWindow.deactivate();
    this.activateBackpackCommandWindow();
  }

  onStampOk() {
    const stamp = this._stampsWindow.selectedStamp();
    if (!stamp) {
      this._stampsWindow.activate();
      return;
    }

    this.showStampSubWindow(stamp.id);
  }

  showStampSubWindow(stampId) {
    switch(stampId) {
      case 'progress':
        this._stampsWindow.deactivate();
        this.showProgressList();
        break;
      // case 'moneyEarned':
      //   break;
      // case 'moneySpent':
      //   break;
      case 'peopleMet':
        this._stampsWindow.deactivate();
        this.showFriendList();
        break;
      case 'animals':
        this._stampsWindow.deactivate();
        this.showAnimalList();
        break;
      case 'areasFound':
        this._stampsWindow.deactivate();
        this.showAreaList();
        break;
      // case 'recipes':
      //   break;
      case 'tasksCompleted':
        this._stampsWindow.deactivate();
        this.showTaskList();
        break;
      case 'shippedItems':
        this._stampsWindow.deactivate();
        this.showShippedItems();
        break;
      default:
        this.showWarning(t("This option was not implemented yet."), this._stampsWindow);
        break;
    }
  }

  enableSubWindow(subWindow) {
    this._subWindow = subWindow;
    subWindow.x = (Graphics.width / 2) - (subWindow.width / 2);
    subWindow.y = (Graphics.height / 2) - (subWindow.height / 2);
    subWindow.show();
    subWindow.activate();
  }

  showFriendList() {
    this.changeDiaryPage('friends');
  }

  showAnimalList() {
    this.changeDiaryPage('animals');
  }

  showAreaList() {
    if (this._areaListWindow) {
      // Move it to the top
      this.addWindow(this._areaListWindow);
    } else {
      this._areaListWindow = new Window_AreaList();
      this.addWindow(this._areaListWindow);
    }

    this.enableSubWindow(this._areaListWindow);
  }

  showProgressList() {
    if (this._progressListWindow) {
      // Move it to the top
      this.addWindow(this._progressListWindow);
    } else {
      this._progressListWindow = new Window_ProgressList();
      this.addWindow(this._progressListWindow);
    }

    this.enableSubWindow(this._progressListWindow);
  }

  showShippedItems() {
    if (this._shippedItemsListWindow) {
      // Move it to the top
      this.addWindow(this._shippedItemsListWindow);
    } else {
      this._shippedItemsListWindow = new Window_ShippedItemsList();
      this.addWindow(this._shippedItemsListWindow);
    }

    this.enableSubWindow(this._shippedItemsListWindow);
  }

  showTaskList() {
    if (this._taskListWindow) {
      // Move it to the top
      this.addWindow(this._taskListWindow);
    } else {
      this._taskListWindow = new Window_TaskList();
      this.addWindow(this._taskListWindow);
    }

    this.enableSubWindow(this._taskListWindow);
  }

  onStampCancel() {
    this.cancelCurrentAction();
  }

  onSettingsCategoryOk() {
    const symbol = this._settingsCategoryWindow.currentSymbol();
    if (symbol == 'save') {
      Managers.Config.save();
      this._settingsCategoryWindow.select(0);
      this.showAlert(t("Saved Settings"));
    } else if (symbol == 'reload') {
      Managers.Config.load();
      this._settingsCategoryWindow.select(0);
      this.showAlert(t("Loaded Settings"));
    } else if (symbol == 'endGame') {
      this.commandEndGame();
    } else {
      this._settingsCategoryWindow.deactivate();
      this._settingsWindow.activate();
    }
  }

  onSettingsCancel() {
    this._settingsWindow.deactivate();
    this._settingsCategoryWindow.activate();
  }

  onBackpackCancel() {
    this.cancelCurrentAction();
  }

  updateInput() {
    if (Engine.Input.isPressed('map')) {
      this.commandShowMap();
      return;
    }

    if (Engine.Input.isTriggered('right_backbutton') || Engine.Input.isTriggered('pagedown')) {
      this.goToNextPage();
      return;
    }

    if (Engine.Input.isTriggered('left_backbutton') || Engine.Input.isTriggered('pageup')) {
      this.goToPreviousPage();
      return;
    }

    if (this.checkForPageShortcuts()) {
      return;
    }
  }

  checkForPageShortcuts() {
    for (let idx = 0; idx <= 5; idx++) {
      if (Engine.Input.isTriggered((idx+1).toString())) {
        this.goToTabIndex(idx);
        return true;
      }
    }

    return false;
  }

  updateMap() {
    if (this.isCancelTriggered()) {
      this._mapDisplayWindow.hide();
      return;
    }

    if (TouchInput.isTriggered() && this._mapDisplayWindow.isMouseOverExitButton()) {
      this._mapDisplayWindow.hide();
      return;
    }

    this._mapDisplayWindow.update();
  }

  update() {
    if (this._changingPage) return;
    if (this._gameEndWindow.visible) {
      this._gameEndWindow.update();
      return;
    }

    if (this._mapDisplayWindow && this._mapDisplayWindow.visible) {
      this.updateMap();
      return;
    }

    super.update();

    if (Managers.Scenes.isSceneChanging()) {
      return;
    }

    if (this._alertWindow && this._alertWindow.visible) {
      if (this._delay > 0) {
        this._delay--;
      } else {
        this._alertWindow.hide();
      }

      return;
    }

    if (this._warningWindow && this._warningWindow.visible) {
      if (this._delay > 0) {
        this._delay--;
      } else if (TouchInput.isTriggered() || Engine.Input.isTriggered('ok')) {
        this.hideWarning();
      }

      return;
    }

    if (this._subWindow && this._subWindow.visible) {
      this.updateSubWindow();
      return;
    }

    if (this._dataContentWindow && this._dataContentWindow.visible && this._dataContentWindow.isLocked()) {
      return;
    }

    this.updateInput();
    if (this._changingPage) return;

    if (this.isCancelTriggered()) {
      this.cancelCurrentAction();
      return;
    }

    this.updateTouch();
    this.updateMovedItem();
    this.updateDataContent();
    this.updateOverlay();
  }

  updateSubWindow() {
    if (this.isCancelTriggered()) {
      this.cancelCurrentAction();
      return;
    }

    if (TouchInput.isTriggered()) {
      if (this._subWindow) {
        if (this._subWindow.isMouseOverCloseButton && this._subWindow.isMouseOverCloseButton()) {
          this.cancelCurrentAction();
          return;
        }

        if (this._subWindow.updateTouch && this._subWindow.updateTouch()) {
          return;
        }
      }
    }
  }

  cancelItemMove() {
    this._action = '';
    this._backpackWindow._movingItemIndex = -1;
    this.activateBackpackWindow();
  }

  cancelCurrentAction() {
    if (this._subWindow && this._subWindow.visible) {
      if (this._subWindow.parentWindow) {
        this._subWindow.parentWindow.show();
        this._subWindow.parentWindow.activate();
        this._subWindow.hide();
        this.enableSubWindow(this._subWindow.parentWindow);
        return;
      }

      this._subWindow.hide();
      this._subWindow = null;

      this.showCurrentLayoutWindows(true);
      return;
    }

    if (this._mapDisplayWindow && this._mapDisplayWindow.visible) {
      this._mapDisplayWindow.hide();
      return;
    }

    if (this.currentTab() == 'items') {
      if (this._action == 'move') {
        this.cancelItemMove();
        $gameTemp.skipSelectableWindowHandling();
        return;
      }
    }

    this.popScene();
  }

  updateMovedItem() {
    if (this.currentTab() != 'items') return;
    if (!this._backpackWindow) return;
    if (!this._backpackWindow._itemSwapAction) return;
    if (!this._itemsContentWindow.isMouseOverTrash()) return;

    const item = this._backpackWindow._itemSwapAction.itemData;

    if (!Managers.Items.canItemBeTrashed(item)) {
      this._backpackWindow._itemSwapAction = false;
      return;
    }

    if (Managers.Items.loseItem(item, item.amount || 1)) {
      this._backpackWindow._itemSwapAction = false;
      this._backpackWindow.refresh();
    }
  }

  updateDataContent() {
    if (this.currentTab() != 'data') return;
    if (!this._dataContentWindow) return;

    this._dataContentWindow.refreshIfNeeded();
    if (this._dataDetailedContentWindow) {
      this._dataDetailedContentWindow.setSaveFilePath(this._dataContentWindow.selectedSaveFilePath());
    }
  }

  updateOverlay() {
    const overlay = this._overlayContentWindow;

    if (overlay) {
      if (this._backpackWindow && this._backpackWindow._itemBeingMoved >= 0) {
        if (!overlay.visible) {
          overlay.show();
        }

        if (overlay._iconUnderCursor != this._backpackWindow._itemBeingMoved || overlay._mouseX != TouchInput.mouseX || overlay._mouseY != TouchInput.mouseY) {
          overlay._iconUnderCursor = this._backpackWindow._itemBeingMoved;
          overlay.refresh();
        }

      } else {
        if (overlay._iconUnderCursor >= 0) {
          overlay._iconUnderCursor = -1;
          overlay.refresh();
        }
      }

      if (this._itemsContentWindow) {
        this._itemsContentWindow._allowTrashSprite = overlay._iconUnderCursor >= 0;
        this._itemsContentWindow.refreshIfNeeded();
      }
    }
  }

  isCancelTriggered() {
    if ($gameTemp.shouldSkipSelectableWindowHandling()) return false;
    if (Engine.Input.isRepeated('cancel')) return true;
    if (Engine.Input.isRepeated('inventory')) return true;

    if (Engine.Input.isRepeated('alternate')) {
      return true;
    }

    return false;
  }

  terminate() {
    const loadSuccess = this._dataContentWindow && this._dataContentWindow._loadSuccess;
    super.terminate();

    if (loadSuccess) {
      $gameSystem.onAfterLoad();
    }

    this.releaseImages();
  }

  showAlert(text, delay = 60) {
    if (this._alertWindow) {
      // Move it to the top
      this.addWindow(this._alertWindow);
    } else {
      this._alertWindow = new Window_Warning();
      this.addWindow(this._alertWindow);
    }

    this._alertWindow.setText(text);
    this._alertWindow.visible = true;
    this._delay = Utils.getFrameCount(delay || 60);
  }

  hideWarning() {
    this._warningWindow.hide();
    if (this._lastActiveWindow && this._lastActiveWindow.activate) {
      this._lastActiveWindow.activate();
    }
    this._lastActiveWindow = null;
    $gameTemp.skipSelectableWindowHandling(true);
  }

  showWarning(warningText, lastActiveWindow) {
    this._lastActiveWindow = lastActiveWindow;
    if (lastActiveWindow && lastActiveWindow.deactivate) {
      lastActiveWindow.deactivate();
    }

    if (this._warningWindow) {
      // Move it to the top
      this.addWindow(this._warningWindow);
    } else {
      this._warningWindow = new Window_Warning();
      this.addWindow(this._warningWindow);
    }

    if (lastActiveWindow) {
      this._warningWindow.x = lastActiveWindow.x + (lastActiveWindow.width / 2) - (this._warningWindow.width / 2);
    } else {
      this._warningWindow.x = (Graphics.width / 2) - (this._warningWindow.width / 2);
    }

    this._warningWindow.setText(warningText);
    this._warningWindow.visible = true;
    this._delay = Utils.getFrameCount(10);
  }

  commandExit() {
    var gui = require('nw.gui');
    var win = gui.Window.get();
    win.close();
  }

  commandToTitle() {
    this.fadeOutAll();
    this._gameEndWindow.hide();

    setTimeout(() => {
      Managers.Scenes.goToScene(GameScenes.Title);
    }, 100);
  }

  commandShowMap() {
    if (!this._mapDisplayWindow) {
      this.createMapDisplayWindow();
    } else {
      this._mapDisplayWindow.show();
    }

    // Move to the top
    this.addWindow(this._mapDisplayWindow);
    this._mapDisplayWindow.refresh();
  }

  commandCancelQuit() {
    this._gameEndWindow.hide();
    this._gameEndWindow.deactivate();

    this.showCurrentLayoutWindows();
  }

  checkCloseButtonClick() {
    if (!TouchInput.isTriggered()) return false;

    const mouseX = TouchInput.mouseX;
    const mouseY = TouchInput.mouseY;

    const closeButtonX = this._bgContentWindow.x + (523 * Graphics.windowZoomLevel);
    if (mouseX < closeButtonX) return false;

    const closeButtonY = this._bgContentWindow.y + (51 * Graphics.windowZoomLevel);
    if (mouseY < closeButtonY) return false;

    const closeButtonW = 16 * Graphics.windowZoomLevel;
    if (mouseX > closeButtonX + closeButtonW) return false;

    const closeButtonH = 16 * Graphics.windowZoomLevel;
    if (mouseY > closeButtonY + closeButtonH) return false;

    return true;
  }

  getLayoutWindows(layoutName = 'main') {
    switch (layoutName) {
      case 'items':
        return this.getItemWindows();
      case 'tools':
        return this.getToolWindows();
      case 'friends':
        return this.getFriendsWindows();
      case 'animals':
        return this.getAnimalWindows();
      case 'calendar':
        return this.getCalendarWindows();
      case 'data':
        return this.getDataWindows();
      case 'settings':
        return this.getSettingsWindow();
      default:
        return this.getMainWindows();
    }
  }

  anyRefreshExpected(pageName) {
    const windows = this.getLayoutWindows(pageName);

    for (let diaryWindow of windows) {
      if (!diaryWindow) {
        continue;
      }

      if (diaryWindow._refreshExpected) {
        return true;
      }

      if (diaryWindow._needsRefresh) {
        return true;
      }

      if (diaryWindow.needsRefresh && diaryWindow.needsRefresh()) {
        return true;
      }
    }

    return false;
  }
}

GameScenes.Diary = Scene_Diary;
module.exports = Scene_Diary;