const Scene_MenuBase = require('engine/scenes/Scene_MenuBase');
const Window_CookingBackground = require('../../windows/crafts/CookingBackground');
const Window_Cooking = require('../../windows/crafts/Cooking');
const Window_BackpackCookingItems = require('../../windows/crafts/BackpackCookingItems');

class Scene_Cooking extends Scene_MenuBase {
  initialize() {
    super.initialize(...arguments);
    this._zoomLevel = Graphics.windowZoomLevel;
    this._inputDelay = 0;
  }

  create() {
    super.create();

    const realWidth = 640 * this._zoomLevel;
    const realHeight = 233 * this._zoomLevel;
    const x = (Graphics.width - realWidth) / 2;
    const y = (Graphics.height - realHeight) / 2;

    this._xOffset = x;
    this._yOffset = y;

    this.createBackgroundContent();
    this.createCookingWindow();
    this.createBackpackWindow();
    this.createCloseButton();

    if (this._selectedRecipe) {
      this._cookingWindow.setRecipe(this._selectedRecipe);
    }
  }

  prepare(selectedRecipe) {
    this._selectedRecipe = selectedRecipe;
  }

  createCookingWindow() {
    const x = 141 * this._zoomLevel;
    const y = 159 * this._zoomLevel;
    const w = 356 * this._zoomLevel;
    const h = 59 * this._zoomLevel;

    this._cookingWindow = new Window_Cooking(this._xOffset + x, this._yOffset + y, w, h);
    this.addWindow(this._cookingWindow);
  }

  createBackpackWindow() {
    const x = 128 * this._zoomLevel;
    const y = 15 * this._zoomLevel;
    const w = 382 * this._zoomLevel;
    const h = 141 * this._zoomLevel;

    this._backpackWindow = new Window_BackpackCookingItems(this._xOffset + x, this._yOffset + y, w, h);
    this.addWindow(this._backpackWindow);

    this._backpackWindow.refresh();

    this._backpackWindow._checkInput = false;
    this._backpackWindow._selectCallback = this.onSelectItem.bind(this);
  }

  createBackgroundContent() {
    const x = 0;
    const y = 0;
    const w = Graphics.width;
    const h = Graphics.height;

    this._bgContentWindow = new Window_CookingBackground(x, y, w, h, this._xOffset, this._yOffset);
    this.addWindow(this._bgContentWindow);

    this._bgContentWindow.refresh();
  }

  createCloseButton() {
    const button = new Sprite_Button();
    button.bitmap = Managers.Images.loadSystem('CloseButton');

    const w = 48;
    const h = 48;

    button.setColdFrame(0, 0, w, h);
    button.setHotFrame(0, h, w, h);
    
    this.addChild(button);

    button.setClickHandler(this.popScene.bind(this));

    button.x = this._xOffset + 490 * Graphics.windowZoomLevel;
    button.y = this._yOffset + 16 * Graphics.windowZoomLevel;
  }

  isCancelTriggered() {
    if (Engine.Input.isRepeated('cancel')) return true;
    if (Engine.Input.isRepeated('alternate')) {
      return true;
    }

    return false;
  }

  update() {
    super.update();

    if (this.isCancelTriggered()) {
      this.popScene();
      return;
    }

    this.updateInput();
  }

  onSelectItem(index) {
    if (index >= 0) {
      this._cookingWindow._index = -1;
      this._cookingWindow.refresh();
    }
  }

  onConfirmItem() {
    const item = this._backpackWindow.item();
    if (!item) {
      return;
    }

    if (!item.canCook) {
      Managers.Sound.playBuzzer();
      return;
    }

    this._cookingWindow.addIngredient(item.id);
    this._inputDelay = Utils.getFrameCount(8);
  }

  onConfirmCookingWindow() {
    if (this._cookingWindow._index < 0) {
      return;
    }

    if (this._cookingWindow._index < 6) {
      this._cookingWindow.confirmUtensil();
      this._inputDelay = Utils.getFrameCount(8);
      return;
    }

    if (this._cookingWindow._index < 12) {
      this._cookingWindow.removeSelectedIngredient();
      this._inputDelay = Utils.getFrameCount(8);
      return;
    }

    this.confirmCooking();
  }

  confirmCooking() {
    if (!Managers.Cooking.isValidUtensil(this._cookingWindow._utensil)) {
      Managers.Sound.playBuzzer();
      return;
    }

    if (!this._cookingWindow._ingredients.length) {
      Managers.Sound.playBuzzer();
      return;
    }

    if (!Managers.Cooking.playerHasAllIngredients(this._cookingWindow._ingredients)) {
      Managers.Sound.playBuzzer();
      return;
    }

    Managers.Cooking.cookWithItems(this._cookingWindow._utensil, this._cookingWindow._ingredients);
    Managers.Scenes.popTo(GameScenes.GameMap);
  }

  updateInput() {
    if (this._inputDelay > 0) {
      this._inputDelay--;
      return;
    }

    let oldBackpackIndex = this._backpackWindow.index();
    let oldCookingIndex = this._cookingWindow._index;
    let newBackpackIndex = oldBackpackIndex;
    let newCookingIndex = oldCookingIndex;

    if (oldCookingIndex >= 0) {
      if (Engine.Input.isPressed('left')) {
        if (oldCookingIndex > 0) {
          newCookingIndex = oldCookingIndex - 1;
        } else {
          newCookingIndex = -1;
          // Last item slot
          newBackpackIndex = this._backpackWindow.maxItems() - 1;
        }
      } else if (Engine.Input.isPressed('right')) {
        if (oldCookingIndex < 12) {
          if (oldCookingIndex === 2) {
            newCookingIndex = 6;
          } else {
            newCookingIndex = oldCookingIndex + 1;
          }
        } else {
          newCookingIndex = -1;
          newBackpackIndex = 0;
        }
      } else if (Engine.Input.isPressed('up')) {
        if (oldCookingIndex >= 3 && oldCookingIndex <= 5) {
          newCookingIndex = oldCookingIndex - 3;
        } else {
          newCookingIndex = -1;
          // First slot of the last row
          newBackpackIndex = (this._backpackWindow.rows() - 1) * this._backpackWindow.maxCols();
        }
      } else if (Engine.Input.isPressed('down')) {
        if (oldCookingIndex <= 2) {
          newCookingIndex = oldCookingIndex + 3;
        } else {
          newCookingIndex = -1;
          newBackpackIndex = 0;
        }
      } else if (Engine.Input.isPressed('ok')) {
        this.onConfirmCookingWindow();
        return;
      } else if (TouchInput.isTriggered() && this._cookingWindow.isTouchedInsideItem()) {
        this.onConfirmCookingWindow();
        return;
      } else if (Engine.Input.isPressed('pageup') || Engine.Input.isPressed('pagedown') || Engine.Input.isPressed('right_backbutton') || Engine.Input.isPressed('left_backbutton')) {
        newCookingIndex = -1;
        newBackpackIndex = 0;
      }
    } else {
      if (Engine.Input.isPressed('left')) {
        if (this._backpackWindow.index() > 0) {
          this._backpackWindow.cursorLeft(Engine.Input.isTriggered('left'));
          newBackpackIndex = this._backpackWindow.index();
        } else {
          newCookingIndex = 12;
          newBackpackIndex = -1;
        }
      } else if (Engine.Input.isPressed('right')) {
        if (this._backpackWindow.index() < (this._backpackWindow.maxItems() -1)) {
          this._backpackWindow.cursorRight(Engine.Input.isTriggered('right'));
          newBackpackIndex = this._backpackWindow.index();
        } else {
          newCookingIndex = 0;
          newBackpackIndex = -1;
        }
      } else if (Engine.Input.isPressed('up')) {
        if (this._backpackWindow.index() < this._backpackWindow.maxCols()) {
          newCookingIndex = 0;
          newBackpackIndex = -1;
        } else {
          this._backpackWindow.cursorUp(Engine.Input.isTriggered('up'));
          newBackpackIndex = this._backpackWindow.index();
        }
      } else if (Engine.Input.isPressed('down')) {
        if (this._backpackWindow.index() >= (this._backpackWindow.rows() -1) * this._backpackWindow.maxCols()) {
          newCookingIndex = 0;
          newBackpackIndex = -1;
        } else {
          this._backpackWindow.cursorDown(Engine.Input.isTriggered('down'));
          newBackpackIndex = this._backpackWindow.index();
        }
      } else if (Engine.Input.isPressed('ok')) {
        this.onConfirmItem();
        return;
      } else if (TouchInput.isTriggered() && this._backpackWindow.isTouchedInsideItem()) {
        this.onConfirmItem();
        return;
      } else if (Engine.Input.isPressed('pageup') || Engine.Input.isPressed('pagedown') || Engine.Input.isPressed('right_backbutton') || Engine.Input.isPressed('left_backbutton')) {
        newCookingIndex = 0;
        newBackpackIndex = -1;
      }
    }

    if (oldCookingIndex !== newCookingIndex) {
      this._cookingWindow._index = newCookingIndex;
    }

    if (oldBackpackIndex !== newBackpackIndex) {
      this._backpackWindow.select(newBackpackIndex);
      if (newBackpackIndex >= 0) {
        this._backpackWindow.activate();
      } else {
        this._backpackWindow.deactivate();
      }
    }

    if (oldBackpackIndex != newBackpackIndex || oldCookingIndex != newCookingIndex) {
      this._inputDelay = Utils.getFrameCount(8);
      this._cookingWindow.refresh();
      Managers.Sound.playCursor();
    }
  }  
}

GameScenes.Cooking = Scene_Cooking;
module.exports = Scene_Cooking;