const Scene_MoveItem = require('./Scene_MoveItem');
const Window_StoreNumber = require('../windows/containers/StoreNumberWindow');
const Window_ModernItemContainer = require('../windows/containers/ModernItemContainer');
const Window_ContainerBackground = require('../windows/containers/ContainerBackground');
const Window_ContainerSummary = require('../windows/containers/ContainerSummary');

//-----------------------------------------------------------------------------
// Scene_TrashCan
//
// The scene class of the trash can screen.
class Scene_TrashCan extends Scene_MoveItem {
  create() {
    this._zoomLevel = Graphics.windowZoomLevel;
    this._contentWidth = 351 * this._zoomLevel;
    this._contentHeight = 306 * this._zoomLevel;
    this._offsetX = (Graphics.width - this._contentWidth) / 2;
    this._offsetY = (Graphics.height - this._contentHeight) / 2;

    this._container = Managers.Items.getTrashContainer(this._containerName);

    super.create();
    this.createSummaryWindow();
    this.createNumberWindow();
    this.updateTotal();

    this._rightWindow.select(0);
    this._rightWindow.activate();
  }

  prepare(containerName) {
    this._containerName = containerName;
  }

  createBackgroundImageWindow() {
    this._bgWindow = new Window_ContainerBackground(0, 0, Graphics.width, Graphics.height, this._offsetX, this._offsetY, 'trash_menu_layout');
    this.addWindow(this._bgWindow);

    this._bgWindow.refresh();
  }

  createLeftWindow() {
    const x = this._offsetX + (33 * this._zoomLevel);
    const y = this._offsetY + (30 * this._zoomLevel);
    const width = 138 * this._zoomLevel;
    const height = 197 * this._zoomLevel;

    const maxItems = 30;

    this._leftWindow = new Window_ModernItemContainer(x, y, width, height, this._container, maxItems);
    this.addWindow(this._leftWindow);
    this._leftWindow._checkInput = false;
    this._leftWindow._allowTouchWhenInactive = true;
    this._leftWindow.margin = 0;
    this._leftWindow.refresh();

    this._leftWindow.setHandler('updateHelp', this.updateTotal.bind(this));
    this._leftWindow.setHandler('beforeSelectByTouch', this.touchOnLeftItem.bind(this));
  }

  createRightWindow() {
    const x = this._offsetX + (182 * this._zoomLevel);
    const y = this._leftWindow.y;
    const width = this._leftWindow.width;
    const height = this._leftWindow.height;
    const maxItems = Managers.Items.maxItemTypes();

    this._rightWindow = new Window_ModernItemContainer(x, y, width, height, Managers.Items.container, maxItems);
    this.addWindow(this._rightWindow);
    this._rightWindow._checkInput = false;
    this._rightWindow._allowTouchWhenInactive = true;
    this._rightWindow.margin = 0;
    this._rightWindow.refresh();

    this._rightWindow.setHandler('updateHelp', this.updateTotal.bind(this));
    this._rightWindow.setHandler('beforeSelectByTouch', this.touchOnRightItem.bind(this));
  }

  touchOnLeftItem() {
    this._rightWindow.select(-1);
  }

  touchOnRightItem() {
    this._leftWindow.select(-1);
  }
  
  createSummaryWindow() {
    const width = 343 * this._zoomLevel;
    const height = 68 * this._zoomLevel;
    const y = this._offsetY + (238 * this._zoomLevel);
    const x = (Graphics.width - width) / 2;

    this._summaryWindow = new Window_ContainerSummary(x, y, width, height);
    this.addWindow(this._summaryWindow);

    this._summaryWindow.refresh();
  }

  moveToRightWindow(all, item) {
    if (!item) {
      return;
    }

    if (!all) {
      this.chooseAmountToSell(item, 'right');
      return;
    }

    this._container.moveItem(item, Managers.Items.container, all ? item.amount : 1);
    this._inputDelay = Utils.getFrameCount(10);
    this._needsRefresh = true;
  }

  moveToLeftWindow(all, item, usedKeyboard) {
    if (!item) {
      return;
    }

    if (!Managers.Items.canItemIdBeTrashed(item.id)) {
      Managers.Sound.playBuzzer();
      return;
    }

    if (!all) {
      this.chooseAmountToSell(item, 'left');
      return;
    }
    
    Managers.Items.container.moveItem(item, this._container, all ? item.amount : 1);
    this._inputDelay = Utils.getFrameCount(10);
    this._needsRefresh = true;
  }

  updateTotal() {
    const leftItem = this._leftWindow.item();
    const rightItem = this._rightWindow.item();

    let selectedItem = (rightItem && rightItem.id) ? rightItem : leftItem;
    if (selectedItem) {
      const itemName = Managers.Text.item(selectedItem.id);
      let text = itemName;

      if (itemName && selectedItem.amount > 1) {
        text = `${ itemName } x ${ selectedItem.amount }`;
      }

      if (text) {
        this._summaryWindow.setText(text);
        return;
      }
    }

    this._summaryWindow.setText('');
  }

  checkCancel() {
    if (this._inputDelay > 0) {
      return false;
    }

    if (Engine.Input.isTriggered('escape') || Engine.Input.isTriggered('alternate') || Engine.Input.isTriggered('start')) {
      if (this._numberWindow.visible) {
        this.onNumberCancel();
        return true;
      }
    }

    return super.checkCancel();
  }

  clampIndex(index, theWindow) {
    if (theWindow === undefined) {
      theWindow = this._rightWindow;
    }
    
    const maxItems = theWindow.maxItems();
    return index.clamp(0, maxItems - 1);
  }

  update() {
    super.update();

    if (TouchInput.isTriggered() && this._bgWindow.isMouseOverCloseButton()) {
      this.onClickClose();
    }
  }

  createNumberWindow() {
    this._numberWindow = new Window_StoreNumber();
    this._numberWindow.hide();
    this._numberWindow.setHandler('ok', this.onNumberOk.bind(this));
    this._numberWindow.setHandler('cancel', this.onNumberCancel.bind(this));
    this.addWindow(this._numberWindow);
  }

  chooseAmountToSell(item, direction) {
    this._direction = direction;

    this._numberWindow.setup(item, item.amount, 0);
    this._numberWindow.setCurrencyUnit('');
    this._numberWindow.show();
    this._numberWindow.activate();

    this._rightWindow.deactivate();
    this._rightWindow._allowTouchWhenInactive = false;
    this._leftWindow.deactivate();
    this._leftWindow._allowTouchWhenInactive = false;
  }

  canMoveCursor() {
    return !this._numberWindow.active;
  }

  onNumberOk() {
    const amount = this._numberWindow.number();
    const item = this._numberWindow.item();

    if (this._direction == 'right') {
      this._container.moveItem(item, Managers.Items.container, amount);
    } else {
      Managers.Items.container.moveItem(item, this._container, amount);
    }

    this._inputDelay = Utils.getFrameCount(10);
    this._needsRefresh = true;
    this.onNumberCancel();
    this.moveHorz(0);
    this.updateTotal();
    this._rightWindow.activate();
    this._leftWindow.activate();
    this._leftWindow._allowTouchWhenInactive = true;
    this._rightWindow._allowTouchWhenInactive = true;
  }

  onNumberCancel() {
    this._inputDelay = Utils.getFrameCount(10);

    this._numberWindow.hide();
    this._numberWindow.deactivate();

    this._rightWindow._allowTouchWhenInactive = false;
    this._leftWindow._allowTouchWhenInactive = false;
    this._leftWindow.activate();
    this._rightWindow.activate();
  }

  updateHover() {
    if (this._numberWindow.visible) {
      if (this._hoverBitmap) {
        this._hoverBitmap.clear();
      }
      return;
    }

    super.updateHover();
  }
}

GameScenes.TrashCan = Scene_TrashCan;
module.exports = Scene_TrashCan;
