const Scene_MoveItem = require('./Scene_MoveItem');
const Window_ShipNumber = require('../windows/containers/ShipNumberWindow');
const Window_ModernItemContainer = require('../windows/containers/ModernItemContainer');
const Window_ContainerBackground = require('../windows/containers/ContainerBackground');
const Window_ContainerSummary = require('../windows/containers/ContainerSummary');

//-----------------------------------------------------------------------------
// Scene_ShippingBin
//
// The scene class of the item export screen.
class Scene_ShippingBin extends Scene_MoveItem {
  create() {
    this._zoomLevel = Graphics.windowZoomLevel;
    this._contentWidth = 351 * this._zoomLevel;
    this._contentHeight = 333 * this._zoomLevel;
    this._offsetX = (Graphics.width - this._contentWidth) / 2;
    this._offsetY = (Graphics.height - this._contentHeight) / 2;

    super.create();
    this.createSummaryWindow();
    this.createNumberWindow();
    this.updateTotal();

    this._rightWindow.select(0);
    this._rightWindow.activate();
  }

  createBackgroundImageWindow() {
    this._bgWindow = new Window_ContainerBackground(0, 0, Graphics.width, Graphics.height, this._offsetX, this._offsetY, 'storage_menu_layout');
    this.addWindow(this._bgWindow);

    this._bgWindow.refresh();
  }

  createLeftWindow() {
    const x = this._offsetX + (33 * this._zoomLevel);
    const y = this._offsetY + (30 * this._zoomLevel);
    const width = 138 * this._zoomLevel;
    const height = 224 * this._zoomLevel;

    const maxItems = Math.ceil($dataItems.length / 30) * 30;

    this._leftWindow = new Window_ModernItemContainer(x, y, width, height, Managers.ShippingBin.container, maxItems);
    this.addWindow(this._leftWindow);
    this._leftWindow._checkInput = false;
    this._leftWindow._allowTouchWhenInactive = true;
    this._leftWindow.margin = 0;
    this._leftWindow.refresh();

    this._leftWindow.setHandler('updateHelp', this.updateTotal.bind(this));
    this._leftWindow.setHandler('beforeSelectByTouch', this.touchOnLeftItem.bind(this));
  }

  createRightWindow() {
    const x = this._offsetX + (182 * this._zoomLevel);
    const y = this._leftWindow.y;
    const width = this._leftWindow.width;
    const height = this._leftWindow.height;
    const maxItems = Managers.Items.maxItemTypes();

    this._rightWindow = new Window_ModernItemContainer(x, y, width, height, Managers.Items.container, maxItems);
    this.addWindow(this._rightWindow);
    this._rightWindow._checkInput = false;
    this._rightWindow._allowTouchWhenInactive = true;
    this._rightWindow.margin = 0;
    this._rightWindow.refresh();

    this._rightWindow.setHandler('updateHelp', this.updateTotal.bind(this));
    this._rightWindow.setHandler('beforeSelectByTouch', this.touchOnRightItem.bind(this));
  }

  touchOnLeftItem() {
    this._rightWindow.select(-1);
  }

  touchOnRightItem() {
    this._leftWindow.select(-1);
  }

  createSummaryWindow() {
    const width = 343 * this._zoomLevel;
    const height = 68 * this._zoomLevel;
    const y = this._offsetY + (265 * this._zoomLevel);
    const x = (Graphics.width - width) / 2;

    this._summaryWindow = new Window_ContainerSummary(x, y, width, height);
    this.addWindow(this._summaryWindow);

    this._summaryWindow.refresh();
  }

  moveToLeftWindow(all, item, usedKeyboard) {
    if (!item) {
      return;
    }
    if (this._numberWindow.visible) {
      return;
    }

    const sellPrice = Managers.Items.getSellPrice(item.id);
    if (sellPrice <= 0) {
      Managers.Sound.playBuzzer();
      return;
    }

    if (!all) {
      this.chooseAmountToSell(item, 'left');
      return;
    }

    Managers.Items.container.moveItem(item, Managers.ShippingBin.container, all ? item.amount : 1);
    this._inputDelay = Utils.getFrameCount(10);
    this._needsRefresh = true;
    this.updateTotal();
  }

  moveToRightWindow(all, item, usedKeyboard) {
    if (!item) {
      return;
    }

    if (this._numberWindow.visible) {
      return;
    }

    if (!all) {
      this.chooseAmountToSell(item, 'right');
      return;
    }

    Managers.ShippingBin.container.moveItem(item, Managers.Items.container, all ? item.amount : 1);
    this._inputDelay = Utils.getFrameCount(10);
    this._needsRefresh = true;
    this.updateTotal();
  }

  updateTotal() {
    const total = Managers.ShippingBin.getItemTotal();
    const line1 = t("Shipping Bin Total: {total}").replace('{total}', Managers.Text.formatMoney(total));

    const leftItem = this._leftWindow.item();
    const rightItem = this._rightWindow.item();

    let selectedItem = (rightItem && rightItem.id) ? rightItem : leftItem;
    let text = line1;

    if (selectedItem) {
      const itemName = Managers.Text.item(selectedItem.id);
      const itemPrice = Managers.Items.getSellPrice(selectedItem.id);
      let line2;

      if (itemName && itemPrice) {
        line2 = `${ itemName }: ${ Managers.Text.formatMoney(itemPrice) }`;
      } else if (itemName) {
        line2 = itemName;
      }

      text = `${ text }<BR>${ line2 }`;
    }

    this._summaryWindow.setText(text);
  }

  checkCancel() {
    if (this._inputDelay > 0) {
      return false;
    }

    if (Engine.Input.isTriggered('escape') || Engine.Input.isTriggered('alternate') || Engine.Input.isTriggered('start')) {
      if (this._numberWindow.visible) {
        this.onNumberCancel();
        return true;
      }
    }

    return super.checkCancel();
  }

  clampIndex(index, theWindow) {
    if (theWindow === undefined) {
      theWindow = this._rightWindow;
    }
    const maxItems = theWindow.maxItems();
    return index.clamp(0, maxItems - 1);
  }

  update() {
    super.update();

    if (this._inputDelay > 0) {
      return false;
    }

    if (TouchInput.isTriggered() && this._bgWindow.isMouseOverCloseButton()) {
      this.onClickClose();
    }
  }

  createNumberWindow() {
    this._numberWindow = new Window_ShipNumber();
    this._numberWindow.hide();
    this._numberWindow.setHandler('ok', this.onNumberOk.bind(this));
    this._numberWindow.setHandler('cancel', this.onNumberCancel.bind(this));
    this.addWindow(this._numberWindow);
  }

  chooseAmountToSell(item, direction) {
    this._direction = direction;

    this._numberWindow.setup(item, item.amount, Managers.Items.getSellPrice(item.id));
    this._numberWindow.setCurrencyUnit(Managers.Text.currencyUnit);
    this._numberWindow.show();
    this._numberWindow.activate();

    this._rightWindow.deactivate();
    this._rightWindow._allowTouchWhenInactive = false;
    this._leftWindow.deactivate();
    this._leftWindow._allowTouchWhenInactive = false;
  }

  canMoveCursor() {
    return !this._numberWindow.active;
  }

  onNumberOk() {
    const amount = this._numberWindow.number();
    const item = this._numberWindow.item();

    if (this._direction == 'right') {
      Managers.ShippingBin.container.moveItem(item, Managers.Items.container, amount);
    } else {
      Managers.Items.container.moveItem(item, Managers.ShippingBin.container, amount);
    }

    this._inputDelay = Utils.getFrameCount(10);
    this._needsRefresh = true;
    this.onNumberCancel();
    this.moveHorz(0);
    this.updateTotal();
    this._rightWindow.activate();
    this._leftWindow.activate();
    this._leftWindow._allowTouchWhenInactive = true;
    this._rightWindow._allowTouchWhenInactive = true;
  }

  onNumberCancel() {
    this._inputDelay = Utils.getFrameCount(10);

    this._numberWindow.hide();
    this._numberWindow.deactivate();

    this._rightWindow._allowTouchWhenInactive = false;
    this._leftWindow._allowTouchWhenInactive = false;
    this._leftWindow.activate();
    this._rightWindow.activate();
  }

  updateHover() {
    if (this._numberWindow.visible) {
      if (this._hoverBitmap) {
        this._hoverBitmap.clear();
      }
      return;
    }

    super.updateHover();
  }
}

GameScenes.ShippingBin = Scene_ShippingBin;
module.exports = Scene_ShippingBin;
