const Scene_MenuBase = require('engine/scenes/Scene_MenuBase');
//-----------------------------------------------------------------------------
// Scene_MoveItem
//
// The scene class of the item screen.
class Scene_MoveItem extends Scene_MenuBase {
  create() {
    super.create();
    this._inputDelay = Utils.getFrameCount(8);
    this._oldMouseX = TouchInput.mouseX;
    this._oldMouseY = TouchInput.mouseY;
    this._leftWindow = null;
    this._rightWindow = null;

    this.createBackgroundImageWindow();
    this.createLeftCategoryWindow();
    this.createLeftWindow();
    this.createRightWindow();
    this.createHoverBitmap();

    if (this._leftWindow) {
      this._leftWindow.onClickItem = this.onClickLeftWindow.bind(this);
    }
    if (this._rightWindow) {
      this._rightWindow.onClickItem = this.onClickRightWindow.bind(this);
    }
  }

  createBackground() {
    const bitmap = Managers.Scenes.backgroundBitmap();
    if (bitmap == null) {
      this._backgroundSprite = new Sprite(this.loadingBgBitmap());
      this._backgroundSprite.scale.x = this._backgroundSprite.scale.y = this.screenRatio;
      this._backgroundSprite.x = this.backgroundX();
    } else {
      this._backgroundSprite = new Sprite(bitmap);
    }

    this.addChild(this._backgroundSprite);
    this.setBackgroundOpacity(this.backgroundOpacity());
  }

  createBackgroundImageWindow() {

  }

  createLeftWindow() {
  }

  createRightWindow() {
  }

  createLeftCategoryWindow() {
  }

  canMoveCursor() {
    return true;
  }

  changeWindow() {
    var col = 0;
    var row = 0;
    var oldWindow;
    var newWindow;

    if (this._leftWindow && this._leftWindow.index() >= 0) {
      col = this._leftWindow.col();
      row = this._leftWindow.row();
      oldWindow = this._leftWindow;
      newWindow = this._rightWindow;
    } else if (this._rightWindow && this._rightWindow.index() >= 0) {
      col = this._rightWindow.col();
      row = this._rightWindow.row();
      oldWindow = this._rightWindow;
      newWindow = this._leftWindow;
    } else {
      newWindow = this._rightWindow;
      oldWindow = this._leftWindow;
    }

    oldWindow.deactivate();
    oldWindow.select(-1);
    newWindow.activate();
    newWindow.selectAt(row, col);
  }

  moveHorz(diff) {
    var leftIndex = this._leftWindow.index();
    var rightIndex = this._rightWindow.index();
    var scene = this;

    if (!this.canMoveCursor()) {
      return false;
    }

    var moveHorz = function (currentWindow, otherWindow, diff /*, vertDirection*/) {
      var currentIndex = currentWindow.index();
      var currentColumns = currentWindow.maxCols();
      var currentRow = currentWindow.row();
      var currentCol = currentWindow.col();
      var maxItems = currentWindow.maxItems();
      var otherIndex = otherWindow.index();
      var otherColumns = otherWindow.maxCols();
      var otherMaxRows = Math.ceil(otherWindow.maxItems() / otherColumns);

      if (diff === -1) {
        if (currentCol === 0) {
          otherIndex = ((currentRow + 1) * otherColumns) - 1;
          if (currentRow >= otherMaxRows) {
            otherIndex = (otherMaxRows + 1) * otherColumns - 1;
          }
          otherIndex = scene.clampIndex(otherIndex, otherWindow);
          otherWindow.select(otherIndex);
          currentWindow.select(-1);
          return;
        }
      }
      else if (diff === 1) {
        if (currentCol === currentColumns - 1 || currentIndex >= maxItems - 1) {
          otherIndex = (currentRow) * otherColumns;
          if (currentRow >= otherMaxRows) {
            otherIndex = (otherMaxRows - 1) * otherColumns;
          }
          otherIndex = scene.clampIndex(otherIndex, otherWindow);
          otherWindow.select(otherIndex);
          currentWindow.select(-1);
          return;
        }
      }

      currentIndex += diff;
      scene.clampIndex(currentIndex, currentWindow);
      currentWindow.select(currentIndex);
      otherWindow.select(-1);
    };

    if (leftIndex >= 0) {
      moveHorz(this._leftWindow, this._rightWindow, diff, -1);
    } else if (rightIndex >= 0) {
      moveHorz(this._rightWindow, this._leftWindow, diff, 1);
    } else {
      this._rightWindow.select(0);
      this._leftWindow.select(-1);
    }

    this._leftWindow.callUpdateHelp();
    this._rightWindow.callUpdateHelp();
  }

  clampIndex(index, theWindow) {
    if (theWindow === undefined) {
      theWindow = this._rightWindow;
    }
    var maxItems = theWindow.maxItems();
    return index.clamp(0, maxItems - 1);
  }

  moveVert(diff) {
    var leftIndex = this._leftWindow.index();
    var rightIndex = this._rightWindow.index();
    var scene = this;
    if (!this.canMoveCursor()) {
      return false;
    }
    var moveVert = function (currentWindow, otherWindow, diff) {
      var currentIndex = currentWindow.index();
      var maxCols = currentWindow.maxCols();
      var maxItems = currentWindow.maxItems();
      var currentRow = Math.floor(currentIndex / maxCols);
      var currentCol = currentIndex % maxCols;
      var maxRows = Math.ceil(maxItems / maxCols);
      currentRow += diff;
      if (currentRow < 0) {
        currentRow = maxRows - 1;
      }
      if (currentRow >= maxRows) {
        currentRow = 0;
      }
      currentIndex = currentRow * maxCols + currentCol;
      if (currentIndex < 0) {
        currentIndex = 0;
      }
      if (currentIndex >= maxItems) {
        currentIndex = maxItems - 1;
      }
      currentIndex = scene.clampIndex(currentIndex, currentWindow);
      currentWindow.select(currentIndex);
      otherWindow.select(-1);
    };
    if (leftIndex >= 0 && !!this._leftWindow) {
      moveVert(this._leftWindow, this._rightWindow, diff);
    }
    else if (rightIndex >= 0 && !!this._rightWindow) {
      moveVert(this._rightWindow, this._leftWindow, diff);
    }
    else {
      this._rightWindow.select(0);
      this._leftWindow.select(-1);
    }
  }

  updateInput() {
    if (this._inputDelay > 0) {
      this._inputDelay--;
      return;
    }

    var oldRightIndex = -1;
    var oldLeftIndex = -1;
    var newRightIndex = -1;
    var newLeftIndex = -1;

    if (this._rightWindow) {
      oldRightIndex = this._rightWindow.index();
    }
    
    if (this._leftWindow) {
      oldLeftIndex = this._leftWindow.index();
    }

    if (Engine.Input.isPressed('left')) {
      this.moveHorz(-1);
    } else if (Engine.Input.isPressed('right')) {
      this.moveHorz(1);
    } else if (Engine.Input.isPressed('up')) {
      this.moveVert(-1);
    } else if (Engine.Input.isPressed('down')) {
      this.moveVert(1);
    } else if (Engine.Input.isPressed('ok')) {
      this.moveCurrentItem(true, true);
    } else if (Engine.Input.isPressed('extra')) {
      this.moveCurrentItem(false, true);
    } else if (Engine.Input.isPressed('pageup') || Engine.Input.isPressed('pagedown') || Engine.Input.isPressed('right_backbutton') || Engine.Input.isPressed('left_backbutton')) {
      this.changeWindow();
    }

    if (this._rightWindow) {
      newRightIndex = this._rightWindow.index();
    }

    if (this._leftWindow) {
      newLeftIndex = this._leftWindow.index();
    }

    if (oldRightIndex != newRightIndex || oldLeftIndex != newLeftIndex) {
      this._inputDelay = Utils.getFrameCount(8);
      this._needsRefresh = true;
      Managers.Sound.playCursor();
    }
  }

  moveCurrentItem(all, usedKeyboard) {
    var rightIndex = this._rightWindow.index();
    var leftIndex = this._leftWindow.index();

    if (rightIndex >= 0) {
      this.moveToLeftWindow(all, this._rightWindow.getItem(rightIndex), usedKeyboard);
    }
    else if (leftIndex >= 0) {
      this.moveToRightWindow(all, this._leftWindow.getItem(leftIndex), usedKeyboard);
    }
  }

  moveToLeftWindow(all, item, usedKeyboard) {

  }

  moveToRightWindow(all, item, usedKeyboard) {

  }

  mouseMoved() {
    return this._oldMouseX != TouchInput.mouseX || this._oldMouseY != TouchInput.mouseY;
  }

  checkCancel() {
    if (this._inputDelay > 0) {
      return false;
    }

    if (Engine.Input.isTriggered('escape') || Engine.Input.isPressed('alternate') || Engine.Input.isPressed('start')) {
      Managers.Sound.playCancel();
      Managers.Scenes.pop();
      return true;
    }

    return false;
  }

  getPositionLabel(x, y) {
    var label;
    if (this._leftWindow) {
      label = this._leftWindow.getPositionLabel(x, y);
    }

    if (label !== '') {
      return label;
    }

    if (this._rightWindow) {
      label = this._rightWindow.getPositionLabel(x, y);
    }

    if (label !== '') {
      return label;
    }
    return '';
  }

  update() {
    super.update(...arguments);

    if (this.checkCancel()) {
      return;
    }

    this.updateInput();
    this.processMapTouch();
    this.updateHover();
    if (this._needsRefresh) {
      this.refresh();
      this._needsRefresh = false;
    }
  }

  refresh() {
    if (this._rightWindow) {
      this._rightWindow.refresh();
    }
    if (this._leftWindow) {
      this._leftWindow.refresh();
    }
    this._needsRefresh = false;
  }

  processMapTouch() {
    if (!TouchInput.isTriggered() && !TouchInput.isRightTriggered()) {
      return;
    }

    this._rightWindow.processMapTouch(TouchInput.mouseX, TouchInput.mouseY);
    this._leftWindow.processMapTouch(TouchInput.mouseX, TouchInput.mouseY);
  }

  onClickRightWindow(index) {
    var item = this._rightWindow.getItem(index);
    if (!item) {
      return;
    }

    this.moveToLeftWindow(TouchInput.button === 2, item);
  }

  onClickLeftWindow(index) {
    var item = this._leftWindow.getItem(index);
    if (!item) {
      return;
    }

    this.moveToRightWindow(TouchInput.button === 2, item);
  }

  onClickClose() {
    Managers.Scenes.pop();
  }
}

module.exports = Scene_MoveItem;