//-----------------------------------------------------------------------------
// Game_Villager
//
// The game object class for villager events

Objects.Villager = class Villager extends Objects.Creature {
  update() {
    super.update();
    if (this.isMoving()) {
      this._stoppingDelay = Utils.getFrameCount(10);
    } else if (this._stoppingDelay > 0) {
      this._stoppingDelay--;
    }
  }

  // X position of the character hitbox (in pixels)
  get hitboxX() {
    if (this._hitboxX === undefined) {
      const actor = this.actor();

      if (actor !== undefined && actor.hitboxX !== undefined) {
        this._hitboxX = actor.hitboxX;
      } else {
        this._hitboxX = 0;
      }
    }

    return this._hitboxX;
  }

  set hitboxX(value) {
    this._hitboxX = value;
  }

  // Y position of the character hitbox (in pixels)
  get hitboxY() {
    if (this._hitboxY === undefined) {
      const actor = this.actor();

      if (actor !== undefined && actor.hitboxY !== undefined) {
        this._hitboxY = actor.hitboxY;
      } else {
        this._hitboxY = 0;
      }
    }

    return this._hitboxY;
  }

  set hitboxY(value) {
    this._hitboxY = value;
  }

  // Width of the character hitbox (in pixels)
  get hitboxWidth() {
    if (this._hitboxWidth === undefined) {
      const actor = this.actor();

      if (actor !== undefined && actor.hitboxWidth !== undefined) {
        this._hitboxWidth = actor.hitboxWidth;
      } else {
        this._hitboxWidth = 1;
      }
    }

    return this._hitboxWidth;
  }
  set hitboxWidth(value) {
    this._hitboxWidth = value;
  }

  // Height of the character hitbox (in pixels)
  get hitboxHeight() {
    if (this._hitboxHeight === undefined) {
      const actor = this.actor();

      if (actor !== undefined && actor.hitboxHeight !== undefined) {
        this._hitboxHeight = actor.hitboxHeight;
      } else {
        this._hitboxHeight = 1;
      }
    }

    return this._hitboxHeight;
  }

  set hitboxHeight(value) {
    this._hitboxHeight = value;
  }

  initMembers() {
    super.initMembers();
    this.allowMovementOnFrozenTime = false;
    this.restoreSpeed();
    this._greetingShown = false;
    this._stoppingDelay = 0;
  }

  isValid(x, y) {
    if ($gameMap.isValid(x, y)) return true;

    // Adds an extra valid position for the villagers, so that they appear to walk out of screen.

    switch(this._direction) {
      case Direction.RIGHT :
        return $gameMap.isValid(x - 1, y);
      case Direction.LEFT :
        return $gameMap.isValid(x + 1, y);
      case Direction.UP :
        return $gameMap.isValid(x, y + 1);
      case Direction.DOWN :
        return $gameMap.isValid(x, y - 1);
      default :
        return false;
    }
  }

  isCollidedWithPlayerCharacters() /*x, y*/{
    return false;
  }

  fixXPosition(x) {
    if (x == -1) x = 0;
    if (x == $gameMap.width()) x = $gameMap.width() - 1;
    return x;
  }

  fixYPosition(y) {
    if (y == -1) y = 0;
    if (y == $gameMap.height()) y = $gameMap.height() - 1;
    return y;
  }

  isMapPassable(x, y, d) {
    let x2 = $gameMap.roundXWithDirection(x, d);
    let y2 = $gameMap.roundYWithDirection(y, d);
    const d2 = this.reverseDir(d);

    x = this.fixXPosition(x);
    y = this.fixYPosition(y);
    x2 = this.fixXPosition(x2);
    y2 = this.fixYPosition(y2);

    return this.checkMapPassable(x, y, d) && this.checkMapPassable(x2, y2, d2);
  }

  actor() {
    return $gameActors.actor(this._eventId);
  }

  restoreSpeed() {
    this._moveSpeed = 2;
  }

  distancePerFrame() {
    if (Managers.Time.paused || !Managers.Time.enabled || $gameSystem.isPlayingCutscene() || $gameSystem.isPlayingDialogue) {
      return super.distancePerFrame(arguments);
    }

    const framesPerTile = Managers.Time.getSecondDuration();
    //Make sure villagers will walk exactly 1 tile per game second
    const baseSpeed = 1 / framesPerTile;

    if (this._moveSpeed > 1) {
      // If the villager is behind schedule, then increase the speed a little bit.
      if (this._behindSchedule) {
        return baseSpeed * (this._moveSpeed - 0.9);
      }

      return baseSpeed * (this._moveSpeed - 1);
    }

    return baseSpeed / 2;
  }

  start(activatedByMouse = false) {
    if (this._assignedEvent) {
      if (this._assignedEvent === 'null') {
        return false;
      }

      this.turnTowardPlayer();
      Managers.CommonEvent.playEvent(this._assignedEvent, this);
    } else {
      this.lock();
      this.turnTowardPlayer();
      if (!Managers.Villagers.talkToCharacter(this._eventData.name, this)) {
        this.unlock();
      }
    }

    return true;
  }

  hasAnythingToRun(activatedByMouse = false) {
    if (this._assignedEvent) {
      if (this._assignedEvent == 'null') return false;
    }
    
    return true;
  }

  isCollidedWithVillagers(x, y) {
    if (this._moveType === 1) {
      return super.isCollidedWithVillagers(x, y);
    }

    return false;
  }

  erase() {
    if ($gameMap.villagerEvent(this._eventId)) {
      $gameMap.removeVillagerFromList(this._eventId);
    }

    super.erase();
  }

  isCollidedWithEvents(x, y) {
    if (this._moveType == 1) {
      return super.isCollidedWithEvents(x, y);
    }

    return false;
  }

  canBeActivatedFrom(d, x, y, playerUsedMouse) {
    return true;
  }

  getSpriteClass() {
    return Sprite_Villager;
  }

  checkMapPassable() /*x, y, d*/{
    return Objects.CustomEvent.prototype.checkMapPassable.apply(this, arguments);
  }

  // Ignore villager's hitboxes or $gameMap.villagersXy won't work.
  pos(x, y) {
    return Math.abs(this._x - x) < 1 && Math.abs(this._y - y) < 1;
  }

  itemToDisplay() {
    return this._currentItem;
  }

  pickItemId(itemId) {
    this._currentItem = Managers.Items.getItemData(itemId);
  }

  hideItem() {
    this._currentItem = undefined;
  }

  characterName() {
    if (this._currentItem !== undefined) {
      return `${this._characterName}_item`;
    }

    if (this.isMoving()) {
      return this.movingCharacterName();
    }

    if (this._stoppingDelay > 0) {
      return this.movingCharacterName();
    }

    return this.stoppedCharacterName();
  }

  maxActivationDistance() {
    return Constants.MAX_EVENT_DISTANCE_MOUSE_OVER * 2;
  }

  isCollidedWithFarmObjects(x, y) {
    if (this._moveType === 1) {
      return super.isCollidedWithFarmObjects(x, y);
    }

    return false;
  }

  setDestination(x, y, d) {
    Objects.Character.prototype.setDestination.call(this, x, y, d);
  }

  checkSchedule(expectedX, expectedY) {
    let diff = 0;

    switch(this._direction) {
      case Direction.DOWN:
        diff = expectedY - this._realY;
        break;
      case Direction.LEFT:
        diff = this._realX - expectedX;
        break;
      case Direction.RIGHT:
        diff = expectedX - this._realX;
        break;
      case Direction.UP:
        diff = this._realY - expectedY;
        break;
    }

    this._behindSchedule = diff >= 0.3;
  }

  findDirectionTo(goalX, goalY) {
    if (this.x === goalX && this.y === goalY) {
      return 0;
    }

    if (this.x === goalX) {
      if (this.y < goalY) {
        return Direction.DOWN;
      } else {
        return Direction.UP;
      }
    } else if (this.y == goalY) {
      if (this.x < goalX) {
        return Direction.RIGHT;
      } else {
        return Direction.LEFT;
      }
    } else {
      return super.findDirectionTo(goalX, goalY);
    }
  }

  movingCharacterName() {
    return this._characterName;
  }

  stoppedCharacterName() {
    return this._characterName;
  }
};
