//-----------------------------------------------------------------------------
// Objects.Object
//
// The game object class for FarmObject events

Objects.Object = class Object extends Objects.CustomEvent {
  get farmObjectData() {
    return this._farmObjectData;
  }
  set farmObjectData(value) {
    this.setFarmObjectData(value);
  }

  initialize() {
    Objects.CustomEvent.prototype.initialize.apply(this, arguments);
    this._farmObjectData = false;
  }

  setFarmObjectData(data) {
    this._farmObjectData = data;

    if (data) {
      const dataType = data.getFarmObjectType();
      if (dataType) {
        const dataTypeData = dataType.getData(data);

        const hitboxX = dataType.isModel ? dataType.getHitboxX(data, dataTypeData) : dataType.hitboxX;
        const hitboxY = dataType.isModel ? dataType.getHitboxY(data, dataTypeData) : dataType.hitboxY;
        const hitboxHeight = dataType.isModel ? dataType.getHitboxHeight(data, dataTypeData) : dataType.hitboxHeight;
        const hitboxWidth = dataType.isModel ? dataType.getHitboxWidth(data, dataTypeData) : dataType.hitboxWidth;
        const offsetX = dataType.isModel ? dataType.getOffsetX(data, dataTypeData) : dataType.offsetX;
        const offsetY = dataType.isModel ? dataType.getOffsetY(data, dataTypeData) : dataType.offsetY;

        if (hitboxY !== undefined) {
          this.hitboxY = hitboxY;
        }
        if (hitboxX !== undefined) {
          this.hitboxX = hitboxX;
        }
        if (hitboxHeight !== undefined) {
          this.hitboxHeight = hitboxHeight;
        }
        if (hitboxWidth !== undefined) {
          this.hitboxWidth = hitboxWidth;
        }

        if (offsetY !== undefined) {
          this.offsetY = offsetY;
        }
        if (offsetX !== undefined) {
          this.offsetX = offsetX;
        }
      }

      if (data.hitboxY !== undefined) {
        this.hitboxY = data.hitboxY;
      }
      if (data.hitboxX !== undefined) {
        this.hitboxX = data.hitboxX;
      }
      if (data.hitboxHeight !== undefined) {
        this.hitboxHeight = data.hitboxHeight;
      }
      if (data.hitboxWidth !== undefined) {
        this.hitboxWidth = data.hitboxWidth;
      }
      if (data.offsetY !== undefined) {
        this.offsetY = data.offsetY;
      }
      if (data.offsetX !== undefined) {
        this.offsetX = data.offsetX;
      }
    }
  }

  createSprite() {
    if (this._sprite && !this._sprite._destroyed) {
      return this._sprite;
    }

    this._sprite = null;

    const spriteClass = this.getSpriteClass();
    const objectSprite = new spriteClass(this);
    return objectSprite;
  }

  createSprites() {
    return [this.createSprite()];
  }

  forceUseOfEventSprite() {
    this._forceEventSprite = true;
    if (!this._sprite) {
      Managers.FarmObjects.updateFarmObjectEvent(this._farmObjectData, this);
    } else if (!this._soilSprite) {
      if (this._farmObjectData.shouldShowSoil()) {
        Managers.FarmObjects.updateFarmObjectEvent(this._farmObjectData, this);
      }
    }
  }

  hasAnythingToRunOnLowPriority(activatedByMouse = false) {
    return false;
  }

  hasAnythingToRun(activatedByMouse = false) {
    const farmObjectData = this._farmObjectData;
    if (!farmObjectData) {
      return false;
    }

    const fixedModels = [FarmObjectState.BARN, FarmObjectState.COOP, FarmObjectState.NEW_FENCE, FarmObjectState.GATE, FarmObjectState.SPRINKLER, FarmObjectState.SUPER_SPRINKLER, FarmObjectState.SMALL_WEED, FarmObjectState.WEED];
    if (fixedModels.indexOf(farmObjectData.modelName) >= 0) {
      return true;
    }

    const handledByContentClass = Managers.Content.hasAnythingToRun(this, farmObjectData, activatedByMouse);
    if (handledByContentClass === true || handledByContentClass === false) {
      return handledByContentClass;
    }

    if (farmObjectData.isItem()) {
      return true;
    }

    const modelName = farmObjectData.modelName;

    // If the farmObject is a crop
    if (farmObjectData.isGrownCrop()) {
      // If it's a fruit tree, return true
      if (modelName.indexOf('-tree') < 0) {
        return true;
      }
    }

    if (activatedByMouse) {
      if (Managers.Tools.shouldToolBeUsedOnEvent(this)) {
        return true;
      }
    }

    return false;
  }

  isStaticPosition() {
    //Returns true for some types of objects that we know will never move, so the game doesn't need to try updating their position

    if (this._isStaticPosition !== undefined) return this._isStaticPosition;

    this._isStaticPosition = (function(){
      const farmObjectData = this._farmObjectData;
      // Without this, we have no idea what this object is
      if (!farmObjectData) {
        return false;
      }

      if (farmObjectData.isStaticPosition()) return true;

      return false;
    })();

    return this._isStaticPosition;
  }

  tryTouchingObject(farmObjectData) {
    if (!Managers.Items.isHoldingItem()) {
      if (farmObjectData.modelName == 'fence') {
        if (Managers.FarmObjects.touchFence(farmObjectData, 'fence')) {
          // huh
        }
        return true;
      }

      if (farmObjectData.modelName == FarmObjectState.SPRINKLER || farmObjectData.modelName == FarmObjectState.SUPER_SPRINKLER) {
        if (Managers.FarmObjects.touchFence(farmObjectData, farmObjectData.modelName)) {
          // huh
        }
        return true;
      }

      if (farmObjectData.modelName == Constants.FarmObjectState.GATE) {
        if (Managers.FarmObjects.touchFence(farmObjectData, 'gate')) {
          return true;
        }

        farmObjectData.pressed = !farmObjectData.pressed;

        const event = this;
        if (farmObjectData.pressed) {
          this._direction = 4;

          $gameTemp.setTimeout(() => {
            event._direction = 8;
          }, 2);
        } else {
          this._direction = 6;
          $gameTemp.setTimeout(() => {
            event._direction = 2;
          }, 2);

          $gamePlayer.leaveTile(event.x, event.y);
        }

        return true;
      }
    }


    if (Managers.Content.touchObject(farmObjectData)) {
      return true;
    }

    return false;
  }

  start(activatedByMouse = false) {
    if (this._doingDoorAnimation) return true;

    const farmObjectData = this._farmObjectData;
    if (!farmObjectData) {
      return false;
    }

    if ($gamePlayer.isRiding()) {
      return false;
    }

    if (farmObjectData.modelName == Constants.FarmObjectState.BARN) {
      this.enterBarn(farmObjectData.targetMapId);
      return true;
    }

    if (farmObjectData.modelName == Constants.FarmObjectState.COOP) {
      this.enterCoop(farmObjectData.targetMapId);
      return true;
    }

    if (!activatedByMouse || TouchInput.isTriggered()) {
      if (this.tryTouchingObject(farmObjectData)) {
        return true;
      }
    }

    if (farmObjectData.isItem() || farmObjectData.isGrownCrop()) {
      if (farmObjectData.pick(this)) {
        return true;
      }
    }

    if (activatedByMouse) {
      if (Managers.Tools.shouldToolBeUsedOnEvent(this)) {
        if ($gamePlayer.useCurrentTool(activatedByMouse, this, true)) {
          return true;
        }
      }
    }

    const modelName = farmObjectData.modelName;

    if (modelName == FarmObjectState.SMALL_WEED || modelName == FarmObjectState.WEED) {
      farmObjectData.pick(this);
      return true;
    }

    return false;
  }

  enterBarn(targetMapId) {
    if ($gameSystem.isPlayingCutscene()) return;

    this.doDoorAnimation();
    $gameTemp.setTimeout(() => {
      $gamePlayer.reserveTransfer(targetMapId, 7, 16, Direction.UP, 0);
    }, 18);
  }

  enterCoop(targetMapId) {
    if ($gameSystem.isPlayingCutscene()) return;
    
    this.doDoorAnimation();
    $gameTemp.setTimeout(() => {
      $gamePlayer.reserveTransfer(targetMapId, 10, 8, Direction.UP, 0);
    }, 18);
  }

  isBigFarmObject() {
    if (this.farmObjectData) {
      return this.farmObjectData.isBigFarmObject();
    } else {
      return false;
    }
  }

  doBreakingAnimation(spriteIndex, spriteName, pattern) {
    const event = this;
    if (pattern === undefined) pattern = 1;
    if (spriteName === undefined) spriteName = 'breaking';

    this._farmObjectData = false;

    if (event._doingBreakingAnimation) return;
    event._doingBreakingAnimation = true;

    $gameTemp.setTimeout(() => {
      event._tileId = 0;
      event._iconIndex = 0;
      event._direction = Direction.DOWN;
      event._characterName = spriteName;
      event._characterIndex = spriteIndex;
      event._pattern = pattern;

      $gameTemp.setTimeout(() => {
        event._direction = Direction.LEFT;

        $gameTemp.setTimeout(() => {
          event._direction = Direction.RIGHT;

          $gameTemp.setTimeout(() => {
            event._direction = Direction.UP;

            $gameTemp.setTimeout(() => {
              event.erase();
            }, 3);
          }, 3);
        }, 3);
      }, 3);
    }, 3);
  }

  isTree() {
    if (!this._farmObjectData) return false;
    return this._farmObjectData.isTree();
  }

  isOutlined() {
    return Managers.FarmObjects.isObjectOutlined(this._farmObjectData);
  }

  getOutlineData() {
    return Managers.FarmObjects.getOutlineData(this._farmObjectData);
  }

  canBeTouchedAt(x, y) {
    if (!this._farmObjectData) return false;

    const modelName = this._farmObjectData.modelName;
    if (this.isTree()) {
      // blackberry and raspberry can't be touched outside it's own tile
      if (modelName && modelName.includes('berry-crop')) return false;
      return true;
    }

    if (modelName == FarmObjectState.BOULDER) {
      if (y > this.bottom) return false;
      if (y < this.top - 2) return false;

      return true;
    }

    if (modelName == 'rock-3' || modelName == 'rock-2') {
      if (y == this.y && (x == this.x || x == this.x + 1)) return true;
      return false;
    }

    return this.isTouchingTile(x, y);
  }

  jumpHeight() {
    return 0;
  }

  getSpriteClass() {
    return Sprite_Object;
  }

  // Disable meta offset lookup for farm objects, as they are never going to be in $dataEvents, 
  // this causes fps to be slightly better on weak computers (2~4fps increase in my carroça test).
  getMetaOffsetX() {
    if (this._metaOffsetX !== undefined) {
      return this._metaOffsetX;
    }

    this._metaOffsetX = false;  
    return false;
  }

  getMetaOffsetY() {
    if (this._metaOffsetY !== undefined) {
      return this._metaOffsetY;
    }

    this._metaOffsetY = false;
    return false;
  }

  updateStop() {
    if (this._isStaticPosition) return;

    Objects.CustomEvent.prototype.updateStop.call(this);
  }

  updateMove() {
    if (this._isStaticPosition) return;

    Objects.CustomEvent.prototype.updateMove.call(this);
  }

  update() {
    if (this.isStaticPosition()) return;

    Objects.CustomEvent.prototype.update.call(this);
  }

  expectsAxe() {
    const modelName = this._farmObjectData.modelName;

    if (modelName == FarmObjectState.LOG) return true;
    const toolLevel = Managers.Tools.toolLevel;

    if (modelName == FarmObjectState.STUMP && toolLevel > 0) return true;
    if (toolLevel > 1) {
      if (this._farmObjectData.isCuttableTree()) {
        if ($gameMap.isFarm() || $gameMap.regionId(this.x, this.y) == Region.FARM) {
          return true;
        }
      }
    }

    return false;
  }

  expectsHammer() {
    const modelName = this._farmObjectData.modelName;

    if (modelName == FarmObjectState.STONE) return true;
    const toolLevel = Managers.Tools.toolLevel;

    if (toolLevel >= 3) {
      if (modelName == FarmObjectState.BOULDER) return true;
    }

    return false;
  }

  expectsSickle() {
    const modelName = this._farmObjectData.modelName;

    if (modelName == FarmObjectState.WEED) return true;
    if (modelName == FarmObjectState.SMALL_WEED) return true;

    return false;
  }

  getBalloonOffset() {
    if (!this._farmObjectData) {
      return 0;
    }

    if (this._farmObjectData.isCuttableTree()) {
      return 96;
    }

    return 0;
  }
};
