//-----------------------------------------------------------------------------
// Game_Fish
//
// The base game object class for fish events

Objects.Fish = class Fish extends Objects.Creature {
  initialize() {
    Objects.Creature.prototype.initialize.apply(this, arguments);
    this._moveType = 1;
    this._fishMoveDirection = 6;
    this._fishMoveDistance = 1;
    this._priorityType = 0;
    this.restoreSpeed();
    this._letPlayerWalkThroughThisEvent = true;
  }

  initMembers() {
    Objects.Creature.prototype.initMembers.call(this);
    this.allowMovementOnFrozenTime = true;
    this._letPlayerWalkThroughThisEvent = true;
  }

  restoreSpeed() {
    this._moveSpeed = 2;
    this._moveFrequency = 5;
  }

  start(activatedByMouse = false) {
    return false;
  }

  runAwayFromPlayer(duration) {
    if (this.isRoundShadow()) return;

    super.runAwayFromPlayer(duration);
  }

  setFishMoveDirection(direction) {
    this._fishMoveDirection = direction;
  }

  setFishMoveDistance(distance) {
    this._fishMoveDistance = distance;
  }

  hasAnythingToRun(activatedByMouse = false) {
    return false;
  }

  canPass(x, y, d) {
    const x2 = $gameMap.roundXWithDirection(x, d);
    const y2 = $gameMap.roundYWithDirection(y, d);
    if (!this.isValid(x2, y2)) {
      return false;
    }
    if (this.isThrough() || this.isDebugThrough()) {
      return true;
    }
    if (!this.isMapPassableForFish(x, y, d)) {
      return false;
    }
    return true;
  }

  isRoundShadow() {
    return this._characterIndex == 3 && this._characterName == 'fishes';
  }

  moveTowardsBait() {
    if (!Managers.Fishing.baitEvent) {
      return false;
    }

    if (this.isRoundShadow()) {
      return false;
    }

    const baitEvent = Managers.Fishing.baitEvent;
    const baitLevel = (baitEvent._direction / 2) - 1;
    if (baitLevel < 1) {
      return false;
    }

    if (Math.abs(baitEvent.x - this.x) + Math.abs(baitEvent.y - this.y) > baitLevel) {
      return false;
    }
    // At least one of the positions must be the same
    if (baitEvent.x != this.x && baitEvent.y != this.y) {
      return false;
    }

    // Found a bait nearby, swim towards it
    if (baitEvent.x == this.x) {
      if (baitEvent.y > this.y) {
        this.setDirection(Direction.DOWN);
      } else if (baitEvent.y < this.y) {
        this.setDirection(Direction.UP);
      }
    } else if (baitEvent.y == this.y) {
      if (baitEvent.x > this.x) {
        this.setDirection(Direction.RIGHT);
      } else if (baitEvent.x < this.x) {
        this.setDirection(Direction.LEFT);
      }
    }

    this._moveSpeed = 2 + (baitLevel / 2);
    this._swimmingTowardsBait = true;
    this.swimTowardsBait();
    return true;
  }

  swimTowardsBait() {
    if (!Managers.Fishing.baitEvent) {
      this._moveSpeed = 1;
      this.fadeOutErase();
    }

    this.moveStraight(this._direction);
  }

  updateSelfMovement() {
    if (this._swimmingTowardsBait) {
      this.swimTowardsBait();
      return;
    }

    super.updateSelfMovement();
  }

  moveTypeRandom() {
    if (isNaN(Number(this._initialX)) || isNaN(Number(this._initialY))) {
      this._initialX = this.x;
      this._initialY = this.y;
      this.setDirection(this._fishMoveDirection);
    }

    if (this.moveTowardsBait()) {
      return;
    }

    switch (this._fishMoveDirection) {
      case Direction.LEFT:
        if (this.x <= this._initialX - this._fishMoveDistance) {
          delete this._initialX;
          this.setFishMoveDirection(Direction.RIGHT);
          return;
        }

        this.moveStraight(Direction.LEFT);

        break;
      case Direction.RIGHT:
        if (this.x >= this._initialX + this._fishMoveDistance) {
          delete this._initialX;
          this.setFishMoveDirection(Direction.LEFT);
          return;
        }

        this.moveStraight(Direction.RIGHT);
        break;
      case Direction.DOWN:
        if (this.y >= this._initialY + this._fishMoveDistance) {
          delete this._initialY;
          this.setFishMoveDirection(Direction.UP);
          return;
        }

        this.moveStraight(Direction.DOWN);
        break;
      case Direction.UP:
        if (this.y <= this._initialY - this._fishMoveDistance) {
          delete this._initialY;
          this.setFishMoveDirection(Direction.DOWN);
          return;
        }

        this.moveStraight(Direction.UP);
        break;
    }
  }

  moveRandom() {
    if (this._horzDirection && this._vertDirection) {
      if (Math.randomInt(2) == 1) {
        this.moveStraight(this._horzDirection);
      } else {
        this.moveStraight(this._vertDirection);
      }

      return;
    }

    const d1 = 2 + Math.randomInt(4) * 2;
    const d2 = 2 + Math.randomInt(4) * 2;

    let horz;
    let vert;

    if (d1 == Direction.LEFT || Direction.RIGHT) {
      horz = d1;
      vert = d2;
    } else {
      vert = d1;
      horz = d2;
    }

    if (vert == Direction.LEFT || vert == Direction.RIGHT || horz == Direction.UP || horz == Direction.DOWN) {
      if (this.canPass(this.x, this.y, d1)) {
        this.moveStraight(d1);
      } else if (this.canPass(this.x, this.y, d2)) {
        this.moveStraight(d2);
      }
    } else {
      if (this.canPassDiagonally(this.x, this.y, horz, vert)) {
        this.moveDiagonally(horz, vert);
      }
    }

  }

  screenY() {
    const y = Objects.Creature.prototype.screenY.call(this);

    return y + 24;
  }

  screenZ() {
    return 3;
  }
};
