const BaseModel = require('./BaseModel');

class Crop extends BaseModel {
  static get modelName() {
    return `${ this.cropName }-crop`;
  }

  static get cropName() {
    return '';
  }

  static isCrop() {
    return true;
  }

  static hasCollision() {
    return false;
  }

  static getRegularSpriteName(gameObject, data) {
    return '';
  }

  static getDeadSpriteName(gameObject, data) {
    return '';
  }

  static getCropSpriteIndex(gameObject, data) {
    return 0;
  }

  static getCropDirection() {
    return 2;
  }

  static getCropPriority(gameObject) {
    return 1;
  }

  static regrows() {
    return false;
  }

  static getStageCaps() {
    return [0, 0, 0];
  }

  static getDryDeathChance() {
    return 20;
  }

  static getMinDeathAge() {
    return 30;
  }

  static canBeCut() {
    return true;
  }

  static getPoints(gameObject) {
    return gameObject.points || 0;
  }

  static stage(gameObject) {
    const caps = this.getStageCaps();
    const points = this.getPoints(gameObject);

    let stage = 0;
    while (stage < caps.length && points >= caps[stage]) {
      stage++;
    }

    return stage;
  }

  static getSeedData(gameObject) {
    const data = {
      direction: 4,
      priority: 0,
      spriteIndex: 5,
      pattern: 2,
      spriteName: '!crops',
    };

    if (gameObject.watered) {
      data.spriteIndex = 6;
      data.pattern = 0;
    }

    if (gameObject.spoiled) {
      data.spriteName = '!dead-crops';
    }

    return data;
  }

  static getCropData(gameObject) {
    const stage = this.stage(gameObject);

    return {
      spriteName: gameObject.spoiled ? this.getDeadSpriteName() : this.getRegularSpriteName(),
      spriteIndex: this.getCropSpriteIndex(),
      direction: this.getCropDirection(),
      pattern: stage -1,
      priority: this.getCropPriority(),
    };
  }

  static getData(gameObject) {
    const stage = this.stage(gameObject);

    if (stage === 0) {
      return this.normalizeData(this.getSeedData(gameObject));
    }

    return this.normalizeData(this.getCropData(gameObject));
  }

  static increasePoints(gameObject) {
    const season = Managers.Map.mapMonth(gameObject.mapId);
    if (this.cropGrownsOnSeason(season)) {
      gameObject.increasePoints(this.getStageCaps());
    } else {
      gameObject.lastFruit = true;
      // If it's now winter, the crops die immediatelly
      // In any other season, it can take up to a few days to die.
      if (season == Seasons.WINTER || Math.randomInt(100) < (this.getDryDeathChance() + 10) * 2) {
        Managers.FarmObjects.markObjectAsSpoiled(gameObject);
      }
    }
  }

  static cropGrownsOnSeason(season) {
    const itemData = Managers.Items.getItemData(this.cropName);
    if (!itemData) return false;
    if (!itemData.seasons) return false;
    const seasonName = Managers.Time.monthNames[season -1];

    return itemData.seasons.indexOf(seasonName) >= 0;
  }
}

Models.Crop = Crop;
module.exports = Crop;
