require('game/core/Constants');
const EventListener = require('engine/helpers/EventListener');

Managers.Weather = class Weather extends EventListener {
  static clear() {
    this.currentWeather = WeatherType.SUN;
    this.currentWeatherStrength = 0;
    this.nextDayWeather = WeatherType.SUN;
    this.previousDayWeather = WeatherType.SUN;
    this.windPosition = 0;
  }

  static nextDayRainChance() {
    let month = Managers.Time.month;
    if (Managers.Time.day == 31) month++;

    month = month % 5;

    // Spring:
    // September: 25% rain, 75% sun
    // October: 33% rain, 67% sun
    // November: 35% rain, 65% sun
    // Summer:
    // December: 49% rain, 49% sun, 2% storm
    // January: 58% rain, 38% sun, 4% storm
    // February: 54% rain, 43% sun, 3% storm
    // Fall:
    // March: 45% rain, 55% sun
    // April: 25% rain, 75% sun
    // May: 20% rain, 80% sun
    // Winter:
    // June: 18% rain, 82% sun
    // July: 16% rain, 84% sun
    // August: 14% rain, 86% sun

    switch(month) {
      case Seasons.SPRING:
        return 0.35;

      case Seasons.SUMMER:
        return 0.15;

      case Seasons.FALL:
        return 0.3;

      case Seasons.WINTER:
        return 0.25;
    }

    return 0;
  }

  static maxRainStrength() {
    return 9;
  }

  static minRainStrength() {
    return 1;
  }

  static canUpdateWeather() {
    return !$gameMap.isInside();
  }

  static onMinuteChange() {
    //#ToDo: randomly increase or decrease weather strength

  }

  static getWeatherData() {
    return {
      weather : this.currentWeather,
      strength : this.currentWeatherStrength,
      nextDay : this.nextDayWeather,
      previousDay : this.previousDayWeather
    };
  }

  static setWeatherData(data) {
    this.clear();

    if (!data) {
      return;
    }

    if (data.weather !== undefined) {
      this.currentWeather = data.weather;
    }

    if (data.strength !== undefined) {
      this.currentWeatherStrength = data.strength;
    }

    if (data.nextDay !== undefined) {
      this.nextDayWeather = data.nextDay;
    }

    if (data.previousDay !== undefined) {
      this.previousDayWeather = data.previousDay;
    }
  }

  static pickNextDayWeather() {
    if (Managers.Festivals.hasFestivalTomorrow()) {
      this.nextDayWeather = WeatherType.SUN;
      return;
    }

    const rainChance = this.nextDayRainChance();
    if (Math.random() < rainChance) {
      let month = Managers.Time.month;
      if (Managers.Time.day == 31) {
        month = (month + 1) % 5;
      }

      if (month == Seasons.WINTER) {
        this.nextDayWeather = WeatherType.SNOW;
        return;
      }

      this.nextDayWeather = WeatherType.RAIN;
      return;
    }

    this.nextDayWeather = WeatherType.SUN;
  }

  static processNewDay() {
    this.previousDayWeather = this.currentWeather;
    const nextWeather = this.nextDayWeather;
    this.changeWeather(nextWeather);

    this.pickNextDayWeather();
  }

  static changeWeather(newWeather) {
    this.currentWeather = newWeather;
    if (Switches.isFestival) {
      this.currentWeather = WeatherType.SUN;
    }
    if (this.currentWeather == WeatherType.SUN) {
      this.currentWeatherStrength = 0;
    } else {
      this.currentWeatherStrength = 5;
    }

    Managers.FarmObjects.updateWeatherEffects();
    this.updateWeather();
  }

  static loadTitleWeather() {
    const globalInfo = Engine.Data.loadGlobalInfo() || {};

    if (globalInfo.weather !== 0) {
      this.currentWeatherStrength = 5;
      this.currentWeather = globalInfo.weather;
    }
  }

  static updateWeather() {
    if (Managers.Scenes._scene instanceof GameScenes.Title) {
      this.loadTitleWeather();
    }

    let type = 0;
    if (this.canUpdateWeather()) {
      type = this.currentWeather;
    }
    if (!$gameMap.isRainAllowed()) {
      type = 0;
    }

    switch(type) {
      case WeatherType.RAIN :
        this.startRaining(this.currentWeatherStrength, 5);
        break;
      case WeatherType.SNOW :
        this.startSnowing(this.currentWeatherStrength, 5);
        break;
      case WeatherType.STORM :
        this.startStorm(this.currentWeatherStrength, 5);
        break;
      default :
        this.resetWeather(5);
        break;
    }

    Switches.isRaining = this.isRaining();
  }

  static getWindPosition() {
    return Math.floor(this.windPosition || 0);
  }

  static updateWind() {
    if ($gameMap === undefined || $gameMap === null) return;

    if (this.windPosition >= 30) {
      this.windPosition = 0;
      return;
    }

    this.windPosition += 1 / Utils.getFrameCount(10);
  }

  static startRaining(strength, duration) {
    if ($gameScreen === undefined || $gameScreen === null) return;

    this.runEvent('startRaining');
    $gameScreen.changeWeather('rain', strength, duration);
  }

  static startSnowing(strength, duration) {
    if ($gameScreen === undefined || $gameScreen === null) return;

    this.runEvent('startSnowing');
    $gameScreen.changeWeather('snow', strength, duration);
  }

  static startStorm(strength, duration) {
    if ($gameScreen === undefined || $gameScreen === null) return;

    this.runEvent('startStorm');
    $gameScreen.changeWeather('storm', strength, duration);
  }

  static resetWeather(duration) {
    if ($gameScreen === undefined || $gameScreen === null) return;

    this.runEvent('normalWeather');
    $gameScreen.changeWeather('none', 0, duration);
  }

  static isRainWeather(weather) {
    return [
      WeatherType.RAIN,
      WeatherType.STORM,
    ].includes(weather);
  }

  static isRaining() {
    return this.isRainWeather(this.currentWeather);
  }

  static isRainingTomorrow() {
    return this.isRainWeather(this.nextDayWeather);
  }

  static isSunny() {
    return this.currentWeather == WeatherType.SUN;
  }

  static isBadWeather(weather) {
    if (weather === undefined) {
      weather = this.currentWeather;
    }

    return [
      WeatherType.RAIN,
      WeatherType.STORM,
      WeatherType.SNOW,
      WeatherType.SNOWSTORM
    ].includes(weather);
  }

  static isBadWeatherTomorrow() {
    return this.isBadWeather(this.nextDayWeather);
  }

  static wasRaining() {
    return this.isRainWeather(this.previousDayWeather);
  }

  static wasBadWeather() {
    return this.isBadWeather(this.previousDayWeather);
  }
};

Managers.Weather.clear();
Managers.Time.on('changeMinute', Managers.Weather.onMinuteChange.bind(Managers.Weather));