Managers.Villagers = class Villagers {
  static maybeKillVillagerEvent(eventId) {
    const event = $gameMap.villagerEvent(eventId);
    if (!event || !!event._erased) return false;

    if (!event._lastDestination) return false;
    if (event._lastDestination.x != event._realX) return false;
    if (event._lastDestination.y != event._realY) return false;

    this.killVillagerEvent(eventId);
    return true;
  }

  static killVillagerEvent(eventId) {
    $gameMap.eraseVillager(eventId);
  }

  static removeVillagerEvent(villagerName) {
    const actorId = $gameActors.findActorId(villagerName);
    if (actorId < 0) {
      console.error('Unknown villager: ', villagerName);
      return undefined;
    }

    this.killVillagerEvent(actorId);
  }

  static refreshMapVillager(villagerData) {
  }

  static getVillagerCurrentMap(villagerName) {
    if (this.isVillagerOnParty(villagerName)) {
      return $gameMap._mapId;
    }

    const position = this.getVillagerPosition(villagerName);
    if (!position) return false;

    return position.mapId;
  }

  static getVillagerLocation(villagerName) {
    const mapId = this.getVillagerCurrentMap(villagerName);
    if (!mapId) {
      return '';
    }

    return Managers.Map.getMapDisplayName(mapId);
  }

  static isVillagerScheduledAtMap(villagerName, mapId, timeBehind) {
    if ($gameSystem.isPlayingCutscene()) return false;

    if (Array.isArray(villagerName)) {
      for (let j = 0; j < villagerName.length; j++) {
        if (this.isVillagerScheduledAtMap(villagerName[j], mapId, timeBehind)) {
          return true;
        }
      }

      return false;
    } else {
      // If the villager is currently on the party, then it isn't scheduled anywhere
      if (this.isVillagerOnParty(villagerName)) return false;
    }

    const position = this.getVillagerPosition(villagerName, timeBehind);
    if (!position) return false;

    if (Array.isArray(mapId)) {
      for (let i = 0; i < mapId.length; i++) {
        if (position.mapId == mapId[i]) return true;
      }

      return false;
    } else {
      return position.mapId == mapId;
    }
  }

  //Same as isVillagerScheduledAtMap, but villager needs to be on it for at least one second
  static wasVillagerScheduledAtMap(villagerName, mapId) {
    return this.isVillagerScheduledAtMap(villagerName, mapId) && this.isVillagerScheduledAtMap(villagerName, mapId, -1);
  }

  static getVillagerPosition(villagerName, timeBehind) {
    let positionGroup = this.getCurrentSchedule(villagerName);
    if (positionGroup === undefined) positionGroup = 'default';

    if (!this._positions) return false;

    const positions = this._positions[villagerName.toLowerCase()];
    if (!positions) return false;

    if (!positions[positionGroup]) {
      positionGroup = 'default';
    }

    if (!positions[positionGroup]) return false;

    const time = Managers.Time.getTimeAsTimeStamp(timeBehind);

    const position = positions[positionGroup][time];
    if (!position) return false;

    return position;
  }

  static getCurrentSchedule(villagerName) {
    const contentManagerSchedule = Managers.Content.getCurrentSchedule(villagerName);
    if (contentManagerSchedule) {
      return contentManagerSchedule;
    }

    if (Switches.isFestival) {
      return 'festival';
    }

    return 'default';
  }

  static updateSchedule(eventData, villagerName) {
    const position = this.getVillagerPosition(villagerName);

    if (!position || position.mapId !== $gameMap._mapId) {
      $gameMap.eraseVillager(eventData._eventId, true);
      return;
    }

    eventData.checkSchedule(position.x, position.y);

    if (position.newX !== undefined && position.newY !== undefined) {
      eventData.setDestination(position.newX, position.newY, position.newD);
    } else if (position.newD !== undefined && position.newX === undefined && position.newY === undefined) {
      eventData.setDirection(position.newD);
    }

    if (position.through) {
      eventData.setThrough(true);
    } else if (position.through === false) {
      eventData.setThrough(false);
    }
  }

  static runVillagerTimedEvents(villagerName) {
    const position = this.getVillagerPosition(villagerName);
    if (!position) {
      return;
    }

    if (position.event && position.event !== '') {
      Managers.CommonEvent.playEventAsync(position.event);
    }
  }

  static createVillagerOnScheduledPosition(villagerName) {
    if (Switches.isInsideFestival) return;
    if (Switches.isInsideFestivalCompetition) return;
    if ($gameSystem.isPlayingCutscene()) return;
    // if ($gamePlayer.isTransferring()) return;

    if (this.isVillagerOnParty(villagerName)) return;

    const actorData = this.getActorData(villagerName);
    if (!actorData) return;

    if (actorData.onlyShowIfKnown) {
      if (!Managers.Relationship.isCharacterKnown(villagerName)) return;
    }

    const position = this.getVillagerPosition(villagerName);
    if (!position) {
      return;
    }

    if (position.mapId !== $gameMap._mapId) {
      if (!position.wildcard) {
        this.removeVillagerEvent(villagerName);
      }
      return;
    }

    if (position.x === undefined || position.y === undefined) {
      return;
    }

    let eventData = $gameMap.createVillagerByNameAt(villagerName, position.x, position.y, position.d, false);
    if (!eventData) {
      eventData = $gameMap.villagerEventByName(villagerName);
      if (eventData) {
        eventData.checkSchedule(position.x, position.y);

        if (!eventData.isMoving()) {
          if (position.newX !== undefined && position.newY !== undefined) {
            eventData.setDestination(position.newX, position.newY, position.newD);
          } else if (position.newD !== undefined && position.newX === undefined && position.newY === undefined) {
            eventData.setDirection(position.newD);
          }

          if (position.through) {
            eventData.setThrough(true);
          } else if (position.through === false) {
            eventData.setThrough(false);
          }
        }

        return eventData;
      }

      return false;
    }

    eventData.checkSchedule(position.x, position.y);

    if (position.newX !== undefined && position.newY !== undefined) {
      eventData.setDestination(position.newX, position.newY, position.newD);
    } else if (position.newD !== undefined) {
      eventData.setDirection(position.newD);
    }

    if (position.through) {
      eventData.setThrough(true);
    } else if (position.through === false) {
      eventData.setThrough(false);
    }

    return eventData;
  }

  static shouldLoadPortrait(characterName) {
    if (characterName == 'Player') return true;

    const actorData = this.getActorData(characterName);
    if (!actorData) return false;
    return !!actorData.loadPortrait;
  }

  static getActorData(characterName) {
    if (!$dataActors) return false;
    const lowerCharacterName = characterName.toLowerCase().trim();

    const len = $dataActors.length;
    for (let i = 0; i < len; i++) {
      const actor = $dataActors[i];
      if (!actor) continue;
      if (actor.name.toLowerCase().trim() == lowerCharacterName) {
        return actor;
      }
    }

    return false;
  }

  static getVillagerGender(characterName) {
    const actorData = this.getActorData(characterName);
    if (!actorData) {
      return false;
    }

    return actorData.gender || false;
  }

  static villagerExists(characterName) {
    if (!$dataActors) return false;
    const lowerCharacterName = characterName.toLowerCase().trim();
    return $dataActors.some(actor => {
      if (!actor) return false;
      return actor.name.toLowerCase().trim() == lowerCharacterName;
    });
  }

  static killAllVillagerEvents() {
    $dataActors.forEach(actor => {
      if (!actor) return;
      if (actor.id < 3) return;

      this.killVillagerEvent(actor.id);
    });
  }

  static clearTimeFrozenFlag() {
    if ($gameSystem.isPlayingCutscene()) {
      return;
    }

    $dataActors.forEach(actor => {
      if (!actor) return;
      if (actor.id < 3) return;

      const event = $gameMap.villagerEvent(actor.id);

      if (event) {
        event.allowMovementOnFrozenTime = false;
      }
    });
  }

  static refreshMapVillagers() {
  }

  static onTimeChange() {
  }

  static walk(villagerName, d) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      if (Managers.Time.isInternallyPaused()) {
        eventData.allowMovementOnFrozenTime = true;
      }

      if (!d) d = eventData._direction;
      eventData.moveStraight(d);
    }
  }

  static walkThrough(villagerName, distance, direction) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      if (Managers.Time.isInternallyPaused()) {
        eventData.allowMovementOnFrozenTime = true;
      }

      if (!direction) direction = eventData._direction;

      switch(direction) {
        case Direction.LEFT :
          eventData._x -= distance;
          break;
        case Direction.RIGHT :
          eventData._x += distance;
          break;
        case Direction.UP :
          eventData._y -= distance;
          break;
        case Direction.DOWN :
          eventData._y += distance;
          break;
        default :
          return;
      }

      eventData.increaseSteps();
    }
  }

  static setStepAnime(villagerName, stepAnime) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.setStepAnime(stepAnime);
    }
  }

  static move(villagerName, x, y, d) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      if (Managers.Time.isInternallyPaused()) {
        eventData.allowMovementOnFrozenTime = true;
      }

      eventData.setDestination(x, y, d || eventData._direction || 2);
    }
  }

  static changeSpriteIndex(villagerName, spriteIndex) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData._characterIndex = spriteIndex;
    }
  }

  static changeSpriteImage(villagerName, spriteName) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData._characterName = spriteName;
    }
  }

  static setPosition(villagerName, x, y) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.setPosition(x, y);
    }
  }

  static showAnimation(villagerName, animationId) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.requestAnimation(animationId);
    }
  }

  static stop(villagerName) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.clearDestination();
    }
  }

  static jump(villagerName, x, y) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      if (x === undefined) x = eventData.x;
      if (y === undefined) y = eventData.y;

      eventData.jumpTo(x, y);
    }
  }

  static setThrough(villagerName, through) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.setThrough(through);
    }
  }

  static isMoving(villagerName) {
    const eventData = this.getVillagerData(villagerName);
    if (!eventData) return false;
    return eventData.isMoving();
  }

  static turn(villagerName, d) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      if (eventData.isMoving()) return;

      if (eventData._eventId == $gameMap._interpreter._eventId && eventData._eventId > 0) return;


      if (typeof d === 'string') {
        const character = this.getVillagerData(d);
        if (character) {
          eventData.turnTowardCharacter(character);
        }
      } else {
        eventData.setDirection(d);
      }
    }
  }

  static villagerEventExists(villagerName) {
    villagerName = this.getVillagerNameByAlias(villagerName);

    if (villagerName.toLowerCase() == 'player') {
      return true;
    }

    const actorId = $gameActors.findActorId(villagerName);
    if (actorId < 0) {
      console.error('Unknown villager: ', villagerName);
      return false;
    }

    const eventData = $gameMap.villagerEvent(actorId);
    if (!eventData) {
      return false;
    }

    return true;
  }

  static getVillagerData(villagerName) {
    villagerName = this.getVillagerNameByAlias(villagerName);

    const lowerVillagerName = villagerName.toLowerCase();
    if (lowerVillagerName == 'player') {
      return $gamePlayer;
    }
    if (lowerVillagerName == 'follower') {
      return $gamePlayer.follower();
    }
    if (lowerVillagerName == 'notice' || lowerVillagerName == 'letter' || lowerVillagerName == 'nobody') {
      return undefined;
    }

    const actorId = $gameActors.findActorId(villagerName);
    if (actorId < 0) {
      console.error('Unknown villager: ', villagerName);
      return undefined;
    }

    const eventData = $gameMap.villagerEvent(actorId);
    if (!eventData) {
      // console.error('Villager ' + villagerName + ' not found on the map.');
      return undefined;
    }

    return eventData;
  }

  static changeSpeed(villagerName, speed) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.setMoveSpeed(speed);
    }
  }

  static changePosition(villagerName, x, y, d) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.setPosition(x, y);

      if (typeof d === 'string') {
        this.turn(villagerName, d);
      } else {
        eventData._direction = d;
      }
    }

  }

  static restoreSpeed(villagerName) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData.restoreSpeed();
    }
  }

  static assignEvent(villagerName, eventName) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData._assignedEvent = eventName;
    }
  }

  static clearAssignedEvent(villagerName) {
    const eventData = this.getVillagerData(villagerName);
    if (eventData) {
      eventData._assignedEvent = null;
    }
  }

  static createItemNearVillager(villagerName, itemName, amount) {
    let eventData = this.getVillagerData(villagerName);
    if (!eventData) {
      eventData = $gamePlayer;
    }

    const tile = TileHelper.findEmptyTileNear(eventData.x, eventData.y, eventData._direction);
    if (tile) {
      Managers.FarmObjects.createItemAt($gameMap._mapId, tile.x, tile.y, itemName, amount || 1);
    }
  }

  static fillGlobalVariablesForCharacter(characterName) {
    window.hearts = $gameSystem.getCharacterHearts(characterName);
    window.fp = $gameSystem.getCharacterFriendPoints(characterName);
    window.characterName = characterName;
    window.companion = $gameParty.getCompanionName();

    Variables.friendName = characterName;
  }

  static playEvent(characterName, eventData, eventName, wrapperName = 'talk_to_villager') {
    this.fillGlobalVariablesForCharacter(characterName);

    if (eventData) {
      eventData._preDialogueDirection = eventData._prelockDirection;
      eventData._prelockDirection = eventData._direction;
    }
    $gameSystem.setLastDialogue(characterName, eventName);

    this.playDialogue(eventData, eventName, wrapperName);
  }

  static playDialogue(eventDataOrName, dialogueName, wrapperName = 'talk_to_villager') {
    if (!Managers.CommonEvent.eventExists(dialogueName)) {
      return;
    }

    if (wrapperName === undefined) {
      wrapperName = 'talk_to_villager';
    }

    if (typeof eventDataOrName == 'string') {
      eventDataOrName = this.getVillagerData(eventDataOrName);
    }

    if ($gameSystem.isPlayingCutscene()) {
      $gameTemp.chatName = dialogueName;
      Managers.CommonEvent.playChildEvent(wrapperName, $gameMap._interpreter);
    } else {
      $gameSystem._isPlayingDialogue = true;

      if (eventDataOrName) {
        $gameTemp.chatName = dialogueName;
        dialogueName = wrapperName;
      }

      if (!Managers.CommonEvent.playEvent(dialogueName, eventDataOrName)) {
        $gameSystem._isPlayingDialogue = false;
      }
    }
  }

  static wasEventPlayedToday(eventName, characterName) {
    characterName = characterName || window.characterName;
    const history = $gameSystem.getCharacterDialogueHistory(characterName);

    if (!history) return false;
    return history.includes(eventName);
  }

  static canPlayEvent(eventName, characterName) {
    return true;
  }

  static isEventPatternOnTheHistory(eventPattern, characterName=window.characterName) {
    const history = $gameSystem.getCharacterDialogueHistory(characterName);

    if (!history) return false;
    if (history.includes(eventPattern)) return true;

    for (let i = 0; i < history.length; i++) {
      if (history[i].includes(eventPattern)) return true;
    }

    return false;
  }

  static getEnabledVillagerEvents(characterName) {
    const events = Managers.CommonEvent.getCharacterEvents(characterName);
    const villagerEvents = {};

    this.fillGlobalVariablesForCharacter(characterName);

    const priorityList = {};
    let highestLevel = 0;

    for (const eventName in events) {
      if (!events.hasOwnProperty(eventName)) {
        continue;
      }

      const condition = events[eventName].condition;
      if (!MVC.safeEval(condition)) {
        continue;
      }

      villagerEvents[eventName] = events[eventName];

      const priorityCondition = events[eventName].priority;
      if (!priorityCondition) {
        continue;
      }

      const priorityLevel = MVC.safeEval(priorityCondition);
      if (!priorityLevel) {
        continue;
      }

      if (priorityLevel === true) {
        const priorityEvents = {};
        priorityEvents[eventName] = events[eventName];
        priorityEvents.__forcePlay = true;

        return priorityEvents;
      }

      if (isNaN(priorityLevel)) {
        continue;
      }

      if (priorityLevel > highestLevel) {
        highestLevel = priorityLevel;
      }

      if (!priorityList[priorityLevel]) {
        priorityList[priorityLevel] = {};
      }

      priorityList[priorityLevel][eventName] = events[eventName];
    }

    if (highestLevel > 0) {
      for (let i = highestLevel; i > 0; i--) {
        if (priorityList[i]) {
          return priorityList[i];
        }
      }
    }

    return villagerEvents;
  }

  static getNextVillagerEvent(characterName) {
    const events = this.getEnabledVillagerEvents(characterName);
    const freshEvents = [];
    const allEvents = [];
    const forcePlay = events.__forcePlay || false;

    if (forcePlay) {
      delete events.__forcePlay;
    }

    for (const eventName in events) {
      if (!events.hasOwnProperty(eventName)) continue;

      allEvents.push(eventName);

      if (!forcePlay) {
        if (!this.canPlayEvent(eventName)) {
          continue;
        }
      }

      freshEvents.push(eventName);
    }

    if (freshEvents.length > 0) {
      const idx = Math.randomInt(freshEvents.length);
      return freshEvents[idx];
    }

    return false;
  }

  static getGiftLevel(villagerName, itemId) {
    const actorData = this.getActorData(villagerName);
    if (actorData && actorData.gifts) {
      const gifts = actorData.gifts;

      const love = gifts.love || [];
      const like = gifts.like || [];
      const dislike = gifts.dislike || [];
      const hate = gifts.hate || [];

      if (love.includes(itemId)) {
        return 2;
      }

      if (like.includes(itemId)) {
        return 1;
      }

      if (dislike.includes(itemId)) {
        return -1;
      }

      if (hate.includes(itemId)) {
        return -2;
      }

      const itemData = Managers.Items.getItemData(itemId);
      if (itemData) {
        switch(itemData.defaultGift) {
          case 'hate' :
            return -2;
          case 'dislike' :
            return -1;
          case 'like' :
            return 1;
          case 'love' :
            return 2;
          default:
            return 0;
        }
      }

      return 0;
    }

    return 0;
  }

  static checkAnimals(characterName, eventData) {
    if (Managers.Content.preventAnimalChecking(characterName, eventData)) {
      return;
    }

    if (Managers.Content.checkAnimals(characterName, eventData)) {
      return true;
    }

    return this._checkAnimals(characterName, eventData);
  }

  static _checkAnimals(characterName, eventData) {
    if (Switches.isInsideFestival) return false;
    if (!Managers.Items.isHoldingAnimal()) return false;

    const animalData = Managers.Items.getItemInfo('animalData');
    if (!animalData) return false;

    let race = animalData.type;

    const lowerCharacterName = characterName.toLowerCase();
    let eventName = `${lowerCharacterName}-animal-${race}`;

    if (Managers.CommonEvent.eventExists(eventName)) {
      this.playEvent(characterName, eventData, eventName);
      return true;
    }

    if (race.indexOf('-') > 0) {
      race = race.split('-').pop();
    }

    eventName = `${lowerCharacterName}-animal-${race}`;
    if (Managers.CommonEvent.eventExists(eventName)) {
      this.playEvent(characterName, eventData, eventName);
      return true;
    }

    return false;
  }


  static checkGifts(characterName, eventData) {
    if (Managers.Content.preventGiftChecking(characterName, eventData)) {
      return;
    }

    if (Managers.Content.checkGifts(characterName, eventData)) {
      return true;
    }

    return this._checkGifts(characterName, eventData);
  }

  static _checkGifts(characterName, eventData) {
    if (Switches.isInsideFestival) return false;
    if (!Managers.Items.isHoldingItem()) return false;
    const itemId = Managers.Items.selectedItem.id;
    if (!Managers.Items.canGiftItem(itemId)) return false;

    if (!Managers.Relationship.canGiveGift(characterName)) return false;
    const level = this.getGiftLevel(characterName, itemId);
    Variables.giftLevel = level;

    const lowerCharacterName = characterName.toLowerCase();
    let eventName = `${lowerCharacterName}-gift-${itemId}`;

    if (Managers.CommonEvent.eventExists(eventName)) {
      Managers.Relationship.currentGiftReceiver = characterName;

      this.playEvent(characterName, eventData, eventName, 'give_gift_to_villager');
      return true;
    }

    eventName = `${lowerCharacterName}-gift-`;

    switch(level) {
      case 2:
        eventName += 'loved';
        break;
      case 1:
        eventName += 'liked';
        break;
      case -1:
        eventName += 'disliked';
        break;
      case -2:
        eventName += 'hated';
        break;
      default:
        eventName += 'normal';
        break;
    }

    if (Managers.CommonEvent.eventExists(eventName)) {
      Managers.Relationship.currentGiftReceiver = characterName;
      this.playEvent(characterName, eventData, eventName, 'give_gift_to_villager');

      return true;
    }

    return false;
  }

  static showMessage(characterName, message) {
    const eventData = this.getVillagerData(characterName);

    $gameMessage.setMessageTarget(eventData);
    $gameMessage.addText(message);

    if ($gameMap && $gameMap._interpreter) {
      $gameMap._interpreter.setWaitMode('message');
    }
  }

  static forceChatWithVillager(characterName) {
    return this.talkToCharacter(characterName, undefined, true);
  }

  static executeTalk(characterName, eventData, skipMapStore) {
    Managers.Player.talkingExp++;
    const lowerCharacterName = characterName.toLowerCase();

    if (!$gameSystem.isCharacterKnown(characterName)) {
      this.playEvent(characterName, eventData, `${lowerCharacterName}_first`);
      $gameSystem.markCharacterAsKnown(characterName);
      return true;
    }

    if ($gameParty.hasCompanion()) {
      const companionName = $gameParty.getCompanionName();

      if (!Managers.Relationship.haveVillagersMetEachOther(characterName, companionName)) {
        const meetingEventName = `${lowerCharacterName}_meets_${companionName.toLowerCase()}`;
        if (Managers.CommonEvent.eventExists(meetingEventName)) {
          this.playEvent(characterName, eventData, meetingEventName);
          return true;
        }
      }
    }

    if (Managers.Tasks.checkTaskDialogues(characterName, eventData)) {
      return true;
    }

    if (Managers.Tasks.checkAutoCompleteTasks(characterName, eventData)) {
      return true;
    }

    if (skipMapStore !== true) {
      if (!Managers.Content.preventMapDialogues(characterName, eventData)) {
        if (Managers.Content.checkMapDialogues(characterName, eventData)) {
          return true;
        }

        //If there's an event set to be used by this character on the current map, always play it
        if (Managers.CommonEvent.eventExists(`${lowerCharacterName}_store_map_${$gameMap._mapId}`)) {
          this.playEvent(characterName, eventData, `${lowerCharacterName}_store_map_${$gameMap._mapId}`);
          return true;
        }
      }
    }

    if (this.checkGifts(characterName, eventData)) {
      return true;
    }

    if (this.checkAnimals(characterName, eventData)) {
      return true;
    }

    if (Managers.Content.performPriorityDialogue(characterName, eventData)) {
      return true;
    }

    const lastEventName = $gameSystem.getLastDialogue(characterName);
    let eventName;

    if (!Managers.Content.preventContinuingConversation(characterName, lastEventName, eventData)) {
      if (lastEventName) {
        eventName = `${lastEventName}_cont`;
        if (Managers.CommonEvent.eventExists(eventName)) {
          this.playEvent(characterName, eventData, eventName);
          return true;
        }
      }
    }

    if (Managers.Content.performDialogueIgnoringHistory(characterName, eventData)) {
      return true;
    }

    eventName = this.getNextVillagerEvent(characterName);
    if (eventName) {
      this.playEvent(characterName, eventData, eventName);
      return true;
    }

    if (!Managers.Content.preventMapDefaultDialogue(characterName, eventData)) {
      //If nothing else worked, try the default events
      const defaultMapEventName = `${lowerCharacterName}_default_map_${$gameMap._mapId}`;
      if (Managers.CommonEvent.eventExists(defaultMapEventName)) {
        this.playEvent(characterName, eventData, defaultMapEventName);
        return true;
      }
    }

    if (!Managers.Content.preventDefaultDialogue(characterName, eventData)) {
      if (Managers.CommonEvent.eventExists(`${lowerCharacterName}_default`)) {
        this.playEvent(characterName, eventData, `${lowerCharacterName}_default`);
        return true;
      }
    }

    if (Managers.Content.performLastPossibleDialogue(characterName, eventData)) {
      return true;
    }

    return false;
  }

  static talkToCharacter(characterName, eventData, skipMapStore) {
    Switches.skipMapStore = skipMapStore;
    if (eventData === undefined) {
      eventData = this.getVillagerData(characterName);
    }

    if (eventData && eventData._toolId) {
      eventData._toolId = undefined;
    }

    const result = this.executeTalk(characterName, eventData, skipMapStore);
    Managers.Relationship.talkToVillager(characterName);
    return result;
  }

  static buildContextChatMethods(villagerName, interpreter) {
    const play = (eventName) => {
      Managers.History.registerConversation(eventName);
      Managers.CommonEvent.includeChild(eventName, interpreter);
      return true;
    };

    const getBestEvent = (events, minDays = 1, repeatSameDay = true, ignoreMinDaysIfNothingLeft = true) => {
      const eventList = [].concat(events);
      const table = { };
      let maxDays = -1;

      for (const eventName of eventList) {
        const days = Managers.History.daysSinceConversation(eventName);
        // If it was played today and we can repeat it, then it has priority
        if (days === 0 && repeatSameDay !== false) {
          return eventName;
        }

        if (days === false) {
          // If we can't repeat the day's event then any event that was never played is good
          if (repeatSameDay === false) {
            return eventName;
          }

          table.best = eventName;
          continue;
        }

        if (days < minDays) {
          continue;
        }

        table[days] = eventName;
      }

      if (table.best) {
        return table.best;
      }

      for (let i = minDays; i <= maxDays; i++) {
        if (table[i]) {
          return table[i];
        }
      }

      if (ignoreMinDaysIfNothingLeft !== false) {
        for (let i = minDays - 1; i >= 0; i--) {
          if (table[i]) {
            return table[i];
          }
        }

        // Uhhh... It should've returned something by now. Just pick whatever
        return eventList.pop() || false;
      }

      return false;
    };

    const playBestEvent = (eventList, minDays, repeatSameDay, ignoreMinDaysIfNothingLeft) => {
      const bestEvent = getBestEvent(eventList, minDays, repeatSameDay, ignoreMinDaysIfNothingLeft);
      if (bestEvent) {
        play(bestEvent);
        return true;
      }
    };

    const spreadCall = (eventName, fn) => {
      if (Array.isArray(eventName)) {
        for (const e of eventName) {
          if (fn(e)) {
            return true;
          }
        }
      } else {
        return fn(eventName);
      }
    };

    const playOnceADay = (eventName) => {
      return spreadCall(eventName, (eventName) => {
        if (Managers.Villagers.wasEventPlayedToday(eventName, villagerName)) {
          return false;
        }

        Managers.History.registerConversation(eventName);
        Managers.CommonEvent.includeChild(eventName, interpreter);
        return true;
      });
    };

    const playOneDayPerWeek = (eventName) => {
      return playBestEvent(eventName, 7, true, false);
    };

    const playOncePerYear = (eventName) => {
      const fn = (eventName) => {
        const year = Managers.History.getConversationYear(eventName);
        if (year === false || year !== Managers.Time.year) {
          return play(eventName);
        }
      };

      if (Array.isArray(eventName)) {
        const bestEvent = getBestEvent(eventName, 0, false, false);
        if (bestEvent) {
          return fn(bestEvent);
        }
      } else {
        return fn(eventName);
      }
    };

    const playTwoDaysPerMonth = (eventName) => {
      if (Array.isArray(eventName)) {
        return playBestEvent(eventName, 15, true, false);
      }

      const days = Managers.History.daysSinceConversation(eventName);
      // If it never played, if it played today or if it played over 15 days ago
      if (days === false || days === 0 || days >= 15) {
        return play(eventName);
      }
    };

    const playDuringOneDayOnly = (eventName) => {
      return spreadCall(eventName, (eventName) => {
        const days = Managers.History.daysSinceConversation(eventName);
        // If it never played or if it played today
        if (days === false || days === 0) {
          if (days === false) {
            // If it's the first time playing it, then add some extra friendship for the unique conversation
            Managers.Relationship.increaseFriendship(villagerName, Constants.FRIENDSHIP_FOR_UNIQUE_CHAT);
          }

          return play(eventName);
        }
      });
    };

    const playIfRepeatingToday = (eventName) => {
      return spreadCall(eventName, (eventName) => {
        const days = Managers.History.daysSinceConversation(eventName);
        // If it played today
        if (days === 0) {
          return play(eventName);
        }
      });
    };

    const playOnce = (eventName) => {
      return spreadCall(eventName, (eventName) => {
        const days = Managers.History.daysSinceConversation(eventName);
        // If it never played
        if (days === false) {
          // If it's the first time playing it, then add some extra friendship for the unique conversation
          Managers.Relationship.increaseFriendship(villagerName, Constants.FRIENDSHIP_FOR_UNIQUE_CHAT);
          return play(eventName);
        }
      });
    };

    const playOnceAWeek = (eventName) => {
      return playBestEvent(eventName, 7, false, false);
    };

    const playOnceAMonth = (eventName) => {
      return playBestEvent(eventName, 31, false, false);
    };

    const playOneDayPerMonth = (eventName) => {
      return playBestEvent(eventName, 31, true, false);
    };

    const playOnePerWeekDay = (eventList) => {
      if (eventList.length <= Managers.Time.weekDay) {
        return false;
      }

      const eventName = eventList[Managers.Time.weekDay];
      if (eventName) {
        return playBestEvent(eventName, 0, true, true);
      }
    };

    const methods = {
      play,
      playOnceADay,
      playOneDayPerWeek,
      playOncePerYear,
      playTwoDaysPerMonth,
      playDuringOneDayOnly,
      playOnce,
      playOnceAWeek,
      playOnceAMonth,
      playOneDayPerMonth,
      playOnePerWeekDay,
      getBestEvent,
      playBestEvent,
      playIfRepeatingToday,
      interpreter
    };

    return methods;
  }

  static callContextChat(villagerName, interpreter) {
    const methods = this.buildContextChatMethods(villagerName, interpreter);
    Managers.Content.callContextChat(villagerName, methods);
  }

  static prepareAnimation(villagerName, toolId) {
    const eventData = this.getVillagerData(villagerName);
    if (!eventData) {
      console.error(`Failed to prepare animation for villager ${villagerName}. Villager not found on the map`);
      return;
    }

    eventData.prepareToolAnimation(toolId);
  }

  static animate(villagerName, toolId, delay) {
    const eventData = this.getVillagerData(villagerName);
    if (!eventData) {
      console.error(`Failed to animate villager ${villagerName}. Villager not found on the map`);
      return;
    }

    eventData.doToolAnimation(toolId, undefined, delay);
  }

  static getRandomVillagerName(onlyVillagersNotOnMap, customFilterFn) {
    if (!$dataActors) return false;

    const names = [];

    const len = $dataActors.length;
    for (let i = 11; i < len; i++) {
      const actor = $dataActors[i];
      if (!actor) continue;
      if (!actor.name) continue;
      if (actor.name === '') continue;

      if (!actor.enabled) continue;

      if (customFilterFn) {
        if (!customFilterFn(actor)) continue;
      }

      if (onlyVillagersNotOnMap) {
        if (this.villagerEventExists(actor.name)) continue;
      }

      names.push(actor.name);
    }

    const idx = Math.randomInt(names.length);
    if (idx < 0 || idx >= names.length) return false;

    return names[idx];
  }

  static randomVillagerMovement(x, y, targetX, targetY, targetD) {
    const name = this.getRandomVillagerName(true);
    if (!name) return;

    $gameMap.createVillagerByNameAt(name, x, y, Direction.DOWN);
    this.move(name, targetX, targetY, targetD);
  }

  static yPosition(villagerName) {
    if (villagerName === undefined) villagerName = 'Player';
    const eventData = this.getVillagerData(villagerName);
    if (!eventData) return -1;

    return eventData.y;
  }

  static xPosition(villagerName) {
    if (villagerName === undefined) villagerName = 'Player';
    const eventData = this.getVillagerData(villagerName);
    if (!eventData) return -1;

    return eventData.x;
  }

  static yPositionBetween(villagerName, min, max) {
    const y = this.yPosition(villagerName);
    return y >= min && y <= max;
  }

  static xPositionBetween(villagerName, min, max) {
    const x = this.xPosition(villagerName);
    return x >= min && x <= max;
  }

  static isPosition(villagerName, x, y) {
    if (villagerName === undefined) villagerName = 'Player';
    const eventData = this.getVillagerData(villagerName);
    if (!eventData) return false;

    return eventData.x == x && eventData.y == y;
  }

  static getVillagerNameByAlias(alias) {
    if (alias.toLowerCase() == 'dog') {
      return Managers.Creatures.getDogType();
    }

    return alias;
  }

  static loadPositionsFile(fullFilePath, lowerVillagerName) {
    if (!Wrapper.existsSync(fullFilePath)) {
      return;
    }

    try {
      const fileData = Wrapper.readFileSync(fullFilePath);
      if (!fileData) {
        console.info('File do not exists: ', lowerVillagerName);
        return false;
      }

      try {
        let data;
        if (fullFilePath.endsWith('.json')) {
          data = JsonEx.parse(fileData);
        } else {
          data = YAML.parse(fileData);
        }

        this._positions[lowerVillagerName] = this.expandPositions(data);
        return true;
      } catch(e) {
        console.error(e);
        console.log(lowerVillagerName);
      }
    }
    catch(e) {
      console.log(...log('Failed to parse villager schedule.'));
      console.error(e);
    }
  }

  static loadPositions(villagerName) {
    if (!villagerName) return;

    if (Managers.Extension.loadPositions(villagerName)) return;

    villagerName = villagerName.toLowerCase();

    const projectFolder = Utils.getProjectFolder();
    const fullPath = Wrapper.joinFileNames(projectFolder, 'data', 'schedules', villagerName);

    const yamlFile = `${ fullPath }.yaml`;
    const jsonFile = `${ fullPath }.json`;

    if (Wrapper.existsSync(yamlFile)) {
      return this.loadPositionsFile(yamlFile, villagerName);
    } else {
      return this.loadPositionsFile(jsonFile, villagerName);
    }
  }

  static onLoadCharacters() {
    this._positions = {};

    for (let i = 0; i < $dataActors.length; i++) {
      const actorData = $dataActors[i];
      if (!actorData) continue;

      actorData.id = i;

      this.loadPositions(actorData.name);
    }
  }

  static expandPositionGroup(positions) {
    const newPositions = {};
    let mapId = 0;
    let x = 0;
    let y = 0;
    let d = 0;
    let minKey = 1500;
    let maxKey = 0;
    let event = '';
    let through;

    for (const key in positions) {
      if (!positions.hasOwnProperty(key)) continue;

      const seconds = Managers.Time.convertStringToSeconds(key);

      let dataStr = positions[key];

      let newData = false;
      if (dataStr) {
        through = undefined;

        let data = dataStr.split(',');

        if (data.length >= 4) {
          const mapIdStr = data[0].trim();
          let wildcard = mapIdStr.includes('*');

          mapId = Managers.Map.parseMapId(mapIdStr.replace('*', ''));
          x = Number(data[1].trim());
          y = Number(data[2].trim());
          d = Number(data[3].trim());

          if (data.length > 4) {
            event = data[4].trim();
          } else {
            event = '';
          }

          if (data.length > 5) {
            const str = data[5].trim().toLowerCase();
            if (str === 'true') through = true;
            if (str === 'false') through = false;
          }

          newData = {
            mapId,
            x,
            y,
            d,
            event,
            through,
            wildcard,
          };
        } else if (data.length === 1 || data.length === 2) {
          const mapIdStr = data[0].trim();
          let wildcard = mapIdStr.includes('*');

          if (mapIdStr == '*') {
            mapId = 0;
          } else {
            mapId = Managers.Map.parseMapId(mapIdStr.replace('*', ''));
          }

          if (data.length > 1) {
            event = data[1].trim();
          } else {
            event = '';
          }

          newData = {
            mapId,
            x : undefined,
            y : undefined,
            d : undefined,
            event,
            through : undefined,
            wildcard,
          };
        }
      }

      if (seconds < minKey) minKey = seconds;
      if (seconds > maxKey) maxKey = seconds;
      newPositions[seconds] = newData;
    }

    const fullData = {};

    for (let newKey in newPositions) {
      if (!newPositions.hasOwnProperty(newKey)) continue;
      const currData = newPositions[newKey];

      if (!currData) continue;

      newKey = Number(newKey);

      if (newKey === maxKey) {
        fullData[newKey] = currData;
      }

      for (let newSeconds = newKey + 1; newSeconds <= maxKey; newSeconds++) {
        if (!newPositions.hasOwnProperty(newSeconds)) continue;

        const nextData = newPositions[newSeconds];

        if (nextData.mapId !== currData.mapId) {
          fullData[newKey] = currData;

          currData.newX = currData.x;
          currData.newY = currData.y;
          currData.newD = currData.d;

          if (!nextData.wildcard && !Managers.Map.isMapInside(currData.mapId)) {
            currData.through = true;

            switch(currData.d) {
              case 2:
                currData.newY += 1;
                break;
              case 4:
                currData.newX -= 1;
                break;
              case 6:
                currData.newX += 1;
                break;
              case 8:
                currData.newY -= 1;
                break;
            }
          } else {
            currData.through = false;
          }

          break;
        }

        currData.newX = nextData.x;
        currData.newY = nextData.y;
        currData.newD = nextData.d;
        currData.wildcard = nextData.wildcard;
        fullData[newKey] = currData;

        for (let newNewSeconds = newKey + 1; newNewSeconds < newSeconds; newNewSeconds++) {
          const newDiff = newNewSeconds - newKey;
          const newData = {
            x : currData.x,
            y : currData.y,
            mapId : currData.mapId,
            d : currData.d,
            newX : nextData.x,
            newY : nextData.y,
            newD : nextData.d,
            event : '',
            wildcard: nextData.wildcard
          };

          if (currData.x !== currData.newX || currData.y !== currData.newY) {
            if (!currData.wildcard) {
              switch(Number(currData.d)) {
                case 2:
                  newData.y += newDiff;
                  break;
                case 4:
                  newData.x -= newDiff;
                  break;
                case 6:
                  newData.x += newDiff;
                  break;
                case 8:
                  newData.y -= newDiff;
                  break;
              }
            }
          }

          fullData[newNewSeconds] = newData;
        }

        break;
      }
    }

    return fullData;
  }

  static expandPositions(groups) {
    for (const groupName in groups) {
      if (!groups.hasOwnProperty(groupName)) continue;

      groups[groupName] = this.expandPositionGroup(groups[groupName]);
    }

    return groups;
  }

  static makeVillagerUseTool(villagerName, toolId, times, delay, soundName) {
    if (!villagerName) return;
    const eventData = this.getVillagerData(villagerName);
    if (!eventData) return;

    if (toolId) {
      eventData.doToolAnimation(toolId, times, delay, undefined, undefined, soundName);
    } else {
      eventData._toolId = undefined;
    }
  }

  static moveVillagerToParty(villagerName) {
    $gameParty.addVillager(villagerName);

    const eventData = this.getVillagerData(villagerName);
    if (!eventData) {
      return;
    }

    const follower = $gamePlayer.follower();
    follower._x = eventData._realX;
    follower._y = eventData._realY;
    follower._realX = eventData._realX;
    follower._realY = eventData._realY;

    eventData.erase();
  }

  static isVillagerOnParty(villagerName) {
    return $gameParty.isVillagerOnParty(villagerName);
  }

  static canAddVillagerToParty(villagerName) {
    if (Switches.isFestival) return false;
    if ($gamePlayer.isRiding()) return false;
    if ($gameParty.hasCompanion()) return false;

    const possibleCompanions = ['Annie', 'Benjamin', 'Brittany', 'Cindy', 'Devin', 'Julia', 'Lucas', 'Nathalia', 'Raphael', 'Stella', 'Viktor'];
    if (!possibleCompanions.includes(villagerName)) {
      return false;
    }

    return true;
  }

  static isVillagerAvailableToHangOut(villagerName) {
    if (!this.canAddVillagerToParty(villagerName)) return false;

    if (Managers.Content.isVillagerAvailableToHangOut(villagerName) === false) {
      return false;
    }

    return true;
  }

  static pickItemId(villagerName, itemId) {
    const villagerEvent = this.getVillagerData(villagerName);
    if (villagerEvent) {
      if (itemId) {
        villagerEvent.pickItemId(itemId);
      } else {
        villagerEvent.hideItem();
      }
    }
  }

  static findVillagersWhoReactToItem(itemId, expectedReaction) {
    const villagers = [];

    $dataActors.forEach(actor => {
      if (!actor) return;
      if (!actor.name) return;
      if (!actor.enabled) return;
      if (!actor.gifts) return;

      const list = actor.gifts[expectedReaction] || [];

      if (list.includes(itemId)) {
        villagers.push(actor.name);
      }
    });

    return villagers;
  }

  static whoLovesItem(itemId) {
    return this.findVillagersWhoReactToItem(itemId, 'love');
  }

  static whoLikesItem(itemId) {
    return this.findVillagersWhoReactToItem(itemId, 'like');
  }

  static whoHatesItem(itemId) {
    return this.findVillagersWhoReactToItem(itemId, 'hate');
  }

  static whoDislikesItem(itemId) {
    return this.findVillagersWhoReactToItem(itemId, 'dislike');
  }

  static showItemBalloon(villagerName, itemId) {
    const iconIndex = Managers.Items.getItemIcon(itemId);
    const villagerEvent = this.getVillagerData(villagerName);
    if (!villagerEvent) {
      return;
    }

    if (iconIndex) {
      villagerEvent._itemIconBalloon = iconIndex;
    } else {
      villagerEvent._itemIconBalloon = false;
    }
  }

  static hideItemBalloon(villagerName) {
    const villagerEvent = this.getVillagerData(villagerName);
    if (!villagerEvent) {
      return;
    }

    villagerEvent._itemIconBalloon = false;
  }

  static getVillagerOccupation(villagerName) {
    return Managers.Content.getFriendOccupation(villagerName) || Managers.Text.translateOnly(`${ villagerName }Occupation`, '');
  }
};
