const EventListener = require('engine/helpers/EventListener');
require('engine/managers/Scenes');
require('engine/core/MVCommons');
require('game/core/Constants');
require('game/core/Variables');

//-----------------------------------------------------------------------------
// TimeManager
//
// The static class that handles the flow of time

Managers.Time = class Time extends EventListener {
  static get secondDuration() {
    return Utils.getFrameCount(this._secondDuration);
  }
  static set secondDuration(value) {
    this._secondDuration = value;
  }

  static get enabled() {
    return this._enabled;
  }
  static set enabled(value) {
    this._enabled = value;
  }
  static get yearLength() {
    return this._yearLength;
  }
  static set yearLength(value) {
    this._yearLength = value;
  }
  static get minuteLength() {
    return this._minuteLength;
  }
  static set minuteLength(value) {
    this._minuteLength = value;
  }
  static get hourLength() {
    return this._hourLength;
  }
  static set hourLength(value) {
    this._hourLength = value;
  }
  static get dayLength() {
    return this._dayLength;
  }
  static set dayLength(value) {
    this._dayLength = value;
  }
  static get weekLength() {
    return this._weekLength;
  }
  static set weekLength(value) {
    this._weekLength = value;
  }
  static get monthLength() {
    return this._monthLength;
  }
  static set monthLength(value) {
    this._monthLength = value;
  }


  static get sunRiseStarts() {
    return this._sunRiseStarts;
  }
  static set sunRiseStarts(value) {
    this._sunRiseStarts = value;
  }
  static get sunRiseEnds() {
    return this._sunRiseEnds;
  }
  static set sunRiseEnds(value) {
    this._sunRiseEnds = value;
  }
  static get sunSetStarts() {
    return this._sunSetStarts;
  }
  static set sunSetStarts(value) {
    this._sunSetStarts = value;
  }
  static get sunSetEnds() {
    return this._sunSetEnds;
  }
  static set sunSetEnds(value) {
    this._sunSetEnds = value;
  }
  static get tilesetList() {
    return this._tilesetList;
  }
  static set tilesetList(value) {
    this._tilesetList = value;
  }
  static get weekDayOffset() {
    return this._weekDayOffset;
  }
  static set weekDayOffset(value) {
    this._weekDayOffset = value;
  }
  static get pauseClockDuringConversations() {
    return this._pauseClockDuringConversations;
  }
  static set pauseClockDuringConversations(value) {
    this._pauseClockDuringConversations = value;
  }
  static get monthNames() {
    return this._monthNames;
  }
  static set monthNames(value) {
    this._monthNames = value;
  }
  static get monthShortNames() {
    return this._monthShortNames;
  }
  static set monthShortNames(value) {
    this._monthShortNames = value;
  }
  static get dayLabels() {
    return this._dayLabels;
  }
  static set dayLabels(value) {
    this._dayLabels = value;
  }
  static get dayShortNames() {
    return this._dayShortNames;
  }
  static set dayShortNames(value) {
    this._dayShortNames = value;
  }
  static get seconds() {
    return this._seconds;
  }
  static set seconds(value) {
    this._seconds = value;
  }
  static get minute() {
    return this._minute;
  }
  static set minute(value) {
    this._minute = value;
  }
  static get hour() {
    return this._hour;
  }
  static set hour(value) {
    this._hour = value;
  }
  static get day() {
    return this._day;
  }
  static set day(value) {
    this._day = value;
  }
  static get month() {
    return this._month;
  }
  static set month(value) {
    this._month = value;
  }
  static get year() {
    return this._year;
  }
  static set year(value) {
    this._year = value;
  }
  static get dayPeriod() {
    return this._dayPeriod;
  }
  static set dayPeriod(value) {
    this._dayPeriod = value;
  }
  static get weekDay() {
    return this._weekDay;
  }
  static set weekDay(value) {
    this._weekDay = value;
  }

  static get mapMonth() {
    if ($gameMap.isEternalSpring()) {
      return Seasons.SPRING;
    }
    if ($gameMap.isEternalSummer()) {
      return Seasons.SUMMER;
    }
    if ($gameMap.isEternalFall()) {
      return Seasons.FALL;
    }
    if ($gameMap.isEternalWinter()) {
      return Seasons.WINTER;
    }

    return this.month;
  }

  static get totalDays() {
    var years = this.year - 1;
    var months = (this.month - 1) + (years * this.yearLength);
    var days = this.day + (months * this.monthLength);

    return days;
  }

  static get totalMonths() {
    var years = this.year - 1;
    var months = (this.month - 1) + (years * this.yearLength);

    return months;
  }

  static get yearDay() {
    var months = (this.month - 1);
    var days = this.day + (months * this.monthLength);

    return days;
  }

  static get monthName() {
    return this.getMonthName(this.month);
  }

  static get monthShortName() {
    return this.monthShortNames[(this.month - 1) % this.monthShortNames.length];
  }

  static get dayName() {
    return this.getDayName();
  }

  static get dayLabel() {
    return this.getDayName(this.weekDay);
  }

  static get dayShortName() {
    return this.dayShortNames[this.weekDay % DayDescriptions.length];
  }

  static get inside() {
    if (Managers.Scenes._scene instanceof GameScenes.GameMap) {
      if (this.tilesetList.length > 0) {
        if ($dataMap !== null) {
          if (this.tilesetList.indexOf($dataMap.tilesetId) >= 0) {
            return true;
          }
        }
      }
    }

    return false;
  }

  static get paused() {
    if (this._paused !== undefined) {
      return this._paused;
    }

    return false;
  }

  static set paused(value) {
    this._paused = value;
  }

  static yearDayToDateObj(yearDay) {
    let month = 1;
    let day = yearDay;

    while (day > this.monthLength) {
      month++;
      day -= this.monthLength;
    }

    return this.validateDateTimeValues({
      month,
      day
    });
  }

  static getWeekDayLabel(weekDay) {
    return this.dayLabels[weekDay % DayDescriptions.length];
  }

  static getMonthName(month) {
    return this.monthNames[(month - 1) % this.monthNames.length];
  }

  static runTimeChangeEvents(oldData) {
    // onChangeTime and onChangeSeconds are called from the progressTime method

    if (this.hour >= 6) {
      if (oldData.hour < 6 || oldData.day < this.day || oldData.month < this.month || oldData.year < this.year) {
        this.processNewDay();
      }
    }

    if (oldData.seconds != this.seconds && (oldData.seconds < this.minuteLength || this.seconds !== 0)) {
      this._onChangeSecond();
    }

    if (oldData.minute !== this.minute && (oldData.minute < this.hourLength || this.minute !== 0)) {
      this._onChangeMinute();
    }

    if (oldData.hour !== this.hour && (oldData.hour < this.dayLength || this.hour !== 0) ) {
      this._onChangeHour();
    }

    if (oldData.day !== this.day) {
      this._onChangeDay();
    }

    if (oldData.month !== this.month) {
      this._onChangeMonth();
    }

    if (oldData.year !== this.year) {
      this._onChangeYear();
    }

    if (oldData.dayPeriod !== this.dayPeriod) {
      this._onChangeDayPeriod();
    }
  }

  static convertConfigToTimestamp(config) {
    let years = config.year;
    let months = config.month;
    let days = config.day;
    let hours = config.hour;
    let minutes = config.minute;
    let seconds = config.seconds;

    years -= 1;
    if (years > 0) {
      months += years * this.yearLength;
    }

    months -= 1;
    if (months > 0) {
      days += months * this.monthLength;
    }

    days -= 1;
    if (days > 0) {
      hours += days * this.dayLength;
    }

    if (hours > 0) {
      minutes += hours * this.hourLength;
    }

    if (minutes > 0) {
      seconds += minutes * this.minuteLength;
    }

    return seconds;
  }

  static convertSecondsToTimeString(seconds) {
    return this.convertConfigToTimeString(this.validateDateTimeValues({
      seconds,
      minute : 0,
      hour : 0,
    }));
  }

  static convertConfigToTimeString(time) {
    const hours = time.hour;
    const minutes = time.minute;
    const seconds = time.seconds;

    let string = '';
    if (hours < 10) {
      string += '0';
    }

    string += hours;
    string += ':';
    string += minutes;
    string += seconds;

    return string;
  }

  static convertStringToConfg(string) {
    const seconds = this.convertStringToSeconds(string);
    return this.convertTimestampToConfig(seconds);
  }

  static convertStringToSeconds(string) {
    const values = string.split(':');
    let seconds = 0;
    let minutes = 0;
    let hours = 0;

    if (values.length > 0) {
      hours = Number(values[0]);

      if (values.length > 1) {
        minutes = Number(values[1]);
      }
    }

    const minuteRatio = 60 / this.hourLength;

    minutes /= minuteRatio;

    minutes += hours * this.hourLength;
    seconds += minutes * this.minuteLength;

    return seconds;
  }

  static isBetweenTimes(stringLow, stringHigh) {
    const low = this.convertStringToSeconds(stringLow);
    const high = this.convertStringToSeconds(stringHigh);
    const time = this.getTimeAsTimeStamp();

    return time >= low && time <= high;
  }

  static isTime(stringTime) {
    const time = this.convertStringToSeconds(stringTime);
    const currTime = this.getTimeAsTimeStamp();

    return time == currTime;
  }

  static convertTimestampToConfig(timestamp) {
    let seconds = timestamp;
    let minutes = 0;
    let hours = 0;
    let days = 0;
    let months = 0;
    let years = 0;

    minutes = (seconds / this.minuteLength).floor();
    seconds -= (minutes * this.minuteLength);

    hours = (minutes / this.hourLength).floor();
    minutes -= (hours * this.hourLength);

    days = (hours / this.dayLength).floor();
    hours -= (days * this.dayLength);

    months = (days / this.monthLength).floor();
    days -= (months * this.monthLength);

    years = (months / this.yearLength).floor();
    months -= (years * this.yearLength);

    return {
      seconds,
      minute : minutes,
      hour : hours,
      day : days + 1,
      month : months + 1,
      year : years + 1
    };
  }

  // Returns the difference in number of seconds
  static compareTimestamps(timestamp1, timestamp2) {
    if (typeof timestamp1 == "object") {
      timestamp1 = this.convertConfigToTimestamp(timestamp1);
    }

    if (typeof timestamp2 == "object") {
      timestamp2 = this.convertConfigToTimestamp(timestamp2);
    }

    const diff = timestamp2 - timestamp1;
    return diff;
  }

  static validateDateTimeValues(date) {
    if (date.seconds === undefined) date.seconds = 0;
    if (date.minute === undefined) date.minute = 0;
    if (date.hour === undefined) date.hour = 0;
    if (date.day === undefined) date.day = 1;
    if (date.month === undefined) date.month = 1;
    if (date.year === undefined) date.year = 1;

    while (date.seconds < 0) {
      date.minute -= 1;
      date.seconds += this.minuteLength;
    }

    while (date.seconds >= this.minuteLength) {
      date.minute += 1;
      date.seconds -= this.minuteLength;
    }

    while (date.minute < 0) {
      date.hour -= 1;
      date.minute += this.hourLength;
    }

    while (date.minute >= this.hourLength) {
      date.hour += 1;
      date.minute -= this.hourLength;
    }

    while (date.hour < 0) {
      date.day -= 1;
      date.hour += this.dayLength;
    }

    while (date.hour >= this.dayLength) {
      date.day += 1;
      date.hour -= this.dayLength;
    }

    date.weekDay = this.calculateWeekDay(date);

    return date;
  }

  static validateDateTimeValuesFurther(date) {
    while (date.day < 0) {
      date.month -= 1;
      date.day += this.monthLength;
    }

    while (date.day > this.monthLength) {
      date.month += 1;
      date.day -= this.monthLength;
    }

    while (date.month < 0) {
      date.year -= 1;
      date.month += this.yearLength;
    }

    while (date.month > this.yearLength) {
      date.year += 1;
      date.month -= this.yearLength;
    }

    let clearDay = false;
    let clearMonth = false;
    let clearYear = false;

    if (date.day <= 0) {
      clearDay = true;
    }

    if (date.month <= 0) {
      clearMonth = true;
      clearDay = true;
    }

    if (date.year <= 0) {
      clearYear = true;
      clearMonth = true;
      clearDay = true;
    }

    if (clearDay) {
      date.day = 1;
      date.hour = 6;
      date.minute = 0;
      date.seconds = 0;
    }

    if (clearMonth) {
      date.month = 1;
    }

    if (clearYear) {
      date.year = 1;
    }

    date.weekDay = this.calculateWeekDay(date);
    return date;
  }

  static updateTime(runEvents, allowDayChangeEvent) {
    if (runEvents === undefined) runEvents = true;
    if (allowDayChangeEvent === undefined) allowDayChangeEvent = false;

    const oldData = this.getDateTime();

    const date = {
      seconds : this.seconds,
      minute : this.minute,
      hour : this.hour,
      day : this.day,
      month : this.month,
      year : this.year
    };

    this.validateDateTimeValues(date);
    this.validateDateTimeValuesFurther(date);

    this.seconds = date.seconds;
    this.minute = date.minute;
    this.hour = date.hour;
    this.day = date.day;
    this.month = date.month;
    this.year = date.year;

    this.updateDayPeriod();

    // Calculate week day
    const previousYears = this.year - 1;
    const previousMonths = previousYears * this.yearLength;
    const numMonths = previousMonths + this.month - 1;
    const numDays = numMonths * this.monthLength + this.day;
    this.weekDay = (numDays + this.weekDayOffset) % this.weekLength;

    Variables.day = this.day;
    Variables.hour = this.hour;
    Variables.minute = this.minute;
    Variables.month = this.month;
    Variables.year = this.year;
    Variables.yearDay = this.yearDay;
    Variables.totalDays = this.totalDays;
    Variables.weekDay = this.weekDay;

    const moddedHour = this.hour % 12;
    const clockMinutes = (moddedHour * this.hourLength) + this.minute - 1;
    const clockSeconds = (clockMinutes * this.minuteLength) + this.seconds - 5;

    Variables.clockHour = Math.round((clockSeconds / this.minuteLength) / this.hourLength);

    if (allowDayChangeEvent) {
      if (this.hour === 6 && this.minute === 0 && this.seconds === 0) {
        $gameSystem.startSleep(() => {
          this.doSleepEffect(0);
          Engine.Data.autoSave();
        }, oldData, this.getDateTime());
      }
    }

    if (runEvents) {
      this.runTimeChangeEvents(oldData);
    }

    this._onUpdateTime();
  }

  static calculateWeekDay(dateConfig) {
    const previousYears = dateConfig.year - 1;
    const previousMonths = previousYears * this.yearLength;
    const numMonths = previousMonths + dateConfig.month - 1;
    const numDays = numMonths * this.monthLength + dateConfig.day;
    return (numDays + this.weekDayOffset) % this.weekLength;
  }

  static updateDayPeriodForDate(date) {
    // Calculate day period
    if (date.hour < this.sunRiseStarts) {
      date.dayPeriod = 4;
    } else if (date.hour < this.sunRiseEnds) {
      date.dayPeriod = 1;
    } else if (date.hour < this.sunSetStarts) {
      date.dayPeriod = 2;
    } else if (date.hour < this.sunSetEnds) {
      date.dayPeriod = 3;
    } else {
      date.dayPeriod = 4;
    }
  }

  static updateDayPeriod() {
    this.updateDayPeriodForDate(this);
  }

  static isMorning() {
    return this.hour >= 6 && this.hour < 12;
  }

  static isNoon() {
    return this.hour >= 11 && this.hour < 13;
  }

  static isAfternoon() {
    return this.hour >= 12 && this.hour < 18;
  }

  static isEvening() {
    return this.hour >= 18 && this.hour < 20;
  }

  static isNight() {
    return this.hour >= 20 || this.hour < 6;
  }

  static isEnabled() {
    return this._enabled;
  }

  static getSecondDuration() {
    const customSecondDuration = Managers.Content.getSecondDuration();
    if (!isNaN(Number(customSecondDuration)) && customSecondDuration > 0) {
      return customSecondDuration;
    }

    return this.secondDuration;
  }

  static resetFrameCount() {
    this._frameCount = this.getSecondDuration() - 1;
  }

  static update() {
    if (!this._enabled) return;
    if (this._frameCount && this._frameCount > 0) {
      this._frameCount--;
      return;
    }

    this.resetFrameCount();
    this.progressTime();
  }

  static enableTime() {
    this._enabled = true;
  }

  static refreshTimeSystem() {
    this.disableTime();
    this.enableTime();
  }

  static disableTime() {
    this._enabled = false;
  }

  static setTime(seconds, minute, hour, day, month, year) {
    const oldData = this.getDateTime();

    if (seconds !== undefined) {
      this.seconds = seconds;
    }

    if (minute !== undefined) {
      this.minute = minute;
    }

    if (hour !== undefined) {
      this.hour = hour;
    }

    if (day !== undefined) {
      this.day = day;
    }

    if (month !== undefined) {
      this.month = month;
    }

    if (year !== undefined) {
      this.year = year;
    }

    this.updateTime(false);
    this.runTimeChangeEvents(oldData);
  }

  static addSeconds(seconds) {
    this.addTime({seconds});
  }

  static addMinutes(minutes) {
    this.addTime({minutes});
  }

  static addHours(hours) {
    this.addTime({hours});
  }

  static addDays(days) {
    this.addTime({days});
  }

  static addMonths(months) {
    this.addTime({months});
  }

  static addYears(years) {
    this.addTime({years});
  }

  static addTime(timeData) {
    this.passTime(timeData.seconds, timeData.minutes, timeData.hours, timeData.days, timeData.months, timeData.years);

    // Whenever the time is modified externally, we need to open or close doors according to the schedules
    $gameMap.closeEveryDoor();
    $gameMap.updateDoorsState();
  }

  static passTime(seconds, minutes, hours, days, months, years) {
    const oldData = this.getDateTime();

    this.seconds += Number(seconds || 0);
    this.minute += Number(minutes || 0);
    this.hour += Number(hours || 0);
    this.day += Number(days || 0);
    this.month += Number(months || 0);
    this.year += Number(years || 0);

    this.updateTime(false);
    this.runTimeChangeEvents(oldData);
  }

  static getDateWeekDay(year, month, day) {
    const previousMonths = (year - 1) * this.yearLength;
    const numMonths = previousMonths + month - 1;
    const numDays = numMonths * this.monthLength + day;
    const weekDay = (numDays + this.weekDayOffset) % this.weekLength;

    return weekDay;
  }

  static forwardTo(newHour) {
    const oldData = this.getDateTime();

    const date = {
      seconds : this.seconds,
      minute : this.minute,
      hour : this.hour,
      day : this.day,
      month : this.month,
      year : this.year
    };

    date.minute = 0;
    date.seconds = 0;
    date.hour++;
    this.validateDateTimeValues(date);

    while (date.hour != newHour) {
      date.hour++;
      this.validateDateTimeValues(date);
    }

    this.seconds = date.seconds;
    this.minute = date.minute;
    this.hour = date.hour;
    this.day = date.day;
    this.month = date.month;
    this.year = date.year;

    this.updateDayPeriod();
    this.runTimeChangeEvents(oldData);
  }

  static isInternallyPaused() {
    if (Switches.isInsideFestival || Switches.holdPlayer) {
      return true;
    }

    if ($gameSystem.isPlayingCutscene()) {
      return true;
    }

    if ($gameSystem.isPlayingDialogue) {
      return true;
    }

    if ($gameSystem.isPlayerSleeping) {
      return true;
    }

    if ($gamePlayer.isTransferring()) return true;
    if ($gamePlayer.isWaitingAfterTransfer()) return true;

    if (Managers.Scenes._scene instanceof GameScenes.GameMap) {
      if (Managers.Scenes._scene.menuCalling) {
        return true;
      }
    }

    if (this.pauseClockDuringConversations === true) {
      if (Managers.Scenes._scene instanceof GameScenes.GameMap) {
        if ($gameMessage.isBusy()) {
          return true;
        }

        if ($gameMap._interpreter.isBalloonPlaying()) {
          return true;
        }

        if ($gamePlayer.isBalloonPlaying()) {
          return true;
        }
      }
    }

    if ($gameSystem.hasLetter()) {
      return true;
    }

    return false;
  }

  static progressTime() {
    if (this.paused) return;
    if (this.isInternallyPaused()) return;
    if (!(Managers.Scenes._scene instanceof GameScenes.GameMap)) return;

    if ($gameMap.isInsideAnyHouse()) {
      this.updateTime(false, false);
      return;
    }

    this.seconds += 1;
    this.updateTime(true, true);

    this._onChangeSecond();
    this._onChangeTime();
  }

  static _onChangeSecond() {
    this.runEvent('changeSecond');
    Managers.Content.onChangeSecond();
  }

  static _onChangeTime() {
    this.runEvent('changeTime');
    Managers.Content.onChangeTime();
  }

  static _onChangeMinute() {
    this.runEvent('changeMinute');
    Managers.Content.onChangeMinute();
  }

  static _onChangeHour() {
    this.runEvent('changeHour');
    Managers.Content.onChangeHour();
  }

  static _onChangeDayPeriod() {
    this.runEvent('changeDayPeriod');
    Managers.Content.onChangeDayPeriod();
  }

  static _onChangeDay() {
    this.runEvent('changeDay');
    Managers.Content.onChangeDay();
  }

  static _onChangeMonth() {
    this.runEvent('changeMonth');
    Managers.Content.onChangeMonth();
  }

  static _onChangeYear() {
    this.runEvent('changeYear');
    Managers.Content.onChangeYear();
  }

  static onTime(callback, hour, minute, second, autoRemove) {
    return this.onDateTime(callback, 0, 0, 0, hour, minute, second, autoRemove);
  }

  static onDate(callback, day, month, year, autoRemove) {
    return this.onDateTime(callback, day, month, year, 0, 0, 0, autoRemove);
  }

  static onDateTime(callback, day, month, year, hour, minute, second, autoRemove) {
    const config = {
      day,
      month,
      year,
      hour,
      minute,
      second,
      callback,
      autoRemove,
      after : false
    };

    if (autoRemove === undefined) {
      autoRemove = false;
    }

    return this.registerTimeEvent(config);
  }

  static onWeekDay(callback, weekDay) {
    return this.registerTimeEvent({
      weekDay,
      callback
    });
  }

  static registerAfterTimeEvent(config) {
    const id = this._afterTimeEventsNextId;
    this._afterTimeEvents[id] = config;
    this._afterTimeEventsNextId++;
    config.key = id;
    return id;
  }

  static runInDateTime(callback, years, months, days, hours, minutes, seconds, autoRemove) {
    const newDate = this.getDateTime();

    newDate.year += Number(years || 0);
    newDate.month += Number(months || 0);
    newDate.day += Number(days || 0);
    newDate.hour += Number(hours || 0);
    newDate.minute += Number(minutes ||0);
    newDate.seconds += Number(seconds || 0);

    this.validateDateTimeValues(newDate);

    if (autoRemove === undefined) {
      autoRemove = true;
    }

    const config = {
      callback,
      year : newDate.year,
      month : newDate.month,
      day : newDate.day,
      hour : newDate.hour,
      minute : newDate.minute,
      seconds : newDate.seconds,
      autoRemove,
      after : true
    };

    return this.registerAfterTimeEvent(config);
  }

  static runInHours(callback, hours, minutes, seconds) {
    return this.runInDateTime(callback, 0, 0, 0, hours, minutes, seconds);
  }

  static runInDays(callback, days) {
    return this.runInDateTime(callback, 0, 0, days);
  }

  static runInMonths(callback, months) {
    return this.runInDateTime(callback, 0, months);
  }

  static runInYears(callback, years) {
    return this.runInDateTime(callback, years);
  }

  static runInMinutes(callback, minutes) {
    return this.runInHours(callback, 0, minutes);
  }

  static runInSeconds(callback, seconds) {
    return this.runInHours(callback, 0, 0, seconds);
  }

  static registerTimeEvent(config) {
    const id = this._timeEventsNextId;
    this._timeEventsNextId++;
    this._timeEvents[id] = config;
    config.key = id;
    return id;
  }

  static checkIfEventShouldRun(config) {
    if (config.day !== undefined && config.day != this.day) return false;
    if (config.month !== undefined && config.month != this.month) return false;
    if (config.year !== undefined && config.year != this.year) return false;
    if (config.hour !== undefined && config.hour != this.hour) return false;
    if (config.minute !== undefined && config.minute != this.minute) return false;
    if (config.second !== undefined && config.second != this.seconds) return false;
    if (config.weekDay !== undefined && config.weekDay != this.weekDay) return false;

    return true;
  }

  static checkEventsToRun(eventList, after) {
    let config;
    const keysToRemove = [];

    const currentTimestamp = this.convertConfigToTimestamp(this.getDateTime());
    for (const key in eventList) {
      config = eventList[key];
      if (!config) continue;
      if (config.callback === undefined) continue;

      if (after) {
        if (config.seconds === undefined) {
          config.seconds = config.second;
        }

        const timestamp = this.convertConfigToTimestamp(config);
        if (timestamp > currentTimestamp) {
          continue;
        }
      } else {
        if (!this.checkIfEventShouldRun(config)) {
          continue;
        }
      }

      this.executeCallback(config.callback);
      if (config.autoRemove === true) {
        config.callback = undefined;
        keysToRemove.push(config.key);
      }
    }

    for (const keyToRemove of keysToRemove) {
      delete eventList[keyToRemove];
    }
  }

  static _onUpdateTime() {
    this.checkEventsToRun(this._timeEvents, false);
    this.checkEventsToRun(this._afterTimeEvents, true);
  }

  static getDayName(weekDay) {
    weekDay = weekDay || this.weekDay;
    return DayDescriptions[weekDay % DayDescriptions.length];
  }

  static getDateTime() {
    return {
      hour: this.hour,
      minute: this.minute,
      seconds: this.seconds,
      day: this.day,
      month: this.month,
      year: this.year,
      dayPeriod: this.dayPeriod,
      weekDay: this.weekDay,
      paused: this.paused
    };
  }

  static getDate() {
    return {
      hour: 6,
      minute: 0,
      seconds: 0,
      day: this.day,
      month: this.month,
      year: this.year,
      dayPeriod: 0,
      weekDay: this.weekDay,
      paused: this.paused
    };
  }

  static getDateAsTimeStamp() {
    const date = this.getDate();

    return this.convertConfigToTimestamp(date);
  }

  static getTimeAsTimeStamp(timeBehind) {
    const time = this.getTime(timeBehind);

    return this.convertConfigToTimestamp(time);
  }

  static getTime(timeBehind) {
    const time = {
      hour : this.hour,
      minute : this.minute,
      seconds : this.seconds,
      day : 0,
      month : 0,
      year : 0,
      dayPeriod : this.dayPeriod,
      weekDay : 0,
      paused : this.paused
    };

    if (timeBehind) {
      time.seconds -= timeBehind;
    }

    return this.validateDateTimeValues(time);
  }

  static getTimeStr() {
    const time = this.getTime();
    return this.convertConfigToTimeString(time);
  }

  static getLaterTimeStr(extraSeconds) {
    const time = this.getTime(-extraSeconds);
    return this.convertConfigToTimeString(time);
  }

  static getTomorrow() {
    const dateObj = this.getDateTime();

    dateObj.day += 1;
    this.validateDateTimeValues(dateObj);

    return dateObj;
  }

  static setDateTime(dateTime) {
    this.hour = dateTime.hour || 0;
    this.minute = dateTime.minute || 0;
    this.seconds = dateTime.seconds || 0;
    this.day = dateTime.day || 1;
    this.month = dateTime.month || 1;
    this.year = dateTime.year || 1;
    this.dayPeriod = dateTime.dayPeriod || DayPeriods.EARLY_MORNING;
    this.weekDay = dateTime.weekDay || 1;

    if (dateTime.paused !== undefined) {
      this.paused = dateTime.paused;
    } else {
      this.paused = false;
    }
  }

  static getTimeDiff(dateTime1, dateTime2, returnDays) {
    if (typeof(dateTime1) == "number") {
      dateTime1 = this.convertTimestampToConfig(dateTime1);
    }
    if (typeof(dateTime2) == "number") {
      dateTime2 = this.convertTimestampToConfig(dateTime2);
    }

    const years = dateTime2.year - dateTime1.year;
    let months = dateTime2.month - dateTime1.month;
    let days = dateTime2.day - dateTime1.day;
    let hours = dateTime2.hour - dateTime1.hour;
    let minutes = dateTime2.minute - dateTime1.minute;
    let seconds = dateTime2.seconds - dateTime1.seconds;

    months += (years * this.yearLength);
    days += (months * this.monthLength);

    if (returnDays === true) return days;

    hours += (days * this.dayLength);
    minutes += (hours * this.hourLength);
    seconds += (minutes * this.minuteLength);

    return seconds;
  }

  static timeSince(oldDateTime) {
    return this.getTimeDiff(oldDateTime, this.getDateTime());
  }

  static daysSince(oldDate) {
    return this.getTimeDiff(oldDate, this.getDate(), true);
  }

  static getDateDescription(dateTime = undefined, stringFormat = null) {
    if (dateTime === undefined) {
      dateTime = this.getDateTime();
    }

    const stringBuilder = t(stringFormat || "{season} {day}, Year {year}");
    const seasonName = t(this.getMonthName(dateTime.month));

    const day = dateTime.day;
    const year = dateTime.year;

    return stringBuilder.replace('{season}', seasonName).replace('{day}', day).replace('{year}', year);
  }

  static getNextMorningTime() {
    const newDateTime = this.getDateTime();
    //Don't increase the day if it's past midnight
    if (newDateTime.hour >= 6) {
      newDateTime.day += 1;
    }

    if (newDateTime.day > 31) {
      newDateTime.day = 1;
      newDateTime.month += 1;
    }

    if (newDateTime.month > 4) {
      newDateTime.month = 1;
      newDateTime.year += 1;
    }

    newDateTime.hour = 6;
    newDateTime.minute = 0;
    newDateTime.seconds = 0;

    return newDateTime;
  }

  static sleep(newDateTime) {
    const oldDateTime = this.getDateTime();

    if (newDateTime === undefined) {
      newDateTime = this.getNextMorningTime();
    }

    const diff = this.getTimeDiff(oldDateTime, newDateTime);

    if (oldDateTime.hour >= 6 && oldDateTime.hour < 8) {
      Managers.ShippingBin.sellBinContents();
    }

    this.setDateTime(newDateTime);
    this.updateTime();
    this.doSleepEffect(diff);
    this.runTimeChangeEvents(oldDateTime);
  }

  static doSleepEffect(diff) {
    Managers.Health.doSleepEffect(diff);
  }

  static processNewDay() {
    Managers.Festivals.processNewDay();

    Managers.CommonEvent.doNewDayProcessing();
    $gameSystem.clearDailyStuff();
    Managers.Weather.processNewDay();
    Managers.FarmObjects.processNewDay();
    Managers.Creatures.processNewDay();
    Managers.Fishing.processNewDay();
    Managers.Mailbox.processNewDay();
    Managers.Relationship.processNewDay();
    Managers.Content.processNewDay();
    Managers.Health.processNewDay();

    this.runEvent('processNewDay');
  }

  static getCallbacksFromList(eventList) {
    const callbackList = [];

    for (const key in eventList) {
      const config = eventList[key];

      if (!config) continue;
      if (config.callback === undefined) continue;

      //can't save functions
      if (typeof(config.callback) == "function") continue;

      callbackList.push(MVC.shallowClone(config));
    }

    return callbackList;
  }

  static getCallbacks() {
    const callbackList = {};

    callbackList.after = this.getCallbacksFromList(this._afterTimeEvents);
    callbackList.normal = this.getCallbacksFromList(this._timeEvents);

    return callbackList;
  }

  static setCallbacksToList(callbackList, nextId) {
    const eventList = {};

    for (let i = 0; i < callbackList.length; i++) {
      const config = MVC.shallowClone(callbackList[i]);

      eventList[nextId] = config;
      config.key = nextId;
      nextId++;
    }

    return {
      list : eventList,
      nextId
    };
  }

  static setCallbacks(callbackList) {
    if (callbackList.after !== undefined) {
      const afterTimeEvents = this._afterTimeEvents = this.setCallbacksToList(callbackList.after, this._afterTimeEventsNextId);
      this._afterTimeEvents = afterTimeEvents.list;
      this._afterTimeEventsNextId = afterTimeEvents.nextId;
    }

    if (callbackList.normal !== undefined) {
      const timeEvents =  this._timeEvents = this.setCallbacksToList(callbackList.normal, this._timeEventsNextId);
      this._timeEvents = timeEvents.list;
      this._timeEventsNextId = timeEvents.nextId;
    }
  }

  static setInitialDateTime() {
    this.setDateTime({
      seconds : 0,
      minute : 0,
      hour : 8,
      day : 1,
      month : 1,
      year : 1,
      weekDay : 1
    });

    //Calls updatetime to set the right weekday before the time system is activated
    this.updateTime();
    this.refreshTimeSystem();
  }

  static getFormattedDate(dateConfig) {
    let format = t("<!dateFormat!>");
    if (format == '<!dateFormat!>') {
      format = "{monthName} {dayNumber}, year {yearNumber}";
    }

    return format.replace(
      '{monthName}',
      t(this.getMonthName(dateConfig.month || 1))
    ).replace(
      '{dayNumber}',
      dateConfig.day || 1
    ).replace(
      '{yearNumber}',
      dateConfig.year || 1
    );
  }

  static getFormattedTimeAge(ageInSeconds) {
    const minutesInAnHour = this.hourLength * this.minuteLength;

    const hours = Math.floor(ageInSeconds / minutesInAnHour);
    const minutes = ageInSeconds - (hours * minutesInAnHour);

    let hourStr = false;
    let minuteStr = false;
    if (hours) {
      if (hours === 1) {
        hourStr = t("One Hour");
      } else {
        hourStr = t("{hours} Hours").replace('{hours}', hours);
      }
    }

    if (minutes) {
      if (minutes === 1) {
        minuteStr = t("One Minute");
      } else {
        minuteStr = t("{minutes} Minutes").replace('{minutes}', minutes);
      }
    }

    if (hourStr && minuteStr) {
      return t("{hours}, {minutes}").replace('{hours}', hourStr).replace('{minutes}', minuteStr);
    }

    return hourStr || minuteStr || '';
  }

  static getFormattedAge(ageInDays) {
    let number;
    let type;

    if (ageInDays < 7) {
      number = ageInDays;
      type = 'days';
    } else if (ageInDays < 31) {
      number = Math.floor(ageInDays / 7);
      type = 'weeks';
    } else if (ageInDays < 124) {
      number = Math.floor(ageInDays / 31);
      type = 'months';
    } else {
      number = Math.floor(ageInDays / 124);
      type = 'years';
    }

    return Managers.Text.translateAge(number, type);
  }

  static secondsUntilDayChange() {
    let hours = 0;

    if (this.hour >= 6) {
      hours = (this.dayLength - this.hour) + 6 - 1;
    } else {
      hours = 6 - this.hour - 1;
    }

    let minutes = this.hourLength - this.minute - 1;
    let seconds = this.minuteLength - this.seconds;

    minutes += hours * this.hourLength;
    seconds += minutes * this.minuteLength;

    return seconds;
  }

  static timeUntilDayChange() {
    return this.convertSecondsToTimeString(this.secondsUntilDayChange());
  }

};

Managers.Time.secondDuration = 40;
Managers.Time.yearLength = 4;
Managers.Time.minuteLength = 10;
Managers.Time.hourLength = 6;
Managers.Time.dayLength = 24;
Managers.Time.weekLength = 7;
Managers.Time.monthLength = 31;
Managers.Time.sunRiseStarts = 6;
Managers.Time.sunRiseEnds = 8;
Managers.Time.sunSetStarts = 18;
Managers.Time.sunSetEnds = 20;

Managers.Time.tilesetList = [];

Managers.Time.weekDayOffset = 6;
Managers.Time.pauseClockDuringConversations = true;
Managers.Time.monthNames = ['Spring', 'Summer', 'Fall', 'Winter'];
Managers.Time.monthShortNames = ['SPR', 'SUM', 'FAL', 'WIN'];
Managers.Time.dayLabels = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
Managers.Time.dayShortNames = ['SUN', 'MON', 'TUE', 'WED', 'THU', 'FRI', 'SAT'];

Managers.Time._timeEvents = {};
Managers.Time._afterTimeEvents = {};
Managers.Time._timeEventsNextId = 0;
Managers.Time._afterTimeEventsNextId = 0;
Managers.Time.seconds = 0;
Managers.Time.minute = 0;
Managers.Time.hour = 0;
Managers.Time.day = 1;
Managers.Time.month = 1;
Managers.Time.year = 1;
Managers.Time.dayPeriod = 0;
Managers.Time.weekDay = 0;

Managers.Time.atDate = Managers.Time.onDate;
Managers.Time.atTime = Managers.Time.onTime;
Managers.Time.atDateTime = Managers.Time.onDateTime;
Managers.Time.atWeekDay = Managers.Time.onWeekDay;

// Managers.Time.setInitialDateTime();

Managers.Time.on('changeSecond', Managers.Sound.updateBgm.bind(Managers.Sound));