//-----------------------------------------------------------------------------
// Managers.Text
//
// The static class that handles terms and messages.

Managers.Text = class Text {
  constructor() {
    throw new Error('This is a static class');
  }

  static get currencyUnit() {
    return '$';
  }

  static basic(basicId) {
    return $dataSystem.terms.basic[basicId] || '';
  }

  static param(paramId) {
    return $dataSystem.terms.params[paramId] || '';
  }

  static command(commandId) {
    return $dataSystem.terms.commands[commandId] || '';
  }

  static message(messageId) {
    return $dataSystem.terms.messages[messageId] || '';
  }

  static item(itemId) {
    if (!itemId) return '';

    return t(`item-${itemId}`);
  }

  static achievement(achievementId) {
    if (!achievementId) return '';

    return t(`achievement-${achievementId}`);
  }

  static stamp(stampId) {
    if (!stampId) return '';

    return t(`stamp-${stampId}`);
  }

  static achievementDescription(achievementId) {
    if (!achievementId) return '';

    return t(`achievement-description-${achievementId}`);
  }

  static creature(creatureType) {
    return t(`creature-${creatureType}`);
  }

  static getTipList() {
    const list = [];

    list.push(t("You can tame wild animals if you talk to them or bring them gifts."));
    // list.push(t("There are some hidden items around the world."));
    list.push(t("You can see a list of your friends or animals on the achievements screen."));
    list.push(t("Press the extra key to check the name of the item or animal you're holding."));
    // list.push(t("Place items on top of the round shadows on rainy days to trade them with the dwarf clouds."));
    list.push(t("There's no point in knocking on a door if the house lights are off. There's no one home to open it."));
    list.push(t("When you cut your crops with a sickle, there's a chance to find new seeds to plant it again."));
    list.push(t("When you weeds with a sickle, it'll be placed on your storage automatically."));
    list.push(t("The Hay Maker can turn weeds and wheat into hay."));
    list.push(t("The Feed Maker can turn corn into bird feed."));
    list.push(t("Viktor can upgrade your tools, but you need to use them a lot first."));

    if (Managers.Config.enableMouse) {
      list.push(t("You can use the mouse wheel to quickly change tools."));
    }

    return list;
  }

  static getRandomTip() {
    const list = Managers.Text.getTipList();
    const idx = Math.randomInt(list.length);
    const title = t("Tip:");

    return `${title} ${list[idx]}`;
  }

  static getString(string, returnKey) {
    // Quick cache for all the situations where it calls the method twice
    if (this._lastStringKey == string && returnKey !== false) {
      return this._lastStringTranslation;
    }

    const language = Managers.Text.language || 'english';
    this._lastStringKey = string;

    if (window.$dataStrings) {
      if (window.$dataLanguages) {
        const languageData = $dataLanguages[language] || $dataLanguages['english'];

        if (languageData && languageData.identifiers) {
          for (let languageId of languageData.identifiers) {
            if ($dataStrings[languageId] && $dataStrings[languageId][string]) {
              this._lastStringTranslation = $dataStrings[languageId][string];
              return $dataStrings[languageId][string];
            }
          }
        }
      }

      if ($dataStrings[language]) {
        if ($dataStrings[language][string]) {
          this._lastStringTranslation = $dataStrings[language][string];
          return $dataStrings[language][string];
        }
      }
    }

    this._lastStringTranslation = string;
    if (returnKey === false) {
      return false;
    }

    return string;
  }

  static stringExists(string) {
    return this.getString(string, false);
  }

  static itemExplanation(itemId) {
    const key = `item-explanation-${itemId}`;
    if (!this.stringExists(key)) {
      const description = Managers.Text.itemDescription(itemId);
      if (description === '') {
        return '';
      }
      
      return description;
    }

    return t(key);
  }

  static itemDescription(itemId) {
    const key = `item-description-${itemId}`;

    if (!this.stringExists(key)) {
      return '';
    }
    
    return t(key);
  }

  static getter(method, param) {
    return {
      get() {
        return t(this[method](param));
      },
      configurable: true
    };
  }

  static replaceBasicTextVariables(text) {
    let regex = /<keys:([^>]*)>/;
    let result = regex.exec(text);
    while (result) {
      if (result.length > 1) {
        const keyNames = result[1];
        text = text.replace(`<keys:${keyNames}>`, Managers.CommonEvent.getKeyValue(keyNames, false));
      }

      result = regex.exec(text);
    }

    regex = /<key:([^>]*)>/;
    result = regex.exec(text);
    while (result) {
      if (result.length > 1) {
        const keyName = result[1];
        text = text.replace(`<key:${keyName}>`, Managers.CommonEvent.getKeyValue(keyName, true));
      }

      result = regex.exec(text);
    }

    regex = /<gamepadIcon:([^>]*)>/;
    result = regex.exec(text);
    while (result) {
      if (result.length > 1) {
        const buttonName = result[1];
        text = text.replace(`<gamepadIcon:${buttonName}>`, Managers.CommonEvent.getGamepadIcon(buttonName));
      }

      result = regex.exec(text);
    }

    regex = /<icon:([^>]*)>/;
    result = regex.exec(text);
    while (result) {
      if (result.length > 1) {
        const iconIndex = result[1];
        text = text.replace(`<icon:${iconIndex}>`, `\\hi[${iconIndex}]`);
      }

      result = regex.exec(text);
    }

    regex = /<color:([^>]*)>/;
    result = regex.exec(text);
    while (result) {
      if (result.length > 1) {
        const colorName = result[1];
        text = text.replace(`<color:${colorName}>`, `\\c[${Managers.Text.getColorValue(result[1])}]`);
      }

      result = regex.exec(text);
    }

    text = text.replace(/<\/color>/g, `\\c[100]`);

    text = text.replace(/<close>/g, '\\^');
    text = text.replace(/<wait>/g, '\\.');
    text = text.replace(/<longwait>/g, '\\|');
    text = text.replace(/<bigger>/g, '\\{');
    text = text.replace(/<smaller>/g, '\\}');

    text = text.replace(/<lastCreatureName>/g, Managers.Creatures.lastCreatureName);

    if (Managers.Time.isMorning()) {
      text = text.replace(/<good morning>/g, Managers.Text.translateOnly('good morning'));
      text = text.replace(/<Good morning>/g, Managers.Text.translateOnly('Good morning'));
    } else if (Managers.Time.isAfternoon()) {
      text = text.replace(/<good morning>/g, Managers.Text.translateOnly('good afternoon'));
      text = text.replace(/<Good morning>/g, Managers.Text.translateOnly('Good afternoon'));
    } else if (Managers.Time.isEvening()) {
      text = text.replace(/<good morning>/g, Managers.Text.translateOnly('good evening'));
      text = text.replace(/<Good morning>/g, Managers.Text.translateOnly('Good evening'));
    } else if (Managers.Time.isNight()) {
      text = text.replace(/<good morning>/g, Managers.Text.translateOnly('good night'));
      text = text.replace(/<Good morning>/g, Managers.Text.translateOnly('Good night'));
    }

    return text;
  }

  static replaceTextVariables(text) {
    if (text.indexOf('<') >= 0) {
      text = Managers.Text.replaceBasicTextVariables(text);
      text = text.replace(/<task>/g, Managers.Text.basicTranslation(Managers.Tasks.lastTaskName || ''));
      text = text.replace(/<taskDesc>/g, Managers.Text.basicTranslation(Managers.Tasks.lastTaskDescription || ''));

      if (window.$dataLanguages) {
        const settings = $dataLanguages[Managers.Text.language];
        if (settings && settings.variables) {
          for (let variableName in settings.variables) {
            const regex = new RegExp(`<${ variableName }>`, 'g');
            let value;

            if (settings.variables[variableName]) {
              const gender = Managers.Player.getGender() || 'male';
              value = settings.variables[variableName][gender] || '';
            } else {
              value = variableName;
            }

            text = text.replace(regex, value || '');
          }
        }
      }

    }
    
    return text;
  }

  static logMissingTranslation(key) {
    Managers.Text.missingTranslations.push(key);
  }

  static getFinalString(key) {
    const data = this.getString(key);

    if (!data) {
      return false;
    }

    if (typeof data == 'object') {
      const gender = Managers.Player.getGender();

      if (data[gender]) {
        return data[gender];
      } else {
        return key;
      }
    }

    return data;
  }

  static translateOnly(key, defaultValue) {
    if (this.stringExists(key)) {
      return this.getFinalString(key);
    }

    const gender = `_${Managers.Player.getGender()}`;

    if (this.stringExists(key + gender)) {
      return this.getFinalString(key + gender);
    }

    if (key.trim() !== '' && !Managers.Text.missingTranslations.includes(key)) {
      Managers.Text.logMissingTranslation(key);
    }

    if (defaultValue === undefined) {
      return key;
    }

    return defaultValue || '';
  }

  static basicTranslation(key) {
    if (key.startsWith('<translated>')) {
      return key.substr(12);
    }

    const text = Managers.Text.translateOnly(key);
    return Managers.Text.replaceBasicTextVariables(text);
  }

  static translate(key) {
    if (key.startsWith('<translated>')) {
      return key.substr(12);
    }

    const translated = Managers.Text.translateOnly(key);
    return Managers.Text.replaceTextVariables(translated);
  }

  static logMissingItemTranslations() {
    const len = $dataItems.length;
    for (let i = 0; i < len; i++) {
      const name = Managers.Text.item($dataItems[i].id);
      const desc = Managers.Text.itemDescription($dataItems[i].id);

      if (name == desc) {
        console.log(...log(`Repeated description for item: ${name}`));
      } else {
        if (name.includes('item-')) {
          console.log(...log(`Missing translation for item: ${name}`));
        }
      }
    }
  }

  static getColorValue(colorName) {
    switch(colorName) {
      case 'important' :
      case 'item' :
        return 1;

      case 'name' :
      case 'task' :
      case 'positive':
        return 2;

      case 'negative':
        return 3;

      case 'letter' :
      case 'black' :
      case 'message':
        return 4;
      case 'default':
        return 100;

      case 'blue':
        return 2;
      case 'red':
        return 3;
      case 'base':
        return 5;
      case 'dark':
        return 6;

      default :
        return 0;
    }
  }

  static getThousandSeparator() {
    if (window.$dataLanguages) {
      const settings = $dataLanguages[Managers.Text.language];

      if (settings && settings.hasOwnProperty('thousandSeparator')) {
        return settings.thousandSeparator;
      }
    }

    return ',';
  }

  static formatMoney(moneyValue) {
    let result = '';
    moneyValue = Math.floor(moneyValue);

    if (moneyValue < 0) {
      result += '-';
      moneyValue = Math.abs(moneyValue);
    }

    result += moneyValue.toString().replace(/(\d)(?=(\d{3})+$)/g, `$1${Managers.Text.getThousandSeparator()}`);
    return `${result}${this.currencyUnit}`;
  }

  static changeLanguage(language) {
    this.language = language.toLowerCase();
  }

  static translateAge(number, type) {
    if (number == 0) {
      return t("0 Days");
    }

    if (number == 1) {
      switch(type) {
        case 'days':
          return t("1 Day");
        case 'weeks':
          return t("1 Week");
        case 'months':
          return t("1 Month");
        default:
          return t("1 Year");
      }
    }

    switch(type) {
      case 'days':
        return t("{ageNumber} Days").replace('{ageNumber}', number);
      case 'weeks':
        return t("{ageNumber} Weeks").replace('{ageNumber}', number);
      case 'months':
        return t("{ageNumber} Months").replace('{ageNumber}', number);
      default:
        return t("{ageNumber} Years").replace('{ageNumber}', number);
    }
  }
};

Managers.Text.missingTranslations = [];
Managers.Text.language = 'english';

window.t = Managers.Text.translate;