//-----------------------------------------------------------------------------
// TaskManager
//

const EventListener = require('engine/helpers/EventListener');
Managers.Tasks = class Tasks extends EventListener {
  static getTaskData(taskId) {
    let data = this.customTasks[taskId];
    if (data) {
      data.taskId = taskId;
      return data;
    }

    data = $dataTasks[taskId];
    if (data) {
      data.taskId = taskId;
      return data;
    }

    return false;
  }

  static clear() {
    this.taskData = {};
    this.customTasks = {};
    this.lastTaskId = '';
    this.lastTaskName = '';
    this.lastTaskDescription = '';
  }

  static isTaskKnown(taskId) {
    return this.taskData[taskId] && this.taskData[taskId].status && this.taskData.status !== TaskStatus.NONE;
  }

  static getTaskStatus(taskId) {
    if (this.taskData[taskId]) {
      return this.taskData[taskId].status;
    } else {
      return TaskStatus.NONE;
    }
  }

  static isTaskComplete(taskId) {
    const status = this.getTaskStatus(taskId);
    return status == TaskStatus.COMPLETED;
  }

  static getTaskGroupStatus(taskGroupId) {
    let started = false;
    let completed = true;

    for (let key in $dataTasks) {
      const taskData = $dataTasks[key];
      if (!taskData) continue;

      const id = taskData.id || key;
      if (id !== taskGroupId) continue;

      const status = Managers.Tasks.getTaskStatus(key);
      if (status === TaskStatus.COMPLETED) {
        started = true;
      } else {
        completed = false;

        if (status == TaskStatus.ACCEPTED) {
          started = true;
        }
      }
    }

    if (started && completed) {
      return TaskStatus.COMPLETED;
    }

    if (started) {
      return TaskStatus.ACCEPTED;
    }

    return TaskStatus.NONE;
  }

  static getTaskGroupTitle(taskGroupId) {
    let taskTitle = '';

    for (let key in $dataTasks) {
      const taskData = $dataTasks[key];
      if (!taskData) continue;

      const id = taskData.id || key;
      if (id !== taskGroupId) continue;

      if (taskData.groupTitle) {
        return t(taskData.groupTitle);
      }

      if (key == id) {
        return t(taskData.title);
      }

      if (!taskTitle) {
        taskTitle = t(taskData.title);
      }
    }

    return taskTitle || t(taskGroupId);
  }

  static getTaskGroupVillager(taskGroupId) {
    let taskVillager = '';

    for (let key in $dataTasks) {
      const taskData = $dataTasks[key];
      if (!taskData) continue;

      const id = taskData.id || key;
      if (id !== taskGroupId) continue;

      if (taskData.groupVillagerName) {
        return taskData.groupVillagerName;
      }

      if (key == id) {
        return taskData.villagerName;
      }

      if (!taskVillager) {
        taskVillager = taskData.villagerName;
      }
    }

    return taskVillager || '';
  }

  static tasksAndStatusesMatch(tasks, statuses) {
    for (let i = 0; i < tasks.length; i++) {
      const status = this.getTaskStatus(tasks[i]);

      if (!statuses.includes(status)) return false;
    }

    return true;
  }

  static countTaskStatus(status) {
    let count = 0;

    for (const taskId in this.taskData) {
      const data = this.taskData[taskId];
      if (!data) continue;

      if (data.status == status) {
        count++;
      }
    }

    return count;
  }

  static setTaskStatus(taskId, status) {
    if (!this.taskData[taskId]) {
      this.taskData[taskId] = {
        status,
        acceptDate : 0,
        completeDate : 0
      };
    }

    this.taskData[taskId] = {};
    this.taskData[taskId].status = status;
    
    if (status == TaskStatus.NONE) {
      this.taskData[taskId].acceptDate = 0;
      this.taskData[taskId].completeDate = 0;
    } else if (status == TaskStatus.ACCEPTED) {
      this.taskData[taskId].acceptDate = Managers.Time.totalDays;
    } else if (status == TaskStatus.COMPLETED) {
      this.taskData[taskId].completeDate = Managers.Time.totalDays;
    }

    const data = this.getTaskData(taskId);
    if (data) {
      this.lastTaskId = taskId;
      this.lastTaskName = data.title;
      this.lastTaskDescription = data.description;
    }
  }

  static getActiveTaskGroups() {
    const list = [];

    for (const taskId in this.taskData) {
      if (!this.taskData.hasOwnProperty(taskId)) continue;

      const data = this.getTaskData(taskId);
      if (!data) continue;
      if (data.hidden) continue;

      const groupId = data.id;
      if (list.indexOf(groupId) >= 0) continue;

      const status = this.getTaskGroupStatus(groupId);
      if (status == TaskStatus.NONE) continue;

      list.push(groupId);
    }

    return list;
  }

  static getActiveTasks(includeCompleted = false) {
    const list = [];

    for (const taskId in this.taskData) {
      if (!this.taskData.hasOwnProperty(taskId)) continue;
      const status = this.taskData[taskId].status;

      if (status != TaskStatus.ACCEPTED) {
        if (!includeCompleted) continue;

        if (status != TaskStatus.COMPLETED) continue;
      }

      list.push(taskId);
    }

    return list;
  }

  static hasAnyActiveTask() {
    for (const taskId in this.taskData) {
      if (!this.taskData.hasOwnProperty(taskId)) continue;
      if (this.taskData[taskId].status != TaskStatus.ACCEPTED) continue;

      return true;    
    }

    return false;
  }

  static replaceTask(oldTaskId, newTaskId) {
    this.completeTask(oldTaskId);
    this.setTaskStatus(newTaskId, TaskStatus.ACCEPTED);
    this.updateVariables();

    Managers.History.registerAcceptedTask(newTaskId);
  }

  static refuseTask(taskId) {
    this.setTaskStatus(taskId, TaskStatus.CANCELED);
  }

  static acceptTask(taskId, useChildEvent) {
    this.setTaskStatus(taskId, TaskStatus.ACCEPTED);
    this.updateVariables();

    if (useChildEvent === undefined) useChildEvent = true;

    Managers.History.registerAcceptedTask(taskId);

    if (useChildEvent) {
      Managers.CommonEvent.playChildEvent('new_task', $gameMap._interpreter);
    } else {
      Managers.CommonEvent.playEvent('new_task');
    }
  }

  static completeTaskAnimation(taskId) {
    if (this.getTaskStatus(taskId) == TaskStatus.COMPLETED) return;

    const data = this.getTaskData(taskId);
    if (data) {
      this.lastTaskId = taskId;
      this.lastTaskName = data.title;
      this.lastTaskDescription = data.description;

      // If there is not onComplete event, give the rewards before the task popup,
      // if there is an event, then leave this for after the popup so it doesn't get in the way
      if (!data.onComplete) {
        this.completeTask(taskId);
      }
      
      Managers.CommonEvent.playEvent('task_complete');
    }
  }

  static completeLasTask() {
    this.completeTask(this.lastTaskId);
  }

  static completeTask(taskId) {
    if (this.getTaskStatus(taskId) == TaskStatus.COMPLETED) return;

    this.setTaskStatus(taskId, TaskStatus.COMPLETED);

    const data = this.getTaskData(taskId);
    const {request, reward, onComplete } = data;

    if (onComplete) {
      Managers.CommonEvent.playEvent(onComplete);
    }

    Managers.History.registerCompletedTask(taskId);

    if (request) {
      for (const requestItem of request) {
        if (!requestItem || requestItem.type != 'item') {
          continue;
        }

        if (requestItem.deliver) {
          Managers.Items.deliverItemId(requestItem.id, requestItem.amount);
        } else if (requestItem.giveaway) {
          Managers.Items.loseItemId(requestItem.id, requestItem.amount);
        }
      }
    }

    if (!reward) return;

    for (const rewardItem of reward) {
      const amount = rewardItem.amount;

      switch (rewardItem.type) {
        case 'item' :
          Managers.Items.forceGainItemId(rewardItem.id, amount);
          break;
        case 'friendship' :
          Managers.Relationship.increaseFriendship(rewardItem.villager, amount);
          break;
        case 'gold' :
          Managers.Items.gainGold(amount);
          break;
        case 'achievement' :
          Managers.Steam.activateAchievement(rewardItem.id);
          break;
        case 'recipe':
          Managers.Cooking.registerLearntRecipe(rewardItem.id);
          break;
        case 'switch':
          Switches[rewardItem.id] = true;
          break;
      }
    }

    Managers.Achievements.checkAchievements();
  }

  static checkIfRequestIsDone(request) {
    for (const requestItem of request) {
      const amount = requestItem.amount;

      switch(requestItem.type) {
        case 'item' :
          if (Managers.Items.numItemsById(requestItem.id) >= amount) continue;
          
          return false;

        case 'gold' :
          if (Managers.Items.gold >= amount) continue;

          return false;

        case 'friendship' :
          // #ToDo
          return false;

        case 'switch' :
          if (Switches[requestItem.id]) continue;

          return false;

        default :
          return false;
      }
    }

    return true;
  }

  static _checkAutoCompleteTasks(villagerName, eventData) {
    const tasks = this.getActiveTasks();

    for (let i = 0; i < tasks.length; i++) {
      const data = this.getTaskData(tasks[i]);
      if (!data) continue;
      if (!data.autoComplete) continue;

      if (data.autoComplete.villager.toLowerCase() !== villagerName.toLowerCase()) continue;

      const request = data.request;
      if (!request) continue;

      if (this.checkIfRequestIsDone(request)) {
        if (data.autoComplete.event) {
          Managers.Villagers.playDialogue(eventData, data.autoComplete.event);
          // Managers.CommonEvent.playEvent(data.autoComplete.event);
          return true;
        } else {
          this.completeTask(tasks[i]);
        }
      }
    }

    return false;
  }

  static checkAutoCompleteTasks(villagerName, eventData) {
    if (Managers.Content.preventAutoCompleteTasks(villagerName, eventData)) {
      return;
    }

    if (Managers.Content.checkAutoCompleteTasks(villagerName, eventData)) {
      return true;
    }

    return this._checkAutoCompleteTasks(villagerName, eventData);
  }

  static checkIfTaskRequestIsDone(taskId) {
    const data = this.getTaskData(taskId);
    if (!data) return false;

    const request = data.request;
    if (!request) return false;

    if (this.checkIfRequestIsDone(request)) {
      return true;
    }

    return false;
  }

  static checkTaskDialogueCondition(condition) {
    return MVCommons.safeEval(condition);
  }

  static _checkTaskDialogues(villagerName, eventData) {
    const tasks = this.getActiveTasks();

    for (let i = 0; i < tasks.length; i++) {
      const data = this.getTaskData(tasks[i]);
      if (!data) continue;
      if (!data.dialogues) continue;

      const dialogues = data.dialogues[villagerName];
      if (!dialogues) continue;

      for (const eventName in dialogues) {
        if (!dialogues.hasOwnProperty(eventName)) continue;
        if (!Managers.CommonEvent.eventExists(eventName)) break;

        const dialogueData = dialogues[eventName];
        if (dialogueData.condition) {
          if (!this.checkTaskDialogueCondition(dialogueData.condition)) break;
        }

        Managers.Villagers.playDialogue(eventData, eventName);
        return true;
      }
    }

    return false;
  }

  static checkTaskDialogues(villagerName, eventData) {
    if (Managers.Content.preventTaskDialogues(villagerName, eventData)) {
      return;
    }

    if (Managers.Content.checkTaskDialogues(villagerName, eventData)) {
      return true;
    }

    return this._checkTaskDialogues(villagerName, eventData);
  }

  static addCustomTask(taskId, title, description, request, reward, autoCompleteData, requestEvent) {
    this.addCustomTaskObject(taskId, {
      title,
      description,
      request,
      reward,
      autoComplete : autoCompleteData,
      requestEvent
    });
  }

  static addCustomTaskObject(taskId, taskData) {
    this.customTasks[taskId] = taskData;
    this.setTaskStatus(taskId, TaskStatus.NONE);
    this.updateVariables();
  }

  static clearCustomTasks() {
    for (const taskId in this.customTasks) {
      if (!this.customTasks.hasOwnProperty(taskId)) continue;
      const status = this.getTaskStatus(taskId);
      
      if (status == TaskStatus.ACCEPTED) {
        this.setTaskStatus(taskId, TaskStatus.NONE);
      }
    }

    this.customTasks = {};
    this.updateVariables();
  }

  static countCustomTasks() {
    let count = 0;
    for (const taskId in this.customTasks) {
      if (!this.customTasks.hasOwnProperty(taskId)) {
        continue;
      }

      const status = this.getTaskStatus(taskId);
      if (status === TaskStatus.NONE) {
        count++;
      }
    }

    return count;
  }

  static getAvailableCustomTasks() {
    const tasks = [];

    for (const taskId in this.customTasks) {
      if (!this.customTasks.hasOwnProperty(taskId)) continue;
      const status = this.getTaskStatus(taskId);

      if (status !== TaskStatus.NONE) continue;
      tasks.push(taskId);
    }

    return tasks;
  }

  static openAutoTaskBy(requester) {
    for (const taskId in this.customTasks) {
      if (!this.customTasks.hasOwnProperty(taskId)) continue;
      const status = this.getTaskStatus(taskId);

      if (status != TaskStatus.NONE) continue;
      const taskData = this.getTaskData(taskId);
      if (!taskData) continue;
      if (taskData.requester != requester) continue;

      Managers.CommonEvent.playEvent(taskData.requestEvent);
      return;
    }
  }

  static getAllTasksOnGroup(taskGroupId) {
    const list = [];

    for (let key in $dataTasks) {
      const taskData = $dataTasks[key];
      if (!taskData) continue;

      const id = taskData.id || key;
      if (id !== taskGroupId) continue;

      list.push(taskData);
    }

    return list;
  }

  static getGroupTaskDate(taskGroupId, completed = false) {
    const tasks = this.getAllTasksOnGroup(taskGroupId);
    if (!tasks || !tasks.length) return 0;

    let firstTask = false;
    let lastTask = false;
    for (let task of tasks) {
      if (task.firstTask) {
        firstTask = task;
      }

      if (task.lastTask) {
        lastTask = task;
      }
    }

    if (!firstTask) {
      firstTask = tasks[0];
    }
    if (!lastTask) {
      lastTask = tasks[tasks.length - 1];
    }

    const relevantTask = (completed && lastTask) ? lastTask : ((!completed && firstTask) ? firstTask : lastTask);
    if (!relevantTask) return 0;

    if (completed) {
      return Managers.History.getTaskCompleteDate(relevantTask.taskId);
    }

    return Managers.History.getTaskAcceptDate(relevantTask.taskId);
  }

  static updateVariables() {
    Variables.customTaskCount = this.countCustomTasks();
    Switches.hasCustomTasks = Variables.customTaskCount > 0;
  }
};