require('engine/core/Utils');

Managers.Steam = class Steam {
  constructor() {
    throw new Error('This is a static class');
  }

  static initialize() {
    this.initialized = false;

    if (!window.fs) {
      return;
    }

    if (Utils._isAutoTestMode) {
      return;
    }

    try {
      const projectPath = Utils.getProjectFolder();
      const fullPath = Wrapper.joinFileNames(projectPath, 'greenworks');

      this.greenworks = require(fullPath);
    }
    catch(e) {
      console.error(e);
      this.greenworks = false;
      return;
    }

    this.initialized = this.greenworks.initAPI();

    if (!this.initialized) {
      console.log(...log('Greenworks failed to initialize.'));
      return;
    }

    this.steamId = this.greenworks.getSteamId();
    console.log('Steam User: ', this.steamId.screenName);

    this.greenworks.on('game-overlay-activated', (is_active) => {
      this.onSteamOverlay(is_active);
    });
  }

  static getScreenName() {
    if (this.initialized) {
      return this.steamId.screenName;
    }

    return 'Play Test';
  }

  static getUILanguage() {
    if (this.initialized) {
      return this.greenworks.getCurrentUILanguage();
    }
    return 'english';
  }

  static getGameLanguage() {
    if (this.initialized) {
      return this.greenworks.getCurrentGameLanguage();
    }

    // On windows we can detect if it is a brazilian computer and then set the language accordingly
    const countryCode = Utils.getCountryCode();
    if (countryCode == 55) return 'brazilian';

    return 'english';
  }

  static canGetAchievement(achievementName) {
    if (!Managers.Achievements.isAchievementPending(achievementName)) return false;

    return true;
  }

  static _achievementSuccess() {
    console.log(...log('Achievement changed', arguments));
  }

  static _achievementError() {
    console.log(...log('Achievement activation error', arguments));
  }

  static activateAchievement(achievementName) {
    if (!this.canGetAchievement(achievementName)) return;

    if (!achievementName) {
      console.log(...log('Achievement name not provided.'));
      return;
    }

    if (!this.isSteamRunning()) {
      console.log(...log(`Can't activate achievement: ${ achievementName }`));
      // console.log(...log('Steam isn\'t running'));
      return;
    }

    this.greenworks.activateAchievement(achievementName, function(){
      console.log(...log('Achievement Activated', achievementName));

      Managers.Achievements.registerActiveAchievement(achievementName);
    }, this._achievementError);
  }

  static clearAchievement(achievementName) {
    if (!achievementName) {
      console.log(...log('Achievement name not provided.'));
      return;
    }

    if (!this.isSteamRunning()) {
      console.log(...log(`Can't clear achievement: ${ achievementName }`));
      return;
    }

    this.greenworks.clearAchievement(achievementName, this._achievementSuccess, this._achievementError);
  }

  static isSteamRunning() {
    if (this.initialized && this.greenworks.isSteamRunning) {
      return this.greenworks.isSteamRunning();
    }

    return this.initialized;
  }

  static getFriendCount() {
    if (this.initialized) {
      return this.greenworks.getFriendCount(this.greenworks.FriendFlags.Immediate);
    }

    return 0;
  }

  static isCloudEnabled() {
    if (this.initialized) {
      return this.greenworks.isCloudEnabled();
    }

    return false;
  }

  static isCloudEnabledForUser() {
    if (this.initialized) {
      return this.greenworks.isCloudEnabledForUser();
    }

    return false;
  }

  static _processSteamSaves(cloudFiles, extraLocalFiles, callback) {
    const successful = [];
    const failed = [];
    const ignored = [];

    if (!this.syncStates) {
      this.syncStates = {};
    }

    let folderPath;
    try {
      folderPath = Managers.Storage.localFileDirectoryPath(true, true);
      if (!Wrapper.existsSync(folderPath)) {
        window.fs.mkdirSync(folderPath);
      }
    }
    catch (e) {
      return;
    }

    const checkResult = () => {
      if (successful.length + failed.length + ignored.length == cloudFiles.length) {
        callback({
          successful,
          failed,
          ignored
        });
      }
    };

    for (const fileName of cloudFiles) {
      this.greenworks.readTextFromFile(fileName, (content) => {
        const fullPath = Wrapper.joinFileNames(folderPath, fileName);
        let data;

        try {
          data = JsonEx.parse(content);
        } catch (e) {
          failed.push(fileName);
          this.syncStates[fileName] = {
            success: false,
            source: 'cloud',
            filePath: fullPath,
            fileName
          };
          return checkResult();
        }

        const cachedData = Engine.Data.getCachedSaveData(fullPath);
        let ignore = true;
        let ignoreSuccess = false;

        if (!Wrapper.existsSync(fullPath)) {
          ignore = false;
        } else if (cachedData && cachedData.info && cachedData.info.timestamp && data && data.info) {
          if (cachedData.info.timestamp < data.info.timestamp) {
            ignore = false;
          } else if (cachedData.info.timestamp == data.info.timestamp) {
            ignoreSuccess = true;
          }
        }

        if (ignore) {
          ignored.push(fileName);
          this.syncStates[fileName] = {
            success: ignoreSuccess,
            source: 'cloud'
          };
        } else {
          window.fs.writeFileSync(fullPath, content);
          Engine.Data.cacheSaveData(fullPath, data);

          successful.push(fileName);
          this.syncStates[fileName] = {
            success : true
          };
        }

        return checkResult();
      }, (err) => {
        failed.push(fileName);
        this.syncStates[fileName] = {
          success: false,
          source: 'cloud',
          fileName,
          err
        };
        return checkResult();
      });
    }

    for (let localFile of extraLocalFiles) {
      const fullPath = Wrapper.joinFileNames(folderPath, localFile);

      if (!Wrapper.existsSync(fullPath)) {
        continue;
      }

      const content = Managers.Storage.loadLocalFile(fullPath);
      this.saveFileOnCloud(localFile, content);
    }

    return true;
  }

  static isCloudUp() {
    if (!this.initialized) {
      return false;
    }

    // Check if user has cloud enabled in their account
    if (!this.greenworks.isCloudEnabledForUser()) {
      return false;
    }

    // Check if user has cloud enabled for orange season
    if (!this.greenworks.isCloudEnabled()) {
      return false;
    }

    return true;
  }

  static deleteFileOnCloud(filePath) {
    if (!this.isCloudUp()) {
      return false;
    }

    const folderPath = Managers.Storage.localFileDirectoryPath(true);
    if (!filePath.startsWith(folderPath)) {
      return false;
    }

    const fileName = Utils.getFileNameFromPath(filePath);

    this.greenworks.deleteFile(fileName, () => {
      console.info('Deleted file from Steam Cloud: ', fileName);
    }, (err) => {
      console.log(...log('Failed to delete file from cloud'));
      console.error(err);
    });
  }

  static saveFileOnCloud(filePath, json) {
    if (!this.isCloudUp()) {
      return false;
    }

    if (!this.syncStates) {
      this.syncStates = {};
    }

    const fileName = Utils.getFileNameFromPath(filePath);
    if (this.syncStates[fileName]) {
      delete this.syncStates[fileName];
    }

    this.greenworks.saveTextToFile(fileName, json, () => {
      this.syncStates[fileName] = {
        success : true
      };
    }, (err) => {
      console.log(...log('Failed to save data on cloud', filePath, fileName));
      console.error(err);

      this.syncStates[fileName] = {
        success: false,
        source: 'local',
        filePath,
        fileName,
        err
      };
    });
  }

  static getCloudFiles() {
    if (!this.isCloudUp()) {
      return false;
    }

    const fileCount = this.greenworks.getFileCount();
    if (!fileCount) {
      return false;
    }

    const fileList = [];

    for (let index = 0; index < fileCount; index++) {
      const data = this.greenworks.getFileNameAndSize(index);
      if (!data) {
        continue;
      }

      fileList.push(data);
    }

    return fileList;
  }

  static syncAllSaves(callback) {
    if (!this.initialized) {
      return false;
    }

    // Check if user has cloud enabled in their account
    if (!this.greenworks.isCloudEnabledForUser()) {
      return false;
    }

    // Check if user has cloud enabled for orange season
    if (!this.greenworks.isCloudEnabled()) {
      return false;
    }

    const fileCount = this.greenworks.getFileCount();
    if (!fileCount) {
      return false;
    }

    const saveList = [];
    const localSaves = Managers.Storage.getAllExistingLocalSaves(false, true).map((fullPath) => {
      return Utils.getFileNameFromPath(fullPath);
    });

    for (let index = 0; index < fileCount; index++) {
      const data = this.greenworks.getFileNameAndSize(index);
      if (!data) {
        continue;
      }

      if (data.name.endsWith('.ffsave')) {
        saveList.push(data.name);

        if (localSaves.indexOf(data.name) >= 0) {
          localSaves.splice(localSaves.indexOf(data.name), 1);
        }

        continue;
      }
    }

    if (!saveList.length) {
      return false;
    }

    return this._processSteamSaves(saveList, localSaves, callback);
  }

  static onSteamOverlay(state) {
    this.showingOvelay = state;
  }
};