const EventListener = require('engine/helpers/EventListener');

Managers.Relationship = class Relationship extends EventListener {
  static clear() {
    this.currentGiftReceiver = '';
    this.data = {
      villagersMetToday: [],
      knownVillagers: [],
      friendship: {},
      villagersFriendships: {},
      interestType: {},
      gifts: {},
      giftDates: {}
    };
  }

  static setData(data) {
    this.clear();

    if (!data) {
      return;
    }

    if (data.villagersMetToday !== undefined) {
      this.data.villagersMetToday = data.villagersMetToday;
    }

    if (data.knownVillagers !== undefined) {
      this.data.knownVillagers = data.knownVillagers;
    }

    if (data.friendship !== undefined) {
      this.data.friendship = data.friendship;
    }

    if (data.villagersFriendships !== undefined) {
      this.data.villagersFriendships = data.villagersFriendships;
    }

    if (data.interestType !== undefined) {
      this.data.interestType = data.interestType;
    }

    if (data.gifts !== undefined) {
      this.data.gifts = data.gifts;
    }

    if (data.giftDates !== undefined) {
      this.data.giftDates = data.giftDates;
    }
  }

  static getData(data) {
    return this.data;
  }

  static updateMetEveryoneSwitch() {
    if (Switches.metAllVillagers) {
      return;
    }

    if (!Managers.Tasks.isTaskKnown('meet-everyone')) {
      return;
    }

    // We don't load this automatically because not all villagers are considered on this quest
    const allVillagers = [
      'Amanda', 'Benjamin', 'Billy', 'Bonnie', 'Brittany', 'Chloe', 'Cindy', 'Devin',
      'Gabriel', 'Ilda', 'Julia', 'Karl', 'Lucas', 'Mia', 'Nathalia', 'Phi', 'Raphael',
      'Richard', 'Rory', 'Serge', 'Stella', 'Thalia', 'Viktor'
    ];

    if (this.data.knownVillagers.length < allVillagers.length) {
      return;
    }

    for (let villagerName of allVillagers) {
      if (!this.isCharacterKnown(villagerName)) {
        return;
      }
    }

    if (Managers.Tasks.isTaskKnown('meet-everyone')) {
      $gameSystem.runWhenAvailable(() => {
        Managers.Tasks.completeTaskAnimation('meet-everyone');
        Switches.metAllVillagers = true;
      });
    }
  }

  static talkToVillager(villagerName) {
    if (!this.data.villagersMetToday.includes(villagerName)) {
      this.data.villagersMetToday.push(villagerName);
    }

    if ($gameParty.size() > 1) {
      const companionName = $gameParty.getCompanionName();

      //Only increase friendship automatically when the two already know each other
      if (this.haveVillagersMetEachOther(villagerName, companionName)) {
        this.increaseVillagerFriendship(villagerName, companionName, 1, true);
      }
    }
  }

  static getFriendship(villagerName) {
    const customFriendship = Managers.Content.getVillagerFriendship(villagerName);
    if (customFriendship !== undefined) {
      return customFriendship;
    }

    if (Managers.Relationship.isVillagerBridgesBurned(villagerName)) {
      return 0;
    }

    if (this.data.friendship[villagerName] === undefined) {
      this.data.friendship[villagerName] = 0;
    }

    return this.data.friendship[villagerName];
  }

  static decreaseFriendship(villagerName, points) {
    this.increaseFriendship(villagerName, -points);
  }

  static isVillagerBridgesBurned(villagerName) {
    return Managers.Content.isVillagerBridgesBurned(villagerName) === true;
  }

  static increaseFriendship(villagerName, points) {
    if (Managers.Content.canIncreaseFriendship(villagerName) === false) {
      return;
    }

    if (Managers.Relationship.isVillagerBridgesBurned(villagerName)) {
      return;
    }

    if (points === undefined || points === null || isNaN(points)) points = 1;

    if (!this.data.friendship[villagerName]) {
      this.data.friendship[villagerName] = 0;
    }

    this.data.friendship[villagerName] += points;
  }

  static increaseVillagerFriendship(villager1Name, villager2Name, points, replicate) {
    if (points === undefined) points = 1;
    if (this.data.villagersFriendships[villager1Name] === undefined) {
      this.data.villagersFriendships[villager1Name] = {};
    }
    if (this.data.villagersFriendships[villager1Name][villager2Name] === undefined) {
      this.data.villagersFriendships[villager1Name][villager2Name] = 0;
    }

    this.data.villagersFriendships[villager1Name][villager2Name] += points;

    if (replicate) {
      this.increaseVillagerFriendship(villager2Name, villager1Name, points, false);
    }
  }

  static decreaseVillagerFriendship(villager1Name, villager2Name, points, replicate) {
    this.increaseVillagerFriendship(villager1Name, villager2Name, -points, replicate);
  }

  static haveVillagersMetEachOther(villager1Name, villager2Name) {
    if (this.data.villagersFriendships[villager1Name] !== undefined) {
      if (this.data.villagersFriendships[villager1Name][villager2Name] !== undefined) {
        return true;
      }
    }

    if (this.data.villagersFriendships[villager2Name] !== undefined) {
      if (this.data.villagersFriendships[villager2Name][villager1Name] !== undefined) {
        return true;
      }
    }

    return false;
  }

  static metToday(villagerName) {
    return this.data.villagersMetToday.includes(villagerName);
  }

  static processNewDay() {
    //If the player went to bed
    if ($gameSystem.isPlayerSleeping && !$gameSystem.isPlayerSleepingStanding && !$gameSystem.isPlayerPassedOut) {
      // And there's someone on the party
      if ($gameParty.hasCompanion()) {
        //Take them out and lose a few friendship points due to awkwardness
        const companionName = $gameParty.getCompanionName();
        this.decreaseFriendship(companionName, 50);
        $gameParty.removeVillager(companionName);
      }
    }

    // Decreases 1 point of friendship from everybody that the player didn't speak to today
    for (const villagerName of this.data.knownVillagers) {
      if (this.metToday(villagerName)) {
        this.increaseFriendship(villagerName, 14);
      } else {
        this.decreaseFriendship(villagerName, 1);
      }
    }

    this.data.villagersMetToday = [];
  }

  static isCharacterKnown(characterName) {
    return this.data.knownVillagers.includes(characterName);
  }

  static getKnownVillagers(filterFn) {
    // Shallow Copy
    const copy = [].concat(this.data.knownVillagers);

    if (filterFn) {
      return copy.filter(filterFn);
    }

    return copy;
  }

  static getUnsortedKnownVillagers(filterForMenu = true) {
    const filterFn = filterForMenu ? (friend) => {
      const data = Managers.Villagers.getActorData(friend);
      return Boolean(data && data.showOnFriendList);
    } : undefined;

    return this.getKnownVillagers(filterFn);
  }

  static getSortedKnownVillagers(filterForMenu = true) {
    const villagers = this.getUnsortedKnownVillagers(filterForMenu);

    villagers.sort((a, b) => this.getFriendship(b) - this.getFriendship(a));

    return villagers;
  }

  static getMenuVillagers() {
    const villagers = this.getSortedKnownVillagers(true);

    for (const actor of $dataActors) {
      if (!actor) {
        continue;
      }

      if (villagers.includes(actor.name)) {
        continue;
      }

      if (actor.showOnFriendList !== true) {
        continue;
      }

      villagers.push(actor.name);
    }

    return villagers;
  }

  static markCharacterAsKnown(characterName) {
    if (this.data.knownVillagers.includes(characterName)) return;
    
    this.data.knownVillagers.push(characterName);
    this.data.interestType[characterName] = 'friendship';

    this.updateMetEveryoneSwitch();
  }

  static getInterestType(characterName) {
    return this.data.interestType[characterName] || 'friendship';
  }

  static getFriendshipLevels() {
    return Constants.FriendshipLevels;
  }

  static getMaxFriendship() {
    return Constants.FriendshipLevels[Constants.FriendshipLevels.length -1];
  }

  static friendshipToHeartNumber(friendship=0) {
    const levels = this.getFriendshipLevels();
    let heartPieces = 0;

    while (heartPieces < 10 && levels[heartPieces] <= friendship) {
      heartPieces++;
    }

    return heartPieces / 2;
  }

  static getHeartsNum(characterName) {
    const friendship = this.getFriendship(characterName);
    return this.friendshipToHeartNumber(friendship);
  }

  static getFriendList(filterFn) {
    const villagers = this.getSortedKnownVillagers();
    const friends = [];

    for (let i = 0; i < villagers.length; i++) {
      if (villagers[i].toLowerCase() == 'mom') continue;

      if (filterFn) {
        if (!filterFn(villagers[i])) continue;
      }

      friends.push(villagers[i]);
    }

    return friends;
  }

  static getFriendData(friendName) {
    const npcData = Managers.Villagers.getActorData(friendName);

    return {
      name : friendName,
      friendship : this.getFriendship(friendName),
      interestType : this.getInterestType(friendName),
      hearts : this.getHeartsNum(friendName),
      spriteName : (npcData ? npcData.characterName : ''),
      spriteIndex : (npcData ? npcData.characterIndex : 0)
    };
  }

  static getBestPossibleGifts(friendName) {
    const items = $dataItems;
    const actor = Managers.Villagers.getActorData(friendName);

    if (!actor || !actor.gifts) {
      return items.map(item => item.id);
    }

    const gifts = actor.gifts;
    const love = gifts.love || [];
    const like = gifts.like || [];

    return love.concat(like);
  }

  static getWorstPossibleGifts(friendName) {
    const items = $dataItems;
    const actor = Managers.Villagers.getActorData(friendName);

    if (!actor || !actor.gifts) {
      return items.map(item => item.id);
    }

    const gifts = actor.gifts;
    const hate = gifts.hate || [];
    const dislike = gifts.dislike || [];

    return hate.concat(dislike);
  }

  static getBestGifts(friendName) {
    const best = this.getBestPossibleGifts(friendName);

    return best.filter(itemId => this.isGiftKnown(friendName, itemId));
  }

  static getWorstGifts(friendName) {
    const worst = this.getWorstPossibleGifts(friendName);

    return worst.filter(itemId => this.isGiftKnown(friendName, itemId));
  }

  static registerGift(villagerName, itemId) {
    if (!this.data.gifts[villagerName]) {
      this.data.gifts[villagerName] = {};
    }

    if (!this.data.gifts[villagerName][itemId]) {
      this.data.gifts[villagerName][itemId] = 0;
    }

    this.data.gifts[villagerName][itemId] += 1;
    this.data.giftDates[villagerName] = Managers.Time.totalDays;

    this.changeFriendshipForGifts(villagerName, itemId);

    Managers.Content.registerGift(villagerName, itemId);
  }

  static changeFriendshipForGifts(villagerName, itemId) {
    const actorData = Managers.Villagers.getActorData(villagerName);
    let type = 'normal';

    if (actorData && actorData.gifts) {
      const gifts = actorData.gifts;

      if (gifts.love && gifts.love.includes(itemId)) {
        type = 'love';
      } else if (gifts.like && gifts.like.includes(itemId)) {
        type = 'like';
      } else if (gifts.dislike && gifts.dislike.includes(itemId)) {
        type = 'dislike';
      } else if (gifts.hate && gifts.hate.includes(itemId)) {
        type = 'hate';
      }
    }


    this.changeFriendshipForGiftType(villagerName, type, itemId);
  }

  static changeFriendshipForGiftType(villagerName, type, itemId) {
    switch(type) {
      case 'love':
        this.changeFriendshipForLovedGift(villagerName, itemId);
        break;
      case 'like':
        this.changeFriendshipForLikedGift(villagerName, itemId);
        break;
      case 'dislike':
        this.changeFriendshipForDislikedGift(villagerName, itemId);
        break;
      case 'hate':
        this.changeFriendshipForHatedGift(villagerName, itemId);
        break;
      default:
        this.changeFriendshipForNormalGift(villagerName, itemId);
        break;
    }
  }

  static changeFriendshipForLovedGift(villagerName, itemId) {
    this.increaseFriendship(villagerName, 60);
  }

  static changeFriendshipForLikedGift(villagerName, itemId) {
    this.increaseFriendship(villagerName, 30);
  }

  static changeFriendshipForNormalGift(villagerName, itemId) {
    this.increaseFriendship(villagerName, 15);
  }

  static changeFriendshipForDislikedGift(villagerName, itemId) {
    this.decreaseFriendship(villagerName, 45);
  }

  static changeFriendshipForHatedGift(villagerName, itemId) {
    this.decreaseFriendship(villagerName, 90);
  }

  static increaseFriendshipForFestival(villagerName) {
    this.increaseFriendship(villagerName, 100);
  }

  static isGiftKnown(villagerName, itemId) {
    if (!this.data.gifts[villagerName]) return false;
    if (!this.data.gifts[villagerName][itemId]) return false;

    return this.data.gifts[villagerName][itemId] > 0;
  }

  static characterGotGiftToday(villagerName) {
    if (!this.data.giftDates[villagerName]) return false;

    const today = Managers.Time.totalDays;
    if (today == this.data.giftDates[villagerName]) return true;

    return false;
  }

  static canGiveGift(characterName) {
    if (!this.data.giftDates[characterName]) return true;

    const today = Managers.Time.totalDays;
    if (today == this.data.giftDates[characterName]) return false;

    return true;
  }

  static onChangeMinute() {
    // If there's a villager on the party, increase friendship with them
    if ($gameParty.hasCompanion()) {
      const companionName = $gameParty.getCompanionName();
      this.increaseFriendship(companionName, 1);
    }
  }

  static whoReceivedGift(itemId) {
    if (!this.data.gifts) return [];

    const list = [];

    for (let name in this.data.gifts) {
      const gifts = this.data.gifts[name];
      if (!gifts) continue;
      if (!gifts[itemId]) continue;

      list.push(name);
    }

    return list;
  }
};


Managers.Relationship.clear();

require('game/managers/Time');
Managers.Time.on('changeMinute', Managers.Relationship.onChangeMinute.bind(Managers.Relationship));