const EventListener = require('engine/helpers/EventListener');

let mapInsideDataCache = {
};

Managers.Map = class Map extends EventListener {
  static registerMapClass(mapId, mapClass) {
    this.mapClasses[mapId] = mapClass;
    mapClass.mapId = mapId;
  }

  static getClass(mapId) {
    return this.mapClasses[this.getRealMapId(mapId)];
  }

  static update() {
    if ($gamePlayer.isTransferring()) return;

    if ($gameSystem.canTeleportNow()) {
      this.checkTeleports();
    }

    if ($gameSystem.isPlayingCutscene()) {
      return;
    }  
    
    this.callMethod('update', []);
    this.runEvent('update', []);
  }

  static updateVillagers(...args) {
    if ($gameSystem.isPlayingCutscene()) {
      return;
    }

    this.runEvent('updateVillagers', args);
  }

  static checkTeleports() {
    if ($gameMessage.isBusy()) return;
    if ($gameTemp.isCutsceneReserved()) return;

    this.callMethod('checkTeleports', []);
  }

  static removeAllTempEvents(mapId) {
    if ($gameSystem.isPlayingCutscene()) return;
    
    const classObj = this.getClass(mapId);

    if (!!classObj && !!classObj.removeAllTempEvents) {
      classObj.removeAllTempEvents.call(classObj);
    }

    this.runEvent('removeAllTempEvents', [mapId]);
  }

  static setupMapEvents(mapId) {
    if ($gameSystem.isPlayingCutscene()) return;
    
    const classObj = this.getClass(mapId);

    if (!!classObj && !!classObj.setupMapEvents) {
      classObj.setupMapEvents.call(classObj);
    }

    this.runEvent('setupMapEvents', [mapId]);
  }

  static afterSetupEvents(mapId) {
    if ($gameSystem.isPlayingCutscene()) return;
    
    const classObj = this.getClass(mapId);

    if (!!classObj && !!classObj.afterSetupEvents) {
      classObj.afterSetupEvents.call(classObj);
    }

    this.runEvent('afterSetupEvents', [mapId]);
  }

  static afterRefreshCreatures() {
    if ($gameSystem.isPlayingCutscene()) return;

    this.callMethod('afterRefreshCreatures', []);
  }

  static setupDailyItems() {
    return this.callMethod('setupDailyItems', [], false);
  }

  static onChangeTime() {
    this.callMethod('onChangeTime', []);

    this.updateMapLights();
  }

  static updateFishes() {
    this.callMethod('updateFishes');
  }

  static updateMapLights() {
    this.callMethod('updateMapLights', []);
  }

  static canDropItemAt(id, x, y) {
    return this.callMethod('canDropItemAt', [id, x, y], true);
  }

  static onDropItem(x, y, id, amount) {
    return this.callMethod('onDropItem', [x, y, id, amount], true);
  }

  static isZoomOutEnabled() {
    return this.callMethod('isZoomOutEnabled', undefined, true) !== false;
  }

  static getRealMapId(mapId) {
    if (mapId >= 1000) {
      return Math.floor(mapId / 1000);
    }

    return mapId;
  }

  static updatePlayerSpritePosition(sprite) {
    this.callMethod('updatePlayerSpritePosition', [sprite]);
  }

  static callMethod(methodName, params, defaultValue) {
    const mapId = $gameMap._mapId;
    if (!mapId) return false;

    const classObj = this.getClass(mapId);

    if (!!classObj && !!classObj[methodName]) {
      return classObj[methodName](...(params || []));
    }

    if (defaultValue !== undefined) return defaultValue;

    return false;
  }

  static checkEventTriggerHere() {
    return this.callMethod('checkEventTriggerHere', []);
  }

  static checkEventTriggerThere() {
    return this.callMethod('checkEventTriggerThere', []);
  }

  static processTileTouch(x, y) {
    return this.callMethod('processTileTouch', [x, y]);
  }

  static mapIsCaveFarm(mapId) {
    mapId = this.getRealMapId(mapId);

    return mapId == Maps.CAVE_FARM;
  }

  static mapIsFarm(mapId) {
    const farmMaps = [];

    farmMaps.push(Maps.FARM);
    farmMaps.push(Maps.FARM_CENTER);

    if (farmMaps.includes(mapId)) return true;

    return false;
  }

  static isGrass(mapId, x, y) {
    if (mapId !== $gameMap._mapId) return false;

    return $gameMap.isGrass(x, y);
  }

  static openStorage(playSound, callback) {
    if (this.callMethod('openStorage', [playSound, callback])) return;

    callback();
  }

  static closeStorage() {
    return this.callMethod('closeStorage', []);
  }

  static canAddTempEvent(x, y, regionId) {
    return this.callMethod('canAddTempEvent', [x, y, regionId], null) !== false;
  }

  static afterDraggingObject(obj) {
    return this.callMethod('afterDraggingObject', [obj]);
  }

  static getHeightLevel(x, y) {
    return this.callMethod('getHeightLevel', [Math.floor(x), Math.floor(y)]) || 0;
  }

  static getSpringMaps() {
    return [
      Maps.SPRING_FIELDS
    ];
  }

  static getSummerMaps() {
    return [
      Maps.SUMMER_AREA
    ];
  }

  static getFallMaps() {
    return [
      Maps.FALL_FIELDS
    ];
  }

  static getWinterMaps() {
    return [
      Maps.WINTER_FARM
    ];
  }

  static mapMonth(mapId) {
    if (mapId == $gameMap._mapId) return Managers.Time.mapMonth;

    const spring = this.getSpringMaps();
    const summer = this.getSummerMaps();
    const fall = this.getFallMaps();
    const winter = this.getWinterMaps();

    if (spring.includes(mapId)) return Seasons.SPRING;
    if (summer.includes(mapId)) return Seasons.SUMMER;
    if (fall.includes(mapId)) return Seasons.FALL;
    if (winter.includes(mapId)) return Seasons.WINTER;

    return Managers.Time.month;
  }

  static getMapIdByLabel(mapLabel) {
    const upperCase = mapLabel.toUpperCase();

    if (upperCase == 'HOME') {
      return $gameSystem.currentHome();
    }

    if (MapAlias[upperCase] !== undefined) {
      return MapAlias[upperCase];
    }

    if (Maps[upperCase] !== undefined) {
      return Maps[upperCase];
    }

    let replacingSpaces = upperCase;
    while (replacingSpaces.includes(' ')) {
      replacingSpaces = replacingSpaces.replace(' ', '_');
    }

    if (MapAlias[replacingSpaces] !== undefined) {
      return MapAlias[replacingSpaces];
    }

    if (Maps[replacingSpaces] !== undefined) {
      return Maps[replacingSpaces];
    }

    let shortName = replacingSpaces;
    while (shortName.includes('_')) {
      shortName = shortName.replace('_', '');
    }

    if (MapAlias[shortName] !== undefined) {
      return MapAlias[shortName];
    }

    if (Maps[shortName] !== undefined) {
      return Maps[shortName];
    }

    return 0;
  }

  static mapExists(mapId) {
    const realMapId = this.getRealMapId(mapId);

    for (const mapName in Maps) {
      if (Maps[mapName] == realMapId) return true;
    }

    return false;
  }

  static getActiveMapIds() {
    return [
      Maps.HOME,
      // Maps.HOME_2,
      // Maps.HOME_3,
      // Maps.HOME_3B,
      Maps.FARM,
      Maps.FARM_CENTER,
      // Maps.BARN,
      // Maps.COOP,

      Maps.BEACH,
      Maps.PETTING_ZOO,
      
      Maps.ORANGE_TOWN,
      Maps.GENERAL_STORE,
      Maps.BLACKSMITH,
      // Maps.BLACKSMITH_ROOMS,
      // Maps.BENJAMINS_HOUSE,
      Maps.CLINIC,
      Maps.LIBRARY,
      Maps.BRITTANYS_FARM,
      Maps.CARPENTER_HOUSE,
      Maps.RESTAURANT,
      Maps.INN,
      Maps.CITY_HALL,
      Maps.RORYS_FARM,
      // Maps.ANIMAL_CLINIC,
      // Maps.LUCAS_HOUSE,
      // Maps.AMANDAS_HOUSE,
      // Maps.ILDAS_HOUSE,
      // Maps.GABRIELS_HOUSE,
      // Maps.MIAS_HOUSE,
      // Maps.KARLS_HOUSE,
      
      Maps.FESTIVAL_RECEPTION,
      Maps.WAR_GROUNDS,
      Maps.RACE_TRACKS,

      Maps.MOUNTAIN,
      Maps.MOUNTAIN_TOP,
      // Maps.MINERS_GUILD,
      Maps.TRAIN_STATION_CAVE,
      Maps.MOUNTAIN_S_SMALL_CAVE,
      Maps.SMALL_CAVE,
      Maps.CAVE_FARM,
      Maps.MAIN_MINE,
      Maps.MINE_LEVEL_2,
      Maps.GOLD_MINE,
      Maps.PATH_TO_RUINS,
      
      Maps.FOREST,
      Maps.FOREST_CENTER,
      Maps.FOREST_S,
      Maps.MAZE,
      Maps.PATH_TO_SUMMER_FIELD,
      Maps.CAVEPATH_TO_FALLFIELDS,
      Maps.PATH_TO_WINTER_FIELDS,
      Maps.SPRING_FIELDS,
      Maps.SUMMER_AREA,
      Maps.FALL_FIELDS,
      Maps.WINTER_FARM
    ];
  }

  static getMapDisplayName(mapId) {
    if (mapId == Maps.FARM && Variables.farmName) {
      return Variables.farmName;
    }

    if (mapId >= 1000) {
      mapId = this.getRealMapId(mapId);
    }

    return t(`map-${ mapId }`, '');
  }

  static getDiscoveredAreaList() {
    const allAreas = this.getActiveMapIds();
    const discoveredAreas = [];

    for (let mapId of allAreas) {
      if ($gameSystem.isMapInitialized(mapId)) {
        discoveredAreas.push(mapId);
      }
    }

    return discoveredAreas;
  }

  static getMissingAreaList() {
    const allAreas = this.getActiveMapIds();
    const missingAreas = [];

    for (let mapId of allAreas) {
      if (!$gameSystem.isMapInitialized(mapId)) {
        missingAreas.push(t(`map-${mapId}`));
      }
    }

    return missingAreas;
  }

  static getTownMapIds() {
    return [
      Maps.ORANGE_TOWN,
      Maps.BLACKSMITH,
      Maps.BENJAMINS_HOUSE,
      Maps.CLINIC,
      Maps.LIBRARY,
      Maps.ANIMAL_CLINIC,
      Maps.LUCAS_HOUSE,
      Maps.BRITTANYS_FARM,
      Maps.BLACKSMITH_ROOMS,
      Maps.RESTAURANT,
      Maps.INN,
      Maps.CITY_HALL,
      Maps.GENERAL_STORE,
      Maps.FESTIVAL_RECEPTION,
      Maps.BAR,
      Maps.WAR_GROUNDS,
      Maps.RACE_TRACKS,
      Maps.RORYS_FARM,
      Maps.AMANDAS_HOUSE,
      Maps.ILDAS_HOUSE,
      Maps.ANDRES_HOUSE,
      Maps.GABRIELS_HOUSE,
      Maps.CARPENTER_HOUSE,
      Maps.MIAS_HOUSE,
      Maps.KARLS_HOUSE,
      Maps.BEACH,
    ];
  }

  static getForestMapIds() {
    return [
      Maps.FOREST,
      Maps.FOREST_CENTER,
      Maps.FOREST_S,
      Maps.MAZE
    ];
  }

  static getMountainMapIds() {
    return [
      Maps.MOUNTAIN,
      Maps.SMALL_CAVE,
      Maps.CAVE_FARM,
      Maps.TRAIN_STATION_CAVE,
      Maps.MAIN_MINE,
      Maps.MINE_LEVEL_2
    ];
  }

  static getMountainTopMapIds() {
    return [
      Maps.MOUNTAIN_TOP,
      Maps.GOLD_MINE,
      Maps.PATH_TO_RUINS,
    ];
  }

  static getFarmMapIds() {
    return [
      Maps.FARM,
      Maps.BARN,
      Maps.FARM_CENTER,
      Maps.COOP,
      Maps.PETTING_ZOO
    ].concat(Managers.Map.getHomeMapIds());
  }

  static getHomeMapIds() {
    return [
      Maps.HOME,
      Maps.HOME_2,
      Maps.HOME_3,
      Maps.HOME_3B,
    ];
  }

  static isHomeMap(mapId) {
    return Managers.Map.getHomeMapIds().includes(mapId);
  }

  static isMapInside(mapId) {
    if (mapId in mapInsideDataCache) {
      return mapInsideDataCache[mapId];
    }

    const map = Engine.Data.loadMapDataSync(mapId);
    if (!map) {
      return false;
    }

    const inside = InsideTilesets.includes(map.tilesetId);
    mapInsideDataCache[mapId] = inside;
    return inside;
  }

  static parseMapId(mapId) {
    const possibleId = this.getMapIdByLabel(mapId);
    if (possibleId) {
      return possibleId;
    }

    return parseInt(mapId, 10);
  }
};

Managers.Map.mapClasses = {};
Managers.Time.on('changeTime', Managers.Map.onChangeTime.bind(Managers.Map));