Managers.History = class History {
  static clear() {
    this._dailyData = {};
    this._totalData = {};
    this._updateData = {};
  }

  static getData() {
    return {
      daily : this._dailyData,
      total : this._totalData,
      updates : this._updateData
    };
  }

  static setData(data) {
    this.clear();

    if (data) {
      if (data.daily) {
        this._dailyData = data.daily;
      }

      if (data.total) {
        this._totalData = data.total;
      }

      if (data.updates) {
        this._updateData = data.updates;
      }
    }
  }

  static checkDate(day, month, year) {
    if (!this._dailyData[year]) {
      this._dailyData[year] = {};
    }
    if (!this._dailyData[year][month]) {
      this._dailyData[year][month] = {};
    }
    if (!this._dailyData[year][month][day]) {
      this._dailyData[year][month][day] = {};
    }

    return this._dailyData[year][month][day];
  }

  static registerTotal(type, value) {
    if (!this._totalData[type]) {
      this._totalData[type] = 0;
    }

    this._totalData[type] += value;
  }

  static registerUpdate(oldVersion, newVersion) {
    this._updateData[oldVersion] = newVersion;
  }

  static registerFishCaught(fishId) {
    this.registerValue(`fish-${ fishId }`, 1);
    this.registerTotal(`fish-${ fishId }`, 1);
    this.registerTotal('fish-caught', 1);
  }

  static readFishCaught(fishId) {
    const { year, month, day } = Managers.Time;

    const data = this.checkDate(day, month, year);
    const type = `fish-${ fishId }`;

    if (!data[type]) {
      return 0;
    }

    return Number(data[type]) || 0;
  }

  static readTotal(type) {
    if (!this._totalData[type]) return 0;

    return this._totalData[type];
  }

  static registerValue(type, value) {
    const year = Managers.Time.year;
    const month = Managers.Time.month;
    const day = Managers.Time.day;

    const data = this.checkDate(day, month, year);

    if (!data[type]) {
      data[type] = 0;
    }

    data[type] += value;
  }

  static readValue(type, value) {
    const year = Managers.Time.year;
    const month = Managers.Time.month;
    const day = Managers.Time.day;

    const data = this.checkDate(day, month, year);

    if (!data[type]) {
      return 0;
    }

    return data[type];
  }

  static registerItem(type, value) {
    const year = Managers.Time.year;
    const month = Managers.Time.month;
    const day = Managers.Time.day;

    const data = this.checkDate(day, month, year);

    if (!data[type]) {
      data[type] = [];
    }

    data[type].push(value);
  }

  static registerEvent(eventName) {
    this.registerItem('events', eventName);
  }

  static registerCutscene(eventName) {
    this.registerEvent(`cutscene-${eventName}`);
  }

  static registerConversation(eventName) {
    const days = this.daysSinceConversation(eventName);

    // Only register the conversation if it hasn't already been registered today
    if (days === false || days >= 1) {
      const id = `conversation-${eventName}`;

      this.registerEvent(id);
      this.registerTotal(id, 1);
    }
  }

  static wasConversationEverPlayed(eventName) {
    return this.readConversationTotal(eventName) > 0;
  }

  static readConversationTotal(eventName) {
    const id = `conversation-${eventName}`;
    return this.readTotal(id);
  }

  static eraseCutsceneRecord(eventName) {
    const id = `cutscene-${eventName}`;

    this.iterateHistory((day, month, year) => {
      const data = this._dailyData[year][month][day];
      if (!data) {
        return false;
      }

      if (!data.events) {
        return false;
      }

      const idx = data.events.indexOf(id);
      if (idx < 0) {
        return false;
      }

      data.events[idx] = `erased__${ id }`;
      return true;
    });
  }

  static iterateHistory(callbackFn) {
    for (let year = Managers.Time.year; year > 0; year--) {
      if (!this._dailyData[year]) continue;

      for (let month = 4; month > 0; month--) {
        if (!this._dailyData[year][month]) continue;

        for (let day = 31; day > 0; day--) {
          if (!this._dailyData[year][month][day]) continue;
          
          if (callbackFn(day, month, year)) {
            return true;
          }
        }
      }
    }

    return false;
  }

  static getEventYear(eventName) {
    let eventYear;

    this.iterateHistory((day, month, year) => {
      const data = this._dailyData[year][month][day];
      if (!data) return false;
      if (!data.events) return false;

      if (data.events.includes(eventName)) {
        eventYear = year;
        return true;
      }

      return false;
    });

    if (eventYear) {
      return eventYear;
    }

    return false;
  }

  static getEventDate(eventName) {
    let eventDate;

    this.iterateHistory((day, month, year) => {
      const data = this._dailyData[year][month][day];
      if (!data) return false;
      if (!data.events) return false;

      if (data.events.includes(eventName)) {
        eventDate = Managers.Time.convertConfigToTimestamp({
          day,
          month,
          year,
          minute : 0,
          seconds : 0,
          hour : 0
        });
        return true;
      }

      return false;
    });

    if (eventDate || eventDate === 0) {
      return eventDate;
    }

    return false;
  }

  static daysSinceEvent(eventName) {
    const timestamp = this.getEventDate(eventName);

    if (!timestamp && timestamp !== 0) return false;
    return Managers.Time.daysSince(timestamp);
  }

  static daysSinceCutscene(eventName) {
    return this.daysSinceEvent(`cutscene-${eventName}`);
  }

  static getConversationYear(eventName) {
    return this.getEventYear(`conversation-${ eventName }`);
  }

  static daysSinceConversation(eventName) {
    return this.daysSinceEvent(`conversation-${ eventName }`);
  }

  static registerMoneyEarned(value) {
    this.registerValue('moneyEarned', value);

    Variables.moneyEarned += value;
  }

  static registerMoneySpent(value) {
    this.registerValue('moneySpent', value);

    Variables.moneySpent += value;
  }

  static registerCropHarvested(cropType, amount) {
    if (amount === undefined) amount = 1;
    
    this.registerValue('cropsHarvested', amount);
    this.registerValue(`harvest-${cropType}`, amount);

    this.registerTotal(`harvest-${cropType}`, amount);

    Variables.cropsHarvested += amount;

    Managers.Achievements.checkAchievements();
  }

  static getNumberOfItemsShippedToday() {
    return this.readValue('itemsShipped');
  }

  static getValueOfItemsShippedToday() {
    return this.readValue('valuesShipped');
  }

  static registerItemShipped(itemId, amount, totalGold) {
    if (amount === undefined) amount = 1;

    this.registerValue('itemsShipped', amount);
    this.registerValue(`ship-${itemId}`, amount);
    // Added in 0.7.0
    this.registerValue('valuesShipped', totalGold);
    
    this.registerTotal(`ship-${itemId}`, amount);
    this.registerTotal(`value-ship-${itemId}`, totalGold);

    Variables.itemsShipped += amount;
  }

  static registerItemTrashed(itemId, amount) {
    if (amount === undefined) amount = 1;

    this.registerValue('itemsTrashed', amount);
    this.registerValue(`trash-${itemId}`, amount);
    this.registerTotal(`trash-${itemId}`, amount);

    Variables.itemsTrashed += amount;
  }

  static getHarvestedCrops(cropType) {
    return this.readTotal(`harvest-${cropType}`);
  }

  static getShippedItems(itemId) {
    return this.readTotal(`ship-${itemId}`);
  }

  static registerAcceptedTask(taskId) {
    this.registerEvent(`accepted-task-${taskId}`);
  }

  static registerCompletedTask(taskId) {
    this.registerEvent(`completed-task-${taskId}`);
  }

  static getTaskAcceptDate(taskId) {
    return this.getEventDate(`accepted-task-${taskId}`);
  }

  static getTaskCompleteDate(taskId) {
    return this.getEventDate(`completed-task-${taskId}`);
  }


};

Managers.History.clear();