Managers.Health = class Health {
  constructor() {
    throw new Error('This is a static class');
  }

  static get stamina() {
    return this._stamina;
  }
  static set stamina(value) {
    if (this._stamina !== value) {
      if (Managers.Scenes._scene._timeHudWindow) {
        Managers.Scenes._scene._timeHudWindow.requestRefresh();
      }
    }

    this._stamina = value;
  }

  static get maxStamina() {
    return this._maxStamina;
  }
  static set maxStamina(value) {
    this._maxStamina = value;
  }

  static get fatigue() {
    return this._fatigue;
  }
  static set fatigue(value) {
    this._fatigue = value;
  }

  static get maxFatigue() {
    return this._maxFatigue;
  }
  static set maxFatigue(value) {
    this._maxFatigue = value;
  }

  static get usedStamina() {
    return this._usedStamina;
  }
  static set usedStamina(value) {
    this._usedStamina = value;
  }

  static get oldStaminaLevel() {
    return this._oldStaminaLevel;
  }
  static set oldStaminaLevel(value) {
    this._oldStaminaLevel = value;
  }

  static get staminaLevel() {
    return Math.floor(this.stamina * 100 / this.maxStamina);
  }

  static get fatigueLevel() {
    return Math.floor(this.fatigue * 100 / this.maxFatigue);
  }

  static get buffs() {
    return this._buffs;
  }

  static set buffs(value) {
    this._buffs = value || [];
  }

  static clear() {
    //When Stamina is low, the player can't work anymore
    this._stamina = 100;

    //Increase max Stamina to be able to work more
    this._maxStamina = 100;

    //When Fatigue is high, the player may pass out
    this._fatigue = 0;

    //Increase max fatigue to be able to stay awake longer
    this._maxFatigue = 100;

    //Keeps tab of how much stamina was used since the character last slept
    this._usedStamina = 0;

    //Identifies the level the stamina was at before the last change
    this._oldStaminaLevel = 0;

    // Stores information about timed buffs
    this._buffs = [];
  }

  static getHealthData() {
    return {
      stamina : this.stamina,
      maxStamina : this.maxStamina,
      fatigue : this.fatigue,
      maxFatigue : this.maxFatigue,
      usedStamina : this.usedStamina,
      buffs : this.buffs
    };
  }

  static setHealthData(healthData) {
    this.stamina = healthData.stamina;
    this.maxStamina = healthData.maxStamina;
    this.fatigue = healthData.fatigue;
    this.maxFatigue = healthData.maxFatigue;
    this.usedStamina = healthData.usedStamina;
    this.buffs = healthData.buffs;

    this._oldStaminaLevel = this.staminaLevel;
  }

  static _doSleepEffect(timeSlept, rate, fatigueRate, staminaRate) {
    if ($gameSystem.isPlayerPassedOut) {
      timeSlept /= 4;
    }

    if (timeSlept <= 0) return;

    let goodRest = timeSlept * (rate || 1);

    const fatigueMultiplier = Math.max(1, Math.min((100 - this.fatigueLevel) / 100, 0.01));
    
    this.fatigue -= (goodRest * 0.5) * (fatigueRate || 1);
    this.stamina += (goodRest * 0.15) * fatigueMultiplier * (staminaRate || 1);

    if ($gameSystem.isPlayerPassedOut) {
      if (this.fatigue < 30) {
        this.fatigue = 30;
      }
      if (this.stamina > 70) {
        this.stamina = 70;
      }
    }

    this.usedStamina = 0;
    this.fixOutOfBoundsValues();
    this._oldStaminaLevel = this.staminaLevel;
  }

  static doSleepEffect(timeSlept) {
    Managers.Player.sleepingExp += timeSlept / 50;

    let rate = 1;
    if ($gameSystem.isPlayerSleepingStanding || $gameSystem.isPlayerPassedOut) {
      rate = 0.5;
    }

    this._doSleepEffect(timeSlept, rate);
  }

  static fixOutOfBoundsValues() {
    if (this.stamina > this.maxStamina) this.stamina = this.maxStamina;
    if (this.stamina < 1) this.stamina = 1;

    if (this.fatigue > this.maxFatigue) this.fatigue = this.maxFatigue;
    if (this.fatigue < 0) this.fatigue = 0;

    this.stamina = Math.fix(this.stamina);
    this.fatigue = Math.fix(this.fatigue);
  }

  static doToolUsageEffect(tool, levelUsed) {
    if (levelUsed === undefined) levelUsed = 0;

    let multiplier = tool.energyRequired - tool.level + levelUsed;
    if (multiplier <= 0) multiplier = 1;
    const stamina = Math.fix(0.1 * multiplier);

    if (Managers.Content.preventStaminaLoss(tool, levelUsed)) {
      return;
    }

    this.loseStamina(stamina);
    const levelMultiplier = (1 + levelUsed) / (1 + tool.level);
    this.fatigue += tool.fatigueRequired * this.getFatigueBaseMultiplier() * levelMultiplier;
    this.fixOutOfBoundsValues();
  }

  static staminaMultiplierForFatigueLevel(fatigueLevel) {
    if (fatigueLevel <= 25) return 0;
    if (fatigueLevel <= 50) {
      return (fatigueLevel / 100);
    }

    if (fatigueLevel <= 75) {
      return 0.5 + (fatigueLevel - 50) / 50;
    }

    if (fatigueLevel <= 80) {
      return 1 + (fatigueLevel - 75) / 10;
    }

    if (fatigueLevel <= 90) {
      return 1.5 + (fatigueLevel - 80) / 8;
    }

    if (fatigueLevel <= 95) {
      return 2.75 + (fatigueLevel - 90) / 5;
    }

    return 4 + (fatigueLevel - 95) / 2.5;
  }

  static getStaminaBaseMultiplier() {
    const buff = this.getActiveBuffType('stamina');

    if (!buff) {
      return 1;
    }

    return buff.rate;
  }

  static loseStamina(amount) {
    this._oldStaminaLevel = this.staminaLevel;
    this.usedStamina += amount;
    
    const multiplier = (1 + this.staminaMultiplierForFatigueLevel(this.fatigueLevel)) * this.getStaminaBaseMultiplier();

    const finalAmount = (amount * multiplier);

    this.stamina -= finalAmount;
    this.fixOutOfBoundsValues();
  }

  static syncStaminaLevel() {
    this._oldStaminaLevel = this.staminaLevel;
  }

  static getFatigueBaseMultiplier() {
    const buff = this.getActiveBuffType('fatigue');

    if (!buff) {
      return 1;
    }

    return buff.rate;
  }

  static increaseFatigue(amount) {
    const finalAmount = amount * this.getFatigueBaseMultiplier();

    this.fatigue += finalAmount;
    this.fixOutOfBoundsValues();
  }

  static onChangeSecond() {
    this.fatigue += 0.005 * this.getFatigueBaseMultiplier();

    if ($gameSystem.isPlayerSleepingStanding) {
      if (this.fatigueLevel > 80) {
        this.fatigue -= (this.maxFatigue * 0.002);
      } else if (this.fatigueLevel > 50) {
        this.fatigue -= (this.maxFatigue * 0.001);
      } else {
        this.fatigue -= (this.maxFatigue * 0.0005);
      }

      if (this.staminaLevel < 25) {
        this.stamina += (this.maxStamina * 0.0025);
      } else if (this.staminaLevel < 50) {
        this.stamina += (this.maxStamina * 0.0018);
      } else {
        this.stamina += (this.maxStamina * 0.0012);
      }
    }

    this.updateBuffs();
    this.fixOutOfBoundsValues();
  }

  static fullyRestoreStamina() {
    this.stamina = this.maxStamina;
  }

  static fullyRestoreFatigue() {
    this.fatigue = 0;
  }

  static restoreHealth(restoreData) {
    if (restoreData.stamina) {
      const staminaRestored = restoreData.stamina;
      this.stamina += staminaRestored;
    }  

    if (restoreData.fatigue) {
      this.fatigue -= restoreData.fatigue;
    }

    if (restoreData.maxStamina) {
      this.maxStamina += restoreData.maxStamina;
    }

    if (restoreData.buffs) {
      for (let buff of restoreData.buffs) {
        if (!buff) {
          continue;
        }

        this.addBuff(buff.type, buff.rate, buff.time);
      }
    }

    this.fixOutOfBoundsValues();
    this.syncStaminaLevel();
  }

  static processNewDay() {
    this.buffs = [];
  }

  static updateBuffs() {
    if (!this.buffs || !this.buffs.length) {
      return;
    }

    const typesAffected = [];
    let clear = false;

    const maxTime = Managers.Time.secondsUntilDayChange();

    for (let i = this.buffs.length -1; i >= 0; i--) {
      const buff = this.buffs[i];

      if (!buff.type) {
        clear = true;
        continue;
      }

      if (typesAffected.includes(buff.type)) {
        continue;
      }

      if (buff.time) {
        if (buff.time > maxTime) {
          buff.time = maxTime;
        } else {
          buff.time--;
        }

        typesAffected.push(buff.type);
        continue;
      }

      buff.type = false;
      clear = true;
    }

    if (clear) {
      this.buffs = this.buffs.filter((buff) => Boolean(buff.type));
    }
  }

  static getActiveBuffType(type) {
    return this.buffs.filter((buff) => buff.type == type).pop();
  }

  static addBuff(type, rate, time) {
    const existingBuff = this.getActiveBuffType(type);
    const maxTime = Managers.Time.secondsUntilDayChange();

    if (existingBuff && existingBuff.rate == rate && existingBuff.time >= 0) {
      existingBuff.time += time;

      if (existingBuff.time > maxTime) {
        existingBuff.time = maxTime;
      }

      return;
    }

    this.buffs.push({
      type,
      rate,
      time: Math.min(time, maxTime)
    });
  }

  static printReport(characterName) {
    const staminaRate = Math.floor(this.stamina * 100 / this.maxStamina);
    const fatigueRate = Math.floor(this.fatigue * 100 / this.maxFatigue);

    let msg = `${t("Stamina")}: ${staminaRate}%`;
    msg += '<BR>';
    msg += `${t("Fatigue")}: ${fatigueRate}%`;

    Managers.Villagers.showMessage(characterName, msg);
  }

  static doPassOutEffect() {
    const oldDateTime = Managers.Time.getDateTime();
    const newDateTime = Managers.Time.getDateTime();
    let shouldShowWindow = false;
    
    newDateTime.hour += 5;
    Managers.Time.validateDateTimeValues(newDateTime);
    if (newDateTime.hour >= 6 && oldDateTime.hour < 6) {
      shouldShowWindow = true;
    }
    // if (newDateTime.day != oldDateTime.day && newDateTime.hour < 6) {
    //   newDateTime.hour = 6;
    //   Managers.Time.validateDateTimeValues(newDateTime);
    // }

    Engine.Audio.fadeOutBgm(3);
    setTimeout(() => {
      // Managers.Sound.playBedEvent();
    }, 200);

    const sleepMethod = this.sleepForPassingOut.bind(this, newDateTime);

    if (shouldShowWindow) {
      $gameSystem.startSleep(sleepMethod, oldDateTime, newDateTime, true);
    } else {
      $gameSystem.isPlayerPassedOut = true;

      sleepMethod();
      $gameScreen.startFadeIn(24);
      $gameSystem.isPlayerPassedOut = false;
      Managers.CommonEvent.playEvent('player_waking_up_from_passing_out');
    }
  }

  static sleepForPassingOut(newDateTime) {
    Managers.Time.sleep(newDateTime);
    $gameMap.autoplay();
    Engine.Audio.fadeInBgm(3);
  }

  static applySpaEffect() {
    this._doSleepEffect(60, 3, 1, 3);
    Variables.bathCount++;
  }
};

Managers.Time.on('changeSecond', Managers.Health.onChangeSecond.bind(Managers.Health));