require('./Time');

Managers.Fishing = class Fishing {
  constructor() {
    throw new Error('This is a static class');
  }

  static contentId() {
    return 'fishing';
  }

  static processNewDay() {
    this.fishList = [];
    this.mapFishList = {};
  }

  static getFishingChance() {
    return this.fishChance;
  }

  static registerFishingTile(tile) {
    this.fishingTile = tile;
  }

  static fishExistsAt(x, y) {
    return this.fishList.some(fish => {
      if (!fish) return false;
      if (fish.x != x) return false;
      if (fish.y != y) return false;

      if (!fish.event) return false;
      if (fish.event._erased) return false;

      if (fish.caught) return false;

      return true;
    });
  }

  static maybeAddFishToMap(x, y, fishType, timeLimit, moveDirection, moveDistance, rewardChance) {
    if (Math.randomInt(10) < 5) {
      return this.addFishToMap(x, y, fishType, timeLimit, moveDirection, moveDistance, rewardChance);
    }
  }

  static getAllAvailableFish() {
    const fish = [];

    for (let item of $dataItems) {
      if (!item) continue;
      if (!item.fish) continue;

      if (item.dailyMax) {
        const caughtToday = Managers.History.readFishCaught(item.id);
        if (caughtToday && caughtToday >= item.dailyMax) {
          continue;
        }
      }

      fish.push(item.id);
    }

    return fish;
  }

  static isFishAvailable(fishId) {
    const fishData = Managers.Items.getItemData(fishId);
    if (!fishData) return false;

    if (!fishData.fish) return true;
    if (!fishData.dailyMax) return true;

    const caughtToday = Managers.History.readFishCaught(fishId);
    if (caughtToday && caughtToday >= fishData.dailyMax) {
      return false;
    }

    return true;
  }

  static addRandomGoodItem(x, y, timeLimit, rewardChance, hiddenShadow) {
    const possibleItems = [
      'sapphire',
      'ruby',
      'emerald',
      'blue-crystal',
      'iron-ore',
      'copper-ore',
      'silver-ore'
    ];

    if ($gameSystem.isMapInitialized(Maps.GOLD_MINE)) {
      possibleItems.push('gold-ore');
    }

    this.addRandomItem(possibleItems, x, y, timeLimit, rewardChance, hiddenShadow);
  }

  static addRandomBadItem(x, y, timeLimit, rewardChance, hiddenShadow) {
    const possibleItems = [
      'old-key',
      'coin',
      'blue-can',
      'green-can',
      'broken-glass',
      'rock',
      'wooden-stick',
      'clay',
      'branch',
      'old-tire'
    ];

    this.addRandomItem(possibleItems, x, y, timeLimit, rewardChance, hiddenShadow);
  }

  static addCaveBadItem(x, y, timeLimit, rewardChance, hiddenShadow) {
    const possibleItems = [
      'rock',
      'clay'
    ];

    this.addRandomItem(possibleItems, x, y, timeLimit, rewardChance, hiddenShadow);
  }

  static addRandomBeachItem(x, y, timeLimit, rewardChance, hiddenShadow) {
    const possibleItems = [
      'red-can',
      'rock',
      'wooden-stick',
      'old-boot',
      'old-tire',
      'clam'
    ];

    this.addRandomItem(possibleItems, x, y, timeLimit, rewardChance, hiddenShadow);
  }

  static addRandomItem(possibleItems, x, y, timeLimit, rewardChance, hiddenShadow) {
    const itemId = possibleItems[Math.randomInt(possibleItems.length)];
    this.addFishToMap(x, y, itemId, timeLimit, 0, 0, rewardChance, hiddenShadow);
  }

  static getMapFish(mapId) {
    if (!this.mapFishList[mapId]) {
      this.mapFishList[mapId] = {
        small: this.getRandomFishes('small', mapId, 5),
        medium: this.getRandomFishes('medium', mapId, 5),
        big: this.getRandomFishes('big', mapId, 5),
      };
    }

    return this.mapFishList[mapId];
  }

  static getRandomFishes(size, mapId, count) {
    if (!mapId) {
      mapId = $gameMap._mapId;
    }

    const sea = mapId === Maps.BEACH;

    const season = Managers.Map.mapMonth(mapId);
    const seasonName = Managers.Time.monthNames[season - 1].toLowerCase();

    const fishList = [];
    for (const item of $dataItems) {
      if (!item || !item.fish) continue;
      if (item.fish.size !== size) continue;
      if (item.fish.sea !== sea) continue;

      let frequency = 0;
      if (item.fish.seasons && item.fish.seasons[seasonName]) {
        frequency = item.fish.seasons[seasonName];
      }

      if (frequency > 10) {
        frequency = 10;
      }

      for (let i = 0; i < frequency; i++) {
        fishList.push(item.id);
      }
    }

    const resultList = [];
    while (resultList.length < count) {
      if (!fishList.length) {
        break;
      }

      const idx = Math.randomInt(fishList.length);
      const id = fishList[idx];

      fishList.splice(idx, 1);
      if (resultList.includes(id)) {
        continue;
      }

      resultList.push(id);
    }

    return resultList;
  }

  static getRandomFishId(size, mapId) {
    const mapFish = this.getMapFish(mapId);

    const list = mapFish[size];
    const idx = Math.randomInt(list.length);
    return list[idx];
  }

  static getAnyFishIdForMap(mapId) {
    if (!mapId) {
      mapId = $gameMap._mapId;
    }

    const sea = mapId === Maps.BEACH;

    const season = Managers.Map.mapMonth(mapId);
    const seasonName = Managers.Time.monthNames[season -1].toLowerCase();
    const fishingLevel = this.fishingLevel();

    const fishList = [];
    for (const item of $dataItems) {
      if (!item || !item.fish) continue;
      if (item.fish.sea !== sea) continue;

      let frequency = 0;
      if (item.fish.seasons && item.fish.seasons[seasonName]) {
        frequency = item.fish.seasons[seasonName];
      }

      if (frequency > 10) {
        frequency = 10;
      }

      switch(item.fish.size) {
        case 'small':
          if (fishingLevel === 1) {
            // Decreases small fish frequency on upgraded rods
            if (frequency > 1) {
              frequency = Math.ceil(frequency * 0.8);
            }
          } else if (fishingLevel === 2) {
            // Decreases small fish frequency further on twice upgraded rods
            if (frequency > 1) {
              frequency = Math.ceil(frequency * 0.6);
            }
          }
          break;
        case 'medium':
          // Make medium fish uncommon when they can't be catched
          if (fishingLevel === 0) {
            if (frequency > 1) {
              frequency = 1;
            }
          } else if (fishingLevel === 2) {
            // Increases chance of medium fish with a rod level 2
            frequency = Math.ceil(frequency * 1.4);
          }
          break;
        case 'big':
          if (fishingLevel === 0) {
            // big fish will never appear for the initial rod
            frequency = 0;
          } else if (fishingLevel === 1) {
            // A level 1 rod can see common big fishes, even if it can't catch it
            if (frequency >= 5) {
              frequency = 1;
            } else {
              frequency = 0;
            }
          }
          break;
      }

      for (let i = 0; i < frequency; i++) {
        fishList.push(item.id);
      }
    }

    if (!fishList.length) {
      return false;
    }

    const idx = Math.randomInt(fishList.length);
    return fishList[idx];
  }

  static addFishToMap(x, y, fishType, timeLimit, moveDirection, moveDistance, rewardChance, hiddenShadow) {
    if (this.fishExistsAt(x, y)) return;
    if (!this.isFishAvailable(fishType)) return;

    if (typeof timeLimit == 'number') {
      timeLimit = Managers.Time.getLaterTimeStr(timeLimit);
    }

    const fishData = {
      x,
      y,
      fishType,
      timeLimit : timeLimit || '23:59',
      moveDirection,
      moveDistance : moveDistance || 1,
      rewardChance : rewardChance || 100,
      hiddenShadow : hiddenShadow || false
    };

    this.fishList.push(fishData);
  }

  static maybeCreateEventForFish(fishData) {
    fishData.event = null;
    if (fishData.caught) return;
    if (fishData.timeLimit && !Managers.Time.isBetweenTimes('00:00', fishData.timeLimit)) return;

    const x = fishData.x;
    const y = fishData.y;
    let event = null;

    if (fishData.hiddenShadow) {
      event = $gameMap.createFishAt(x, y, 7);
    } else {
      const itemData = Managers.Items.getItemData(fishData.fishType);
      if (!itemData.fish) return;
      switch (itemData.fish.size) {
        case 'small':
          event = $gameMap.createTinyFishAt(x, y);
          break;
        case 'medium':
          event = $gameMap.createFishAt(x, y);
          break;
        case 'big':
          event = $gameMap.createLargeFishAt(x, y);
          break;
      }
    }

    fishData.event = event;

    if (event) {
      if (!fishData.hiddenShadow) {
        event.fadeIn();
      }
      event._fishData = fishData;
      event.setFishMoveDirection(fishData.moveDirection);
      event.setFishMoveDistance(fishData.moveDistance);
    }
  }

  static maybeClearEventForFish(fishData) {
    if (fishData.caught) {
      fishData.event.erase();
      return;
    }

    if (fishData.timeLimit) {
      if (!Managers.Time.isBetweenTimes('00:00', fishData.timeLimit)) {
        if (fishData.hiddenShadow) {
          fishData.event.erase();
        } else {
          fishData.event.fadeOutErase();
        }
        return;
      }
    }
  }

  static updateFishEvents() {
    for (const fishData of this.fishList) {
      if (fishData.event && !fishData.event._erased && !fishData.event._fadingOut) {
        this.maybeClearEventForFish(fishData);
      } else {
        this.maybeCreateEventForFish(fishData);
      }
    }
  }

  static setupMapEvents() {
    this.fishList = [];
    this.updateFishes();
  }

  static updateFishes() {
    Managers.Map.updateFishes();

    this.updateFishEvents();
  }

  static startFishing(level) {
    if (!this.fishingTile) {
      this.endFishing();
      return;
    }

    Switches.isFishing = true;

    this.baitEvent = $gameMap.createNormalEventAt('!$fishing', 0, this.fishingTile.x, this.fishingTile.y, (level + 1) * 2, false, true, Objects.CustomEvent);
    this.baitEvent._through = true;
    this.baitEvent._directionFix = true;
    this.baitEvent._stepAnime = true;

    this.firstCheck = false;
  }

  static checkIfThereIsAFishOnTheBait() {
    if (!this.fishingTile) {
      this.endFishing();
      return;
    }

    if (this.itemFound) {
      return;
    }

    const creatures = $gameMap.creaturesXy(this.fishingTile.x, this.fishingTile.y);
    if (creatures && creatures.length) {
      const len = creatures.length;
      for (let i = 0; i < len; i++) {
        const creature = creatures[i];

        if (!this.firstCheck) {
          if (creature.isMoving()) continue;
        }
        this.firstCheck = false;

        if (creature instanceof Objects.Fish) {
          creature.erase();
          this._currentFishData = creature._fishData;
          this.foundFish(this._currentFishData);
          return;
        }
      }
    }

    if (Math.randomInt(1000) > 995) {
      this._currentFishData = null;
      this.foundFish(null);
      return;
    }
  }

  static foundLargeCaveTreasure(/*better*/) {
    const chance = Math.randomInt(10);

    switch(chance) {
      case 0:
        this.itemFound = 'diamond';
        this.rewardChance = 4;
        break;
      case 1:
        this.itemFound = 'sapphire';
        this.rewardChance = 10;
        break;
      case 2:
        this.itemFound = 'ruby';
        this.rewardChance = 9;
        break;
      case 3:
        this.itemFound = 'emerald';
        this.rewardChance = 8;
        break;
      case 4:
        this.itemFound = 'iron-ore';
        this.rewardChance = 50;
        break;
      case 5:
        this.itemFound = 'blue-crystal';
        this.rewardChance = 20;
        break;
      case 6:
        this.itemFound = 'iron-ingot';
        this.rewardChance = 35;
        break;
      case 7:
        this.itemFound = 'copper-ingot';
        this.rewardChance = 20;
        break;
      case 8:
        this.itemFound = 'silver-ingot';
        this.rewardChance = 10;
        break;
      case 9:
        this.itemFound = 'gold-ingot';
        this.rewardChance = 5;
        break;
    }
  }

  static fishingLevel() {
    return (Managers.Tools.isToolTypeEquipped('fishing-rod') ? Managers.Tools.toolLevel : 0).clamp(0, 2);
  }

  static foundFish(fishData) {
    Managers.Player.fishingExp++;

    if (fishData) {
      this.itemFound = fishData.fishType;
      this.rewardChance = fishData.rewardChance || 100;
    } else {
      this.itemFound = this.getAnyFishIdForMap($gameMap._mapId);
      const itemData = Managers.Items.getItemData(this.itemFound);
      if (itemData && itemData.fish) {
        const toolLevel = (Managers.Tools.isToolTypeEquipped('fishing-rod') ? Managers.Tools.toolLevel : 0).clamp(0, 2);

        switch(itemData.fish.size) {
          case 'small':
            this.rewardChance = [75, 90, 100][toolLevel];
            break;
          case 'medium':
            this.rewardChance = [0, 75, 90][toolLevel];
            break;
          default:
            this.rewardChance = [0, 0, 75][toolLevel];
            break;
        }
      } else {
        this.rewardChance = 0;
      }
    }

    this.baitEvent._characterName = '$sink';
    this.baitEvent._direction = Direction.LEFT;
    this.timeLeft = 60 + Math.randomInt(60);

    $gamePlayer.requestBalloon(Balloons.EXCLAMATION);
  }

  static update() {
    if (!Switches.isFishing) return;
    if (!this.fishingTile) {
      this.endFishing();
      return;
    }

    this.checkIfThereIsAFishOnTheBait();

    if (Engine.Input.isTriggered('use') || Engine.Input.isTriggered('ok') || TouchInput.isTriggered() || TouchInput.isRightTriggered()) {
      if (Switches.isFishing) {
        this.checkIfFishedAnything();
        this.endFishing();
      }
    }

    this.checkIfFishEscaped();
  }

  static rewardPlayer(itemId) {
    Managers.Player.fishingExp++;

    if (Math.randomInt(100) < this.rewardChance) {
      Managers.History.registerFishCaught(itemId);
      Managers.Items.pickItemId(itemId);
    } else {
      Variables.fishThatEscapedId = itemId;
      Variables.fishThatEscaped = Managers.Text.item(itemId);
      Managers.CommonEvent.playEvent('fishing_failed');
    }
  }

  static checkIfFishedAnything() {
    if (this.itemFound) {
      this.rewardPlayer(this.itemFound);
    }

    if (this._currentFishData) {
      this._currentFishData.caught = true;
    }
  }

  static checkIfFishEscaped() {
    if (!Switches.isFishing) return;
    if (!this.itemFound) return;

    if (this.timeLeft > 0) {
      this.timeLeft--;
      return;
    }

    this.baitEvent._direction = Direction.DOWN;
    this._currentFishData = false;
    this.itemFound = false;
  }

  static canCheckInput() {
    if (Switches.isFishing) return false;
    // return !Switches.isFishing;
  }

  static endFishing() {
    if (this.baitEvent) {
      this.baitEvent.erase();
      this.baitEvent = false;
    }

    this.itemFound = false;
    this._currentFishData = false;
    $gamePlayer.endToolAnimation();
    Switches.isFishing = false;
  }
};


Managers.Fishing.fishingTile = false;
Managers.Fishing.baitEvent = false;
Managers.Fishing.firstCheck = false;
Managers.Fishing.itemFound = false;
Managers.Fishing.timeLeft = 0;
Managers.Fishing.rewardChance = 0;
Managers.Fishing.fishChance = 0.2;
Managers.Fishing.fishList = [];
Managers.Fishing.mapFishList = {};

Managers.Content.registerContentClass(Managers.Fishing);

Managers.Time.on('changeMinute', Managers.Fishing.updateFishes.bind(Managers.Fishing));