const largeModels = [];
const wideModels = [];
const tallModels = [];
const modelsWithCollision = [];

Managers.FarmObjects = class FarmObjects {
  constructor() {
    throw new Error('This is a static class');
  }

  static clear() {
    this._farmObjects = {};
    // Protected tiles are tiles that can't receive farmObjects because they have something else over them.
    this._protectedTiles = {};
    // Flag to stop the protected tiles from being updated - used to prevent the update method from being called inside a loop.
    this._protectedTilesLocked = false;
    this._outlinedTiles = [];
    this._barns = [];
    this._coops = [];
  }

  static setOutlinedTiles(tileList) {
    this._outlinedTiles = tileList;

    if (tileList && Graphics.isWebGL()) {
      for (let i = 0; i < tileList.length; i++) {
        if (!tileList[i]) continue;
        if (!tileList[i].farmObject) continue;

        const event = tileList[i].farmObject.findEvent();
        if (event) {
          event.forceUseOfEventSprite();
        }
      }
    }
  }

  static clearOutlinedTiles() {
    this._outlinedTiles = [];
  }

  static getOutlinedTiles() {
    return this._outlinedTiles || [];
  }

  static isObjectOutlined(object) {
    if (!object) return false;
    
    const outlinedTiles = this.getOutlinedTiles();
    for (let i = 0; i < outlinedTiles.length; i++) {
      if (outlinedTiles[i].farmObject == object) return true;
    }

    return false;
  }

  static getOutlineData(object) {
    if (!object) return false;

    const outlinedTiles = this.getOutlinedTiles();
    for (let i = 0; i < outlinedTiles.length; i++) {
      if (outlinedTiles[i].farmObject == object) {
        return outlinedTiles[i];
      }
    }

    return false;
  }

  static clearProtectedTiles(mapId) {
    this._protectedTiles[mapId] = {};
  }

  static lockProtectedTiles() {
    this._protectedTilesLocked = true;
  }

  static unlockProtectedTiles() {
    this._protectedTilesLocked = false;
  }

  static processNewDay() {
    Managers.Mine.processNewDay();
    
    for (const mapId in this._farmObjects) {
      if (isNaN(mapId)) continue;

      for (const farmObject of this._farmObjects[mapId]) {
        if (!farmObject) continue;

        farmObject.processNewDay();
      }
    }

    this.clearEmptyItems();
    this.refreshMapEvents();
  }

  static clearMapObjectsByFilter(mapId, filterFn) {
    this.checkMapFarmObjects(mapId);

    for (let i = 0; i < this._farmObjects[mapId].length; i++) {
      if (filterFn(this._farmObjects[mapId][i]) === true) {
        this._farmObjects[mapId][i].clear();
      }
    }

    this.clearEmptyItems();
  }

  static refreshMapEvents() {
    if ($gameMap === null || $gameMap === undefined) return;

    const farmObjects = this._farmObjects[$gameMap._mapId];
    if (farmObjects) {
      for (let i = 0; i < farmObjects.length; i++) {
        if (farmObjects[i]) {
          farmObjects[i].updateEvents();
        }
      }
    }

    this.updateProtectedTiles($gameMap._mapId);
  }

  static getFarmObjectType(modelName) {
    const type = this.farmObjectTypes[modelName];
    if (type !== undefined) {
      return type;
    }

    for (let modelProp in Models) {
      const model = Models[modelProp];

      if (model.modelName == modelName) {
        return model;
      }
    }

    return Models.BaseModel;
  }

  static markObjectAsSpoiled(farmObject) {
    if (farmObject.spoiled) return;
    if (farmObject.stage < 1) return;

    farmObject.spoiled = true;
  }

  static findOldCoopKey() {
    for (let i = 0; i < this._coops.length; i++) {
      const coopData = this._coops[i];
      if (!!coopData && !coopData.used) {
        return i;
      }
    }

    return false;
  }

  static findOldBarnKey() {
    for (let i = 0; i < this._barns.length; i++) {
      const barnData = this._barns[i];

      if (!!barnData && !barnData.used) {
        return i;
      }
    }

    return false;
  }

  static findCoopData(targetMapId) {
    const len = this._coops.length;
    for (let i = 0; i < len; i++) {
      if (this._coops[i].targetMapId == targetMapId) {
        return this._coops[i];
      }
    }

    return false;
  }

  static findBarnData(targetMapId) {
    const len = this._barns.length;
    for (let i = 0; i < len; i++) {
      if (this._barns[i].targetMapId == targetMapId) {
        return this._barns[i];
      }
    }

    return false;
  }

  static deactivateBarn(targetMapId) {
    for (let i = 0; i < this._barns.length; i++) {
      if (this._barns[i].targetMapId != targetMapId) continue;

      this._barns[i].used = false;
      return;
    }
  }

  static deactivateCoop(targetMapId) {
    for (let i = 0; i < this._coops.length; i++) {
      if (this._coops[i].targetMapId != targetMapId) continue;

      this._coops[i].used = false;
      return;
    }
  }

  static registerCoop(mapId, x, y) {
    const oldCoopKey = this.findOldCoopKey();

    if (oldCoopKey !== false) {
      this._coops[oldCoopKey].used = true;
      this._coops[oldCoopKey].x = x;
      this._coops[oldCoopKey].y = y;
      this._coops[oldCoopKey].mapId = mapId;

      return this._coops[oldCoopKey].targetMapId;
    }

    const newId = this._coops.length;

    const coopData = {
      used : true,
      targetMapId : Maps.COOP * 1000 + newId,
      mapId,
      x,
      y
    };

    this._coops.push(coopData);
    return coopData.targetMapId;
  }

  static registerBarn(mapId, x, y) {
    const oldBarnKey = this.findOldBarnKey();
    if (oldBarnKey !== false) {
      this._barns[oldBarnKey].used = true;
      this._barns[oldBarnKey].x = x;
      this._barns[oldBarnKey].y = y;
      this._barns[oldBarnKey].mapId = mapId;

      return this._barns[oldBarnKey].targetMapId;
    }

    const newId = this._barns.length;

    const barnData = {
      used : true,
      targetMapId : Maps.BARN * 1000 + newId,
      mapId,
      x,
      y
    };

    this._barns.push(barnData);
    return barnData.targetMapId;
  }

  static checkMapFarmObjects(mapId) {
    if (this._farmObjects[mapId] === undefined) {
      this._farmObjects[mapId] = [];
    }
    if (this._protectedTiles[mapId] === undefined) {
      this._protectedTiles[mapId] = {};
    }
  }

  static clearMapObjects(mapId, forceErasure) {
    if (mapId == $gameMap._mapId && !forceErasure) return;
    
    this._farmObjects[mapId] = [];
  }

  static eraseFarmObjectsXy(mapId, x, y, keepPlayerObjects = false) {
    const objects = this.getFarmObjectsXy(mapId, x, y);
    let anyFailure = false;

    for (const obj of objects) {
      // To determine if this object was added by the player, we check if there's any tool with the same modelName
      if (keepPlayerObjects === true) {
        if (Managers.Tools.isToolValid(obj.modelName)) {
          anyFailure = true;
          continue;
        }
      }

      obj.clear();
      this.eraseFarmObject(obj);
      obj.updateEvents(true, false);
    }

    if (anyFailure) {
      return false;
    }
  }

  static destroyEverythingOnPositionList(mapId, positionList) {
    for (const { x, y } of positionList) {
      this.eraseFarmObjectsXy(mapId, x, y, false);
    }
  }

  static eraseItemFromPosition(mapId, x, y, itemId) {
    const objects = this.getFarmObjectsXy(mapId, x, y, (farmObject) => {
      return farmObject.itemName == itemId;
    });

    for (const item of objects) {
      this.eraseFarmObject(item, true);
    }
  }

  static replaceObjectOnTile(mapId, x, y, creationMethod) {
    this.eraseFarmObjectsXy(mapId, x, y);
    return creationMethod.call(this, mapId, x, y);
  }

  static eraseFarmObject(farmObject, forceErasure) {
    this.checkMapFarmObjects();

    // Unless the forceErasure arg is true, prevent the method from erasing item events from items without sell price
    if (!forceErasure) {
      if (farmObject && farmObject.itemName) {
        const itemData = Managers.Items.getItemData(farmObject.itemName);
        if (itemData) {
          if (!itemData.sellPrice) return false;
        }
      }
    }

    for (const mapId in this._farmObjects) {
      const len = this._farmObjects[mapId].length;
      for (let i = 0; i < len; i++) {
        if (this._farmObjects[mapId][i] == farmObject) {
          delete this._farmObjects[mapId][i];
          return;
        }
      }
    }
  }

  static clearEmptyItems() {
    this.checkMapFarmObjects();

    for (const mapId in this._farmObjects) {
      for (let i = 0; i < this._farmObjects[mapId].length; i++) {
        const farmObjectData = this._farmObjects[mapId][i];
        if (farmObjectData === undefined || !farmObjectData || farmObjectData.isEmpty()) {
          delete this._farmObjects[mapId][i];
        }
      }
    }

    this.updateProtectedTiles($gameMap._mapId);
  }

  static getFarmObjectList() {
    return this._farmObjects;
  }

  static getData() {
    const farmObjects = {};

    for (const mapId in this._farmObjects) {
      if (!mapId || !this._farmObjects[mapId] || !this._farmObjects[mapId].length) continue;

      farmObjects[mapId] = this._farmObjects[mapId].filter((obj) => !!obj);
    }

    return {
      farmObjects : farmObjects,
      barns : this._barns,
      coops : this._coops
    };
  }

  static setData(data) {
    this.clear();
    if (data) {
      if (data.farmObjects) {
        this._farmObjects = {};
        for (const mapId in data.farmObjects) {
          if (!mapId || !data.farmObjects[mapId] || !data.farmObjects[mapId].length) continue;

          this._farmObjects[mapId] = data.farmObjects[mapId].filter((obj) => !!obj);
        }
      }
      if (data.barns) {
        this._barns = data.barns;
      }
      if (data.coops) {
        this._coops = data.coops;
      }
    }
  }

  static setFarmObjectList(farmObjectList) {
    this._farmObjects = farmObjectList;
  }

  static getBarnList() {
    return this._barns || [];
  }

  static getCoopList() {
    return this._coops || [];
  }

  static setBarnList(barnList) {
    this._barns = barnList;
  }

  static setCoopList(coopList) {
    this._coops = coopList;
  }

  static getMapFarmObjects(mapId) {
    this.checkMapFarmObjects(mapId);

    return this._farmObjects[mapId];
  }

  static isTileUsed(mapId, x, y) {
    if (this.isTileProtected(mapId, x, y)) return true;
    
    const farmObject = this.getFarmObject(mapId, x, y);
    if (farmObject) {
      if (farmObject.hasCollision()) return true;
      if (farmObject.isItem()) return true;
    }

    return false;
  }

  static isTileUsedByAnything(mapId, x, y) {
    if (this.isTileProtected(mapId, x, y)) return true;
    
    const farmObject = this.getFarmObject(mapId, x, y);
    if (farmObject) {
      return true;
    }

    return false;
  }

  static isTileBlockingAnimalMovement(mapId, x, y) {
    if (this.isTileProtected(mapId, x, y)) return true;
    const farmObject = this.getFarmObject(mapId, x, y);
    if (!farmObject) {
      return false;
    }

    if (farmObject.hasCollision()) {
      // #ToDo: ensure open gates don't block this
      return true;
    }

    return false;
  }

  // Check for objects ignoring custom hitboxes;
  // Use this in combination with other filters when you're only interested in objects that you know won't have custom sizes
  static quickFarmObjectsCheck(mapId, x, y, { preFilter, postFilter }, returnFirstObject) {
    return this.farmObjectLookup(mapId, x, y, {
      preFilter,
      postFilter,
      positionFilter: (farmObject) => {
        return farmObject.x == x && farmObject.y == y;
      }
    }, returnFirstObject);
  }

  static getFarmObject(mapId, x, y) {
    return this.farmObjectLookup(mapId, x, y, {}, true);
  }

  //////////////////////
  //////////////////////
  // farmObjectLookup //
  //////////////////////
  //////////////////////
  // Advanced lookup method, provides the possibility of filtering objects at many different stages of the query
  // preFilter - this filter method is run before even checking the positions
  // postFilter - this filter method is run after all basic checks were successful
  // positionFilter - this filter method replaces the default position check
  // typeFilter - this filter method receives the farmObjectType along with the object
  // largeTypeFilter - same as typeFilter but only for large objects; ignored if typeFilter or positionFilter exists
  static farmObjectLookup(mapId, x, y, { preFilter, postFilter, positionFilter, typeFilter, largeTypeFilter }, returnFirstObject) {
    this.checkMapFarmObjects(mapId);

    const list = [];

    for (const farmObject of this._farmObjects[mapId]) {
      if (!farmObject) {
        continue;
      }

      if (preFilter && preFilter(farmObject) === false) {
        continue;
      }

      if (positionFilter) {
        if (positionFilter(farmObject) === false) {
          continue;
        } 

        if (typeFilter) {
          const farmType = farmObject.getFarmObjectType();
          if (!farmType) {
            continue;
          }

          if (typeFilter(farmObject, farmType) === false) {
            continue;
          }
        }
      } else {
        let farmType;
        if (typeFilter) {
          farmType = farmObject.getFarmObjectType();
          if (!farmType) {
            continue;
          }

          if (typeFilter(farmObject, farmType) === false) {
            continue;
          }
        }

        if (farmObject.isLarge()) {
          if (!typeFilter) {
            farmType = farmObject.getFarmObjectType();
          }
          if (!farmType) {
            continue;
          }

          if (!typeFilter && largeTypeFilter && largeTypeFilter(farmObject, farmType) === false) {
            continue;
          }

          const data = farmType.getData(farmObject);
          const hitboxX = farmType.isModel ? farmType.getHitboxX(farmObject, data) : farmType.hitboxX;
          const hitboxY = farmType.isModel ? farmType.getHitboxY(farmObject, data) : farmType.hitboxY;
          const hitboxWidth = farmType.isModel ? farmType.getHitboxWidth(farmObject, data) : farmType.hitboxWidth;
          const hitboxHeight = farmType.isModel ? farmType.getHitboxHeight(farmObject, data) : farmType.hitboxHeight;

          const left = farmObject.x + (hitboxX || 0);
          const right = left + (hitboxWidth || 1);

          const top = farmObject.y + (hitboxY || 0);
          const bottom = top + (hitboxHeight || 1);

          if (Math.ceil(right) < x) continue;
          if (Math.floor(left) > x) continue;
          if (Math.floor(top) > y) continue;
          if (bottom < y) continue;
        } else {
          if (farmObject.x !== x) continue;
          if (farmObject.y !== y) continue;
        }
      }

      if (postFilter && postFilter(farmObject) === false) {
        continue;
      }

      if (returnFirstObject === true) {
        return farmObject;
      }

      list.push(farmObject);
    }

    if (returnFirstObject) {
      return undefined;
    }

    return list;
  }

  static getFarmObjectsByModelAndPosition(mapId, x, y, acceptedModelNames, returnFirstObject) {
    if (Array.isArray(acceptedModelNames)) {
      return this.farmObjectLookup(mapId, x, y, {
        preFilter: (farmObject) => {
          return acceptedModelNames.indexOf(farmObject.modelName) >= 0;
        }
      }, returnFirstObject);
    }

    return this.farmObjectLookup(mapId, x, y, {
      preFilter: (farmObject) => {
        return farmObject.modelName == acceptedModelNames;
      }
    }, returnFirstObject);
  }

  static getFarmObjectsXy(mapId, x, y, preFilter, returnFirstObject) {
    return this.farmObjectLookup(mapId, x, y, { preFilter }, returnFirstObject || false);
  }

  static hasFarmObjectXy(mapId, x, y, filterFn) {
    const anyObject = this.farmObjectLookup(mapId, x, y, {
      postFilter: filterFn
    }, true);

    return Boolean(anyObject);
  }

  static addFarmObject(mapId, newFarmObject) {
    this.checkMapFarmObjects(mapId);

    this._farmObjects[mapId].push(newFarmObject);
    this.updateProtectedTiles($gameMap._mapId);
  }

  static moveFarmObject(oldMapId, newMapId, obj, skipProtectedTiles) {
    this.checkMapFarmObjects(oldMapId);
    this.checkMapFarmObjects(newMapId);

    obj.mapId = newMapId;

    this._farmObjects[newMapId].push(obj);
    const idx = this._farmObjects[oldMapId].indexOf(obj);
    if (idx >= 0) {
      this._farmObjects[oldMapId][idx] = null;
    }

    if (skipProtectedTiles !== false) {
      this.updateProtectedTiles(newMapId);
      this.updateProtectedTiles(oldMapId);
    }
  }

  static createFarmObjectAt(mapId, x, y, modelName) {
    const farmObject = new FarmObject();
    farmObject.mapId = mapId;
    farmObject.x = x;
    farmObject.y = y;

    if (modelName !== undefined) {
      farmObject.modelName = modelName;
    }

    this.addFarmObject(mapId, farmObject);
    return farmObject;
  }

  static createBarnAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = Constants.FarmObjectState.BARN;
    farmObject.offsetY = 6;
    farmObject.width = 5;
    farmObject.height = 6;
    farmObject.protectedAreaX = -2;
    farmObject.protectedAreaY = -5;
    farmObject.targetMapId = this.registerBarn(mapId, x, y);

    if (mapId == $gameMap._mapId) {
      farmObject.updateEvents(false, false);
    }
    this.updateProtectedTiles(mapId);
    return farmObject;
  }

  static createCoopAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = Constants.FarmObjectState.COOP;
    farmObject.offsetY = 6;
    farmObject.width = 3;
    farmObject.height = 3;
    farmObject.protectedAreaX = -1;
    farmObject.protectedAreaY = -2;
    farmObject.targetMapId = this.registerCoop(mapId, x, y);

    if (mapId == $gameMap._mapId) {
      farmObject.updateEvents(false, false);
    }
    this.updateProtectedTiles(mapId);
    return farmObject;
  }

  static createItemAt(mapId, x, y, itemName, amount) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.itemName = itemName;
    farmObject.amount = amount;

    // Don't generate item events while the map isn't fully loaded, because events will be generated automatically at the end.
    if ($gameMap.isMapFullyLoaded()) {
      farmObject.updateEvents(false, false);
    }

    return farmObject;
  }

  static tileIsWater(x, y) {
    const tag = $gameMap.terrainTag(x, y);

    return tag == TerrainTags.WATER;
  }

  static tileIsSoil(x, y) {
    const regionId = $gameMap.regionId(x, y);

    if (regionId >= Region.BENJAMINS_FARM && regionId <= Region.TOWN_FARM) {
      return true;
    }

    const allowedRegions = [
      Region.FARM,
      Region.FARM_WEEDS,
      Region.FARM_ROCK,
      Region.FARM_GRASS,
      Region.FARM_EMPTY,
      Region.FARM_NON_EMPTY,
      Region.FARM_OBSTACLE,
      Region.FARM_FRUIT_TREE,
      Region.FARM_NEW_GRASS,
      Region.FARM_FLOWER
    ];

    return allowedRegions.includes(regionId);
  }

  static createFarmObjectEvent(farmObject, classType) {
    if (farmObject.state == FarmObjectState.NONE) return false;
    if (classType === undefined) classType = Objects.Object;

    $gameTemp.requestFarmObjectsRefresh();
    
    const x = farmObject.x;
    const y = farmObject.y;

    const farmObjectType = farmObject.getFarmObjectType();
    const farmObjectData = farmObjectType.getData(farmObject);

    if (farmObjectType.getObjectClassType && classType == Objects.Object) {
      classType = farmObjectType.getObjectClassType();
    }

    let spriteName = farmObjectType.getSpriteName(farmObject, farmObjectData);
    const spriteIndex = farmObjectType.getSpriteIndex(farmObject, farmObjectData);
    const direction = farmObjectType.getDirection(farmObject, farmObjectData);
    const pattern = farmObjectType.getPattern(farmObject, farmObjectData);
    const priority = farmObjectType.getPriority(farmObject, farmObjectData);
    let tileId = farmObjectType.getTileId(farmObject, farmObjectData);
    let iconIndex = farmObjectType.getIconIndex(farmObject, farmObjectData);
    const through = farmObjectType.getThrough(farmObject, farmObjectData);

    let event;

    if (iconIndex > 0) {
      if (farmObject.invisible) {
        iconIndex = 0;
      }
      event = $gameMap.createStaticIconEventAt(iconIndex, x, y, priority, true, classType, through);
    } else if (tileId > 0) {
      if (farmObject.invisible) {
        tileId = 0;
      }
      event = $gameMap.createStaticTileEventAt(tileId, x, y, priority, true, classType, through);
    } else {
      if (farmObject.invisible) {
        spriteName = '';
      }
      event = $gameMap.createStaticEventAt(spriteName, spriteIndex, x, y, direction, pattern, priority, true, classType, through);
    }

    if (farmObject.invisible) {
      farmObject._noGraphic = true;
    }

    //Offset will be filled in the setFarmObjectData method
    event.offsetX = 0;
    event.offsetY = 0;

    event.farmObjectData = farmObject;
    this.configureCustomTile(farmObjectType, tileId, event, farmObject, false);

    //Make sure the offset is updated
    const sprites = $gameMap.findAllObjectSprites(event);
    if (sprites && sprites.length) {
      for (const sprite of sprites) {
        if (sprite) {
          sprite.updatePosition(true);
        }
      }
    }

    return event;
  }

  static configureCustomTile(farmObjectType, tileId, event, farmObject, isUpdate) {
    const forceSprite = event._forceEventSprite;

    if (!forceSprite && farmObjectType._useCustomTileId && tileId) {
      event._useCustomTileId = true;
      $gameTemp.setCustomTile(farmObject.x, farmObject.y, tileId);

      if (isUpdate) {
        $gameMap.refreshTilemap();
      }
    } else if (!forceSprite && farmObject.shouldShowSoil()) {
      let soilTileId = 382;
      if (farmObject.watered) {
        soilTileId = 383;
      }

      $gameTemp.setCustomTile(farmObject.x, farmObject.y, soilTileId);

      if (isUpdate) {
        $gameMap.refreshTilemap();
      }
    } else {
      event._useCustomTileId = false;
      $gameTemp.clearCustomTile(farmObject.x, farmObject.y);
    }
  }

  static updateFarmObjectEvent(farmObject, event) {
    $gameTemp.requestFarmObjectsRefresh();

    const farmObjectType = farmObject.getFarmObjectType();
    const farmObjectData = farmObjectType.getData(farmObject);

    if (farmObject.getState(false) == FarmObjectState.NONE) {
      event.erase();
      $gameTemp.clearCustomTile(farmObject.x, farmObject.y);
      return event;
    }

    let spriteName = farmObjectType.getSpriteName(farmObject, farmObjectData);
    const spriteIndex = farmObjectType.getSpriteIndex(farmObject, farmObjectData);
    const direction = farmObjectType.getDirection(farmObject, farmObjectData);
    const pattern = farmObjectType.getPattern(farmObject, farmObjectData);
    const priority = farmObjectType.getPriority(farmObject, farmObjectData);
    let tileId = farmObjectType.getTileId(farmObject, farmObjectData);
    let iconIndex = farmObjectType.getIconIndex(farmObject, farmObjectData);
    const through = farmObjectType.getThrough(farmObject, farmObjectData);

    if (farmObject.invisible) {
      spriteName = '';
      tileId = 0;
      iconIndex = 0;
      event._noGraphic = true;
    }

    this.configureCustomTile(farmObjectType, tileId, event, farmObject, true);

    const page = event.page();
    if (page) {
      page.image.characterName = spriteName;
      page.image.characterIndex = spriteIndex;
      page.image.tileId = tileId;
      page.image.direction = direction;
      page.image.pattern = pattern;
      page.image.iconIndex = iconIndex;
      page.priorityType = priority;
      page.through = through;
    }

    event.setupPage();
    event.farmObjectData = farmObject;

    if ((!event._sprite || event._sprite._destroyed) && !event.usesNoGraphic()) {
      $gameMap.addObjectSprite(event);
    } else if (event._sprite && !event._sprite._destroyed) {
      event._sprite.onImageChanged();
    }

    if (event._forceEventSprite && farmObject.shouldShowSoil()) {
      $gameMap.addSoilSprite(event);
    }

    return event;
  }

  static createGrassObjectAt(mapId, x, y) {
    // var farmObject = this.createFarmObjectAt(mapId, x, y);
    // farmObject.modelName = FarmObjectState.GRASS;
  }

  static createBoulderObjectAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = FarmObjectState.BOULDER;
  }

  static createDeadTreeCropAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = 'orange-crop';
    farmObject.stage = 3;
    farmObject.points = 120;
    farmObject.spoiled = true;
  }

  static createBlackberryCropAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = 'blackberry-crop';
    farmObject.stage = 3;
    farmObject.points = 63;
  }

  static createRaspberryCropAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = 'raspberry-crop';
    farmObject.stage = 3;
    farmObject.points = 63;
  }

  static createRandomFruitTreeAt(mapId, x, y) {
    const fruitTypes = ['orange', 'apple', 'peach'];
    const fruitType = fruitTypes[Math.randomInt(fruitTypes.length)];

    return this.createCropAt(mapId, x, y, `${ fruitType }-crop`, 4, 120);
  }

  static createOrangeCropAt(mapId, x, y) {
    return this.createCropAt(mapId, x, y, 'orange-crop', 4, 120);
  }

  static createAppleCropAt(mapId, x, y) {
    return this.createCropAt(mapId, x, y, 'apple-crop', 4, 120);
  }

  static createPeachCropAt(mapId, x, y) {
    return this.createCropAt(mapId, x, y, 'peach-crop', 4, 120);
  }

  static createCoconutCropAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = 'coconut-crop';

    if (Managers.Map.mapMonth(mapId) == Seasons.SUMMER) {
      farmObject.stage = 4;
      farmObject.points = 120;
    } else {
      farmObject.stage = 3;
      farmObject.points = 69;
    }
  }

  static createGrownCoconutCropAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = 'coconut-crop';
    farmObject.stage = 4;
    farmObject.points = 120;
  }

  static createDesertTreeAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = 'desert-tree';
    farmObject.stage = 3;
    farmObject.points = 120;
  }

  static createCropAt(mapId, x, y, modelName, stage=3, points=2000) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = modelName;
    farmObject.stage = stage;
    farmObject.points = points;

    return farmObject;
  }

  static createGreenTreeCropAt(mapId, x, y) {
    const types = ['regular-tree-3', 'regular-tree-6'];
    return this.createCropAt(mapId, x, y, types[Math.randomInt(types.length)], 3, 2000);
  }

  static createPinkTreeCropAt(mapId, x, y) {
    const types = ['regular-tree', 'regular-tree-5'];
    return this.createCropAt(mapId, x, y, types[Math.randomInt(types.length)], 3, 2000);
  }

  static createMediumGreenTreeCropAt(mapId, x, y) {
    const types = ['medium-tree-3', 'medium-tree-6'];
    return this.createCropAt(mapId, x, y, types[Math.randomInt(types.length)], 2, 10);
  }

  static createMediumPinkTreeCropAt(mapId, x, y) {
    const types = ['medium-tree', 'medium-tree-5'];
    return this.createCropAt(mapId, x, y, types[Math.randomInt(types.length)], 2, 10);
  }

  static createSmallGreenTreeCropAt(mapId, x, y) {
    const types = ['small-tree-3', 'small-tree-6'];
    return this.createCropAt(mapId, x, y, types[Math.randomInt(types.length)], 2, 10);
  }

  static createSmallPinkTreeCropAt(mapId, x, y) {
    const types = ['small-tree', 'small-tree-5'];
    return this.createCropAt(mapId, x, y, types[Math.randomInt(types.length)], 2, 10);
  }

  static createGiantTreeCropAt(mapId, x, y) {
    return this.createCropAt(mapId, x, y, 'giant-tree', 3, 2000);
  }

  static createWeedObjectAt(mapId, x, y) {
    if (mapId == Maps.CAVE_FARM) return;

    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = FarmObjectState.WEED;
  }

  static createLogObjectAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = FarmObjectState.LOG;
  }

  static createStumpObjectAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = FarmObjectState.STUMP;
  }

  static createTermiteNestAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = FarmObjectState.TERMITE_NEST;
  }

  static createMiningRockObjectAt(mapId, x, y, rockType) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.rockType = rockType;
  }

  static createDailyMineItemAt(mapId, x, y, levelType) {
    const chance = Math.randomInt(100);
    //#ToDo: Create some kind of luck system that changes this
    let baseChance = 70;

    if (chance < (baseChance - levelType * 2)) return;

    const methodName = `createLevel${levelType}MineItemAt`;
    if (!this[methodName]) return;

    this[methodName].call(this, mapId, x, y);
  }

  static createLevel0MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);

    if (chance < 66) {
      this.createStoneObjectAt(mapId, x, y);
      return;
    }

    if (chance < 70) {
      // this.createItemAt(mapId, x, y, 'rock', 1);
      return;
    }

    if (chance < 88) {
      this.createMiningRockObjectAt(mapId, x, y, 'clay');
      return;
    }

    if (chance < 90) {
      this.createItemAt(mapId, x, y, 'clay', 1);
      return;
    }

    if (chance < 100) {
      this.createMiningRockObjectAt(mapId, x, y, 'iron-ore');
      return;
    }
  }

  static createLevel1MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);

    if (chance < 46) {
      this.createStoneObjectAt(mapId, x, y);
      return;
    }

    if (chance < 50) {
      return;
    }

    if (chance < 73) {
      this.createMiningRockObjectAt(mapId, x, y, 'clay');
      return;
    }

    if (chance < 75) {
      this.createItemAt(mapId, x, y, 'clay', 1);
      return;
    }

    if (chance < 85) {
      return;
    }
    
    if (chance < 100) {
      this.createMiningRockObjectAt(mapId, x, y, 'copper-ore');
      return;
    }
  }

  static createLevel2MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);

    if (chance < 40) {
      return;
    }

    if (chance < 55) {
      this.createMiningRockObjectAt(mapId, x, y, 'clay');
      return;
    }

    if (chance < 58) {
      this.createItemAt(mapId, x, y, 'clay', 1);
      return;
    }

    if (chance < 67) {
      return;
    }

    if (chance < 83) {
      if (Math.randomInt(100) >= 96) {
        this.createMiningRockObjectAt(mapId, x, y, 'blue-copper-ore');
      } else {
        this.createMiningRockObjectAt(mapId, x, y, 'copper-ore');
      }
      return;
    }
    
    if (chance < 100) {
      if (Math.randomInt(100) >= 99) {
        this.createMiningRockObjectAt(mapId, x, y, 'blue-silver-ore');
      } else {
        this.createMiningRockObjectAt(mapId, x, y, 'silver-ore');
      }
      return;
    }
  }

  static createLevel3MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);

    if (chance < 30) {
      return;
    }

    if (chance < 50) {
      this.createMiningRockObjectAt(mapId, x, y, 'copper-ore');
      return;
    }

    if (chance < 70) {
      return;
    }

    if (chance < 100) {
      this.createMiningRockObjectAt(mapId, x, y, 'silver-ore');
      return;
    }
  }

  static createLevel4MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);

    if (chance < 80) return;

    if (chance < 90) {
      this.createMiningRockObjectAt(mapId, x, y, 'clay');
      return;
    }

    if (chance < 100) {
      this.createMiningRockObjectAt(mapId, x, y, 'gold-ore');
      return;
    }
  }

  static createLevel5MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);

    if (chance < 84) return;

    if (chance < 99) {
      this.createMiningRockObjectAt(mapId, x, y, 'gold-ore');
      return;
    }

    if (chance < 100) {
      this.createMiningRockObjectAt(mapId, x, y, 'ruby');
      return;
    }
  }

  static createLevel6MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);

    if (chance < 84) return;

    if (chance < 90) {
      this.createMiningRockObjectAt(mapId, x, y, 'gold-ore');
      return;
    }

    if (chance < 100) {
      this.createMiningRockObjectAt(mapId, x, y, 'ruby');
      return;
    }
  }

  static createLevel7MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);
    if (chance < 98) return;

    this.createMiningRockObjectAt(mapId, x, y, 'sapphire');
  }

  static createLevel8MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);
    if (chance < 95) return;

    this.createMiningRockObjectAt(mapId, x, y, 'emerald');
  }

  static createLevel9MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);
    if (chance < 70) return;
    this.createMiningRockObjectAt(mapId, x, y, 'blue-copper-ore');
  }

  static createLevel10MineItemAt(mapId, x, y) {
    const chance = Math.randomInt(100);
    if (chance < 70) return;
    this.createMiningRockObjectAt(mapId, x, y, 'blue-silver-ore');
  }

  static createLevel11MineItemAt(mapId, x, y) {
  }

  static createLevel12MineItemAt(mapId, x, y) {
  }

  static createLevel13MineItemAt(mapId, x, y) {
  }

  static createLevel14MineItemAt(mapId, x, y) {
  }

  static createLevel15MineItemAt(mapId, x, y) {
  }

  static createLevel16MineItemAt(mapId, x, y) {
  }

  static createLevel17MineItemAt(mapId, x, y) {
  }

  static createLevel18MineItemAt(mapId, x, y) {
  }

  static createLevel19MineItemAt(mapId, x, y) {
  }

  static createButtonObjectAt(mapId, x, y, invisible, immuneToPlayer, oneTime) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = FarmObjectState.BUTTON;
    farmObject.invisible = !!invisible;
    farmObject.immuneToPlayer = !!immuneToPlayer;
    farmObject.oneTime = !!oneTime;
    farmObject.pressed = false;

    return farmObject;
  }

  static createStoneObjectAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);
    farmObject.modelName = FarmObjectState.STONE;
  }

  static createFlowerObjectAt(mapId, x, y) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);

    switch (Managers.Map.mapMonth(mapId)) {
      case Seasons.SPRING:
        farmObject.modelName = 'pink-daisy-crop';
        break;
      case Seasons.SUMMER:
        farmObject.modelName = 'sunflower-crop';
        break;
      case Seasons.FALL:
        farmObject.modelName = 'rose-crop';
        break;
      case Seasons.WINTER:
        farmObject.modelName = 'white-rose-crop';
        break;
    }
  }

  static setupTempObject(x, y, d, characterName, characterIndex, offsetY) {
    $gameMap.destroyAnythingAt(x, y);
    const event = $gameMap.createStaticEventAt(characterName, characterIndex, x, y, d, 0, EventPriorities.NORMAL, true, Objects.PushableObject);
    if (offsetY) {
      event.offsetY = offsetY;
    }

    return event;
  }

  static setupTempEvents() {
    for (let x = 0; x < $gameMap.width(); x++) {
      for (let y = 0; y < $gameMap.height(); y++) {
        const regionId = $gameMap.regionId(x, y);

        if (Managers.Map.canAddTempEvent(x, y, regionId) === false) continue;

        switch(regionId) {
          case Region.OPEN_BARREL :
            this.setupTempObject(x, y, 4, '$objects', 0, -8);
            break;
          case Region.CLOSED_BARREL :
            this.setupTempObject(x, y, 2, '$objects', 0, -8);
            break;
          case Region.BOX :
            this.setupTempObject(x, y, 6, '$objects', 0, -8);
            break;
          case Region.RARE_SHINY_ITEM :
            if (Utils.chance(Math.ceil(Variables.shinyTrinketSpawnRate / 50))) {
              Variables.shinyTrinketSpawnRate = 0;
              $gameMap.makeShinyEvent(x, y);
            } else {
              Variables.shinyTrinketSpawnRate++;
            }
            break;
          case Region.COMMON_SHINY_ITEM :
            if (Utils.chance(Math.ceil(Variables.shinyTrinketSpawnRate / 10))) {
              Variables.shinyTrinketSpawnRate = 0;
              $gameMap.makeShinyEvent(x, y);
            } else {
              Variables.shinyTrinketSpawnRate += 5;
            }
            break;
        }
      }
    }
  }

  static setupDailyFarm() {
    for (let x = 0; x < $gameMap.width(); x++) {
      for (let y = 0; y < $gameMap.height(); y++) {

        if (!this.tileIsSoil(x, y)) continue;

        if (!TileHelper.checkIfTileCanGetObject(x, y, false, '', false)) continue;
        if ($gameMap.eventsXy(x, y).length > 0) continue;
        if (this.isTileUsedByAnything($gameMap._mapId, x, y)) continue;

        const idx = Math.randomInt(100);
        if (idx >= 5) continue;

        this.setupRandomFarmEventAt($gameMap._mapId, x, y, true);
      }
    }
  }

  static regionIdIsItem(regionId) {
    if (regionId >= Region.DAILY_MINE_LEVEL_0 && regionId <= Region.DAILY_MINE_LEVEL_19) {
      return true;
    }

    switch(regionId) {
      case Region.DAILY_ITEM :
      case Region.DAILY_FLOWER :
      case Region.RANDOM_ITEM :
      case Region.DAILY_STICK :
      case Region.DAILY_GOOD_ITEM :
      case Region.DAILY_BEACH_ITEM :
      case Region.RANDOM_GOOD_ITEM :
      case Region.DAILY_BAD_ITEM :
      case Region.RANDOM_BAD_ITEM :
      case Region.DAILY_CAVE_ITEM :
      case Region.RANDOM_CAVE_ITEM :
      case Region.DAILY_GOLD_MINE_ITEM :
      case Region.RANDOM_GOLD_MINE_ITEM :
      case Region.DAILY_RUINS_ITEM :
      case Region.DAILY_ROCK :
      case Region.DAILY_CLAY_ROCK :
      case Region.RANDOM_CLAY_ROCK :
      case Region.CLAY_ROCK_ON_STARTUP :
      case Region.DAILY_IRON_ROCK :
      case Region.RANDOM_IRON_ROCK :
      case Region.DAILY_SILVER_ROCK :
      case Region.DAILY_LOG :
      case Region.DAILY_WEED :
      case Region.DAILY_GRASS :
      case Region.DAILY_STUMP :
      case Region.RANDOM_TERMITE_NEST :
      case Region.DAILY_TERMITE_NEST :
      case Region.DAILY_FALLEN_TREE :
        return true;
    }

    return false;
  }

  static createRegionItem(mapId, x, y, regionId) {
    if (regionId >= Region.DAILY_MINE_LEVEL_0 && regionId <= Region.DAILY_MINE_LEVEL_19) {
      this.createDailyMineItemAt(mapId, x, y, regionId - Region.DAILY_MINE_LEVEL_0);
      return;
    }

    const randomWeekDay = (Managers.Time.weekDay + Math.randomInt(7)) % 7;
    const rnd = Math.randomInt(10);

    switch(regionId) {
      case Region.DAILY_ITEM :
        if (rnd > 3) {
          this.createSimpleItemAt(mapId, x, y, randomWeekDay % 4);
        }
        break;
      case Region.DAILY_FLOWER :
        if (rnd === 5) {
          this.createFlowerItemAt(mapId, x, y, Managers.Time.weekDay % 5);
        } else {
          this.createFlowerItemAt(mapId, x, y, Managers.Time.weekDay % 4);
        }

        break;
      case Region.RANDOM_ITEM :
        if (rnd > 7) {
          this.createSimpleItemAt(mapId, x, y, 1);
        }
        break;
      case Region.DAILY_STICK :
        this.createItemAt(mapId, x, y, 'wooden-stick', 1);
        break;
      case Region.DAILY_GOOD_ITEM :
        if (rnd > 3) {
          if (rnd <= 6) {
            this.createFruitItemAt(mapId, x, y, Managers.Time.weekDay % rnd);
          } else {
            this.createFruitItemAt(mapId, x, y, Managers.Time.weekDay % 4);
          }
        }
        break;
      case Region.DAILY_BEACH_ITEM :
        if (rnd > 3) {
          this.createSimpleItemAt(mapId, x, y, 5);
        }
        break;
      case Region.RANDOM_GOOD_ITEM :
        if (rnd > 7) {
          this.createFruitItemAt(mapId, x, y, randomWeekDay % 4);
        }
        break;
      case Region.DAILY_BAD_ITEM :
        this.createSimpleItemAt(mapId, x, y, 0);
        
        break;
      case Region.RANDOM_BAD_ITEM :
        if (rnd > 8) {
          this.createSimpleItemAt(mapId, x, y, 0);
        }
        break;

      case Region.DAILY_CAVE_ITEM :
        this.createCaveItemAt(mapId, x, y, 0);
        break;
      case Region.RANDOM_CAVE_ITEM :
        if (rnd > 6) {
          this.createCaveItemAt(mapId, x, y, 2);
        }
        break;
      case Region.DAILY_GOLD_MINE_ITEM :
        if (rnd > 3) {
          this.createCaveItemAt(mapId, x, y, 4);
        }
        break;
      case Region.RANDOM_GOLD_MINE_ITEM :
        if (rnd > 8) {
          this.createCaveItemAt(mapId, x, y, 5);
        }
        break;

      case Region.DAILY_RUINS_ITEM :
        if (rnd > 3) {
          this.createSimpleItemAt(mapId, x, y, 4);
        }
        
        break;

      case Region.DAILY_ROCK :
        this.createStoneObjectAt(mapId, x, y);
        break;
      case Region.DAILY_CLAY_ROCK :
        this.createMiningRockObjectAt(mapId, x, y, 'clay');
        break;
      case Region.RANDOM_CLAY_ROCK :
      case Region.CLAY_ROCK_ON_STARTUP :
        if (rnd > 6) {
          this.createMiningRockObjectAt(mapId, x, y, 'clay');
        }
        break;
      case Region.DAILY_IRON_ROCK :
        this.createMiningRockObjectAt(mapId, x, y, 'iron-ore');
        break;
      case Region.RANDOM_IRON_ROCK :
        if (rnd > 6) {
          this.createMiningRockObjectAt(mapId, x, y, 'iron-ore');
        }
        break;
      case Region.DAILY_SILVER_ROCK :
        this.createMiningRockObjectAt(mapId, x, y, 'silver-ore');
        break;
      case Region.DAILY_LOG :
        this.createLogObjectAt(mapId, x, y);
        break;
      case Region.DAILY_WEED :
        if (Managers.Map.mapMonth(mapId) != Seasons.WINTER) {
          this.createWeedObjectAt(mapId, x, y);
        }
        break;
      case Region.DAILY_GRASS :
        this.createGrassObjectAt(mapId, x, y);
        break;
      case Region.DAILY_STUMP :
        this.createStumpObjectAt(mapId, x, y);
        break;
      case Region.RANDOM_TERMITE_NEST :
        if (Math.randomInt(10) >= 8) {
          this.createTermiteNestAt(mapId, x, y);
        }
        break;
      case Region.DAILY_TERMITE_NEST :
        this.createTermiteNestAt(mapId, x, y);
        break;
      case Region.DAILY_FALLEN_TREE :
        this.createFarmObjectAt(mapId, x, y).modelName = 'fallen-tree';
        break;
    }
  }

  static setupDailyRegionItem(mapId, x, y, regionId) {
    const allowProtected = [Region.DAILY_GRASS].includes(regionId);

    if (!this.regionIdIsItem(regionId)) {
      return;
    }

    // If it fails to erase the object currently in that spot, then give up trying to add a daily item to it
    if (this.eraseFarmObjectsXy(mapId, x, y, true) === false) {
      return;
    }

    if (!TileHelper.checkIfTileCanGetObject(x, y, false, '', allowProtected)) return;
    if ($gameMap.eventsXy(x, y).length > 0) return;
    if (this.isTileUsed($gameMap._mapId, x, y)) return;

    this.createRegionItem($gameMap._mapId, x, y, regionId);
  }

  static createCaveItemAt(mapId, x, y, itemLevel) {
    const items = ['clay', 'rock'];

    if (itemLevel >= 1) {
      items.push('iron-ore');
      
      if (itemLevel == 2 || itemLevel >= 6) {
        items.push('copper-ore');
      }

      if (itemLevel == 3  || itemLevel >= 6) {
        if (itemLevel == 3) {
          items.push('clay');
          items.push('clay');
        }
        items.push('silver-ore');
      }

      if (itemLevel >= 4) {
        if (itemLevel < 6) {
          items.push('clay');
          items.push('clay');
          items.push('clay');
        }

        items.push('gold-nuggets');
        
        if (itemLevel >= 5) {
          if (itemLevel < 6) {
            items.push('clay');
            items.push('clay');
            items.push('clay');
          }

          items.push('gold-ore');
        }
      }
    }

    const idx = Math.randomInt(items.length);
    const itemId = items[idx];

    if (!itemId) return false;
    return this.createItemAt(mapId, x, y, itemId);
  }

  static createSimpleItemAt(mapId, x, y, itemLevel) {
    const item1 = ['wooden-stick', 'orange', 'orange', 'peanut'];
    const item2 = ['peanut', 'half-coconut', 'mushroom', 'coin'];
    let item3 = [false, false, 'red-mushroom', 'broken-pearl'];
    let item4 = ['orange', 'currant', 'leaf', 'coin'];
    
    if (Managers.Time.year >= 2 && Utils.whatever()) {
      item4 = ['cocoa', 'cocoa-beans', 'green-mushroom', false];
    }
    if (Managers.Time.year >= 3 && Utils.whatever()) {
      item3 = [false, false, 'blue-mushroom', 'white-pearl'];
      
      item2[3] = 'papaya';
      item4[3] = 'silver-coin';
    }

    const itemRuins = ['spoiled-orange', 'snake-venom', 'spoiled-apple', 'frog-legs'];
    if (Utils.whatever()) {
      itemRuins[1] = 'snake-skin';
    }
    const itemBeach = ['seashell', 'clam'];

    let itemId;

    const month = Managers.Map.mapMonth(mapId) - 1;

    switch(itemLevel) {
      case 0:
        itemId = item1[month];
        break;
      case 1:
        itemId = item2[month];
        break;
      case 2:
        itemId = item3[month];
        break;
      case 3:
        itemId = item4[month];
        break;
      case 4:
        itemId = itemRuins[Math.randomInt(itemRuins.length)];
        break;
      case 5:
        itemId = itemBeach[Math.randomInt(itemBeach.length)];
        break;
      default:
        itemId = false;
        break;
    }

    if (!itemId) return false;
    return this.createItemAt(mapId, x, y, itemId);
  }

  static createFruitItemAt(mapId, x, y, fruitLevel) {
    const fruit1 = ['peach', 'banana', 'apple', 'orange'];
    let fruit2 = ['cherry', 'coconut', 'blackberry', 'persimmon'];
    let fruit3 = ['raspberry', 'pear', 'plum', 'starfruit'];
    let fruit4 = ['apricot', 'passion-fruit', 'chestnut', 'pine-nut'];
    
    if (Managers.Time.year >= 2 && Utils.whatever()) {
      fruit4 = ['acai-berry', 'jackfruit', 'pistachio', 'cupuacu'];
      fruit2 = ['guarana', 'fig', 'golden-honeydew', 'pine-nut'];
    }

    let fruit5 = ['yellow-mombin', 'blueberry', 'blackberry', 'persimmon'];

    if (Managers.Time.year >= 3 && Utils.whatever()) {
      fruit3 = ['jabuticaba', 'guava', 'golden-honeydew', 'cupuacu'];
      fruit5 = ['cashew', 'sour-sugar-apple', 'sugar-apple', 'papaya'];
    }

    const fruit6 = ['mango', 'passion-fruit', 'chestnut', 'pine-nut'];


    let itemId;
    const month = Managers.Map.mapMonth(mapId) - 1;

    switch(fruitLevel) {
      case 0:
        itemId = fruit1[month];
        break;
      case 1:
        itemId = fruit2[month];
        break;
      case 2:
        itemId = fruit3[month];
        break;
      case 3:
        itemId = fruit4[month];
        break;
      case 4:
        itemId = fruit5[month];
        break;
      case 5:
        itemId = fruit6[month];
        break;
      default:
        itemId = false;
        break;
    }

    if (!itemId) return false;

    return this.createItemAt(mapId, x, y, itemId);
  }

  static createFlowerItemAt(mapId, x, y, flowerLevel) {
    const flower1 = ['jasmine', 'yellow-daisy', 'daisy', false];
    const flower2 = ['pink-daisy', 'yellow-daisy', false, false];
    const flower3 = ['daisy', false, false, false];
    const flower4 = ['pink-daisy', 'sunflower', false, 'white-rose'];
    const flower5 = ['pink-daisy', 'sunflower', 'rose', 'white-rose'];

    let itemId;
    const month = Managers.Map.mapMonth(mapId) - 1;

    switch(flowerLevel) {
      case 0:
        itemId = flower1[month];
        break;
      case 1:
        itemId = flower2[month];
        break;
      case 2:
        itemId = flower3[month];
        break;
      case 3:
        itemId = flower4[month];
        break;
      case 4:
        itemId = flower5[month];
        break;
      default:
        itemId = false;
        break;
    }

    if (!itemId) return false;

    return this.createItemAt(mapId, x, y, itemId);
  }

  static setupDailyItems() {
    Managers.Map.setupDailyItems();

    for (let x = 0; x < $gameMap.width(); x++) {
      for (let y = 0; y < $gameMap.height(); y++) {
        const regionId = $gameMap.regionId(x, y);

        this.setupDailyRegionItem($gameMap._mapId, x, y, regionId);

        this.checkButtonRegions($gameMap._mapId, x, y, regionId, true);
      }
    }

    this.setupDailyFarm();
  }

  static restoreMapTrees() {
    const width = $gameMap.width();
    const height = $gameMap.height();

    for (let x = 0; x < width; x++) {
      for (let y = 0; y < height; y++) {
        const regionId = $gameMap.regionId(x, y);

        this.checkTreeRegions($gameMap._mapId, x, y, regionId, true);
      }
    }
  }

  static setupInitialEvents() {
    const checkForExisting = this._farmObjects[$gameMap._mapId] && this._farmObjects[$gameMap._mapId].length;
    const width = $gameMap.width();
    const height = $gameMap.height();

    for (let x = 0; x < width; x++) {
      for (let y = 0; y < height; y++) {
        const regionId = $gameMap.regionId(x, y);
        const regionsAllowProtected = [Region.FARM_GRASS, Region.GRASS_ON_STARTUP];

        const allowProtected = regionsAllowProtected.includes(regionId);
        if (!allowProtected) {
          if (this.isTileProtected($gameMap._mapId, x, y)) continue;
        }

        if (checkForExisting) {
          if (!TileHelper.checkIfTileCanGetObject(x, y, false, '', false)) continue;
          if ($gameMap.eventsXy(x, y).length > 0) continue;
          if (this.isTileUsedByAnything($gameMap._mapId, x, y)) continue;
        }

        switch(regionId) {
          case Region.FARM :
            this.setupRandomFarmEventAt($gameMap._mapId, x, y, false);
            break;
          case Region.GRASSY_FARM :
            this.setupRandomGrassyFarmEventAt($gameMap._mapId, x, y);
            break;
          case Region.FARM_OBSTACLE :
            this.createRandomObstacleAt($gameMap._mapId, x, y);
            break;
          case Region.FARM_NON_EMPTY :
            this.createNonEmptyFarmItemAt($gameMap._mapId, x, y);
            break;
          case Region.FARM_NEW_GRASS :
            this.createNewGrassObjectAt($gameMap._mapId, x, y);
            break;
          case Region.BOULDER_ON_STARTUP :
            this.createBoulderObjectAt($gameMap._mapId, x, y);
            break;            
          case Region.THICK_WEEDS_ON_STARTUP :
            this.createFarmObjectAt($gameMap._mapId, x, y).modelName = 'thick-weed';
            break;            
          case Region.LIGHT_THICK_WEEDS_ON_STARTUP :
            this.createFarmObjectAt($gameMap._mapId, x, y).modelName = 'light-thick-weed';
            break;            
          case Region.ROCK1_ON_STARTUP :
            this.createFarmObjectAt($gameMap._mapId, x, y).modelName = 'rock-1';
            break;            
          case Region.ROCK2_ON_STARTUP :
            this.createFarmObjectAt($gameMap._mapId, x, y).modelName = 'rock-2';
            break;            
          case Region.ROCK3_ON_STARTUP :
            this.createFarmObjectAt($gameMap._mapId, x, y).modelName = 'rock-3';
            break;            
          case Region.ROCK4_ON_STARTUP :
            this.createFarmObjectAt($gameMap._mapId, x, y).modelName = 'rock-4';
            break;
          case Region.FALLEN_TREE_ON_STARTUP :
            this.createFarmObjectAt($gameMap._mapId, x, y).modelName = 'fallen-tree';
            break;
          case Region.ROCK_ON_STARTUP :
          case Region.FARM_ROCK :
            this.createStoneObjectAt($gameMap._mapId, x, y);
            break;
          case Region.FARM_FLOWER :
            this.createFlowerObjectAt($gameMap._mapId, x, y);
            break;
          case Region.CLAY_ROCK_ON_STARTUP :
            this.createMiningRockObjectAt($gameMap._mapId, x, y, 'clay');
            break;
          case Region.STUMP_ON_STARTUP :
            this.createStumpObjectAt($gameMap._mapId, x, y);
            break;
          case Region.LOG_ON_STARTUP :
            this.createLogObjectAt($gameMap._mapId, x, y);
            break;
          case Region.GRASS_ON_STARTUP :
          case Region.FARM_GRASS :
            this.createGrassObjectAt($gameMap._mapId, x, y);
            break;
          case Region.WEEDS_ON_STARTUP :
            if (Managers.Time.mapMonth != Seasons.WINTER) {
              this.createWeedObjectAt($gameMap._mapId, x, y);
            }
            break;
          case Region.FARM_WEEDS :
            if (Managers.Time.mapMonth != Seasons.WINTER) {
              this.createWeedObjectAt($gameMap._mapId, x, y);
            }
            break;
          case Region.FARM_FRUIT_TREE :
            this.createRandomFruitTreeAt($gameMap._mapId, x, y);
            break;
          case Region.BLACKBERRY_BUSH_ON_STARTUP :
            this.createBlackberryCropAt($gameMap._mapId, x, y);
            break;
          case Region.RASPBERRY_BUSH_ON_STARTUP :
            this.createRaspberryCropAt($gameMap._mapId, x, y);
            break;
        }

        this.checkTreeRegions($gameMap._mapId, x, y, regionId, false);
        this.checkButtonRegions($gameMap._mapId, x, y, regionId, false);
      }
    }
  }

  static checkButtonRegions(mapId, x, y, regionId, checkIfExists) {
    if (checkIfExists) {
      const existingButton = this.getFarmObjectsByModelAndPosition(mapId, x, y, FarmObjectState.BUTTON, true);
      if (existingButton) return;
    }

    switch (regionId) {
      case Region.BUTTON :
        this.createButtonObjectAt(mapId, x, y);
        break;
      case Region.INVISIBLE_BUTTON :
        this.createButtonObjectAt(mapId, x, y, true);
        break;
      case Region.NO_PLAYER_BUTTON :
        this.createButtonObjectAt(mapId, x, y, false, true);
        break;
      case Region.ONE_TIME_BUTTON :
        this.createButtonObjectAt(mapId, x, y, false, false, true);
        break;
    }
  }

  static checkTreeRegions(mapId, x, y, regionId, checkIfExists) {
    if (checkIfExists) {
      const existingTree = this.farmObjectLookup(mapId, x, y, {
        preFilter: (farmObject) => {
          return farmObject.isTree();
        }
      }, true);

      if (existingTree) return;
    }

    switch(regionId) {
      case Region.DEAD_TREE_ON_STARTUP :
        this.createDeadTreeCropAt(mapId, x, y);
        break;
      case Region.ORANGE_TREE_ON_STARTUP :
        this.createOrangeCropAt(mapId, x, y);
        break;
      case Region.APPLE_TREE_ON_STARTUP :
        this.createAppleCropAt(mapId, x, y);
        break;
      case Region.PEACH_TREE_ON_STARTUP :
        this.createPeachCropAt(mapId, x, y);
        break;
      case Region.COCONUT_TREE_ON_STARTUP :
        this.createCoconutCropAt(mapId, x, y);
        break;
      case Region.TREE_ON_STARTUP_BACKWARD_COMPATIBILITY :
      case Region.TREE_ON_STARTUP :
      case Region.GREEN_TREE_ON_STARTUP:
        this.createGreenTreeCropAt(mapId, x, y);
        break;

      case Region.PINK_TREE_ON_STARTUP:
        this.createPinkTreeCropAt(mapId, x, y);
        break;
      case Region.BLUE_TREE_ON_STARTUP:
        this.createCropAt(mapId, x, y, 'regular-tree-2');
        break;
      case Region.HIGHER_GREEN_TREE_ON_STARTUP:
        this.createCropAt(mapId, x, y, 'regular-tree-4');
        break;

      case Region.MEDIUM_GREEN_TREE_ON_STARTUP :
        this.createMediumGreenTreeCropAt(mapId, x, y);
        break;
      case Region.MEDIUM_PINK_TREE_ON_STARTUP :
        this.createMediumPinkTreeCropAt(mapId, x, y);
        break;
      case Region.MEDIUM_BLUE_TREE_ON_STARTUP :
        this.createCropAt(mapId, x, y, 'medium-tree-2');
        break;
      case Region.HIGHER_MEDIUM_GREEN_TREE_ON_STARTUP:
        this.createCropAt(mapId, x, y, 'medium-tree-4');
        break;

      case Region.SMALL_GREEN_TREE_ON_STARTUP :
        this.createSmallGreenTreeCropAt(mapId, x, y);
        break;
      case Region.SMALL_PINK_TREE_ON_STARTUP :
        this.createSmallPinkTreeCropAt(mapId, x, y);
        break;
      case Region.SMALL_BLUE_TREE_ON_STARTUP :
        this.createCropAt(mapId, x, y, 'small-tree-2');
        break;
      case Region.HIGHER_SMALL_GREEN_TREE_ON_STARTUP:
        this.createCropAt(mapId, x, y, 'small-tree-4');
        break;

      case Region.IPE_TREE:
        this.createCropAt(mapId, x, y, 'ipe').ipeIndex = Math.randomInt(6);
        break;

      case Region.GIANT_TREE_ON_STARTUP :
        this.createGiantTreeCropAt(mapId, x, y);
        break;
      case Region.DESERT_TREE_ON_STARTUP :
        this.createDesertTreeAt(mapId, x, y);
        break;
    }
  }

  static setupRandomVillagerFarm(farmObject, villagerName) {
    const crops = ['turnip', 'cucumber', 'potato', 'cabbage', 'strawberry'];
    const idx = Math.randomInt(crops.length);
    let crop = crops[idx];

    if (villagerName == 'Lucas') {
      crop = 'potato';
    }

    farmObject.modelName = `${ crop}-crop`;
    farmObject.tilled = true;
    farmObject.stage = 3;
    farmObject.points = 120;
  }

  static addVillagerFarmAt(villagerName, mapId, x, y, regionId) {
    const farmObject = this.createFarmObjectAt(mapId, x, y);

    switch(regionId) {
      case Region.TOWN_TREE:
        farmObject.modelName = 'regular-tree';
        farmObject.stage = 3;
        farmObject.points = 120;
        break;
      case Region.TOWN_ORANGE_TREE:
        farmObject.modelName = 'orange-crop';
        farmObject.stage = 3;
        farmObject.points = 120;
        break;
      default:
        this.setupRandomVillagerFarm(farmObject, villagerName);
        break;
    }
  }

  static initializeVillagerFarms() {
    for (let x = 0; x < $gameMap.width(); x++) {
      for (let y = 0; y < $gameMap.height(); y++) {
        const regionId = $gameMap.regionId(x, y);
        let villagerName = '';

        switch(regionId) {
          case Region.BENJAMINS_FARM :
            villagerName = 'Benjamin';
            break;
          case Region.BRITTANYS_FARM :
            villagerName = 'Brittany';
            break;
          case Region.MIAS_FARM :
            villagerName = 'Mia';
            break;
          case Region.NATHALIAS_FARM :
            villagerName = 'Nathalia';
            break;
          case Region.DEVINS_FARM :
            villagerName = 'Devin';
            break;
          case Region.SERGES_FARM :
            villagerName = 'Serge';
            break;
          case Region.BONNIES_FARM :
            villagerName = 'Bonnie';
            break;
          case Region.LUCAS_FARM :
            villagerName = 'Lucas';
            break;
          case Region.JULIAS_FARM :
            villagerName = 'Julia';
            break;
          case Region.TOWN_TREE :
          case Region.TOWN_ORANGE_TREE :
            villagerName = 'Town';
            break;
          default:
            if (regionId > Region.BENJAMINS_FARM && regionId < Region.TOWN_FARM) {
              villagerName = 'Someone';
            }
            continue;
        }

        this.addVillagerFarmAt(villagerName, $gameMap._mapId, x, y, regionId);
      }
    }
  }

  static createNonEmptyFarmItemAt(mapId, x, y) {
    if (Math.floor(Math.random() * 4) === 0) {
      this.createGrassObjectAt(mapId, x, y);
    } else {
      this.createRandomObstacleAt(mapId, x, y);
    }
  }

  static createRandomObstacleAt(mapId, x, y) {
    switch(Math.randomInt(3)) {
      case 0 :
        this.createStoneObjectAt(mapId, x, y);
        break;
      case 1 :
        this.createLogObjectAt(mapId, x, y);
        break;
      case 2 :
        if (Managers.Map.mapMonth(mapId) != Seasons.WINTER) {
          this.createWeedObjectAt(mapId, x, y);
        } else {
          this.createLogObjectAt(mapId, x, y);
        }
        break;
    }
  }

  static setupRandomFarmEventAt(mapId, x, y, harderChances) {
    if (this.isTileProtected(mapId, x, y)) return;
      
    //Replacing grass with something else while I think about restoring or removing the grass completely
    // var randomNumber = Math.randomInt(600);
    // var randomNumber = Math.randomInt(1000);
    const randomNumber = Math.randomInt(700);

    if (randomNumber < 5 && randomNumber > 2 && !harderChances) {
      this.createRandomFruitTreeAt($gameMap._mapId, x, y);
      return;
    }
    
    if (randomNumber > 600) {
      return;
    }

    if (randomNumber >= 350 && randomNumber < 600) {
      if (randomNumber < 450) {
        this.createGrassObjectAt($gameMap._mapId, x, y);
      }

      if (Managers.Map.mapMonth(mapId) != Seasons.WINTER) {

        if (randomNumber > 500) {
          this.createWeedObjectAt(mapId, x, y);
        }
      }

      return;
    }

    if (randomNumber >= 100 && randomNumber < 140) {
      if (randomNumber < 120) {
        this.createGrassObjectAt($gameMap._mapId, x, y);
      }

      if (harderChances && randomNumber < 139) return;

      this.createStoneObjectAt(mapId, x, y);
      return;
    }

    if (randomNumber >= 140 && randomNumber < 190) {
      if (randomNumber < 160) {
        this.createGrassObjectAt($gameMap._mapId, x, y);
      }
      return;
    }

    if (randomNumber < 30) {
      if (randomNumber < 15) {
        this.createGrassObjectAt($gameMap._mapId, x, y);
      }

      if (harderChances) return;
      this.createStumpObjectAt(mapId, x, y);
      return;
    }

    if (randomNumber < 70) {
      if (randomNumber < 30) {
        this.createGrassObjectAt($gameMap._mapId, x, y);
      }

      if (harderChances && randomNumber < 78) return;
      this.createLogObjectAt(mapId, x, y);
      return;
    }
  }

  static setupRandomGrassyFarmEventAt(mapId, x, y) {
    if (this.isTileProtected(mapId, x, y)) return;
      
    const randomNumber = Math.randomInt(700);

    if (randomNumber >= 600) {
      return;
    }
    
    if (randomNumber >= 500) {
      this.createWeedObjectAt(mapId, x, y);
      return;
    }

    if (randomNumber >= 155) {
      return;
    }

    if (randomNumber >= 150) {
      this.createGreenTreeCropAt(mapId, x, y);
      return;
    }

    if (randomNumber >= 140) {
      const farmObject = this.createFarmObjectAt(mapId, x, y);
      farmObject.modelName = 'rock-2';
      return;
    }

    if (randomNumber >= 100) {
      this.createStoneObjectAt(mapId, x, y);
      return;
    }

    if (randomNumber < 20) {
      return;
    }

    if (randomNumber < 40) {
      this.createStumpObjectAt(mapId, x, y);
      return;
    }

    if (randomNumber < 70) {
      this.createLogObjectAt(mapId, x, y);
      return;
    }
  }

  static removeAllTempEvents() {
    Managers.Map.removeAllTempEvents($gameMap._mapId);
  }

  static generateStoneAt(x, y) {
    
  }

  static setupMapEvents() {
    this.removeAllTempEvents();

    //Updates to make sure the correct list will be used
    this.updateProtectedTiles($gameMap._mapId);
    this.lockProtectedTiles();

    Managers.Map.setupMapEvents($gameMap._mapId);

    if (!$gameSystem.isMapInitialized($gameMap._mapId)) {
      this.setupInitialEvents();
      this.initializeVillagerFarms($gameMap._mapId);
      $gameSystem.initializeMap($gameMap._mapId);
    }

    if (!$gameSystem.hasMapGeneratedDailyEvents($gameMap._mapId)) {
      this.setupDailyItems();
      $gameSystem.generatedDailyMapEvents($gameMap._mapId);
    }

    this.setupTempEvents();

    this.updateMapEvents();

    this.unlockProtectedTiles();
    this.updateProtectedTiles($gameMap._mapId);
  }

  static updateMapEvents() {
    const mapId = $gameMap._mapId;
    this.checkMapFarmObjects(mapId);

    for (const farmObject of this._farmObjects[mapId]) {
      if (farmObject) {
        farmObject.updateEvents(false);
      }
    }
  }

  static updateButtonEvents() {
    const mapId = $gameMap._mapId;
    this.checkMapFarmObjects(mapId);

    for (const farmObject of this._farmObjects[mapId]) {
      if (!farmObject) continue;
      if (farmObject.modelName != FarmObjectState.BUTTON) continue;

      // farmObject.getState(true);

      farmObject.updateEvents(true);
    }
  }

  static updateEventsAt(mapId, x, y) {
    const farmObjects = this.getFarmObjectsXy(mapId, x, y);
    for (let i = 0; i < farmObjects.length; i++) {
      farmObjects[i].updateEvents(true, false);
    }
  }

  static isTileTilled(mapId, x, y) {
    const farmObject = this.getFarmObject(mapId, x, y);
    if (!farmObject) return false;

    return farmObject.state == Constants.FarmObjectState.TILLED_SOIL;
  }

  static modelCanBeJumpedOver(modelName, farmObject) {
    const jumpable = [
      FarmObjectState.FENCE,
      FarmObjectState.GATE
    ];

    if (jumpable.indexOf(modelName) >= 0) {
      return true;
    }

    return false;
  }

  static positionCanBeJumpedOver(mapId, x, y) {
    const anyObjectToJump = this.farmObjectLookup(mapId, x, y, {
      preFilter: (farmObject) => {
        return this.modelCanBeJumpedOver(farmObject.modelName, farmObject);
      }
    }, true);

    return Boolean(anyObjectToJump);
  }

  static isFenceObject(mapId, x, y, allowGate) {
    const anyFence = this.getFarmObjectsXy(mapId, x, y, (farmObject) => {
      if (farmObject.modelName == 'fence') return true;

      if (allowGate) {
        if (farmObject.modelName == FarmObjectState.GATE) {
          if (allowGate === 'open') {
            // #ToDo: check if the gate is open
            return true;
          }

          if (allowGate === 'closed') {
            // #ToDo: check if the gate is closed
            return false;
          }

          return true;
        }
      }

      return false;
    }, true);

    return Boolean(anyFence);
  }

  static isGateObject(mapId, x, y) {
    const anyGate = this.getFarmObjectsXy(mapId, x, y, (farmObject) => {
      return farmObject.modelName == FarmObjectState.GATE;
    }, true);

    return Boolean(anyGate);
  }

  static isBushObject(mapId, x, y) {
    const anyBush = this.getFarmObjectsXy(mapId, x, y, (farmObject) => {
      return farmObject.modelName == FarmObjectState.GRASS;
    }, true);

    return Boolean(anyBush);
  }

  static isTilePassable(mapId, x, y) {
    const objects = this.getFarmObjectsXy(mapId, x, y);

    for (const farmObject of objects) {
      if (farmObject.hasCollision()) return false;
    }

    return !this.isTileProtected(mapId, x, y);
  }

  static updateProtectedTiles(mapId) {
    this.checkMapFarmObjects(mapId);
    this._protectedTiles[mapId] = {};

    for (const farmObject of this._farmObjects[mapId]) {
      if (farmObject === null || farmObject === undefined) continue;

      let width = 1;
      let height = 1;
      if (farmObject.width) width = farmObject.width;
      if (farmObject.height) height = farmObject.height;

      if (width > 1 || height > 1) {
        let minX = farmObject.x;
        let maxX = farmObject.x + width;

        if (farmObject.protectedAreaX) {
          minX += farmObject.protectedAreaX;
          maxX += farmObject.protectedAreaX;
        }

        for (let x = minX; x < maxX; x++) {
          this._protectedTiles[mapId][x] = this._protectedTiles[mapId][x] || {};

          let minY = farmObject.y;
          let maxY = farmObject.y + height;
          if (farmObject.protectedAreaY) {
            minY += farmObject.protectedAreaY;
            maxY += farmObject.protectedAreaY;
          }

          for (let y = minY; y < maxY; y++) {
            this._protectedTiles[mapId][x][y] = farmObject;
          }
        }
      }
    }
  }

  static isTileProtected(mapId, x, y) {
    if (!this._protectedTiles[mapId]) return false;
    if (!this._protectedTiles[mapId][x]) return false;
    if (!this._protectedTiles[mapId][x][y]) return false;

    return true;
  }

  static countStateOnMap(mapId, state, detailed, basicState) {
    const objects = this.getMapFarmObjects(mapId);
    const len = objects.length;
    let count = 0;

    for (let i = 0; i < len; i++) {
      if (!objects[i]) continue;

      if (basicState) {
        const objectBasicState = objects[i].getState(false);

        if (objectBasicState != basicState) continue;
      }
      
      if (objects[i].getState(detailed) == state) {
        count++;
      }
    }

    return count;
  }

  static _isObjectAnException(object, exceptions) {
    const { x, y } = object;

    for (let except of exceptions) {
      if (except.x !== undefined && except.x !== x) {
        continue;
      }
      if (except.y !== undefined && except.y !== y) {
        continue;
      }

      if (except.minX !== undefined && except.minX > x) {
        continue;
      }
      if (except.maxX !== undefined && except.maxX < x) {
        continue;
      }

      if (except.minY !== undefined && except.minY > y) {
        continue;
      }
      if (except.maxY !== undefined && except.maxY < y) {
        continue;
      }

      return true;
    }

    return false;
  }

  static countModelNameOnMap(mapId, modelName, exceptions = []) {
    const objects = this.getMapFarmObjects(mapId);
    const len = objects.length;
    let count = 0;

    for (let i = 0; i < len; i++) {
      if (!objects[i]) continue;

      if (objects[i].modelName === modelName) {
        if (exceptions.length && this._isObjectAnException(objects[i], exceptions)) {
          continue;
        }

        count++;
      }
    }

    return count;
  }

  static removeModelNameOnMap(mapId, modelName) {
    this.clearMapObjectsByFilter(mapId, farmObject => farmObject && farmObject.modelName === modelName);
  }

  static getProtectedTileOwner(mapId, x, y) {
    if (!this._protectedTiles[mapId]) return false;
    if (!this._protectedTiles[mapId][x]) return false;
    if (!this._protectedTiles[mapId][x][y]) return false;

    return this._protectedTiles[mapId][x][y];
  }

  static triggerProtectedTile(x, y) {
    const mapId = $gameMap._mapId;

    const protectedTileObj = this.getProtectedTileOwner(mapId, x, y);
    if (protectedTileObj !== false) {

      switch(protectedTileObj.getState(false)) {
        case FarmObjectState.BARN :
          return this.triggerBarnTouch(protectedTileObj, x, y);
        case FarmObjectState.COOP :
          return this.triggerCoopTouch(protectedTileObj, x, y);
      }
    }

    return false;
  }

  static triggerBuildingTouch(buildingData, minX, maxX, originY) {
    if (buildingData.y === originY) {
      if (buildingData.x >= minX && buildingData.x <= maxX) {
        const events = $gameMap.farmObjectsXy(buildingData.x, buildingData.y);

        for (let i = 0; i < events.length; i++) {
          if (events[i].start()) {
            return true;
          }
        }
      }
    }

    return false;
  }

  static triggerCoopTouch(coopData, originX, originY) {
    if (this.touchCoop(coopData)) {
      return true;
    }

    return false;
  }

  static triggerBarnTouch(barnData, originX, originY) {
    if (this.triggerBuildingTouch(barnData, originX - 1, originX + 1, originY)) return true;

    if (this.touchBarn(barnData)) {
      return true;
    }

    return false;
  }

  static touchCoop(coopData) {
    // To remove a coop, you have to press the action button on it twice.
    // This method is called everytime you press the action button to check if it's the first or second time.

    const now = (new Date()).getTime();
    if (coopData.lastTouch) {
      if (now - coopData.lastTouch < 500) {
        return this.removeCoop(coopData);
      }
    }

    coopData.lastTouch = (new Date()).getTime();
    return false;
  }

  static touchBarn(barnData) {
    // To remove a barn, you have to press the action button on it twice.
    // This method is called everytime you press the action button to check if it's the first or second time.

    const now = (new Date()).getTime();
    if (barnData.lastTouch) {
      if (now - barnData.lastTouch < 500) {
        return this.removeBarn(barnData);
      }
    }

    barnData.lastTouch = (new Date()).getTime();
    return false;
  }

  static touchFence(fenceData, fenceType) {
    // To remove a fence, you have to press the action button on it twice.
    // This method is called everytime you press the action button to check if it's the first or second time.

    const now = (new Date()).getTime();
    if (fenceData.lastTouch) {
      if (now - fenceData.lastTouch < 500) {
        return this.removeFence(fenceData, fenceType);
      }
    }

    fenceData.lastTouch = (new Date()).getTime();
    // Add a small mouse delay with the only intention of preventing the default delay
    $gameMap._mouseDelay = 1;
    return false;
  }

  static removeCoop(coopData) {
    if (Managers.Items.pickItemId('coop')) {
      this.deactivateCoop(coopData.targetMapId);

      coopData.clear();
      coopData.updateEvents(true, true);
      this.clearEmptyItems();
      this.updateProtectedTiles();

      return true;
    }

    return false;
  }

  static removeBarn(barnData) {
    if (Managers.Items.pickItemId('barn')) {
      this.deactivateBarn(barnData.targetMapId);

      barnData.clear();
      barnData.updateEvents(true, true);
      this.clearEmptyItems();
      this.updateProtectedTiles();

      return true;
    }

    return false;
  }

  static removeFence(fenceData, fenceType) {
    if (fenceType === undefined) fenceType = 'fence';

    if (Managers.Items.pickItemId(fenceType)) {
      fenceData.clear();
      fenceData.updateEvents(true, true);
      this.clearEmptyItems();

      return true;
    }

    return false;
  }

  static updateWeatherEffects() {    
  }

  static updateSiblings(mapId, x, y) {
    this.updateEventsAt(mapId, x - 1, y);
    this.updateEventsAt(mapId, x + 1, y);
    this.updateEventsAt(mapId, x, y - 1);
    this.updateEventsAt(mapId, x, y + 1);
  }

  static moveMapContent(oldMapId, newMapId) {
    const farmObjects = this.getMapFarmObjects(oldMapId);
    if (farmObjects && farmObjects.length) {
      this.checkMapFarmObjects(newMapId);

      const newFarmObjects = this._farmObjects[newMapId];

      for (let j = 0; j < farmObjects.length; j++) {
        if (!farmObjects[j]) continue;

        farmObjects[j].mapId = newMapId;
        newFarmObjects.push(farmObjects[j]);
        delete farmObjects[j];
      }
    }

    if ($gameMap._mapId == oldMapId || $gameMap._mapId == newMapId) {
      this.updateMapEvents();
    }
  }

  static isTileWithinRangeOfASprinkler(mapId, x, y) {
    const farmObjects = this.getMapFarmObjects(mapId);
    if (!farmObjects || !farmObjects.length) return false;

    for (let i = 0; i < farmObjects.length; i++) {
      if (!farmObjects[i]) continue;

      let range = 0;

      if (farmObjects[i].modelName == FarmObjectState.SPRINKLER) {
        range = 1;
      } else if (farmObjects[i].modelName == FarmObjectState.SUPER_SPRINKLER) {
        range = 2;
      } else {
        continue;
      }

      if (farmObjects[i].x < x - range) continue;
      if (farmObjects[i].x > x + range) continue;
      if (farmObjects[i].y < y - range) continue;
      if (farmObjects[i].y > y + range) continue;

      return true;
    }

    return false;
  }

  static registerLargeModel(modelName) {
    largeModels.push(modelName);
  }

  static registerWideModel(modelName) {
    wideModels.push(modelName);
  }

  static registerTallModel(modelName) {
    tallModels.push(modelName);
  }

  static isLargeModel(modelName) {
    if (!modelName) return false;

    return largeModels.includes(modelName);
  }

  static isWideModel(modelName) {
    if (!modelName) return false;

    return wideModels.includes(modelName);
  }

  static isTallModel(modelName) {
    if (!modelName) return false;

    return tallModels.includes(modelName);
  }

  static registerCollisionModel(modelName) {
    modelsWithCollision.push(modelName);
  }

  static isModelWithCollision(modelName) {
    if (!modelName) return false;

    return modelsWithCollision.indexOf(modelName) >= 0;
  }

  static getCustomTileId(x, y) {
    return $gameTemp.getCustomTile(x, y);

    // return 0;
    // var mapId = $gameMap._mapId;
    // var objects = this.getFarmObjectsXy(mapId, x, y);

    // if (!objects || !objects.length) return 0;

    // for (var i = 0; i < objects.length; i++) {
    //   var obj = objects[i];
    //   if (!obj) continue;

    //   if (obj._useCustomTileId !== undefined) {
    //     return obj._useCustomTileId;
    //   }

    //   var objType = obj.getFarmObjectType();
    //   if (!objType) continue;

    //   if (objType._useCustomTileId) {
    //     obj._useCustomTileId = objType._useCustomTileId;
    //     return objType._useCustomTileId;
    //   }

    //   obj._useCustomTileId = false;
    // }

    // return 0;
  }

  static createAnotherMapEvents(mapId, originalRect, targetPoint) {
    Managers.FarmObjects.checkMapFarmObjects(mapId);

    for (const farmObject of Managers.FarmObjects._farmObjects[mapId]) {
      if (!farmObject) {
        continue;
      }
      if ((originalRect.left && farmObject.x < originalRect.left) || (originalRect.right !== undefined && farmObject.x > originalRect.right)) {
        continue;
      }
      if ((originalRect.top && farmObject.y < originalRect.top) || (originalRect.bottom !== undefined && farmObject.y > originalRect.bottom)) {
        continue;
      }

      const clone = new FarmObject();
      Object.assign(clone, farmObject);
      clone.mapId = $gameMap._mapId;
      clone.x = clone.x - (originalRect.left || 0) + (targetPoint.x || 0);
      clone.y = clone.y - (originalRect.top || 0) + (targetPoint.y || 0);

      clone.updateEvents(false);
    }

  }

  static autoRegisterAllModels() {
    for (let modelProp in Models) {
      const model = Models[modelProp];

      const modelName = model.modelName;
      if (!this.farmObjectTypes[modelName]) {
        this.farmObjectTypes[modelName] = model;
      }
    }
  }

  static registerModel(model) {
    this.farmObjectTypes[model.modelName] = model;
  }
};

Managers.FarmObjects.clear();
Managers.FarmObjects.farmObjectTypes = {};