//-----------------------------------------------------------------------------
// Extension Manager
//
// The base class of the mods and packages managers

const imageRedirects = new Map();
const extensionFolders = new Set();

Managers.Extension = class Extension {
  constructor() {
    throw new Error('This is a static class');
  }

  static getImageRedirectMap() {
    return imageRedirects;
  }

  static getExtensionFolders() {
    return extensionFolders;
  }

  static addTranslationFileEx(fullPath, language) {
    let loaderFn = 'loadDataFile';

    if (fullPath.toLowerCase().endsWith('.yaml')) {
      loaderFn = 'loadYamlFile';
    }

    Managers.Storage[loaderFn](fullPath, loadedJson => {
      let count = 0;
      for (const key in loadedJson) {
        if (loadedJson.hasOwnProperty(key)) {
          this.addTranslation(language, key, loadedJson[key]);
          count++;
        }
      }

      this.log('[Extension] Added ' + count + ' new translations for ' + language + ' language.');
    });
  }

  static addTranslationFolder(i18nPath) {
    const fileList = Wrapper.readdirSync(i18nPath);

    for (let i = 0; i < fileList.length; i++) {
      const fileName = fileList[i];
      const lowerFileName = fileName.toLowerCase();
      const language = lowerFileName.replace('.json', '').replace('.yaml', '');

      const fullPath = Wrapper.joinFileNames(i18nPath, fileName);
      this.addTranslationFileEx(fullPath, language);
    }
  }

  static addDataFolder(dataFolder) {
    const fileList = Wrapper.readdirSync(dataFolder);

    for (const fileName of fileList) {
      const fullPath = Wrapper.joinFileNames(dataFolder, fileName);

      switch(fileName.toLowerCase()) {
        case 'languages.yaml':
          this.addLanguageListFile(fullPath);
          break;
        case 'tasks.yaml':
          this.addTaskListFile(fullPath);
          break;
        case 'letters.yaml':
          this.addLetterListFile(fullPath);
          break;
        case 'visitors.yaml':
          this.addVisitorListFile(fullPath);
          break;
        case 'schedules':
          this.addSchedulesFolder(fullPath);
          break;
      }
    }
  }

  static addLanguageListFile(fullPath) {
    Managers.Storage.loadYamlFile(fullPath, loadedJson => {
      let count = 0;
      for (const languageId in loadedJson) {
        if (loadedJson.hasOwnProperty(languageId)) {
          $dataLanguages[languageId] = loadedJson[languageId];
          count++;

          if (loadedJson[languageId] && loadedJson[languageId].activateAutomatically) {
            Managers.Text.changeLanguage(languageId);
          }
        }
      }
      
      this.log('[Extension] Added ' + count + ' languages.');
    });
  }

  static addTaskListFile(fullPath) {
    Managers.Storage.loadYamlFile(fullPath, loadedJson => {
      let count = 0;
      for (const taskId in loadedJson) {
        if (loadedJson.hasOwnProperty(taskId)) {
          $dataTasks[taskId] = loadedJson[taskId];
          count++;
        }
      }
      
      this.log('[Extension] Added ' + count + ' tasks.');
    });
  }

  static addLetterListFile(fullPath) {
    Managers.Storage.loadYamlFile(fullPath, loadedJson => {
      let count = 0;
      for (const letterId in loadedJson) {
        if (loadedJson.hasOwnProperty(letterId)) {
          $dataLetters[letterId] = loadedJson[letterId];
          count++;
        }
      }
      
      this.log('[Extension] Added ' + count + ' letters.');
    });
  }

  static addVisitorListFile(fullPath) {
    Managers.Storage.loadYamlFile(fullPath, loadedJson => {
      let count = 0;
      for (const eventId in loadedJson) {
        if (loadedJson.hasOwnProperty(eventId)) {
          $dataVisitors[eventId] = loadedJson[eventId];
          count++;
        }
      }
      
      this.log('[Extension] Added ' + count + ' visitors.');
    });
  }

  static addSchedulesFolder(schedulesFolder) {
    for (let actor of $dataActors) {
      if (!actor) continue;

      const name = actor.name.toLowerCase();
      const fullPath = Wrapper.joinFileNames(schedulesFolder, name);
      const yamlFile = `${ fullPath }.yaml`;
      const jsonFile = `${ fullPath }.json`;

      if (Wrapper.existsSync(yamlFile)) {
        Managers.Villagers.loadPositionsFile(yamlFile, name);
      }
      
      if (Wrapper.existsSync(jsonFile)) {
        Managers.Villagers.loadPositionsFile(jsonFile, name);
      }
    }
  }

  static addEventsFolder(eventsFolder) {
    Managers.EventScript.generateCommonEventsForFolder(eventsFolder);
  }

  static getPath() {
    throw new Error('Extension.getPath was not overwritten.');
  }


  static loadYamlFile(pluginName, relativeFilePath, callback) {
    const baseFolder = this.getPath();
    const fullPath = Wrapper.joinFileNames(baseFolder, pluginName, relativeFilePath);

    Managers.Storage.loadYamlFile(fullPath, callback);
  }

  static addTaskFileEx(fullPath, taskId, extensionName) {
    Managers.Storage.loadYamlFile(fullPath, loadedJson => {
      $dataTasks[taskId] = loadedJson;
    });
    
    // console.log(...log('[Extension] Loaded ' + taskId + ' task. (' + (extensionName || 'Unknown Mod') + ')'));
  }


  static loadDataFile(pluginName, relativeFilePath, callback) {
    const baseFolder = this.getPath();
    const fullPath = Wrapper.joinFileNames(baseFolder, pluginName, relativeFilePath);

    Managers.Storage.loadDataFile(fullPath, callback);
  }
  
  static addTranslationFile(pluginName, language, filePath) {
    const baseFolder = this.getPath();
    const fullPath = Wrapper.joinFileNames(baseFolder, pluginName, filePath);
    this.addTranslationFileEx(fullPath, language);
  }

  static addIcon(pluginName, relativePath, fileName) {
    const extensionFolder = pluginName !== undefined ? Wrapper.joinFileNames(this.getPath(), pluginName) : 'img';
    let imagePath = extensionFolder;
    if (relativePath) {
      imagePath = Wrapper.joinFileNames(extensionFolder, relativePath);
    }

    imagePath = Wrapper.joinFileNames(imagePath, fileName);

    return Managers.Images.createIconFromImage(imagePath);
  }

  static addObjectData(objectData) {
    if (!objectData) return;
    if (!objectData.id) {
      console.log(...log("[Extension] Item is missing it's id."));
      return;
    }
    if (!objectData.type) {
      console.log(...log(`[Extension] Item ${objectData.id} is missing it's type.`));
      return;
    }

    const validTypes = ['food', 'item', 'tool-item', 'seed', 'tool', 'object', 'animal', 'medicine', 'unique', 'seed-box', 'other'];
    if (!validTypes.includes(objectData.type)) {
      console.log(...log(`[Extension] Item ${objectData.id} has invalid type: ${objectData.type}`));
      return;
    }
    
    if (!objectData.iconIndex) {
      objectData.iconIndex = -1;
    }

    if (objectData.iconIndex == -1) {
      if (!objectData.characterName) {
        console.log(...log(`[Extension] Item ${objectData.id} has no sprite information`));
        return;
      }

      if (!objectData.characterIndex) {
        objectData.characterIndex = 0;
      }
    }

    const validCategories = ['tools', 'crops', 'fish', 'loot', 'food', 'material'];
    if (!objectData.category || validCategories.indexOf(objectData.category) < 0) {
      objectData.category = 'loot';
    }

    if (!objectData.possibleGift) {
      objectData.possibleGift = false;
    } else {
      objectData.possibleGift = true;
      const validGiftTypes = ['hate', 'dislike', 'normal', 'like', 'love'];
      if (!objectData.defaultGift || !validGiftTypes.includes(objectData.defaultGift)) {
        objectData.defaultGift = 'normal';
      }
    }

    if (!objectData.buyPrice) {
      objectData.buyPrice = 0;
    }
    if (!objectData.sellPrice) {
      objectData.sellPrice = 0;
    }
    if (!objectData.taskPrice) {
      objectData.taskPrice = 0;
    }

    if (!objectData.ingredients) {
      objectData.ingredients = false;
    }
    if (!objectData.restores) {
      objectData.restores = false;
    }
    if (!objectData.seasons) {
      objectData.seasons = false;
    }
    if (!objectData.daysToComplete) {
      objectData.daysToComplete = false;
    }
    if (!objectData.daysToGrow) {
      objectData.daysToGrow = 0;
    }
    if (!objectData.daysToRegrow) {
      objectData.daysToRegrow = false;
    }
    if (!objectData['placeholder-for']) {
      objectData['placeholder-for'] = false;
    }
    if (!objectData['placeholder-quantity']) {
      objectData['placeholder-quantity'] = false;
    }

    objectData.useOnRequests = Boolean(objectData.useOnRequests);
    objectData.ignoredBySalesman = Boolean(objectData.ignoredBySalesman);
    objectData.consumable = Boolean(objectData.consumable);
    objectData.fish = Boolean(objectData.fish);
    objectData.drink = Boolean(objectData.drink);
    objectData.candy = Boolean(objectData.candy);
    objectData.canTrash = Boolean(objectData.canTrash);
    objectData.canCook = Boolean(objectData.canCook);
    objectData.fruit = Boolean(objectData.fruit);
    $dataItems.push(objectData);

    Managers.Items.expandItemIngredients(objectData);
  }

  static setVillagerScheduleData(villagerName, scheduleData) {
    const lowerVillagerName = villagerName.toLowerCase();
    Managers.Villagers._positions[lowerVillagerName] = Managers.Villagers.expandPositions(scheduleData);
  }

  static addVillagerSchedule(villagerName, scheduleName, scheduleData) {
    const lowerVillagerName = villagerName.toLowerCase();
    if (!Managers.Villagers._positions[lowerVillagerName]) {
      Managers.Villagers._positions[lowerVillagerName] = {};
    }

    Managers.Villagers._positions[lowerVillagerName][scheduleName] = Managers.Villagers.expandPositionGroup(scheduleData);
  }

  static addActorData(actorData) {
    if (!actorData.name) return;

    actorData.id = $dataActors.length;
    $dataActors.push(actorData);
    Managers.Villagers.loadPositions(actorData.name);
  }

  static addCustomEvents(characterName, scriptText) {
    const newEvents = Managers.EventScript.parseScript(scriptText.replace(/`/g, '"'));

    for (let i = 0; i < newEvents.length; i++) {
      Managers.EventScript.saveEvent(characterName, newEvents[i]);
    }
  }

  static addTranslation(language, string, translation) {
    if (!$dataStrings[language]) {
      $dataStrings[language] = {};
    }

    $dataStrings[language][string] = translation;
  }

  static loadEventFile(pluginName, relativeFilePath, characterName) {
    const baseFolder = this.getPath();
    const fullPath = Wrapper.joinFileNames(baseFolder, pluginName, relativeFilePath);

    Managers.EventScript.parseExtraFile(fullPath, characterName);
  }




  static loadScript(url, extensionName) {
    const script = Wrapper.readFileSync(url);
    if (script) {
      this.parseExtensionScript(script, extensionName);
    }
  }

  static parseExtensionScript(script, extensionName) {
    try {
      console.log(...log('[Extension] Loading extension', extensionName));
      const fullScript = "//@ sourceURL=" + extensionName + "\n" + script;
      eval(fullScript); //jshint ignore:line
      console.log(...log('[Extension] Success'));
    }
    catch(e) {
      console.log(...log('[Extension] Error'));
      Managers.Scenes.showExplicitException('Mod Error', `Failed to load mod ${extensionName}`);
      console.error(e);
    }
  }

  static loadFileExtension(fileName, filePath, extensionName) {
    const ext = Wrapper.extname(filePath);
    if (ext.toLowerCase() != '.js') return;
    const nameExt = Wrapper.extname(fileName);

    if (extensionName === undefined) {
      if (ext.length > 0 && nameExt.length > 0 && ext == nameExt) {
        extensionName = fileName.slice(0, 0 - ext.length);
      } else {
        extensionName = fileName;
      }
    }

    this.loadScript(filePath, extensionName);
  }

  static registerExtensionImage(imagePath, newPath) {
    const normalizedPath = imagePath.replace(/\\/g, '/').replace(/\/\//g, '/');

    imageRedirects.set(normalizedPath, newPath);
  }

  static requestBitmap(folder, filename, hue, smooth) {
    const imagePath = folder + `/${filename}.png`;
    const normalizedPath = imagePath.replace(/\\/g, '/').replace(/\/\//g, '/');
    if (!imageRedirects.has(normalizedPath)) {
      return false;
    } 

    const newImagePath = imageRedirects.get(normalizedPath);

    if (!folder.endsWith('/')) {
      folder += '/';
    }
    const cachePath = `${folder + encodeURIComponent(filename)}.png`;
    const localPath = `file:///${ newImagePath }`;
    const bitmap = Managers.Images.requestNormalBitmap(localPath, hue || 0, cachePath);
    bitmap.smooth = smooth;
    return bitmap;
  }

  static loadBitmap(folder, filename, hue, smooth) {
    const imagePath = folder + `/${filename}.png`;
    const normalizedPath = imagePath.replace(/\\/g, '/').replace(/\/\//g, '/');
    if (!imageRedirects.has(normalizedPath)) {
      return false;
    } 

    const newImagePath = imageRedirects.get(normalizedPath);

    if (!folder.endsWith('/')) {
      folder += '/';
    }
    const cachePath = `${folder + encodeURIComponent(filename)}.png`;

    const localPath = `file:///${ newImagePath }`;
    const bitmap = Managers.Images.loadNormalBitmap(localPath, hue || 0, cachePath);
    bitmap.smooth = smooth;
    return bitmap;
  }

  static registerImages(packageName, imgPath) {
    this.registerImageFolder('img/', imgPath);
  }

  static registerImageFolder(relativePath, fullPath) {
    const fileList = Wrapper.readdirSync(fullPath);

    for (let i = 0; i < fileList.length; i++) {
      const filePath = Wrapper.joinFileNames(fullPath, fileList[i]);

      if (Wrapper.isDirectory(filePath)) {
        const relativeFilePath = Wrapper.joinFileNames(relativePath, fileList[i]);
        this.registerImageFolder.call(this, relativeFilePath, filePath);
      } else {
        this.registerExtensionImage(`${relativePath}/${fileList[i]}`, filePath);
      }
    }
  }

  static registerExtensionFolder(extensionPath, extensionName, extensionType) {
    extensionFolders.add({
      extensionPath,
      extensionName,
      extensionType
    });
  }

  static loopExtensionFolders(callback) {
    for (let extension of extensionFolders) {
      const result = callback.call(this, extension);
      if (result !== undefined) return result;
    }
  }

  static loadPositions(villagerName) {
    if (!villagerName) return;
    villagerName = villagerName.toLowerCase();

    return this.loopExtensionFolders(extension => {
      const fullPath = Wrapper.joinFileNames(extension.extensionPath, 'data', 'schedules', villagerName);
      const jsonFile = `${ fullPath }.json`;
      const yamlFile = `${ fullPath }.yaml`;

      if (Wrapper.existsSync(yamlFile)) {
        return Managers.Villagers.loadPositionsFile(yamlFile, villagerName);
      }

      if (Wrapper.existsSync(jsonFile)) {
        return Managers.Villagers.loadPositionsFile(jsonFile, villagerName);
      }
    }) || false;
  }

  static loadExtensionFolderContent(folder, name, extensionType) {
    if (!Utils._isAutoTestMode) {
      console.log(`[ExtensionManager] Loading: ${ name }`);
    }
    let anyKnownFolder = false;

    const i18nPath = Wrapper.joinFileNames(folder, 'i18n');
    if (Wrapper.existsSync(i18nPath)) {
      this.addTranslationFolder(i18nPath);
      anyKnownFolder = true;
    }

    const eventsFolder = Wrapper.joinFileNames(folder, 'events');
    if (Wrapper.existsSync(eventsFolder)) {
      this.addEventsFolder(eventsFolder);
      anyKnownFolder = true;
    }

    const dataPath = Wrapper.joinFileNames(folder, 'data');
    if (Wrapper.existsSync(dataPath) && Wrapper.isDirectory(dataPath)) {
      this.addDataFolder(dataPath);
      anyKnownFolder = true;
    }

    const imgPath = Wrapper.joinFileNames(folder, 'img');
    if (Wrapper.existsSync(imgPath) && Wrapper.isDirectory(imgPath)) {
      this.registerImages(name, imgPath);
      anyKnownFolder = true;
    }

    const audioPath = Wrapper.joinFileNames(folder, 'audio');
    if (Wrapper.existsSync(audioPath) && Wrapper.isDirectory(audioPath)) {
      anyKnownFolder = true;
    }

    this.registerExtensionFolder(folder, name, extensionType);
    return anyKnownFolder;
  }

  static imgIsInvalid(filename, folder) {
    const imagePath = 'img/' + folder + `/${filename}.png`;
    const normalizedPath = imagePath.replace(/\\/g, '/').replace(/\/\//g, '/');
    return !imageRedirects.has(normalizedPath);
  }

  static log(text) {
    // console.log(...log(`[Extension] ${ text }`));
  }
};