Managers.EventScript = class EventScript {
  static parseChoiceLine(line) {
    const spacePos = line.indexOf(' ');
    const name = line.substr(0, spacePos);

    return {
      name,
      type : 'choices',
      line
    };
  }

  static parseDialogueLine(line, checkIfOwnerExists) {
    const spacePos = line.indexOf(' ');
    const bracketPos = line.indexOf('(');
    const beginQuotePos = line.indexOf('"');

    if (spacePos < 0) return undefined;
    if (beginQuotePos < spacePos) return undefined;

    if (bracketPos > 0 && bracketPos < beginQuotePos) {
      return this.parseChoiceLine(line);
    }
    
    const name = line.substr(0, spacePos).trim();

    const regex = / "(.*)"/;
    const result = regex.exec(line);
    if (!result) return;
    if (result.length < 2) return undefined;

    if (checkIfOwnerExists) {
      if (!this.isValidDialogueOwner(name)) {
        return undefined;
      }
    }

    const lineData = {
      name,
      type : 'dialogue',
      line : result[1]
    };

    return lineData;
  }

  static removeStringsAndParamsFromLine(lineData) {
    let regex = /"([^"]*)"/;

    let newLine = lineData.line;
    let result = regex.exec(newLine);
    while (result) {
      newLine = newLine.replace(result[0], `string(${lineData.strings.length})`);
      lineData.strings.push(result[0]);
      result = regex.exec(newLine);
    }
    
    regex = /'([^']*)'/;
    result = regex.exec(newLine);
    while (result) {
      newLine = newLine.replace(result[0], `string(${lineData.strings.length})`);
      lineData.strings.push(result[0]);
      result = regex.exec(newLine);
    }

    regex = /\(([^()]*)\)/;
    result = regex.exec(newLine);

    while (result) {
      newLine = newLine.replace(result[0], `{${lineData.params.length}}`);
      lineData.params.push(result[0]);

      result = regex.exec(newLine);
    }

    lineData.line = newLine;
  }

  static parseRMConditionLine(line) {
    const lineData = {
      type : 'rm-condition',
      params : [],
      strings : [],
      line : '',
      strict : false
    };

    const newLine = line.substr(3);
    
    lineData.line = newLine;
    lineData.original = line;

    return lineData;
  }

  static parseConditionLine(line) {
    const lineData = {
      type : 'condition',
      params : [],
      strings : [],
      line : '',
      strict : false
    };

    let newLine = line;
    if (newLine.toLowerCase().startsWith('$ ')) {
      newLine = newLine.substr(2);
    }
    if (newLine.toLowerCase().startsWith('js ')) {
      newLine = newLine.substr(3);
      lineData.strict = true;
    }
    if (newLine.toLowerCase().startsWith('if ')) {
      newLine = newLine.substr(3);
    }
  
    lineData.line = newLine;
    lineData.original = line;

    if (!lineData.strict) {
      this.removeStringsAndParamsFromLine(lineData);
    }
    return lineData;
  }

  static parseCommandLine(line, strict, isConcat) {
    const lineData = {
      type : isConcat ? 'script-concat' : 'script',
      params : [],
      strings : [],
      line : '',
      strict : strict || false
    };

    let newLine = line;
    if (newLine.toLowerCase().startsWith('$ ')) {
      newLine = newLine.substr(2);
    }
    if (newLine.toLowerCase().startsWith('do ')) {
      newLine = newLine.substr(3);
    }

    if (newLine.toLowerCase().startsWith('js ')) {
      newLine = newLine.substr(3);
      lineData.strict = true;
    }

    lineData.line = newLine;
    lineData.original = line;

    if (!lineData.strict) {
      this.removeStringsAndParamsFromLine(lineData);
    }
    return lineData;
  }

  static parseCommentLine(group, line, isConcat) {
    if (line.substr(0, 1) == '#') {
      line = line.substr(1, line.length - 1);
    }

    group.lines.push({
      type : isConcat ? 'comment-concat' : 'comment',
      line
    });
  }

  static addToPreviousLine(group, newLine) {
    let cleanLine = newLine.substr(1, newLine.length - 1);
    if (cleanLine.substr(0, 1) == ' ') {
      cleanLine = cleanLine.substr(1, newLine.length - 1);
    }

    if (group.lines.length <= 0) {
      this.parseCommentLine(group, cleanLine);
      return;
    }

    const previousLine = group.lines[group.lines.length - 1];

    if (previousLine.type == 'script' || previousLine.type == 'script-concat') {
      const newLineData = this.parseCommandLine(cleanLine, previousLine.strict, true);
      group.lines.push(newLineData);

      return;
    }

    if (previousLine.type == 'comment' || previousLine.type == 'comment-concat') {
      this.parseCommentLine(group, cleanLine, true);
      return;
    }
  }

  static parseScriptLine(group, line) {
    const lower = line.toLowerCase();

    if (lower.startsWith('#')) {
      this.parseCommentLine(group, line);
      return;
    }

    if (group.name === undefined) {
      if (!lower.startsWith('priority') && !lower.startsWith('$ ') && !lower.startsWith('do ') && !lower.startsWith('js ')) {
        group.name = line;
      }
      return;
    }

    if (lower.startsWith('+')) {
      this.addToPreviousLine(group, line);
      return;
    }

    if (lower.startsWith('if-')) {
      const conditionLineDataRM = this.parseRMConditionLine(line);
      if (conditionLineDataRM) {
        group.lines.push(conditionLineDataRM);
      }
      return;
    }

    if (lower.startsWith('if ') || lower.startsWith('$ if ') || lower.startsWith('$ js if ') || lower.startsWith('js if ')) {
      const conditionLineData = this.parseConditionLine(line);
      if (conditionLineData) {
        group.lines.push(conditionLineData);
      }
      return;
    }

    if (lower.startsWith('else')) {
      group.lines.push({
        type : 'else',
        line : ''
      });
      return;
    }

    if (lower.startsWith('end event') || lower.startsWith('return') || lower.startsWith('exit')) {
      group.lines.push({
        type : 'exit',
        line : ''
      });
      return;
    }

    if (lower.startsWith('loop')) {
      group.lines.push({
        type : 'loop',
        line : ''
      });
      return;
    }

    if (lower.startsWith('break loop')) {
      group.lines.push({
        type : 'breakloop',
        line : ''
      });
      return;
    }

    if (lower.startsWith('end loop')) {
      group.lines.push({
        type : 'endloop',
        line : ''
      });
      return;
    }

    if (lower.startsWith('buy item')) {
      group.lines.push({
        type : 'buyItem',
        line : lower.replace('buy item', '').trim()
      });
      return;
    }

    if (lower.startsWith('label')) {
      group.lines.push({
        type : 'label',
        line : lower.replace('label', '').trim()
      });
      return;
    }

    if (lower.startsWith('go to label')) {
      group.lines.push({
        type : 'goto-label',
        line : lower.replace('go to label', '').trim()
      });
      return;
    }

    if (lower.startsWith('input name for actor')) {
      group.lines.push({
        type : 'inputName',
        line : lower.replace('input name for actor', '').replace('digits in variable', ',')
      });
      return;
    }

    if (lower.startsWith('input')) {
      group.lines.push({
        type : 'input',
        line : lower.replace('input', '').replace('digits in variable', ',')
      });
      return;
    }

    if (lower.startsWith('wait for')) {
      group.lines.push({
        type : 'waitfor',
        line : lower.replace('wait for', '').trim()
      });
      return;
    }

    if (lower.startsWith('wait')) {
      group.lines.push({
        type : 'wait',
        line : lower.replace('wait', '').trim()
      });
      return;
    }

    if (lower.startsWith('fade out')) {
      group.lines.push({
        type : 'fadeout',
        line : lower.replace('fade out', '').trim()
      });
      return;
    }

    if (lower.startsWith('fade in')) {
      group.lines.push({
        type : 'fadein',
        line : lower.replace('fade in', '').trim()
      });
      return;
    }

    if (lower.startsWith('plugin ')) {
      group.lines.push({
        type : 'plugin',
        line : line.substr(7)
      });
      return;
    }

    if (lower.startsWith('* ')) {
      group.lines.push({
        type : 'plugin',
        line : line.substr(2)
      });
      return;
    }

    if (lower.startsWith('erase event')) {
      group.lines.push({
        type : 'erase',
        line : ''
      });
      return;
    }

    if (lower.trim() == 'end') {
      group.lines.push({
        type : 'end',
        line : ''
      });
      return;
    }

    if (lower.startsWith('say ')) {
      const dialogueLineData = this.parseDialogueLine(line, false);
      if (dialogueLineData) {
        group.lines.push(dialogueLineData);
        return;
      }

    }

    if (lower.startsWith('priority')) {
      return;
    }

    if (lower.startsWith('$ ') || lower.startsWith('do ') || lower.startsWith('js ')) {
      const commandData = this.parseCommandLine(line);
      if (commandData) {
        group.lines.push(commandData);
      }
      return;
    }

    const lineData = this.parseDialogueLine(line, true);
    if (lineData) {
      group.lines.push(lineData);
      return;
    }

    //If all else failed, treat it as a plugin command
    group.lines.push({
      type : 'plugin',
      line
    });
    return;
  }

  static parseScriptGroup(group) {
    const lines = group.split('\n');

    const groupData = {
      name : undefined,
      priority : undefined,
      condition : undefined,
      lines : [],
      trigger : 0,
      switchId : 0
    };

    for (let j = 0; j < lines.length; j++) {
      let line = lines[j].trim();

      if (line.length === 0) continue;
      if (line.substr(0, 1) == '-') continue;
      if (line.substr(0, 1) == '#') {
        this.parseCommentLine(groupData, line);
        continue;
      }

      if (line.startsWith('trigger ')) {
        line = line.replace('trigger', '').trim();
        const trigger = Number(line);
        if (!isNaN(trigger) && trigger >= 0 && trigger <= 2) {
          groupData.trigger = trigger;
        }
        continue;
      }

      if (line.startsWith('switch id')) {
        line = line.replace('switch id', '').trim();
        const switchId = Number(line);
        if (!isNaN(switchId) && switchId >= 0 && switchId <= 2) {
          groupData.switchId = switchId;
        }
        continue;
      }

      if (line.startsWith('$ do ')) {
        line = line.replace('$ do ', '$ ');
      }
      if (line.startsWith('do ')) {
        line = line.replace('do ', '$ ');
      }

      this.parseScriptLine(groupData, line);
    }

    if (groupData.name === undefined) {
      return undefined;
    }

    if (groupData.trigger > 0 && groupData.switchId === 0) {
      groupData.switchId = Switches._alwaysOn;
    }

    return groupData;
  }

  static parseScript(script) {
    script = script.trim();
    script = `#\n${script}`;

    const groups = script.split('\n:');
    const validChats = [];
    const invalidChats = [];

    for (let i = 0; i < groups.length; i++) {
      const groupData = this.parseScriptGroup(groups[i]);

      if (groupData === undefined) {
        continue;
      }

      if (groupData.lines.length > 0) {
        validChats.push(groupData);
      } else {
        invalidChats.push(groupData);
      }
    }

    return validChats;
  }

  static createCommandsForChoiceLine(characterName, line, commands, indent, eventData) {
    commands.push({
      code : 356,
      indent,
      parameters : [`CHOICE ${line.line}`]
    });
  }

  static createCommandsForPluginLine(characterName, line, commands, indent, eventData) {
    commands.push({
      code : 356,
      indent,
      parameters : [line.line]
    });
  }

  static createCommandsForEraseLine(characterName, line, commands, indent, eventData) {
    commands.push({
      code : 214,
      indent,
      parameters : []
    });
  }

  static createCommandsForCommentLine(line, commands, indent) {
    let code = 108;
    if (line.type == 'comment-concat') {
      code = 408;
    }

    commands.push({
      code,
      indent,
      parameters : [line.line]
    });
  }

  static createCommandsForDialogueLine(characterName, line, commands, indent, eventData) {
    commands.push({
      code : 356,
      indent,
      parameters : [`DIALOGUE ${line.name} ${line.line}`]
    });
  }

  static createCommandsForBalloonLine(characterName, line, commands, indent, eventData) {
    let eventId;

    if (line.name == 'Player') {
      eventId = -1;
    } else if (line.name.toLowerCase() == characterName.toLowerCase()) {
      eventId = eventData;
    } else {
      eventId = line.name;
    }

    commands.push({
      code : 213,
      indent,
      parameters : [eventId, line.ballonId, !!line.wait]
    });
  }

  static getScriptLineWithParamsAndStrings(lineData) {
    let script = lineData.line;

    if (lineData.params !== undefined && lineData.params.length > 0) {
      script = script.formatList(lineData.params);
    }
    
    if (lineData.strings !== undefined && lineData.strings.length > 0) {
      const len = lineData.strings.length;
      for (let i = 0; i < len; i++) {
        script = script.replace(`string(${i})`, lineData.strings[i]);
      }
    }

    return script;
  }

  static createCommandsForConditionalLine(characterName, line, commands, indent, eventData) {
    let script;
    if (!line.strict) {
      script = this.getScriptLineWithParamsAndStrings(line);
    } else {
      script = line.line;
    }

    commands.push({
      code : 111,
      indent,
      parameters : [12, script]
    });
  }

  static createCommandsForRMConditionalLine(characterName, line, commands, indent, eventData) {
    let script = line.line.trim();
    let params = [];

    if (script.toLowerCase().startsWith('switch')) {
      script = script.substr(6).trim();
      
      let switchStatus = 0;
      if (script.toLowerCase().endsWith('is on')) {
        script = script.substr(0, script.length - 5).trim();
      } else if (script.toLowerCase().endsWith('is off')) {
        script = script.substr(0, script.length - 6).trim();
        switchStatus = 1;
      }

      if (Switches[`_${script}`] !== undefined) {
        params = [0, Switches[`_${script}`], switchStatus];
      } else {
        console.error('Invalid switch: ', script);
        return false;
      }
    } else if (script.toLowerCase().startsWith('variable')) {
      script = script.substr(8).trim();

      let lineData;
      let symbol = 0;
      const lowerScript = script.toLowerCase();

      if (lowerScript.indexOf(' is not ') > 0) {
        symbol = 5;
        lineData = script.split(' is not ');
      } else if (lowerScript.indexOf(' is ') > 0) {
        symbol = 0;
        lineData = script.split(' is ');
      } else if (lowerScript.indexOf('==') > 0) {
        symbol = 0;
        lineData = script.split('==');
      } else if (lowerScript.indexOf('!=') > 0) {
        symbol = 5;
        lineData = script.split('!=');
      } else if (lowerScript.indexOf('<>') > 0) {
        symbol = 5;
        lineData = script.split('<>');
      } else if (lowerScript.indexOf('>=') > 0) {
        symbol = 1;
        lineData = script.split('>=');
      } else if (lowerScript.indexOf('<=') > 0) {
        symbol = 2;
        lineData = script.split('<=');
      } else if (lowerScript.indexOf('=') > 0) {
        symbol = 0;
        lineData = script.split('=');
      } else if (lowerScript.indexOf('<') > 0) {
        symbol = 4;
        lineData = script.split('<');
      } else if (lowerScript.indexOf('>') > 0) {
        symbol = 3;
        lineData = script.split('>');
      }

      if (!lineData) {
        console.error('Invalid condition data: ', script);
        return false;
      }

      const variableName = lineData[0].trim();
      let variableValueType = 0;
      let variableId;
      if (Variables[`_${variableName}`] !== undefined) {
        variableId = Variables[`_${variableName}`];
      } else {
        console.error('Invalid variable: ', variableName);
        return false;
      }

      let variableValue = lineData[1].trim();
      if (variableValue.toLowerCase().startsWith('variable')) {
        variableValueType = 1;
        variableValue = variableValue.substr(8).trim();
        if (Variables[`_${variableValue}`] === undefined) {
          console.error('Invalid variable:', variableValue);
          return false;
        }

        variableValue = Variables[`_${variableValue}`];
      }

      params = [1, variableId, variableValueType, variableValue, symbol];
    } else {
      console.error('Invalid RM condition: ', script);
      return false;
    }

    commands.push({
      code : 111,
      indent,
      parameters : params
    });

    return true;
  }

  static createCommandsForScriptLine(characterName, line, commands, indent, eventData) {
    let script;
    let code = 355;
    if (line.type == 'script-concat') {
      code = 655;
    }

    if (!line.strict) {
      script = this.getScriptLineWithParamsAndStrings(line);
    } else {
      script = line.line;
    }

    commands.push({
      code,
      indent,
      parameters : [script]
    });
  }

  static createCommandsForWaitLine(line, commands, indent, eventData) {
    commands.push({
      code : 230,
      indent,
      parameters : [line.line]
    });
  }

  static createCommandsForLoopLine(line, commands, indent, eventData) {
    commands.push({
      code : 112,
      indent,
      parameters : []
    });
  }

  static createCommandsForEndLoopLine(line, commands, indent, eventData) {
    commands.push({
      code : 0,
      indent,
      parameters : []
    });

    commands.push({
      code : 413,
      indent : indent -1,
      parameters : []
    });
  }

  static createCommandsForBreakLoopLine(line, commands, indent, eventData) {
    commands.push({
      code : 113,
      indent,
      parameters : []
    });
  }

  static createCommandsForWaitForLine(line, commands, indent, eventData) {
    commands.push({
      code : 356,
      indent,
      parameters : [`WaitFor ${line.line}`]
    });
  }

  static createCommandsForInputLine(line, commands, indent, eventData) {
    const params = line.line.split(',');

    commands.push({
      code : 103,
      indent,
      parameters : [parseInt(params[1], 10), parseInt(params[0], 10)]
    });
  }

  static createCommandsForInputNameLine(line, commands, indent, eventData) {
    commands.push({
      code : 303,
      indent,
      parameters : [parseInt(line.line, 10), 20]
    });
  }

  static createCommandsForFadeOutLine(line, commands, indent, eventData) {
    commands.push({
      code : 221,
      indent,
      parameters : []
    });
  }

  static createCommandsForFadeInLine(line, commands, indent, eventData) {
    commands.push({
      code : 222,
      indent,
      parameters : []
    });
  }

  static createCommandsForExitLine(line, commands, indent, eventData) {
    commands.push({
      code : 115,
      indent,
      parameters : []
    });
  }

  static createCommandsForLabelLine(line, commands, indent, eventData) {
    commands.push({
      code : 118,
      indent,
      parameters : [line.line]
    });
  }

  static createCommandsForGoToLabelLine(line, commands, indent, eventData) {
    commands.push({
      code : 119,
      indent,
      parameters : [line.line]
    });
  }

  static createCommandsForBuyItemLine(line, commands, indent, eventData) {
    commands.push({
      code : 356,
      indent,
      parameters : [`BuyItem ${line.line}`]
    });
  }

  static createCommandsForEndLine(commands, indent, eventData) {
    commands.push({
      code : 0,
      indent,
      parameters : []
    });

    if (indent > 0) {
      commands.push({
        code : 412,
        indent : indent - 1,
        parameters : []
      });
    }
  }

  static createCommandsForElseLine(line, commands, indent, eventData) {
    commands.push({
      code : 0,
      indent,
      parameters : []
    });
    commands.push({
      code : 411,
      indent : indent - 1,
      parameters : []
    });
  }

  static addAllCommands(characterName, commands, lineGroup, eventData) {
    const len = lineGroup.length;
    let indent = 0;

    for (let i = 0; i < len; i++) {
      const line = lineGroup[i];

      switch(line.type) {
        case 'comment' :
        case 'comment-concat' :
          this.createCommandsForCommentLine(line, commands, indent);
          break;
        case 'dialogue' :
          this.createCommandsForDialogueLine(characterName, line, commands, indent, eventData);
          break;
        case 'choices' :
          this.createCommandsForChoiceLine(characterName, line, commands, indent, eventData);
          break;
        case 'plugin' :
          this.createCommandsForPluginLine(characterName, line, commands, indent, eventData);
          break;
        case 'erase' :
          this.createCommandsForEraseLine(characterName, line, commands, indent, eventData);
          break;
        case 'balloon' :
          this.createCommandsForBalloonLine(characterName, line, commands, indent, eventData);
          break;
        case 'condition' :
          this.createCommandsForConditionalLine(characterName, line, commands, indent, eventData);
          indent++;
          break;
        case 'rm-condition' :
          this.createCommandsForRMConditionalLine(characterName, line, commands, indent, eventData);
          indent++;
          break;
        case 'loop' :
          this.createCommandsForLoopLine(line, commands, indent, eventData);
          indent++;
          break;
        case 'breakloop' :
          this.createCommandsForBreakLoopLine(line, commands, indent, eventData);
          break;
        case 'script' :
        case 'script-concat' :
          this.createCommandsForScriptLine(characterName, line, commands, indent, eventData);
          break;
        case 'wait' :
          this.createCommandsForWaitLine(line, commands, indent, eventData);
          break;
        case 'waitfor' :
          this.createCommandsForWaitForLine(line, commands, indent, eventData);
          break;
        case 'fadeout' :
          this.createCommandsForFadeOutLine(line, commands, indent, eventData);
          break;
        case 'fadein' :
          this.createCommandsForFadeInLine(line, commands, indent, eventData);
          break;
        case 'else' :
          this.createCommandsForElseLine(line, commands, indent, eventData);
          break;
        case 'exit' :
          this.createCommandsForExitLine(line, commands, indent, eventData);
          break;
        case 'label' :
          this.createCommandsForLabelLine(line, commands, indent, eventData);
          break;
        case 'input' :
          this.createCommandsForInputLine(line, commands, indent, eventData);
          break;
        case 'inputName' :
          this.createCommandsForInputNameLine(line, commands, indent, eventData);
          break;
        case 'buyItem' :
          this.createCommandsForBuyItemLine(line, commands, indent, eventData);
          break;
        case 'goto-label' :
          this.createCommandsForGoToLabelLine(line, commands, indent, eventData);
          break;
        case 'end' :
          this.createCommandsForEndLine(commands, indent, eventData);
          indent--;
          break;
        case 'endloop' :
          this.createCommandsForEndLoopLine(line, commands, indent, eventData);
          indent--;
          break;
      }
    }

    while (indent > 0) {
      this.createCommandsForEndLine(commands, indent, eventData);
      indent--;      
    }
  }

  static log(text) {
    // console.log(...log(`[ModManager] ${ text }`));
  }

  static generateCommonEventsForFolder(eventsFolder) {
    const allFiles = Wrapper.readdirSync(eventsFolder);
    let count = 0;

    for (const fileName of allFiles) {
      const fileContent = Wrapper.readFileSync(Wrapper.joinFileNames(eventsFolder, fileName));
      const newEvents = this.parseScript(fileContent);

      const characterName = fileName.replace('.txt', '');

      for (const newEvent of newEvents) {
        this.saveEvent(characterName, newEvent);
        count++;
      }
    }

    this.log('[ModManager] Added ' + count + ' events.');
  }

  static parsePluginFile(pluginName, fileName) {
    const projectFolder = Utils.getProjectFolder();
    const fullPath = Wrapper.joinFileNames(projectFolder, 'plugins', pluginName, fileName);

    const fileContent = Wrapper.readFileSync(fullPath);
    const newEvents = this.parseScript(fileContent);

    const characterName = fileName.replace('.txt', '');

    for (let j = 0; j < newEvents.length; j++) {
      this.saveEvent(characterName, newEvents[j]);
    }
  }

  static parseExtraFile(fullFilePath, characterName) {
    const fileContent = Wrapper.readFileSync(fullFilePath);
    const newEvents = this.parseScript(fileContent);

    for (let i = 0; i < newEvents.length; i++) {
      this.saveEvent(characterName, newEvents[i]);
    }
  }

  static addExtraEventFolder(folderPath, identifier) {
    if (!this._extraEventFolders) {
      this._extraEventFolders = [];
    }

    this._extraEventFolders.push({path: folderPath, identifier});
  }

  static generateCommonEvents() {
    if (!Utils.isNwjs()) return;

    this.saveEmptyCommonEvent('0');

    return 'Done';
  }

  static saveCommonEventsFile() {
    if (Utils.isOptionValid('test')) {
      try {
        const projectFolder = Utils.getProjectFolder();
        const dataFolder = Wrapper.joinFileNames(projectFolder, 'data');
        window.fs.writeFileSync(Wrapper.joinFileNames(dataFolder, 'CommonEvents.json'), JSON.stringify($dataCommonEvents));
      }
      catch(e) {
        console.log(...log('Failed to generate CommonEvents.json'));
        console.log(...log(e));
      }
    }
  }

  static saveEmptyCommonEvent(name) {
    const commonEvent = {
      "id" : $dataCommonEvents.length,
      "name" : name,
      "switchId" : 0,
      "trigger" : 0,
      "list" : []
    };
    $dataCommonEvents.push(commonEvent);    
  }

  static saveEvent(characterName, lineGroup) {
    const commands = [];

    if (characterName && Managers.Villagers.villagerExists(characterName)) {
      commands.push({
        code : 108,
        indent : 0,
        parameters : [`[CHARACTER] ${characterName}`]
      });
    }

    this.addAllCommands(characterName, commands, lineGroup.lines, undefined, true);

    commands.push({
      code : 0,
      parameters : [],
      indent : 0
    });

    const commonEvent = {
      "id" : $dataCommonEvents.length,
      "name" : lineGroup.name,
      "switchId" : lineGroup.switchId,
      "trigger" : lineGroup.trigger,
      "list" : commands
    };
    $dataCommonEvents.push(commonEvent);
  }

  static isValidDialogueOwner(name) {
    name = name.toUpperCase();

    if (name == 'NOBODY') return true;
    if (name == 'NOTICE') return true;
    if (name == 'LETTER') return true;
    if (name == 'PLAYER') return true;

    return Managers.Villagers.villagerExists(name);
  }

  static parseList(line) {
    if (line.startsWith('(')) {
      line = line.substring(1);
    }
    if (line.endsWith(')')) {
      line = line.substring(0, line.length - 1);
    }

    return line.split(',');
  }
};
