let alternateDirection = false;

Managers.Creatures = class Creatures {
  constructor() {
    throw new Error('This is a static class');
  }

  static registerCreature(creatureClass) {
    this.creatureList[creatureClass.creatureName] = creatureClass;
  }

  static processNewDay() {
    for (const mapId in this._creatures) {
      if (isNaN(mapId)) continue;

      let i = 0;
      const len = this._creatures[mapId].length;
      let creature;

      for (i = 0; i < len; i++) {
        creature = this._creatures[mapId][i];
        if (!creature) continue;

        if (creature.processNewDay) {
          creature.processNewDay();
        }
      }

      for (i = 0; i < len; i++) {
        creature = this._creatures[mapId][i];
        if (!creature) continue;

        if (creature.processLateNewDay) {
          creature.processLateNewDay();
        }
      }
    }

    this.refreshMapEvents();
  }

  static refreshMapEvents() {
    if ($gameMap === null || $gameMap === undefined) return;

    const creatures = this._creatures[$gameMap._mapId];
    if (creatures) {
      for (let i = 0; i < creatures.length; i++) {
        if (!creatures[i]) continue;

        creatures[i].updateEvents();
      }
    }

    Managers.Map.afterRefreshCreatures();
  }

  static animalSorterFn(item1, item2) {
    return item2.friendship - item1.friendship;
  }

  static getAnimalList(filterFn, sorterFn, mapFilterFn) {
    const animals = [];

    for (const mapId in this._creatures) {
      if (isNaN(mapId)) continue;

      if (mapFilterFn && !mapFilterFn(mapId)) {
        continue;
      }

      for (let i = 0; i < this._creatures[mapId].length; i++) {
        if (!this._creatures[mapId][i]) continue;

        if (filterFn) {
          if (!filterFn(this._creatures[mapId][i])) continue;
        }

        animals.push(this._creatures[mapId][i]);
      }
    }

    animals.sort(sorterFn || this.animalSorterFn);

    return animals;
  }

  static getCreatureType(creatureName, acceptDefault) {
    const type = this.creatureList[creatureName];
    if (type !== undefined) {
      return type;
    }

    if (acceptDefault === false) return false;

    return Creatures.CreatureType;
  }

  static checkMapCreatures(mapId) {
    if (this._creatures[mapId] === undefined) {
      this._creatures[mapId] = [];
    }

    if (this._eggs[mapId] === undefined) {
      this._eggs[mapId] = [];
    }
  }

  static registerEggMom(mapId, eventId, momId) {
    this.checkMapCreatures(mapId);

    if (this._eggs[mapId][eventId] === undefined) {
      this._eggs[mapId][eventId] = {};
    }

    this._eggs[mapId][eventId].momId = momId;
    if (momId) {
      this._eggs[mapId][eventId].layDate = Managers.Time.getDateAsTimeStamp();
    } else {
      this._eggs[mapId][eventId].layDate = 0;
    }
  }

  static clearEggMomData(mapId, eventId) {
    this.registerEggMom(mapId, eventId, undefined);
  }

  static getEggMom(mapId, eventId) {
    this.checkMapCreatures(mapId);

    const data = this._eggs[mapId][eventId];
    if (!data) return undefined;

    return data.momId;
  }

  static getEggAge(mapId, eventId) {
    this.checkMapCreatures(mapId);
    const data = this._eggs[mapId][eventId];
    if (!data) return 0;

    return Managers.Time.daysSince(data.layDate);
  }

  static clearEmptyItems() {
    for (const mapId in this._creatures) {
      if (!this._creatures[mapId]) continue;

      this._creatures[mapId] = this._creatures[mapId].filter(creatureData => !!creatureData && !creatureData.isEmpty());
    }
  }

  static getCreatureList() {
    return this._creatures;
  }

  static hasAnyCreatureOfRace(race, minFriendshipLevel) {
    let count = 0;
    const minFriendship = minFriendshipLevel !== undefined ? Managers.Relationship.getFriendshipLevels()[minFriendshipLevel] : undefined;

    for (const mapId in this._creatures) {
      for (const creatureData of this._creatures[mapId]) {
        if (!creatureData) continue;
        if (!creatureData.creatureName) continue;

        if (this.isSameRace(race, creatureData.type)) {
          if (minFriendship !== undefined) {
            if (creatureData.friendship < minFriendship) continue;
          }

          count++;
        }
      }
    }

    return count;
  }

  static getCreaturesData() {
    return {
      creatures : this._creatures,
      eggs : this._eggs
    };
  }

  static setCreaturesData(data) {
    this.clear();

    if (!data) return;

    if (data.creatures) {
      this._creatures = data.creatures;
    }

    if (data.eggs) {
      this._eggs = data.eggs;
    }
  }

  static getHomeMapCreatures(homeMapId) {
    const list = [];

    for (const mapId in this._creatures) {
      if (!this._creatures.hasOwnProperty(mapId)) continue;

      const len = this._creatures[mapId].length;
      for (let i = 0; i < len; i++) {
        const creature = this._creatures[mapId][i];

        if (!creature) continue;
        if (creature.homeMapId !== homeMapId) continue;

        list.push(creature);
      }
    }

    return list;
  }

  static getMapCreatures(mapId) {
    this.checkMapCreatures(mapId);

    return this._creatures[mapId];
  }

  static hasAnyChicken(chicksCount) {
    for (let mapId in this._creatures) {
      const animals = this._creatures[mapId];

      for (let animal of animals) {
        if (!animal) continue;

        const type = this.getCreatureType(animal.type);
        if (!type) continue;
        if (type.isChicken) return true;

        if (chicksCount && type.isChick) return true;
      }
    }

    return false;
  }

  static countChickenOnMap(mapId) {
    const animals = this.getMapCreatures(mapId);
    let count = 0;

    for (let i = 0; i < animals.length; i++) {
      if (!animals[i]) continue;

      const type = this.getCreatureType(animals[i].type);
      if (!type) continue;

      if (type.isChicken) count++;
    }

    return count;
  }

  static getCreature(mapId, x, y) {
    this.checkMapCreatures(mapId);

    for (const creature of this._creatures[mapId]) {
      if (creature.x !== x) continue;
      if (creature.y !== y) continue;

      return creature;
    }

    return undefined;
  }

  static addCreature(mapId, newCreature) {
    this.checkMapCreatures(mapId);

    this._creatures[mapId].push(newCreature);
  }

  static test() {
    this.createBarnCreature('dog', 'Toddy');

    this.createCoopCreature('chicken', 'Gertrudes');
    this.createBarnCreature('cow', 'Mimosa');
  }

  static generateNewGuid() {
    return Utils.uniqueId();
  }

  static getCreatureByPosition(mapId, x, y) {
    const mapCreatures = this.getMapCreatures(mapId);
    const len = mapCreatures.length;
    for (let i = 0; i < len; i++) {
      if (mapCreatures[i].x == x && mapCreatures[i].y == y) {
        return mapCreatures[i];
      }
    }

    return null;
  }

  static prepareRandomBarnCreature(allowedTypes) {
    const typeIdx = Math.randomInt(allowedTypes.length);
    return this.prepareBarnCreature(allowedTypes[typeIdx]);
  }

  static prepareRandomCoopCreature(allowedTypes) {
    const typeIdx = Math.randomInt(allowedTypes.length);

    return this.prepareCoopCreature(allowedTypes[typeIdx]);
  }

  static addNewCreature(id, type, name, homeMapId, homeX, homeY, mapId, x, y) {
    if (mapId === undefined) mapId = homeMapId;
    if (x === undefined) x = homeX;
    if (y === undefined) y = homeY;

    const creature = new Creature();
    creature.id = id;
    creature.homeMapId = homeMapId;
    creature.homeX = homeX;
    creature.homeY = homeY;
    creature.mapId = mapId;
    creature.type = type;
    creature.x = x;
    creature.y = y;
    creature.creatureName = name;

    this.addCreature(homeMapId, creature);

    return creature;
  }

  static createDog() {
    const type = 'shiba';
    const homeX = 35;
    const homeY = 9;

    return this.addNewCreature('dog', type, Variables.dogName, Maps.FARM, homeX, homeY, Maps.FARM, homeX, homeY);
  }

  static changeDogName(newName) {
    const dog = this.getCreatureById('dog');
    if (!dog) return;

    dog.creatureName = newName;
  }

  static getDogType() {
    const dog = this.getCreatureById('dog');

    if (!dog) return '';

    return dog.type;
  }

  static prepareCoopCreatureAt(type, onePerMap, coop, age) {
    const position = this.findCoopRoomForCreature(type, onePerMap, coop);
    if (!position) return false;

    const data = {
      type,
      mapId : position.mapId,
      x : position.x,
      y : position.y,
      age : age || 0
    };

    this._lastPreparedData = data;
    return data;
  }

  static prepareCreature(type, addAge) {
    if (this.isCoopCreature(type)) {
      return this.prepareCoopCreature(type, addAge);
    } else {
      return this.prepareBarnCreature(type, addAge);
    }
  }

  static prepareCoopCreature(type, addAge) {
    const coops = Managers.FarmObjects.getCoopList();

    const creatureType = this.getCreatureType(type);
    const onePerMap = creatureType.isOnePerMap();
    let age = 0;

    if (addAge) {
      if (creatureType.creatureKindName == 'chicken' && !type.match(/chicken/)) {
        age = 1;
      } else {
        age = 10;
      }
    }

    for (let i = 0; i < coops.length; i++) {
      if (!coops[i].used) continue;

      const data = this.prepareCoopCreatureAt(type, onePerMap, coops[i], age);
      if (data) return data;
    }

    return false;
  }

  static addWildCreature(type, mapId, x, y) {
    const createEvent = mapId == $gameMap._mapId;

    return this.createCreatureAt(type, mapId, x, y, '', createEvent);
  }

  static isSameRace(creatureType1, creatureType2) {
    // ATTENTION:
    // This method will treat goats and mountain goats as same race

    // Remove numbers from the names:
    creatureType1 = creatureType1.replace(/\d*/g, '');
    creatureType2 = creatureType2.replace(/\d*/g, '');

    if (creatureType1 == creatureType2) return true;
    if (creatureType1.includes(creatureType2)) return true;
    if (creatureType2.includes(creatureType1)) return true;

    // If there's a dash in the first name, get everything after the dash and check if the second name contains that
    let idx = creatureType1.indexOf('-');
    if (idx > 0) {
      creatureType1 = creatureType1.substr(idx + 1);
      return creatureType2.includes(creatureType1);
    }

    // Same test with the variables reversed
    idx = creatureType2.indexOf('-');
    if (idx > 0) {
      creatureType2 = creatureType2.substr(idx + 1);
      return creatureType1.includes(creatureType2);
    }

    return false;
  }

  static isCoopCreature(type) {
    return this.isSameRace(type, 'chick') || this.isSameRace(type, 'chicken') || this.isSameRace(type, 'rooster');
  }

  static isBarnCreature(type) {
    return this.isSameRace(type, 'horse') || this.isSameRace(type, 'cow') || this.isSameRace(type, 'goat') || this.isSameRace(type, 'yak') || this.isSameRace(type, 'sheep');
  }

  static isPetCreature(type) {
    return this.isSameRace(type, 'bunny') || this.isSameRace(type, 'dog') || this.isSameRace(type, 'cat') || this.isSameRace(type, 'monkey') || this.isSameRace(type, 'squirrel') || this.isSameRace(type, 'shiba') || this.isSameRace(type, 'turtle');
  }

  static isWildCreature(type) {
    return this.isSameRace(type, 'bunny') || this.isSameRace(type, 'dog') || this.isSameRace(type, 'cat') || this.isSameRace(type, 'monkey') || this.isSameRace(type, 'squirrel') || this.isSameRace(type, 'duck') || this.isSameRace(type, 'turtle');
  }

  static getAnimalsForCampList() {
    return this.getAnimalList((animal) => {
      return this.isCoopCreature(animal._type) || this.isBarnCreature(animal._type);
    }, undefined, (mapId) => {
      return Number(mapId) !== Maps.ANIMAL_CAMP;
    });
  }

  static hasRoomForAnimal(type) {
    return !!this.findRoomForCreature(type);
  }

  static findCoopRoomForCreature(type, onePerMap, coop) {
    const mapId = coop.targetMapId;
    const mapClass = Managers.Map.getClass(Maps.COOP);

    const mapCreatures = this.getHomeMapCreatures(mapId);
    if (mapCreatures.length >= Constants.COOP_MAX_BIRDS) return false;

    if (onePerMap) {
      for (const creature of mapCreatures) {
        if (!creature) continue;

        if (this.isSameRace(type, creature.type)) {
          return false;
        }
      }
    }

    const positions = mapClass.getValidPositions(mapId);

    for (let k = 0; k < positions.length; k++) {
      let foundAny = false;
      for (let j = 0; j < mapCreatures.length; j++) {
        if (mapCreatures[j].homeX == positions[k].x && mapCreatures[j].homeY == positions[k].y) {
          foundAny = true;
          break;
        }
      }

      if (!foundAny) {
        const data = {
          mapId,
          x : positions[k].x,
          y : positions[k].y
        };

        return data;
      }
    }

    return false;
  }

  static findRoomForCreature(type) {
    const isCoop = this.isCoopCreature(type);

    if (isCoop) {
      const coops = Managers.FarmObjects.getCoopList();

      const coopCreatureType = this.getCreatureType(type);
      const onePerCoop = coopCreatureType.isOnePerMap();

      for (let i = 0; i < coops.length; i++) {
        if (!coops[i].used) continue;

        const position = this.findCoopRoomForCreature(type, onePerCoop, coops[i]);
        if (position) {
          return position;
        }
      }

      return false;
    } else {
      const barns = Managers.FarmObjects.getBarnList();
      const barnCreatureType = this.getCreatureType(type);

      const onePerBarn = barnCreatureType.isOnePerMap();

      for (let j = 0; j < barns.length; j++) {
        if (!barns[j].used) continue;

        const position = this.findBarnRoomForCreature(type, onePerBarn, barns[j]);
        if (position) {
          return position;
        }
      }

      return false;
    }
  }

  static coopHasRoomForCreature(type, onePerMap, coop) {
    const data = this.findCoopRoomForCreature(type, onePerMap, coop);
    return !!data;
  }

  static findBarnRoomForCreature(type, onePerMap, barn) {
    const mapId = barn.targetMapId;
    const mapClass = Managers.Map.getClass(Maps.BARN);

    const mapCreatures = this.getHomeMapCreatures(mapId);
    if (mapCreatures.length >= 20) return false;

    if (onePerMap) {
      for (const creature of mapCreatures) {
        if (!creature) continue;

        if (this.isSameRace(type, creature.type)) {
          return false;
        }
      }
    }

    const positions = mapClass.getValidPositions();

    for (let k = 0; k < positions.length; k++) {
      let foundAny = false;
      for (let j = 0; j < mapCreatures.length; j++) {
        if (mapCreatures[j].homeX == positions[k].x && mapCreatures[j].homeY == positions[k].y) {
          foundAny = true;
          break;
        }
      }

      if (!foundAny) {
        return {
          mapId,
          x : positions[k].x,
          y : positions[k].y
        };
      }
    }

    return false;
  }

  static barnHasRoomForCreature(type, onePerMap, barn) {
    const data = this.findBarnRoomForCreature(type, onePerMap, barn);
    return !!data;
  }

  static prepareBarnCreatureAt(type, onePerMap, barn, age) {
    const position = this.findBarnRoomForCreature(type, onePerMap, barn);
    if (!position) return false;

    const data = {
      type,
      mapId : position.mapId,
      x : position.x,
      y : position.y,
      age: age || 0
    };

    this._lastPreparedData = data;
    return data;
  }

  static prepareBarnCreature(type, addAge) {
    const barns = Managers.FarmObjects.getBarnList();
    const creatureType = this.getCreatureType(type);

    const onePerMap = creatureType.isOnePerMap();
    let age = 0;

    if (addAge) {
      age = 10;
    }

    for (let i = 0; i < barns.length; i++) {
      if (!barns[i].used) continue;

      const data = this.prepareBarnCreatureAt(type, onePerMap, barns[i], age);
      if (data) {
        if (creatureType.isSheep) {
          data.woolDays = 7;
          data.hasWool = true;
        }

        return data;
      }
    }

    return false;
  }

  static createCoopCreature(type, name) {
    const creatureData = this.prepareCoopCreature(type);
    if (creatureData === false) return false;

    return this.createCreatureAt(type, creatureData.mapId, creatureData.x, creatureData.y, name);
  }

  static createBarnCreature(type, name) {
    const creatureData = this.prepareBarnCreature(type);
    if (creatureData === false) return false;

    return this.createCreatureAt(type, creatureData.mapId, creatureData.x, creatureData.y, name);
  }

  static createPreparedCreature() {
    if (!this._lastPreparedData) return;

    const data = this._lastPreparedData;
    this._lastPreparedData = false;

    return this.createCreature(data);
  }

  static createPurchasedCreature() {
    const creature = this.createPreparedCreature();

    if (!creature) return;

    const typeData = this.getCreatureType(creature.type);
    if (!typeData) return;

    if (typeData._price) {
      Managers.Items.loseGold(typeData._price);
    }
  }

  static createPreparedRooster() {
    this.createPreparedCreature();
    Managers.Achievements.checkAchievements();
  }

  static dropPreparedCreature() {
    this._lastPreparedData = false;
  }

  static createCreature(data) {
    return this.createCreatureAt(data.type, data.mapId, data.x, data.y, data.creatureName, undefined, data.age);
  }

  static changeCreatureName(creature, callback, currentName) {
    this._lastPreparedData = creature;
    this._inputNameCompletedCallbackEvent = callback;

    Managers.Scenes.push(GameScenes.AnimalName);
    Managers.Scenes.prepareNextScene(currentName, creature);
  }

  static inputNameForAnimal(completedCallbackEvent) {
    // if (!this._lastPreparedData) {
    //   this._lastPreparedData = this.prepareBarnCreature();
    // }
    this._inputNameCompletedCallbackEvent = completedCallbackEvent;

    // if (!completedCallbackEvent) return false;

    Managers.Scenes.push(GameScenes.AnimalName);
    Managers.Scenes.prepareNextScene(undefined, this._lastPreparedData);
  }

  static inputNameForDog(completedCallbackEvent) {
    this._lastPreparedData = this.getCreatureById('dog');
    this._inputNameCompletedCallbackEvent = completedCallbackEvent;

    Managers.Scenes.push(GameScenes.AnimalName);
    Managers.Scenes.prepareNextScene(undefined, this._lastPreparedData);
  }

  static applyInputedName(name) {
    if (!this._lastPreparedData) return;
    if (!this._inputNameCompletedCallbackEvent) return;

    this._lastPreparedData.creatureName = name;

    this.lastCreatureName = name;
    Managers.CommonEvent.playEvent(this._inputNameCompletedCallbackEvent);
  }

  static createCreatureAt(type, mapId, x, y, name, createEvent, age) {
    if (createEvent !== false) createEvent = true;

    const creature = new Creature();
    creature.id = this.generateNewGuid();
    creature.homeMapId = mapId;
    creature.homeX = x;
    creature.homeY = y;
    creature.mapId = mapId;
    creature.type = type;
    creature.x = x;
    creature.y = y;
    creature.creatureName = name;
    creature.age = age || 0;

    this.addCreature(mapId, creature);

    if (createEvent) {
      creature.updateEvents(false);
    }

    const typeData = this.getCreatureType(type);
    if (typeData) {
      if (typeData.processNewCreatureInstance) {
        typeData.processNewCreatureInstance(creature);
      }
    }

    return creature;
  }

  static getCreatureClassType(creatureType) {
    if (!creatureType.creatureName) {
      return Objects.Animal;
    }

    if (creatureType.creatureName.includes('shiba')) {
      return Objects.Shiba;
    }

    if (creatureType.creatureName.includes('chick')) {
      return Objects.Chicken;
    }

    if (creatureType.creatureName.includes('duck')) {
      return Objects.Duck;
    }

    if (creatureType.creatureName.includes('cow')) {
      return Objects.Cow;
    }

    if (creatureType.creatureName.includes('horse')) {
      return Objects.Horse;
    }

    if (creatureType.creatureName.includes('dog')) {
      return Objects.Dog;
    }

    if (creatureType.creatureName.includes('cat')) {
      return Objects.Cat;
    }

    if (creatureType.creatureName.includes('turtle')) {
      return Objects.Turtle;
    }

    if (creatureType.creatureName.includes('bunny')) {
      return Objects.Bunny;
    }

    if (creatureType.creatureName.includes('squirrel')) {
      return Objects.Squirrel;
    }

    if (creatureType.creatureName.includes('monkey')) {
      return Objects.Monkey;
    }

    // if (creatureType.creatureName.includes('pig')) {
    //   return Objects.Pig;
    // }

    if (creatureType.creatureName.includes('sheep')) {
      return Objects.Sheep;
    }

    if (creatureType.creatureName.includes('rooster')) {
      return Objects.Rooster;
    }

    if (creatureType.creatureName.includes('goat')) {
      return Objects.Goat;
    }

    if (creatureType.creatureName.includes('yak')) {
      return Objects.Yak;
    }

    return Objects.Animal;
  }

  static createCreatureEvent(creature) {
    const x = creature.x;
    const y = creature.y;

    const creatureType = creature.getCreatureType();

    const spriteName = creatureType.getSpriteName(creature);
    const spriteIndex = creatureType.getSpriteIndex(creature);
    const direction = creatureType.getDirection(creature);

    const classType = this.getCreatureClassType(creatureType);

    const event = $gameMap.createNormalEventAt(spriteName, spriteIndex, x, y, direction, "", true, classType);
    event.setCreatureData(creature);

    return event;
  }

  static updateCreatureEvent(creature, event) {
    if (!event) return;
    if (event._erased) return;

    const x = creature.x;
    const y = creature.y;

    const creatureType = creature.getCreatureType();
    const spriteName = creatureType.getSpriteName(creature);
    const spriteIndex = creatureType.getSpriteIndex(creature);
    const direction = creatureType.getDirection(creature);

    const page = event.page();

    if (page) {
      page.image.characterName = spriteName;
      page.image.characterIndex = spriteIndex;
      page.image.direction = direction;
    }

    event.setupPage();
    event.restoreMoveType();

    event.setPosition(x, y);

    return event;
  }

  static getMapCreatureById(mapId, id) {
    this.checkMapCreatures(mapId);

    const len = this._creatures[mapId].length;
    for (let i = 0; i < len; i++) {
      const creature = this._creatures[mapId][i];

      if (!creature) continue;
      if (creature.id !== id) continue;

      return creature;
    }

    return null;
  }

  static getCreatureById(id) {
    for (const key in this._creatures) {
      if (!this._creatures.hasOwnProperty(key)) continue;

      const creature = this.getMapCreatureById(key, id);
      if (creature) return creature;
    }

    return null;
  }

  static setupMapEvents() {
    this.updateMapEvents(false);
  }

  static updateMapEvents(checkIfExists) {
    const mapId = $gameMap._mapId;
    this.checkMapCreatures(mapId);

    for (const creature of this._creatures[mapId]) {
      if (!creature) continue;

      creature.updateEvents(checkIfExists);
    }
  }

  static onChangeTime() {
    for (const creatureName in this.creatureList) {
      const creature = this.creatureList[creatureName];

      if (creature.onChangeTime) {
        creature.onChangeTime();
      }
    }
  }

  static setDogPosition(x, y, d) {
    const data = this.getCreatureById('dog');

    data.x = x;
    data.y = y;
    data.d = d;
  }

  static moveAnimal(animalData, mapId, x, y, d) {
    const oldMapId = animalData.mapId;

    animalData.x = x;
    animalData.y = y;
    animalData.d = d;

    this.checkMapCreatures(oldMapId);
    let idx = -1;
    for (let i = 0; i < this._creatures[oldMapId].length; i++) {
      if (!this._creatures[oldMapId][i]) continue;

      if (this._creatures[oldMapId][i]._id == animalData._id) {
        idx = i;
        break;
      }
    }

    if (oldMapId != mapId) {
      this.checkMapCreatures(mapId);

      if (idx < 0) return;
      animalData.mapId = mapId;
      this._creatures[oldMapId].splice(idx, 1);
      this._creatures[mapId].push(animalData);
    } else {
      if (idx < 0) return;

      this._creatures[mapId][idx] = animalData;
    }
  }

  static canChangeHomeMap(animalData, newMapId) {
    const animalType = animalData.type;
    const realMapId = Managers.Map.getRealMapId(newMapId);

    if (this.isCoopCreature(animalType)) {
      // Only change the home map if we are on a coop
      if (realMapId !== Maps.COOP) {
        return false;
      }
    }
    
    return true;
  }

  static dropAnimalAt(mapId, x, y, animalId, animalData) {
    if (this.creatureList[animalId] === undefined) return false;
    if (!animalData) return false;

    this.moveAnimal(animalData, mapId, x, y, $gamePlayer._direction);

    // if (this.canChangeHomeMap(animalData, mapId)) {
    //   animalData.homeMapId = mapId;
    //   animalData.homeX = x;
    //   animalData.homeY = y;
    // }

    animalData.updateEvents(false, true);

    if (mapId == $gameMap._mapId) {
      const creatureEvents = $gameMap.creaturesXy(x, y);
      if (creatureEvents.length > 0) {
        creatureEvents[0].doAnimalDropReaction();
        return creatureEvents[0];
      }
    }

    return true;
  }

  static clear() {
    this._creatures = {};
    this._eggs = {};
  }

  static calculateChickenPrice(animalData) {
    if (!animalData) return false;

    const price = Creatures.Chicken._price + (Math.min(animalData.friendship, 10000) / 5);
    const multiplier = ((animalData._chickenSpeed / 2.5) + (animalData._chickenLuck / 2.5) + (animalData._chickenEggChance / 2.5)) / 3;

    Variables.chickenName = animalData.creatureName;
    Variables.chickenPrice = Math.ceil(price + (1000 * multiplier));

    return true;
  }

  static getAnimalSellPrice(animalData) {
    const animalType = this.getCreatureType(animalData.type);
    if (!animalType) return 0;
    if (!animalType._price) return 0;

    const basePrice = animalType._price / 2;
    let bonusPrice = 0;
    let friendshipPrice = (Math.min(animalData.friendship, 10000) / 5000) * animalType._price;

    if (animalType.creatureKindName == 'chicken') {
      const luck = animalData._chickenLuck;
      const speed = animalData._chickenSpeed;
      const eggChance = animalData._chickenEggChance;
      const evaluation = (luck + speed + eggChance) / 5;

      bonusPrice = Math.min(evaluation, 3) * animalType._price / 3;
      friendshipPrice /= 4;
    }

    const price = Math.floor(basePrice + friendshipPrice + bonusPrice);
    return price;
  }

  static sellCreature(animalData) {
    const price = this.getAnimalSellPrice(animalData);
    if (price <= 0) return false;

    Managers.Items.gainGold(price);
    this.removeCreature(animalData);
    return true;
  }

  static sellChicken(animalData) {
    if (!this.calculateChickenPrice(animalData)) return false;

    Managers.Items.gainGold(Variables.chickenPrice);
    this.removeCreature(animalData);
    return true;
  }

  static removeCreature(creatureToRemove) {
    for (const mapId in this._creatures) {
      for (let i = 0; i < this._creatures[mapId].length; i++) {
        const creatureData = this._creatures[mapId][i];
        if (creatureData === creatureToRemove) {
          delete this._creatures[mapId][i];
        }
      }
    }

    if (Managers.Items.isHoldingSpecialItem()) {
      const holdingItem = Managers.Items.getItemInfo('animalData');
      if (holdingItem == creatureToRemove) {
        Managers.Items.specialItem = undefined;
        Managers.Items.requestHudRefresh();
      }
    }

    this.clearEmptyItems();
  }

  static moveMapCreatures(oldMapId, newMapId) {
    const creatures = this._creatures[oldMapId];
    if (creatures && creatures.length) {
      let newCreatures = this._creatures[newMapId];
      if (!newCreatures) {
        newCreatures = [];
        this._creatures[newMapId] = newCreatures;
      }

      for (let i = 0; i < creatures.length; i++) {
        if (!creatures[i]) continue;

        creatures[i].mapId = newMapId;
        newCreatures.push(creatures[i]);
        delete creatures[i];
      }
    }

    if ($gameMap._mapId == oldMapId || $gameMap._mapId == newMapId) {
      this.updateMapEvents(true);
    }
  }

  static getNumberOfCreatureKindsOwned() {
    const animals = Managers.Creatures.getAnimalList((data) => Boolean(data._creatureName)); 

    let ownedKinds = [];

    for (let animal of animals) {
      const creatureType = animal.getCreatureType();
      if (!creatureType) continue;
      if (!creatureType.creatureKindEnabled) continue;

      const name = creatureType.creatureKindName;
      if (!name) continue;

      if (ownedKinds.indexOf(name) >= 0) continue;
      ownedKinds.push(name);
    }

    return ownedKinds.length;
  }

  static getNumberOfCreatureKinds() {
    let allKinds = [];

    for (let kindName in Managers.Creatures.creatureList) {
      const kind = Managers.Creatures.creatureList[kindName];
      if (!kind) continue;
      if (!kind.creatureKindEnabled) continue;

      const name = kind.creatureKindName;
      if (!name) continue;

      if (allKinds.indexOf(name) >= 0) continue;
      allKinds.push(name);
    }

    return allKinds.length;
  }

  static getCreatureStatistics() {
    const owned = Managers.Creatures.getNumberOfCreatureKindsOwned();
    const all = Managers.Creatures.getNumberOfCreatureKinds();

    return [owned, all];
  }

  static sendBuildingAnimalsOutside() {
    const mapId = $gameMap._mapId;

    const barns = Managers.FarmObjects.getBarnList();
    const coops = Managers.FarmObjects.getCoopList();

    const buildings = [].concat(barns).concat(coops);
    let farmMapId = 0;
    let x = 0;
    let y = 0;

    for (const building of buildings) {
      if (building.targetMapId !== mapId) {
        continue;
      }

      x = building.x;
      y = building.y;
      farmMapId = building.mapId;

      break;
    }

    const buildingCreatures = [].concat(Managers.Creatures._creatures[mapId] || []);
    if (!buildingCreatures.length) {
      return;
    }

    for (const animal of buildingCreatures) {
      if (!animal) {
        continue;
      }

      Managers.Creatures.moveAnimal(animal, farmMapId, x, y + 1, 2);
    }

    $gameMap.refreshCreatures();
  }

  static sendBuildingAnimalsInside() {
    const mapId = $gameMap._mapId;

    const barns = Managers.FarmObjects.getBarnList();
    const coops = Managers.FarmObjects.getCoopList();

    const buildings = [].concat(barns).concat(coops);

    for (const building of buildings) {
      if (building.targetMapId !== mapId) {
        continue;
      }

      const mapCreatures = [].concat(Managers.Creatures._creatures[building.mapId] || []);
      if (!mapCreatures.length) {
        return;
      }

      for (const animal of mapCreatures) {
        if (!animal) {
          continue;
        }

        if (animal._homeMapId === mapId) {
          Managers.Creatures.moveAnimal(animal, animal._homeMapId, animal._homeX, animal._homeY, 2);
          continue;
        }
      }

      $gameMap.refreshCreatures();
      return;
    }
  }

  static moveAnimalsInside() {
    const mapId = $gameMap._mapId;
    const realMapId = Managers.Map.getRealMapId(mapId);

    if ([Maps.COOP, Maps.BARN, Maps.EXPANDED_COOP, Maps.EXPANDED_BARN].includes(realMapId)) {
      return this.sendBuildingAnimalsInside();
    }

    const barns = Managers.FarmObjects.getBarnList();
    const coops = Managers.FarmObjects.getCoopList();

    const buildings = [].concat(barns).concat(coops);
    const mapBuildings = [];

    // Check if there's any barn or coop in the current map
    for (const building of buildings) {
      if (building.mapId !== mapId) {
        continue;
      }

      mapBuildings.push(building);
    }

    if (!mapBuildings.length) {
      return;
    }

    const mapCreatures = [].concat(Managers.Creatures._creatures[mapId] || []);
    if (!mapCreatures.length) {
      return;
    }

    // Try moving animals inside
    const targetMapIds = [];
    for (const building of mapBuildings) {
      const { targetMapId } = building;
      targetMapIds.push(targetMapId);
    }

    const animalsForHome = [];
    const coopAnimals = [];
    const barnAnimals = [];

    for (const animal of mapCreatures) {
      if (!animal) {
        continue;
      }

      if (targetMapIds.includes(animal._homeMapId)) {
        Managers.Creatures.moveAnimal(animal, animal._homeMapId, animal._homeX, animal._homeY, 2);
        continue;
      }

      if (Managers.Map.isHomeMap(animal._homeMapId)) {
        animalsForHome.push(animal);
        continue;
      }

      if (this.isCoopCreature(animal.type)) {
        coopAnimals.push(animal);
        continue;
      }

      if (this.isBarnCreature(animal.type)) {
        barnAnimals.push(animal);
        continue;
      }

      if (this.isPetCreature(animal.type)) {
        animalsForHome.push(animal);
        continue;
      }

      // Probably something I forgot about, let's move to the barn to be safe
      barnAnimals.push(animal);
    }

    for (const animal of coopAnimals) {
      if (!animal) {
        continue;
      }

      for (const coop of coops) {
        if (!coop) {
          continue;
        }

        const pos = this.findCoopRoomForCreature(animal.type, false, coop);
        if (pos) {
          Managers.Creatures.moveAnimal(animal, pos.mapId, pos.x, pos.y, 2);
          animal.homeMapId = pos.mapId;
          animal.homeX = pos.x;
          animal.homeY = pos.y;
          
          break;
        }
      }
    }

    for (const animal of barnAnimals) {
      if (!animal) {
        continue;
      }

      for (const barn of barns) {
        if (!barn) {
          continue;
        }

        const pos = this.findBarnRoomForCreature(animal.type, false, barn);
        if (pos) {
          Managers.Creatures.moveAnimal(animal, pos.mapId, pos.x, pos.y, 2);

          animal.homeMapId = pos.mapId;
          animal.homeX = pos.x;
          animal.homeY = pos.y;

          break;
        }
      }
    }

    this.moveAnimalsToHome(animalsForHome);
    $gameMap.refreshCreatures();  
  }

  static moveAnimalsToHome(animalList) {
    const homeMapId = $gameSystem.currentHome();
    const homePositions = [];
    const usedPositions = [];
    for (let x = 16; x <= 20; x++) {
      for (let y = 5; y <= 9; y++) {
        homePositions.push({x, y});
      }
    }

    for (const animal of animalList) {
      if (!animal) {
        continue;
      }

      if (homePositions.length < usedPositions.length) {
        break;
      }

      while (true) {
        const pos = this._getRandomHomePosition(homePositions, usedPositions);
        if (!pos) {
          break;
        }

        usedPositions.push(pos);
        const { x, y } = pos;

        if (Managers.FarmObjects.isTileBlockingAnimalMovement(homeMapId, x, y)) {
          continue;
        }

        Managers.Creatures.moveAnimal(animal, homeMapId, x, y, 2);
        break;
      }
    }
  }

  static _getRandomHomePosition(possiblePositions, usedPositions) {
    if (usedPositions.length >= possiblePositions.length) {
      return false;
    }

    const idx = Math.randomInt(possiblePositions.length);
    const pos = possiblePositions[idx];

    if (this._isPositionOnHistory(pos.x, pos.y, usedPositions)) {
      return this._getRandomHomePosition(possiblePositions, usedPositions);
    }

    return pos;
  }

  static moveAnimalsOutside() {
    const mapId = $gameMap._mapId;
    const realMapId = Managers.Map.getRealMapId(mapId);

    if ([Maps.COOP, Maps.BARN, Maps.EXPANDED_COOP, Maps.EXPANDED_BARN].includes(realMapId)) {
      return this.sendBuildingAnimalsOutside();
    }

    const barns = Managers.FarmObjects.getBarnList();
    const coops = Managers.FarmObjects.getCoopList();

    const buildings = [].concat(barns).concat(coops);
    const mapBuildings = [];

    // Check if there's any barn or coop in the current map
    for (const building of buildings) {
      if (building.mapId !== mapId) {
        continue;
      }

      mapBuildings.push(building);
    }

    if (mapBuildings.length) {
      // Try moving animals outside
      for (const building of mapBuildings) {
        this.moveBuildingAnimalsOutside(building);
      }
    }

    if ($gameMap._mapId === Maps.FARM) {
      this.moveMapAnimalsOutside(Maps.HOME, 34, 9);
      this.moveMapAnimalsOutside(Maps.HOME_2, 34, 9);
      this.moveMapAnimalsOutside(Maps.HOME_3, 34, 9);
      this.moveMapAnimalsOutside(Maps.HOME_3B, 34, 9);
    }

    $gameMap.refreshCreatures();
  }

  static moveBuildingAnimalsOutside(building) {
    const { targetMapId, x, y } = building;
    this.moveMapAnimalsOutside(targetMapId, x, y);
  }

  static moveMapAnimalsOutside(mapId, buildingX, buildingY) {
    const buildingCreatures = [].concat(Managers.Creatures._creatures[mapId] || []);

    if (!buildingCreatures.length) {
      return;
    }

    for (const animal of buildingCreatures) {
      if (!animal) {
        continue;
      }

      if (!this.moveSingleAnimalOutisde(animal, buildingX, buildingY + 1)) {
        // If there's no space for this animal, then skip the whole building.
        return;
      }
    }
  }

  static moveSingleAnimalOutisde(animal, baseX, baseY) {
    const mapId = $gameMap._mapId;
    
    // If there's something blocking the movement of animals right in front of the gate, then nothing moves out
    if (Managers.FarmObjects.isTileBlockingAnimalMovement(mapId, baseX, baseY)) {
      return false;
    }

    const position = this.getNextTileForAnimal(baseX, baseY, 2);
    if (!position) {
      return false;
    }

    const { x, y, d } = position;
    Managers.Creatures.moveAnimal(animal, mapId, x, y, d);
    return true;
  }

  static _isPositionOnHistory(x, y, history) {
    for (const position of history) {
      if (position.x === x && position.y === y) {
        return true;
      }
    }

    return false;
  }

  static getNextTileForAnimal(baseX, baseY, preferredDirection, previousPositions = null) {
    const positionHistory = [].concat(previousPositions || []);
    positionHistory.push({x: baseX, y: baseY});

    const directionPreference = [];
    switch (preferredDirection) {
      case 2:
        if (alternateDirection) {
          directionPreference.push(2, 4, 6, 8);
        } else {
          directionPreference.push(2, 6, 4, 8);
        }
        break;
      case 4:
        if (alternateDirection) {
          directionPreference.push(4, 2, 8, 6);
        } else {
          directionPreference.push(4, 8, 2, 6);
        }
        break;
      case 6:
        if (alternateDirection) {
          directionPreference.push(6, 2, 8, 4);
        } else {
          directionPreference.push(6, 8, 2, 4);
        }
        break;
      case 8:
        if (alternateDirection) {
          directionPreference.push(8, 4, 6, 2);
        } else {
          directionPreference.push(8, 6, 4, 2);
        }
        break;
    }
    alternateDirection = !alternateDirection;

    for (const d of directionPreference) {
      let currentX = baseX;
      let currentY = baseY;

      switch(d) {
        case 2:
          currentY++;
          break;
        case 4:
          currentX--;
          break;
        case 6:
          currentX++;
          break;
        case 8:
          currentY--;
          break;
      }

      // If we already tried this position before on this iteration, skip to the next direction instead
      if (this._isPositionOnHistory(currentX, currentY, positionHistory)) {
        continue;
      }

      // If this path is blocked, try the next direction
      if ($gameMap.anyCollisionXy(currentX, currentY, true, true, true) || Managers.FarmObjects.isTileBlockingAnimalMovement($gameMap._mapId, currentX, currentY)) {
        continue;
      }

      // If there's already an animal here, move on to the next position
      const creatures = $gameMap.creaturesXy(currentX, currentY);
      const creature = this.getCreature($gameMap._mapId, currentX, currentY);

      if (creatures.length > 0 || creature) {
        try {
          const next = this.getNextTileForAnimal(currentX, currentY, d, positionHistory);
          if (next) {
            return next;
          }
        } catch (e) {
          // Just to be safe, let's skip any errors here and try the next direction instead.
          console.error(...error(e));
        }

        // If the next position got to a dead-end, try the next direction
        continue;
      }

      return {
        x: currentX,
        y: currentY,
        d,
      };
    }

    // No valid position found
    return false;
  }
};

Managers.Creatures._creatures = {};
Managers.Creatures._eggs = {};
Managers.Creatures.creatureList = {};

require('game/managers/Time');
Managers.Time.on('changeTime', Managers.Creatures.onChangeTime.bind(Managers.Creatures));