require('engine/core/MVCommons');

//-----------------------------------------------------------------------------
// ConfigManager
//
// The static class that manages the configuration data.

Managers.Config = class Config {
  constructor() {
    throw new Error('This is a static class');
  }

  static get masterVolume() {
    return Math.fix(Engine.Audio._masterVolume * 100, 0);
  }
  static set masterVolume(value) {
    Engine.Audio.masterVolume = Math.fix(value / 100, 0);
  }

  static get bgmVolume() {
    return Engine.Audio._bgmVolume;
  }
  static set bgmVolume(value) {
    Engine.Audio.bgmVolume = value;
    Engine.Audio.meVolume = value;
  }

  static get bgsVolume() {
    return Engine.Audio.bgsVolume;
  }

  static set bgsVolume(value) {
    Engine.Audio.bgsVolume = value;
  }

  static get seVolume() {
    return Engine.Audio.seVolume;
  }
  static set seVolume(value) {
    Engine.Audio.seVolume = value;
  }

  static get fullscreen() {
    return this._fullscreen;
  }
  static set fullscreen(value) {
    if (value !== this._fullscreen) {
      this._fullscreen = value;
    }
  }

  static get stretchToFit() {
    return this._stretchToFit;
  }
  static set stretchToFit(value) {
    if (value !== this._stretchToFit) {
      this._stretchToFit = value;
    }
    if (this._stretchToFit != Graphics._stretchEnabled) {
      Graphics._switchStretchMode();
    }
  }

  static get enableMouse() {
    return this._enableMouse;
  }
  static set enableMouse(value) {
    if (value !== this._enableMouse) {
      this._enableMouse = value;
      if (value) {
        TouchInput._mouseUsed = true;
      } else {
        TouchInput.clearMouseUsed();
      }

      Graphics.updateBodyClass();
    }
  }

  static get showAreaOfEffect() {
    return this._showAreaOfEffect;
  }
  static set showAreaOfEffect(value) {
    this._showAreaOfEffect = value;
  }
  static get drawItemBar() {
    return this._drawItemBar;
  }
  static set drawItemBar(value) {
    this._drawItemBar = value;
  }
  static get drawToolBar() {
    return this._drawToolBar;
  }
  static set drawToolBar(value) {
    this._drawToolBar = value;
  }
  static get quickText() {
    return this._quickText;
  }
  static set quickText(value) {
    this._quickText = value;
  }
  static get smallHud() {
    return this._smallHud;
  }
  static set smallHud(value) {
    this._smallHud = value;
    Graphics.hudZoomLevel = Utils.getResolutionHudZoomLevel(Graphics._resolution, this._smallHud);
    Graphics.timeHudZoomLevel = Utils.getResolutionHudZoomLevel(Graphics._resolution, this._smallHud);
  }
  static get enableAnimations() {
    return this._enableAnimations;
  }
  static set enableAnimations(value) {
    this._enableAnimations = value;
  }
  static get forceResolution() {
    return this._forceResolution;
  }
  static set forceResolution(value) {
    this._forceResolution = value;
  }
  static get useScreenRefreshRate() {
    return this._useScreenRefreshRate;
  }
  static set useScreenRefreshRate(value) {
    this._useScreenRefreshRate = value;
  }

  static get resolutionLabel() {
    const width = this.resolutionWidth;
    const height = this.resolutionHeight;

    return `${width} x ${height}`;
  }

  static get resolutionWidth() {
    return Utils.getResolutionWidth(this.resolution);
  }

  static get resolutionHeight() {
    return Utils.getResolutionHeight(this.resolution);
  }

  static get windowWidth() {
    return Utils.getResolutionWidth(this.windowSize);
  }

  static get windowHeight() {
    return Utils.getResolutionHeight(this.windowSize);
  }

  static _load() {
    let json;
    let config = {};
    try {
      json = Managers.Storage.loadFileFromGameFolder('config.ffdata');
    } catch (e) {
      console.error(e);
    }
    if (json) {
      config = JSON.parse(json);
    }
  
    return config;    
  }

  static load() {
    const config = this._load();

    this.loadedConfig = config;
    this.applyData(config);
  }

  static save() {
    //Don't try to save the config if it was never loaded or initialized
    if (!this.loadedConfig) return;

    const content = JSON.stringify(this.makeData(), undefined, 2);

    const result = Managers.Storage.saveFileToGameFolder('config.ffdata', content, false);
    if (result && result.error) {
      throw result.error;
    }
  }

  static makeData() {
    const config = {};
    config.alwaysDash = this.alwaysDash;
    config.masterVolume = this.masterVolume;
    config.bgmVolume = this.bgmVolume;
    config.bgsVolume = this.bgsVolume;
    config.seVolume = this.seVolume;
    config.fullscreen = this.fullscreen;
    config.stretchToFit = this.stretchToFit;
    config.renderer = this.renderer;
    config.resolution = this.resolution;
    config.windowSize = this.windowSize;
    config.gameFps = this.gameFps;
    config.fpsCalibration = this.fpsCalibration;
    config.keys = Engine.Input.keyMapper;
    config.gamepadProfiles = Engine.Input.getGamepadProfileData();
    config.showAreaOfEffect = this.showAreaOfEffect;
    config.drawItemBar = this.drawItemBar;
    config.drawToolBar = this.drawToolBar;
    config.quickText = this.quickText;
    config.smallHud = this.smallHud;
    config.enableMouse = this.enableMouse;
    config.enableAnimations = this.enableAnimations;
    config.frameSkip = this.maxAutoFrameSkip;
    config.forceResolution = this.forceResolution;
    config.zoom = this.zoom || 'default';
    config.useScreenRefreshRate = this.useScreenRefreshRate;

    config.configVersion = Managers.Updates.currentVersion;
    Managers.Content.configManager_makeData(config);

    return config;
  }

  static applyData(config) {
    this.masterVolume = this.readNumber(config, 'masterVolume', 30);
    this.bgmVolume = this.readVolume(config, 'bgmVolume');
    this.bgsVolume = this.readVolume(config, 'bgsVolume');
    this.seVolume = this.readVolume(config, 'seVolume');
    this.fullscreen = this.readFlag(config, 'fullscreen', false);
    this.renderer = this.readFlag(config, 'renderer', 'auto');
    this.enableMouse = this.readFlag(config, 'enableMouse', true);
    this.enableAnimations = this.readFlag(config, 'enableAnimations', true);
    this.maxAutoFrameSkip = this.readFlag(config, 'frameSkip', 0);

    this.resolution = this.readFlag(config, 'resolution', 'default');
    this.windowSize = this.readFlag(config, 'windowSize', 'default');
    this.showAreaOfEffect = this.readFlag(config, 'showAreaOfEffect', true);
    this.drawItemBar = this.readFlag(config, 'drawItemBar', true);
    this.drawToolBar = this.readFlag(config, 'drawToolBar', true);
    this.gameFps = [40, 48, 60].includes(config.gameFps) ? config.gameFps : 48;
    this.fpsCalibration = this.readNumber(config, 'fpsCalibration', 0);
    this.quickText = this.readFlag(config, 'quickText', false);
    this.smallHud = this.readFlag(config, 'smallHud', false);
    this.forceResolution = this.readFlag(config, 'forceResolution', false);
    this.stretchToFit = this.readFlag(config, 'stretchToFit', undefined);
    this.useScreenRefreshRate = this.readFlag(config, 'useScreenRefreshRate', true);

    if (Utils.isOptionValid('test')) {
      this.zoom = this.readFlag(config, 'zoom', 'default');
    } else {
      this.zoom = 'default';
    }

    const keys = this.readFlag(config, 'keys', Engine.Input.keyMapper);
    if (typeof keys == 'object') {
      Engine.Input.loadKeyMapper(keys);
    }

    const profiles = this.readFlag(config, 'gamepadProfiles', Engine.Input.gamepadProfiles);
    if (typeof profiles == 'object') {
      this.applyGamepadProfiles(profiles);
    } else {
      this.applyGamepadProfiles(Engine.Input.defaultGamepadProfiles);
    }

    const configVersion = this.readFlag(config, 'configVersion', 0);
    this.validateConfig(configVersion);

    Managers.Scenes.updateGameSpeed(60 / this.gameFps);

    Graphics._width = this.resolutionWidth;
    Graphics._height = this.resolutionHeight;
  }

  static updateOldConfig(configVersion) {
    if (configVersion < 0.5003) {
      this.maxAutoFrameSkip = 0;
    }
  }

  static validateConfig(configVersion) {
    if (configVersion < Managers.Updates.currentVersion) {
      this.updateOldConfig(configVersion);
    }

    if (this.renderer == 'webgl') {
      this.renderer = 'auto';
    }

    if (this.masterVolume > 0 && this.masterVolume < 1) {
      this.masterVolume = 30;
    }

    if (isNaN(Number(this.maxAutoFrameSkip))) {
      this.maxAutoFrameSkip = 0;
    }
    this.maxAutoFrameSkip = Number(this.maxAutoFrameSkip).clamp(0, 15);

    if (!this.forceResolution) {
      const allResolutions = Utils.getResolutionList();
      if (!allResolutions.includes(this.resolution)) {
        if (allResolutions.includes('8k')) {
          this.resolution = '4k';
        } else if (allResolutions.includes('4k')) {
          this.resolution = 'full-hd';
        } else if (allResolutions.includes('hd')) {
          this.resolution = 'hd';
        } else if (allResolutions.length > 0) {
          this.resolution = allResolutions[allResolutions.length -1];
        }
      }

      const allWindowSizes = Utils.getWindowSizeList();
      if (allWindowSizes.length > 0) {
        if (!allWindowSizes.includes(this.windowSize)) {
          if (allWindowSizes.includes(this.resolution)) {
            this.windowSize = this.resolution;
          } else {
            this.windowSize = allWindowSizes[allWindowSizes.length - 1];
          }
        }
      } else {
        this.windowSize = 'disabled';
      }
    }
  }

  static reloadKeys() {
    const savedConfigs = this._load();
    if (savedConfigs) {
      Engine.Input.loadKeyMapper(savedConfigs);
    }
  }

  static applyGamepadProfiles(profiles) {
    Engine.Input.gamepadProfiles = MVC.deepClone(Engine.Input.defaultGamepadProfiles);

    if (!profiles) {
      return;
    }

    const validActions = ['ok', 'alternate', 'use', 'extra', 'left_backbutton', 'right_backbutton', 'left_trigger', 'right_trigger', 'map', 'start', 'inventory', 'change_backpack'];

    for (const key in profiles) {
      const buttons = profiles[key];
      const profile = Engine.Input.gamepadProfiles[key] || MVC.shallowClone(Engine.Input.defaultGamepadProfiles.unknown);

      for (const buttonCode in buttons) {
        if (buttonCode == 'type') {
          profile.type = buttons[buttonCode];
          continue;
        }

        // Do not load icons and labels from config file unless the default profile is missing it
        if (buttonCode == 'icons' || buttonCode == 'labels') {
          profile[buttonCode] = profile[buttonCode] || buttons[buttonCode];
          continue;
        }

        const buttonAction = buttons[buttonCode];
        if (!validActions.includes(buttonAction)) {
          continue;
        }

        profile[buttonCode] = buttonAction;
      }

      Engine.Input.gamepadProfiles[key] = profile;
    }
  }

  static getButtonIconIndex(symbol, profileName) {
    const buttonName = symbol.replace('gamepad_', '');

    const profile = profileName ? Engine.Input.gamepadProfiles[profileName.toLowerCase()] : Engine.Input.getActiveGamepadProfile();
    if (!profile) {
      return '';
    }

    for (const button in profile) {
      if (['type', 'icons', 'labels'].includes(button)) {
        continue;
      }

      if (profile[button] == buttonName) {
        const { icons } = profile;
        return icons[button];
      }
    }

    return false;
  }

  static getButtonDescriptions(symbol, allowIcon, profileName) {
    const buttonName = symbol.replace('gamepad_', '');

    const profile = profileName ? Engine.Input.gamepadProfiles[profileName.toLowerCase()] : Engine.Input.getActiveGamepadProfile();
    if (!profile) {
      return '';
    }

    for (const button in profile) {
      if (['type', 'icons', 'labels'].includes(button)) {
        continue;
      }

      if (profile[button] == buttonName) {
        const { icons, labels } = profile;
        if (allowIcon && icons[button]) {
          return `\\i[${ icons[button] }]`;
        }

        if (labels && labels[button]) {
          return t(labels[button]);
        }

        return t('Button {0}').replace('{0}', button);
      }
    }

    return '';
  }

  static sortKeyValues(keyName, keyValues) {
    const priorityKey = Engine.Input.priorityKeys[keyName];

    if (!priorityKey) return keyValues;

    const idx = keyValues.indexOf(String(priorityKey));
    if (idx <= 0) return keyValues;

    keyValues.splice(idx, 1);
    keyValues.splice(0, 0, priorityKey);

    return keyValues;
  }

  static removeKey(key) {
    Engine.Input.removeKey(key);
  }

  static setKey(keyName, key) {
    Engine.Input.setKey(key, keyName);
  }

  static changeKey(keyName, index, newKey) {
    let keyValues = [];
    for (const key in Engine.Input.keyMapper) {
      if (Engine.Input.keyMapper[key] == keyName) {
        keyValues.push(key);
      }
    }

    if (keyValues.length > index) {
      const keyToRemove = keyValues[index];
      if (keyToRemove == newKey) return;

      this.removeKey(keyToRemove);
    }

    this.setKey(keyName, newKey);
  }

  static getKeyDescriptions(keyNameOrSymbol, useArray) {
    const keyName = keyNameOrSymbol.replace('key_', '');
    const keyDescs = [];
    const keyObj = {};

    if (useArray === undefined) useArray = true;

    let keyValues = [];
    for (const key in Engine.Input.keyMapper) {

      if (Engine.Input.keyMapper[key] == keyName) {
        keyValues.push(key);
      }
    }

    keyValues = this.sortKeyValues(keyNameOrSymbol, keyValues);

    for (let i = 0; i < keyValues.length; i++) {
      let keyDesc = Engine.Input.keyDescriptions[keyValues[i]];
      if (!keyDesc) {
        keyDesc = `${t('Key')} ${keyValues[i]}`;
      }

      keyDescs.push(keyDesc);
      keyObj[keyValues[i]] = keyDesc;
    }

    if (useArray) {
      return keyDescs;
    }

    return keyObj;
  }

  static getKeyDescription(keyName, all) {
    const keyNames = keyName.split(',');
    let allDescriptions = [];

    for (let i = 0; i < keyNames.length; i++) {
      const descs = this.getKeyDescriptions(keyNames[i], true);
      if (descs.length > 0) {
        if (all) {
          allDescriptions = allDescriptions.concat(descs);
        } else {
          allDescriptions.push(descs[0]);
        }
      } else {
        allDescriptions.push(keyNames[i]);
      }

    }

    if (allDescriptions.length > 1) {
      return allDescriptions;
    }
    if (allDescriptions.length == 1) {
      return allDescriptions[0];
    }

    return keyName;
  }

  static readFlag(config, name, defaultValue) {
    const value = config[name];
    if (value !== undefined) {
      return value;
    } else {
      return defaultValue;
    }
  }

  static readVolume(config, name) {
    const value = config[name];
    if (value !== undefined) {
      return Number(value).clamp(0, 100);
    } else {
      return 100;
    }
  }

  static readNumber(config, name, defaultValue) {
    const value = config[name];
    if (value !== undefined) {
      return Number(value);
    } else {
      return defaultValue || 0;
    }
  }
};

Managers.Config.renderer = 'auto';
Managers.Config.resolution = 'default';
Managers.Config.windowSize = 'default';
Managers.Config.gameFps = 48;
Managers.Config.fpsCalibration = 0;
Managers.Config.language = undefined;
Managers.Config.loadedConfig = undefined;
Managers.Config.maxAutoFrameSkip = 0;
Managers.Config.forceResolution = false;
Managers.Config.stretchToFit = undefined;
Managers.Config.useScreenRefreshRate = true;
