Managers.CommonEvent = class CommonEvent {
  static getKeyValue(keyName, singleKey) {
    const description = Managers.Config.getKeyDescription(keyName, false);

    if (description instanceof Array) {
      let descStr = '';
      for (let i = 0; i < description.length; i++) {
        if (singleKey) {
          descStr = description[i];
          break;
        } else {
          if (descStr !== '') {
            descStr += ', ';
          }
        }

        descStr += description[i];
      }

      if (singleKey) {
        return this.createKeyString(descStr);
      } else {
        return this.createKeysString(descStr);
      }
    }

    return this.createKeyString(description);
  }

  static getGamepadIcon(buttonName, profileName) {
    return Managers.Config.getButtonDescriptions(buttonName, true, profileName);
  }

  static createKeyString(keyDescription) {
    return t("{key} key").replace('{key}', keyDescription);
  }

  static createKeysString(keyDescription) {
    return t("{keys} keys").replace('{keys}', keyDescription);
  }

  static isHoldingItem() {
    return Managers.Items.isHoldingItem();
  }

  static getCurrentTool() {
    if (Managers.Tools.toolId) {
      return Managers.Tools.toolId;
    }
    
    return 'none';
  }

  static getSelectedItem() {
    return Managers.Items.getSelectedItemId();
  }

  static getSelectedItemName() {
    if (Managers.Items.selectedItem) {
      return Managers.Text.item(Managers.Items.selectedItem.id);
    }
    
    return t('Nothing');
  }

  static getGender() {
    return Managers.Player.getGender();
  }

  static getDateTime() {
    return Managers.Time.getDateTime();
  }

  static getCharacterName() {
    return "\\P[1]";
  }

  static getFarmName() {
    return Variables.farmName;
  }

  static setVariableValue(variableName, variableValue) {
    Variables[variableName] = variableValue;
  }

  static getVariableValue(variableName) {
    return Variables[variableName];
  }

  static replaceVariables(line) {
    let newLine = line;

    newLine = newLine.replace(/<name>/ig, this.getCharacterName());
    newLine = newLine.replace(/<farm>/ig, this.getFarmName());
    newLine = newLine.replace(/<current_item>/ig, this.getSelectedItem());
    newLine = newLine.replace(/<gift>/ig, this.getSelectedItemName());

    while (true) {
      const variableMatches = newLine.match(/<var:(.+?)>/);
      if (!variableMatches) break;
      if (variableMatches.length < 2) break;

      const variableName = variableMatches[1];
      let variableValue = this.getVariableValue(variableName);

      if (variableValue === undefined) variableValue = '';

      newLine = newLine.replace(variableMatches[0], variableValue);
    }

    while (true) {
      const itemMatches = newLine.match(/<item:(.+?)>/);
      if (!itemMatches) break;
      if (itemMatches.length < 2) break;

      const itemId = itemMatches[1];
      if (itemId) {
        const itemName = Managers.Text.item(itemId) || '';
        newLine = newLine.replace(itemMatches[0], itemName);
      }
    }

    while (true) {
      const scriptMatches = newLine.match(/<script:(.+?)>/);
      if (!scriptMatches) break;
      if (scriptMatches.length < 2) break;

      const scriptCode = scriptMatches[1];
      let scriptValue = MVC.safeEval(scriptCode);

      if (scriptValue === undefined) scriptValue = '';
      newLine = newLine.replace(scriptMatches[0], scriptValue);
    }

    const dateTime = this.getDateTime();
    newLine = newLine.replace(/<hour>/ig, dateTime.hour);
    newLine = newLine.replace(/<minute>/ig, `${dateTime.minute}0`);
    newLine = newLine.replace(/<second>/ig, `${dateTime.seconds}0`);
    newLine = newLine.replace(/<day>/ig, dateTime.day);
    newLine = newLine.replace(/<month>/ig, dateTime.month);
    newLine = newLine.replace(/<year>/ig, dateTime.year);
    newLine = newLine.replace(/<date>/ig, '');
    newLine = newLine.replace(/<time>/ig, '');
    newLine = newLine.replace(/<quiet>/ig, "\\QUIET");
    newLine = newLine.replace(/<resetfont>/ig, "\\RESET");
    newLine = newLine.replace(/<bigger>/ig, '\\{');
    newLine = newLine.replace(/<smaller>/ig, '\\}');

    const gender = this.getGender();
    switch(gender) {
      case 'male' :
        newLine = newLine.replace(/<he>/g, 'he');
        newLine = newLine.replace(/<He>/g, 'He');
        newLine = newLine.replace(/<his>/g, 'his');
        newLine = newLine.replace(/<His>/g, 'His');
        newLine = newLine.replace(/<him>/g, 'him');
        newLine = newLine.replace(/<Him>/g, 'Him');
        newLine = newLine.replace(/<himself>/ig, 'himself');
        newLine = newLine.replace(/<guy>/ig, 'guy');
        
        newLine = newLine.replace(/<o>/ig, 'o');
        break;
      case 'female' :
        newLine = newLine.replace(/<he>/g, 'she');
        newLine = newLine.replace(/<He>/g, 'She');
        newLine = newLine.replace(/<his>/g, 'hers');
        newLine = newLine.replace(/<His>/g, 'Hers');
        newLine = newLine.replace(/<him>/g, 'her');
        newLine = newLine.replace(/<Him>/g, 'Her');
        newLine = newLine.replace(/<himself>/ig, 'herself');
        newLine = newLine.replace(/<o>/ig, 'a');
        newLine = newLine.replace(/<guy>/ig, 'girl');
        break;
      default :
        newLine = newLine.replace(/<he>/g, 'they');
        newLine = newLine.replace(/<He>/g, 'They');
        newLine = newLine.replace(/<his>/g, 'their');
        newLine = newLine.replace(/<His>/g, 'Their');
        newLine = newLine.replace(/<him>/g, 'them');
        newLine = newLine.replace(/<Him>/g, 'Them');
        newLine = newLine.replace(/<himself>/ig, 'themself');
        newLine = newLine.replace(/<o>/ig, 'o');
        newLine = newLine.replace(/<guy>/ig, 'person');
        break;
    }

    return newLine;
  }

  static getFinalName(eventName) {
    for (const key in this._redirects) {
      if (!this._redirects.hasOwnProperty(key)) continue;

      if (key == eventName) {
        return this.getFinalName(this._redirects[key]);
      }
    }

    return eventName;
  }

  static findEventIdByName(eventName) {
    eventName = this.getFinalName(eventName);

    const len = $dataCommonEvents.length;
    for (let i = 0; i < len; i++) {
      const data = $dataCommonEvents[i];
      if (!data) continue;
      if (data.name != eventName) continue;

      return i;
    }
  }

  static eventExists(eventName) {
    if (this.isEventDisabled(eventName)) return false;

    const eventId = this.findEventIdByName(eventName);

    return eventId >= 0;
  }

  static playChildEvent(eventName, interpreter) {
    const eventId = this.findEventIdByName(eventName);
    if (eventId >= 0) {
      if (interpreter === undefined) {
        if (window.evalInterpreter) {
          interpreter = window.evalInterpreter;
        } else {
          interpreter = $gameMap._interpreter;
        }
      }

      const commonEvent = $dataCommonEvents[eventId];
      if (commonEvent) {
        interpreter.setupChild(commonEvent.list, eventId);
      }
    }
  }

  static includeChild(eventName, interpreter) {
    if (interpreter === undefined) {
      if (window.evalInterpreter) {
        interpreter = window.evalInterpreter;
      } else {
        interpreter = $gameMap._interpreter;
      }
    }

    const eventId = this.findEventIdByName(eventName);
    if (eventId >= 0) {
      const commonEvent = $dataCommonEvents[eventId];

      if (commonEvent) {
        interpreter.includeChild(commonEvent.list);
        return true;
      }
    }

    return false;
  }

  static playEvent(eventName, parentEvent, onlyOnce) {
    onlyOnce = onlyOnce !== false;

    const eventId = this.findEventIdByName(eventName);
    if (eventId >= 0) {
      if (parentEvent) {
        $gameMap.setupVillagerEvent(parentEvent, eventId);
        return true;
      } else {
        if (!onlyOnce || !$gameTemp.isThisCommonEventReserved(eventId)) {
          $gameTemp.reserveCommonEvent(eventId);
        }
        return true;
      }
    }

    return false;
  }

  static playEventAsync(eventName) {
    const eventId = this.findEventIdByName(eventName);
    if (eventId >= 0) {
      $gameTemp.reserveAsyncCommonEvent(eventId);
    }
  }

  static runCommand(commandLine, interpreter = undefined) {
    if (interpreter === undefined) {
      if (window.evalInterpreter) {
        interpreter = window.evalInterpreter;
      } else {
        interpreter = $gameMap._interpreter;
      }
    }

    const args = commandLine.split(' ');
    const command = args.shift();
    interpreter.pluginCommand(command, args);
    return true;
  }

  static doNewDayProcessing() {
    $gameSystem.eraseDialogueHistories();

    Switches.receivedDailyGiftFromChloe = false;
  }

  static getDialogueLineSettings(line) {
    const settings = {
      stance : 'normal',
      position : 'low',
      invisible : false,
      hideName : false,
      input : false
    };

    const stances = this.possibleStances;
    for (let i = 0; i < stances.length; i++) {
      const stanceTag = `<${stances[i]}>`;

      if (!line.includes(stanceTag)) continue;

      settings.stance = stances[i];
      line = line.replace(stanceTag, '');
    }
    
    const positions = this.possiblePositions;
    const len = positions.length;
    for (let j = 0; j < len; j++) {
      let positionTag = `<${positions[j]}>`;

      if (line.includes(positionTag)) {
        settings.position = positions[j];
        line = line.replace(positionTag, '');
      }

      positionTag = `<${positions[j]}, invis>`;
      if (line.includes(positionTag)) {
        settings.position = positions[j];
        settings.invisible = true;
        line = line.replace(positionTag, '');
      }
    }

    if (line.includes('<invis>')) {
      settings.invisible = true;
      line = line.replace('<invis>', '');
    }

    if (line.includes('<nar>')) {
      settings.hideName = true;
      line = line.replace('<nar>', '');
    }

    if (line.includes('<hideName>')) {
      settings.hideName = true;
      line = line.replace('<hideName>', '');
    }

    if (line.includes('<input:')) {
      const params = line.match(/<input:(\d+),(\d+)>/);
      if (params && params.length === 3) {
        settings.input = [Number(params[1]), Number(params[2])];
      }

      line = line.replace(/<input:\d+,\d+>/, '');
    }

    if (line.includes('<inputVar:')) {
      const inputParams = line.match(/<inputVar:(.+),(.+)>/);
      if (inputParams && inputParams.length === 3) {
        settings.input = [inputParams[1], inputParams[2], true];
      }

      line = line.replace(/<inputVar:.+?,.+?>/, '');
    }

    settings.line = line.trim();
    return settings;
  }

  static isSystemName(name) {
    const lowerName = name.toLowerCase();
    if (lowerName == 'letter') return true;
    if (lowerName == 'notice') return true;
    if (lowerName == 'nobody') return true;
    if (lowerName == 'player') return true;

    return false;
  }

  static getNameString(name, settings) {
    if (name) {
      const lowerName = name.toLowerCase();

      if (lowerName == 'letter') {
        return '';
      }

      if (lowerName == 'player') {
        if (settings.invisible) {
          return "\\NDR<\\n[1]>";
        } else {
          return "\\NR<\\n[1]>";
        }
      }

      if (this.isSystemName(name)) {
        name = '';
      }

      const translatedName = Managers.Text.translateOnly(name);

      if (settings.invisible) {
        return `\\ND<${translatedName}>`;
      } else {
        return `\\N<${translatedName}>`;
      }
    }

    return '';
  }

  static getPortraitString(name) {
    if (name) {
      const lowerName = name.toLowerCase();
      if (lowerName == 'letter') return '';
      if (lowerName == 'notice') return '';
      if (lowerName == 'nobody') return '';

      if (lowerName == 'player') {
        return "\x1bNP<Player>";
      }
      
      return `\x1bNP<${name}>`;
    }

    return '';
  }

  static makeConditionTable() {
    this.conditionTable = {};
    this.deactivationConditions = {};

    const len = $dataCommonEvents.length;
    for (let i = 0; i < len; i++) {
      if (!$dataCommonEvents[i]) continue;
      const commands = $dataCommonEvents[i].list;
      let characterName = undefined;
      let condition = undefined;
      let priority = undefined;
      let disableIf = undefined;
      const eventName = $dataCommonEvents[i].name;

      const len2 = commands.length;
      for (let j = 0; j < len2; j++) {
        const commandData = commands[j];

        if (commandData.code != 108) continue;
        if (commandData.parameters.length < 1) continue;

        let line = commandData.parameters[0];

        if (line.substr(0, 1) == '#') {
          line = line.substr(1, line.length - 1);
        }

        if (line.includes('[CHARACTER]')) {
          line = line.replace('[CHARACTER]', '').trim();
          characterName = line.toLowerCase();
        } else if (line.indexOf('[CONDITION]', '') >= 0) {
          condition = line.replace('[CONDITION]', '').trim();
        } else if (line.indexOf('[PRIORITY]', '') >= 0) {
          priority = line.replace('[PRIORITY]', '').trim();
        } else if (line.indexOf('[DISABLEIF]', '') >= 0) {
          disableIf = line.replace('[DISABLEIF]', '').trim();
        }
      }

      if (!!condition && !!characterName) {
        if (!this.conditionTable[characterName]) {
          this.conditionTable[characterName] = {};
        }

        this.conditionTable[characterName][eventName] = {
          condition,
          priority : priority || false
        };
      }

      if (disableIf) {
        this.deactivationConditions[eventName] = disableIf;
      }
    }
  }

  static getCharacterEvents(characterName) {
    return this.conditionTable[characterName.toLowerCase()] || {};
  }

  static isEventDisabled(eventName) {
    if (!this.deactivationConditions[eventName]) return false;

    const script = this.deactivationConditions[eventName];
    return Boolean(MVC.safeEval(script));
  }

  static checkBalloons(fullLine) {
    const line = fullLine.trim().toLowerCase();
    const balloonNames = this.possibleBalloons;
    const len = balloonNames.length;

    for (let i = 0; i < len; i++) {
      const balloonName = `<${balloonNames[i]}>`;
      const balloonNameAsync = `<${balloonNames[i]}:async>`;

      //If the only thing on the message is a balloon, then it's a regular ballon, wait = true by default, unless async is specified
      if (line == balloonName || line == balloonNameAsync) {
        return {
          balloonId : i + 1,
          wait : line == balloonName,
          newLine : ''
        };
      } 

      //If the line starts with a balloon but there's more, then it's a balloon with wait = false, regardless if it was specified as async or not
      if (line.includes(balloonName) || line.includes(balloonNameAsync)) {
        // lineData.line = lineData.line.replace(balloonName + ' ', '');

        fullLine = fullLine.replace(balloonName, '').replace(balloonNameAsync, '');
        return {
          balloonId : i + 1,
          wait : false,
          newLine : fullLine
        };
      }
    }

    return false;
  }

  static generateDataForChoiceLine(name, line, extraChoices) {
    if (line.startsWith(`${name} `)) {
      line = line.substr(name.length + 1);
    }

    let regex = /\((.*)\)/;
    const result = regex.exec(line);
    if (!result) return;
    if (result.length < 2) return undefined;

    const paramsStr = result[1];
    regex = /(["])((?:\\\1|.)*?)\1/ig;

    const params = paramsStr.match(regex);
    const len = params.length;
    let extraLen = 0;
    if (extraChoices) extraLen = extraChoices.length;
    if (len + extraLen < 3) return undefined;

    let i;
    for (i = 0; i < len; i++) {
      params[i] = params[i].substr(1, params[i].length - 2);
    }

    const settings = this.getDialogueLineSettings(params[0]);
    line = settings.line;

    const balloonData = this.checkBalloons(line);
    let balloonId = false;
    let waitForBalloon = false;

    if (balloonData) {
      line = balloonData.newLine;
      balloonId = balloonData.balloonId;
      waitForBalloon = balloonData.wait;
    }

    let translatedLine = t(line);
    const displayName = settings.invisible ? '' : ((settings.hideName && !this.isSystemName(name)) ? '?' : name);
    const portraitName = settings.invisible ? '' : name;

    let msg = '';
    if (line.trim() !== '') {
      msg += this.getNameString(displayName, settings);
      msg += this.getPortraitString(portraitName);
      
      translatedLine = this.replaceVariables(translatedLine);
      msg += translatedLine;
    }

    const position = 2;
    let background = 0;
    if (settings.invisible) {
      background = 2;
    }

    let choices = [];
    if (extraChoices) choices = choices.concat(extraChoices);
    let cancelChoiceIndex = -1;

    // Adds the choices
    for (i = 1; i < len; i++) {
      choices.push(params[i]);
    }

    // Look for the cancel index
    const choiceLen = choices.length;
    for (i = 0; i < choiceLen; i++) {
      for (let j = i + 1; j < choiceLen; j++) {
        if (choices[i].trim() == choices[j].trim()) {
          cancelChoiceIndex = i;
          choices.splice(j, 1);
          break;
        }
      }

      if (cancelChoiceIndex >= 0) break;
    }

    return {
      faceName : '',
      faceIndex : 0,
      backgroundType : background,
      positionType : position,
      message : msg,
      balloonId,
      waitForBalloon,
      choices,
      cancelChoiceIndex
    };
  }

  static generateDataForDialogueLine(name, line) {
    if (line.startsWith(`${name} `)) {
      line = line.substr(name.length + 1);
    }

    const settings = this.getDialogueLineSettings(line);
    line = settings.line;
    let balloonId = false;
    let waitForBalloon = false;

    const balloonData = this.checkBalloons(line);
    if (balloonData) {
      line = balloonData.newLine;
      balloonId = balloonData.balloonId;
      waitForBalloon = balloonData.wait;
    }

    let msg = '';

    const displayName = settings.invisible ? '' : ((settings.hideName && !this.isSystemName(name)) ? '?' : name);
    const portraitName = settings.invisible ? '' : name;

    if (line.trim() === '') {
      msg = '';
    } else {
      msg += this.getNameString(displayName, settings);
      msg += this.getPortraitString(portraitName);

      let translatedLine = t(line);
      translatedLine = this.replaceVariables(translatedLine);

      msg += translatedLine;
    }

    let position = 2;
    let background = 0;

    switch(settings.position) {
      case 'high' :
        position = 0;
        break;
      case 'mid' :
        position = 1;
        break;
      case 'low' :
        position = 2;
        break;
      default :
        if ($gamePlayer.screenY() > Graphics.height / 2) {
          position = 0;
        } else {
          position = 2;
        }
        break;
    }

    if (settings.invisible) {
      background = 2;
    }

    return {
      faceName : '',
      faceIndex : 0,
      backgroundType : background,
      positionType : position,
      message : msg,
      balloonId,
      waitForBalloon,
      input : settings.input
    };
  }

  static enableSkip(callback) {
    this._skipCallback = callback;
    this._skipped = false;
  }

  static disableSkip() {
    this._skipCallback = false;
    this._skipped = false;
  }

  static skipCutscene() {
    if (!this._skipCallback) {
      return false;
    }

    this._skipped = true;
    $gameMessage.clear();
    return true;
  }

  static doSkip() {
    this.playEvent(this._skipCallback);

    this._skipCallback = false;
    this._skipped = false;
  }

  static makeAlias(oldName, newName) {
    this._aliases[oldName] = newName;
  }

  static redirectEvent(oldName, newName) {
    this._redirects[oldName] = newName;
  }
};


Managers.CommonEvent.possibleStances = ['smiling', 'sad', 'angry', 'worried', 'laughing', 'sighing', 'dreamy', 'surprised'];
Managers.CommonEvent.possibleBalloons = ['!', '?', 'music', 'heart', 'anger', 'sweatdrop', 'cobweb', '...', 'light', 'zzz', 'cry', 'item', 'thinking', 'quickItem', 'backpack', 'failure', 'talking', 'secret'];
Managers.CommonEvent.possiblePositions = ['high', 'mid', 'low'];
Managers.CommonEvent.conditionTable = {};
Managers.CommonEvent.deactivationConditions = {};

Managers.CommonEvent._skipCallback = false;
Managers.CommonEvent._skipped = false;
Managers.CommonEvent._aliases = {};
Managers.CommonEvent._redirects = {};
