var TileHelper = {};

(function($){
  $.Tile = function(x, y) {
    var tile = this;

    this.x = x;
    this.y = y;

    this.westTile = function(distance) {
      distance = distance || 1;
      return $.newTile(tile.x - distance, tile.y);
    };

    this.eastTile = function(distance){
      distance = distance || 1;
      return $.newTile(tile.x + distance, tile.y);
    };

    this.northTile = function(distance){
      distance = distance || 1;
      return $.newTile(tile.x, tile.y - distance);
    };

    this.southTile = function(distance){
      distance = distance || 1;
      return $.newTile(tile.x, tile.y + distance);
    };

    this.tileAtDirection = function(d, distance) {
      if (distance === undefined) distance = 1;

      switch (d) {
        case Direction.DOWN :
          return this.southTile(distance);
        case Direction.LEFT :
          return this.westTile(distance);
        case Direction.RIGHT :
          return this.eastTile(distance);
        case Direction.UP :
          return this.northTile(distance);
        default :
          return this;
      }
    };

    this.isValid = function() {
      return $.isValid(this.x, this.y);
    };

    this.isSoil = function() {
      return Managers.FarmObjects.tileIsSoil(this.x, this.y);
    };

    this.isWater = function() {
      return Managers.FarmObjects.tileIsWater(this.x, this.y);
    };
  };

  $.newTile = function(x, y) {
    if (this.isValid(x, y)) {
      return new this.Tile(x, y);
    } else {
      return undefined;
    }
  };  

  $.isValid = function(x, y) {
    return $gameMap.isValid(x, y);
  };

  $.getTileAtDirectionFromPosition = function(left, right, top, bottom, distance, direction, useFloat) {
    var mapX = useFloat ? left : Math.round(left);
    var mapY = useFloat ? top : Math.round(top);

    if (distance > 0) {
      if (distance < 1) {
        switch (direction) {
          case Direction.DOWN :
            mapY = Math.round(bottom + distance - 0.7);
            break;
          case Direction.LEFT :
            mapX = Math.floor(left - distance);
            mapY = Math.round(top - 0.3);
            break;
          case Direction.RIGHT :
            mapX = Math.round(right + distance);
            mapY = Math.round(top - 0.3);
            break;
          case Direction.UP :
            mapY = Math.floor(top - distance);
            break;
          default :
            break;
        }
      } else {
        switch (direction) {
          case Direction.DOWN :
            mapY += distance;
            break;
          case Direction.LEFT :
            mapX -= distance;
            break;
          case Direction.RIGHT :
            mapX += distance;
            break;
          case Direction.UP :
            mapY -= distance;
            break;
          default :
            break;
        }
      }
    } else {
      switch (direction) {
        case Direction.DOWN :
          mapY = Math.ceil(bottom);
          break;
        case Direction.LEFT :
          mapX = Math.floor(left);
          break;
        case Direction.RIGHT :
          mapX = Math.ceil(right);
          break;
        case Direction.UP :
          mapY = Math.floor(top);
          break;
        default :
          break;
      }
    }

    return new $.Tile(mapX, mapY);  
  };

  $.getTileAtDirection = function(distance, direction, character, useFloat) {
    if (distance === undefined) distance = 1;
    if (character === undefined) character = $gamePlayer;
    if (direction === undefined) direction = character._direction;

    return $.getTileAtDirectionFromPosition(character.left, character.right, character.top, character.bottom, distance, direction, useFloat);
  };

  $.getTilesAtDirection = function(distance, direction, character) {
    if (distance === undefined) distance = 1;
    if (character === undefined) character = $gamePlayer;
    if (direction === undefined) direction = character._direction;

    let left1 = character.left;
    let left2 = left1;
    let right1 = character.right;
    let right2 = right1;
    let top1 = character.top;
    let top2 = top1;
    let bottom1 = character.bottom;
    let bottom2 = bottom1;

    switch(direction) {
      case Direction.UP:
      case Direction.DOWN:
        left1 = Math.floor(left1);
        left2 = Math.ceil(left2);
        right1 = Math.floor(right1);
        right2 = Math.ceil(right2);

        if (left2 > right1) {
          left2 = right1;
        }

        break;
      case Direction.LEFT:
      case Direction.RIGHT:
        top1 = Math.floor(top1);
        top2 = Math.ceil(top2);
        bottom1 = Math.floor(bottom1);
        bottom2 = Math.ceil(bottom2);

        if (top2 > bottom1) {
          top2 = bottom1;
        }
        break;
    }

    var tiles = [];
    var tile1 = $.getTileAtDirectionFromPosition(left1, right1, top1, bottom1, distance, direction);
    if (tile1) {
      tiles.push(tile1);
    }
    if (left1 != left2 || right1 != right2 || bottom1 != bottom2 || top1 != top2) {
      var tile2 = $.getTileAtDirectionFromPosition(left2, right2, top2, bottom2, distance, direction);
      if (tile2) {
        tiles.push(tile2);
      }
    }

    return tiles;
  };

  $.getCircularTiles = function(x, y, radius) {
    var tiles = [];

    for (var currentY = 0; currentY <= radius; currentY++) {
      var maxX = Math.floor(Math.sqrt(Math.pow(radius, 2) - Math.pow(currentY, 2)));

      for (var currentX = -maxX; currentX <= maxX; currentX++) {
        tiles.push(new $.Tile(x + currentX, y - currentY));
        if (currentY !== 0) {
          tiles.push(new $.Tile(x + currentX, y + currentY));
        }
      }
    }

    return tiles;
  };

  $.getConeTiles = function(x, y, direction, distance) {
    var tiles = [];

    switch(direction) {
      case Direction.LEFT :
        break;
      case Direction.RIGHT :
        break;
      case Direction.UP :
        break;
      case Direction.DOWN :
        break;
    }

    return tiles;
  };

  $.getAlternativeFrontTile = function(baseTile, character, direction) {
    if (character === undefined) character = $gamePlayer;
    if (direction === undefined) direction = character._direction;

    if (direction == Direction.DOWN || direction == Direction.UP) {
      if (Math.floor(character._x) == Math.round(character._x)) {
        return baseTile.tileAtDirection(Direction.RIGHT);
      } else {
        return baseTile.tileAtDirection(Direction.LEFT);
      }
    } else {
      if (Math.floor(character._y) == Math.round(character._y)) {
        return baseTile.tileAtDirection(Direction.DOWN);
      } else {
        return baseTile.tileAtDirection(Direction.UP);
      }
    }
  };

  $.getSmartFrontTile = function(distance, character, direction, useFloat = false) {
    if (character === undefined) character = $gamePlayer;
    if (direction === undefined) direction = character._direction;

    var mapX;
    var mapY;
    
    // Down, Left and Right had to be changed because of the player's offset position and to ease tile selection on tight spaces
    if (distance && distance >= 1) {
      if (direction == Direction.DOWN) {
        mapX = useFloat ? character._x : Math.round(character._x);

        mapY = Math.round(character._y + 0.5);
        mapY += (distance - 1);

        return new $.Tile(mapX, mapY);
      } else if (direction == Direction.RIGHT) {
        mapY = useFloat ? character._y : Math.round(character._y - 0.25);

        var diffX = character._x - Math.floor(character._x);
        if (diffX <= 0.5 && diffX > 0) {
          mapX = Math.ceil(character._x);
        } else {
          mapX = Math.ceil(character._x) + 1;
        }

        mapX += (distance - 1);
        return new $.Tile(mapX, mapY);
      } else if (direction == Direction.LEFT) {
        mapY = useFloat ? character._y : Math.round(character._y - 0.25);
        mapX = Math.round(character._x) - distance;

        return new $.Tile(mapX, mapY);
      }
    }

    return this.getFrontTile(distance, character, direction, useFloat);
  };


  $.getFrontTile = function(distance, character, direction, useFloat) {
    return this.getTileAtDirection(distance, direction, character, useFloat);
  };

  $.getBackTile = function(distance) {
    return this.getTileAtDirection(distance, $gamePlayer.backDirection());
  };

  $.getLeftTile = function(distance) {
    return this.getTileAtDirection(distance, $gamePlayer.leftDirection());
  };

  $.getRightTile = function(distance) {
    return this.getTileAtDirection(distance, $gamePlayer.rightDirection());
  };

  $.getTileRegion = function(mapX, mapY) {
    return $gameMap.regionId(mapX, mapY);
  };

  $.checkIfTileHasUpperLayers = function(mapX, mapY) {
    var tile0 = $gameMap.tileId(mapX, mapY, 0);
    var tile1 = $gameMap.tileId(mapX, mapY, 1);
    var tile2 = $gameMap.tileId(mapX, mapY, 2);
    var tile3 = $gameMap.tileId(mapX, mapY, 3);

    if (tile1 > 0 && tile1 !== tile0) return true;
    if (tile2 > 0) return true;
    if (tile3 > 0) return true;

    return false;
  };

  $.checkIfTileCanGetObject = function(mapX, mapY, allowPlayer, objectType, allowProtected, allowItems, allowSeeds) {
    const tile0 = $gameMap.tileId(mapX, mapY, 0);
    const tile1 = $gameMap.tileId(mapX, mapY, 1);
    const tile2 = $gameMap.tileId(mapX, mapY, 2);
    const tile3 = $gameMap.tileId(mapX, mapY, 3);

    if (tile0 > 0 && $gameMap.getPassabilityFromTileId(tile0) === false) return false;
    if (tile1 > 0 && tile1 !== tile0 && $gameMap.getPassabilityFromTileId(tile1) === false) return false;
    if (tile2 > 0 && $gameMap.getPassabilityFromTileId(tile2) === false) return false;
    if (tile3 > 0 && $gameMap.getPassabilityFromTileId(tile3) === false) return false;

    if (objectType == 'animal') {
      return $.isTilePriorityEmpty(mapX, mapY, allowPlayer, allowProtected);
    }

    return $.isTileEmpty(mapX, mapY, allowPlayer, allowProtected, allowItems, allowSeeds);
  };

  $.isTilePriorityEmpty = function(x, y, allowPlayer, allowProtected) {
    if (allowPlayer === undefined) allowPlayer = false;
    if (allowProtected === undefined) allowProtected = false;

    if (!$gameMap.isTilePassable(x, y)) return false;

    if (!allowProtected) {
      if (Managers.FarmObjects.isTileProtected($gameMap._mapId, x, y)) return false;
    }

    const objects = $gameMap.anythingXy(x, y);
    for (var i = 0; i < objects.length; i++) {
      if (objects[i]._priorityType == 1) return false;
    }

    if (!allowPlayer) {
      if ($gamePlayer.isTouchingTile(x, y)) return false;
    }

    return true;
  };

  $.isTileEmpty = function(x, y, allowPlayer, allowProtected, allowItems, allowSeeds) {
    if (allowPlayer === undefined) allowPlayer = false;
    if (allowProtected === undefined) allowProtected = false;
    if (allowItems === undefined) allowItems = false;

    if (allowSeeds === undefined) allowSeeds = true;

    if (!$gameMap.isTilePassable(x, y)) return false;

    if (!allowProtected) {
      if (Managers.FarmObjects.isTileProtected($gameMap._mapId, x, y)) return false;
    }

    if ($gameMap.anyCollisionXy(x, y, allowItems, allowSeeds)) return false;

    if (!allowPlayer) {
      if ($gamePlayer.isTouchingTile(x, y)) return false;
    }

    return true;
  };

  $.findEmptyTileNear = function(x, y, preferredDirection, maxDistance) {
    //#ToDo: Check the ceiling too
    x = Math.floor(x);
    y = Math.floor(y);

    if ($.checkIfTileCanGetObject(x, y)) {
      return $.newTile(x, y);
    }

    if (preferredDirection === undefined) preferredDirection = Direction.LEFT;
    if (maxDistance === undefined) maxDistance = 1;

    var currentDistance = 0;
    while (currentDistance < maxDistance) {
      currentDistance++;

      var positions = [];

      switch(preferredDirection) {
        case Direction.DOWN :
          positions.push({ x : x, y : y + currentDistance});
          positions.push({ x : x - currentDistance, y : y});
          positions.push({ x : x + currentDistance, y : y});
          positions.push({ x : x, y : y - currentDistance});
          break;
        case Direction.LEFT :
          positions.push({ x : x - currentDistance, y : y});
          positions.push({ x : x, y : y + currentDistance});
          positions.push({ x : x, y : y - currentDistance});
          positions.push({ x : x + currentDistance, y : y});
          break;
        case Direction.RIGHT :
          positions.push({ x : x + currentDistance, y : y});
          positions.push({ x : x, y : y + currentDistance});
          positions.push({ x : x, y : y - currentDistance});
          positions.push({ x : x - currentDistance, y : y});
          break;
        case Direction.UP:
          positions.push({ x : x, y : y - currentDistance});
          positions.push({ x : x - currentDistance, y : y});
          positions.push({ x : x + currentDistance, y : y});
          positions.push({ x : x, y : y + currentDistance});
          break;
      }

      var len = positions.length;
      for (var i = 0; i < len; i++) {
        if ($.checkIfTileCanGetObject(positions[i].x, positions[i].y)) {
          return $.newTile(positions[i].x, positions[i].y);
        }
      }
    }

    return false;
  };
})(TileHelper);