var reportTimes = false;

class OrangeProfiler {
  static overrideMethod(classType, methodName, identifier) {
    const oldMethod = classType.prototype[methodName];

    classType.prototype[methodName] = function() {
      const data = OrangeProfiler.times[identifier] || {};

      const start = performance.now();
      const result = oldMethod.apply(this, arguments);
      const end = performance.now();

      const elapsedTime = end - start;
      if (!data.maxTime) {
        data.maxTime = elapsedTime;
        data.lastTime = elapsedTime;
        data.moment = end;
      } else {
        if (elapsedTime > 1 && elapsedTime > (data.maxTime * 1.2)) {
          reportTimes = true;
        }

        data.maxTime = Math.max(data.maxTime, elapsedTime);
        data.lastTime = elapsedTime;
        data.moment = end;
      }
      OrangeProfiler.times[identifier] = data;

      return result;
    };
  }

  static overrideClass(classType, identifier) {
    const descriptors = Object.getOwnPropertyDescriptors(classType.prototype);

    for (let methodName of Object.getOwnPropertyNames(classType.prototype)) {
      if (methodName == 'constructor') continue;
      const desc = descriptors[methodName];
      if (!desc) continue;
      if (desc.get || desc.set) continue;

      const method = classType.prototype[methodName];
      if (typeof method == 'function') {
        OrangeProfiler.overrideMethod(classType, methodName, `${ identifier }-${ methodName }`);
      }
    }
  }

  static overrideNamespace(namespace, identifier) {
    for (let className in namespace) {
      const classType = namespace[className];
      if (!classType) continue;

      this.overrideClass(classType, `${ identifier }-${ className }`);
    }
  }

  static overrideAll() {
    this.overrideNamespace(Objects, 'objects');
    this.overrideNamespace(Managers, 'manages');
    this.overrideNamespace(Core, 'core');
    this.overrideNamespace(Creatures, 'creatures');
    this.overrideNamespace(Models, 'models');
    this.overrideNamespace(Tools, 'tools');
  }

  static enableProfiler() {
    this.times = {};

    this.overrideAll();

    const oldSceneManagerUpdate = Managers.Scenes.update;
    Managers.Scenes.update = function() {
      // OrangeProfiler.times = {};
      reportTimes = false;
      const start = performance.now();

      oldSceneManagerUpdate.apply(this, arguments);

      const end = performance.now();
      const frameTime = end - start;

      if (reportTimes) {
        OrangeProfiler.logLastTimesSince(start, frameTime);
      }
    };
  }

  static logLastTimesSince(startTime, totalTime) {
    if (totalTime < 14) return;
    // If it took really long, it's likely useless
    if (totalTime > 1000) return;

    const logData = [];

    let maxTime = 0;
    let idx = -1;

    for (let identifier in this.times) {
      idx++;
      const data = this.times[identifier];
      if (data.moment < startTime) continue;
      const lastTime = data.lastTime;
      if (lastTime < 0.15) continue;
      if (lastTime < data.maxTime * 0.8) continue;

      maxTime = Math.max(maxTime, lastTime);

      logData.push({
        identifier,
        lastTime,
        idx
      });
    }

    if (maxTime < 1) return;
    if (logData.length < 5) return;

    logData.sort((item1, item2) => item2.lastTime - item1.lastTime);
    if (logData.length > 50) {
      logData.splice(0, 50);
    }

    console.log(...log(totalTime, logData));
  }
}

window.OrangeProfiler = OrangeProfiler;