//-----------------------------------------------------------------------------
/**
 * The weather effect which displays rain, storm, or snow.
 *
 * @class Weather
 * @constructor
 */
class Weather extends PIXI.Container {
  constructor() {
    super(...arguments);
    this.initialize(...arguments);
  }

  initialize() {
    this._width = Graphics.width;
    this._height = Graphics.height;
    this._sprites = [];

    this._createBitmaps();
    this._createDimmer();

    /**
     * The type of the weather in ['none', 'rain', 'storm', 'snow', 'leaves', 'embers', 'shine'].
     *
     * @property type
     * @type String
     */
    this.type = 'none';

    /**
     * The power of the weather in the range (0, 9).
     *
     * @property power
     * @type Number
     */
    this.power = 0;

    /**
     * The origin point of the weather for scrolling.
     *
     * @property origin
     * @type Point
     */
    this.origin = new Point();
  }

  /**
   * Updates the weather for each frame.
   *
   * @method update
   */
  update() {
    this._updateDimmer();
    var times = 1;

    if ($gameSystem.isPlayerSleepingStanding) {
      times = Constants.MAGIC_SLEEP_FACTOR;
    }

    for (var i = 0; i < times; i++) {
      this._updateAllSprites();
    }
  }

  _createRainBitmap() {
    this._rainBitmap = Managers.Images.loadPicture('weather/rain');
  }

  _createStormBitmap() {
    this._stormBitmap = Managers.Images.loadPicture('weather/storm');
  }

  _createSnowBitmap() {
    this._snowBitmap = Managers.Images.loadPicture('weather/snow');
  }

  _createLeafBitmap() {
    this._leafBitmap = Managers.Images.loadPicture('weather/leaf');
  }

  _createHeatBitmap() {
    // this._heatBitmap = Managers.Images.loadPicture('weather/embers');
  }

  _createShineBitmap() {
    // this._shineBitmap = Managers.Images.loadPicture('weather/shine');
  }

  /**
   * @method _createBitmaps
   * @private
   */
  _createBitmaps() {
    this._createRainBitmap();
    this._createStormBitmap();
    this._createSnowBitmap();
    this._createLeafBitmap();
    this._createHeatBitmap();
    this._createShineBitmap();
  }

  /**
   * @method _createDimmer
   * @private
   */
  _createDimmer() {
    this._dimmerSprite = new ScreenSprite();
    this._dimmerSprite.setColor(80, 80, 80);
    this.addChild(this._dimmerSprite);
  }

  /**
   * @method _updateDimmer
   * @private
   */
  _updateDimmer() {
    this._dimmerSprite.opacity = Math.floor(this.power * 6);
  }

  _getMaxSpriteCount() {
    switch (this.type) {
      case 'rain':
        return 12 + (30 * this.power);
      case 'storm':
        return 75 + (35 * this.power);
      case 'snow':
        return 12 + (32 * this.power);
      case 'leaves':
        return 10 + (10 * this.power);
      case 'embers':
        return 20 + (16 * this.power);
      case 'shine':
        return 20 + (36 * this.power);
      default:
        return Math.floor(this.power * 10);
    }
  }

  /**
   * @method _updateAllSprites
   * @private
   */
  _updateAllSprites() {
    const maxSprites = this._getMaxSpriteCount();

    while (this._sprites.length < maxSprites) {
      this._addSprite();
    }

    while (this._sprites.length > maxSprites) {
      this._removeSprite();
    }

    this._sprites.forEach(function(sprite) {
      this._updateSprite(sprite);
      sprite.x = sprite.ax - this.origin.x;
      sprite.y = sprite.ay - this.origin.y;
    }, this);
  }

  /**
   * @method _addSprite
   * @private
   */
  _addSprite() {
    const Sprite_Weather = require('engine/sprites/Weather');

    const sprite = new Sprite_Weather(this.viewport);
    switch(this.type) {
      case 'rain':
        this._setupRain(sprite);
        break;
      case 'storm':
        this._setupStorm(sprite);
        break;
      case 'snow':
        this._setupSnow(sprite);
        break;
      case 'leaves':
        this._setupLeaves(sprite);
        break;
      case 'embers':
        this._setupEmbers(sprite);
        break;
      case 'shine':
        this._setupShine(sprite);
        break;
    }

    sprite.opacity = 0;
    this._sprites.push(sprite);
    this.addChild(sprite);
  }

  _setupRain(sprite) {
    const w = Constants.Weather;
    const life = w.RaindropLifetime;
    const size = 0.01 * ((100 - w.RainSizeVariation) + Math.randomInt(1 + w.RainSizeVariation * 2));
    const getangle = ((w.RainAngleC - w.RainAngleV) + Math.randomInt(1 + w.RainAngleV * 2));
    const alpha = ((w.RainAlphaC - w.RainAlphaV) + Math.randomInt(1 + w.RainAlphaV * 2));
    sprite.setup('rain', life, size, getangle, alpha);
    sprite.blendMode = PIXI.BLEND_MODES.NORMAL;
  }

  _setupStorm(sprite) {
    const w = Constants.Weather;
    const life = w.StormLifetime;
    const size = 0.01 * ((100 - w.StormSizeV) + Math.randomInt(1 + w.StormSizeV * 2));
    const getangle = ((w.StormAngleC - w.StormAngleV) + Math.randomInt(1 + w.StormAngleV * 2));
    const alpha = ((w.StormAlphaC - w.StormAlphaV) + Math.randomInt(1 + w.StormAlphaV * 2));

    sprite.setup('storm', life, size, getangle, alpha);
    sprite.blendMode = PIXI.BLEND_MODES.NORMAL;
  }

  _setupSnow(sprite) {
    const w = Constants.Weather;
    const life = w.SnowLifetime;
    const size = 0.01 * ((100 - w.SnowSizeV) + Math.randomInt(1 + w.SnowSizeV * 2));
    const getangle = (-1 + Math.randomInt(3)) * 0.267;
    const alpha = ((w.SnowAlphaC - w.SnowAlphaV) + Math.randomInt(1 + w.SnowAlphaV * 2));
    const snowspeed = 1.33 + Math.randomInt(2);
    sprite.setup('snow', life, size, getangle, alpha, snowspeed);
    sprite.blendMode = PIXI.BLEND_MODES.NORMAL;
  }

  _setupLeaves(sprite) {
    const w = Constants.Weather;
    const life = w.LeafLifetime;
    const size = 0.007 * ((100 - w.LeafSizeV) + Math.randomInt(1 + w.LeafSizeV * 2));
    const getangle = w.LeafSpeedV;
    const alpha = 255;
    const leafspeed = 0.83 + Math.randomInt(2);

    sprite.setup('leaves', life, size, getangle, alpha, leafspeed);
    sprite.blendMode = PIXI.BLEND_MODES.NORMAL;
  }

  _setupEmbers(sprite) {
    const w = Constants.Weather;
    const life = w.HeatLifetime;
    const size = 0.007 * ((100 - w.HeatSizeV) + Math.randomInt(1 + w.HeatSizeV * 2));
    const getangle = w.HeatSpeedV;
    const alpha = 255;
    const emberspeed = 0.83 + Math.randomInt(2);
    sprite.setup('embers', life, size, getangle, alpha, emberspeed);
    sprite.blendMode = PIXI.BLEND_MODES.NORMAL;
  }

  _setupShine(sprite) {
    const w = Constants.Weather;
    const life = w.ShineLifetime;
    const size = 0.007 * ((100 - w.ShineSizeV) + Math.randomInt(1 + w.ShineSizeV * 2));
    const getangle = 12;
    const alpha = ((w.ShineAlphaC - w.ShineAlphaV) + Math.randomInt(1 + w.ShineAlphaV * 2));
    const shinespeed = 0.25 * (2 + Math.randomInt(3));

    sprite.setup('shine', life, size, getangle, alpha, shinespeed);
    sprite.rotation = Math.random();
    sprite.opacity = 0;
  }

  /**
   * @method _removeSprite
   * @private
   */
  _removeSprite() {
    this.removeChild(this._sprites.pop());
  }

  /**
   * @method _updateSprite
   * @param {Sprite} sprite
   * @private
   */
  _updateSprite(sprite) {
    sprite.update();

    switch (this.type) {
      case 'rain':
        this._updateRainSprite(sprite);
        break;
      case 'storm':
        this._updateStormSprite(sprite);
        break;
      case 'snow':
        this._updateSnowSprite(sprite);
        break;
      case 'leaves':
        this._updateLeafSprite(sprite);
        break;
      case 'embers':
        this._updateHeatSprite(sprite);
        break;
      case 'shine':
        this._updateShineSprite(sprite);
        break;
    }

    if (sprite._lifetime <= 0 && sprite.opacity <= 6) {
      this._rebornSprite(sprite);
    }
  }

  /**
   * @method _updateRainSprite
   * @param {Sprite} sprite
   * @private
   */
  _updateRainSprite(sprite) {
    sprite.bitmap = this._rainBitmap;
    sprite.rotation = sprite._anglev * (Math.PI / 180);

    const velocity = sprite.getVelocity(Constants.Weather.RainSpeed);
    sprite.ax += velocity[0];
    sprite.ay += velocity[1];
  }

  /**
   * @method _updateStormSprite
   * @param {Sprite} sprite
   * @private
   */
  _updateStormSprite(sprite) {
    sprite.bitmap = this._stormBitmap;
    sprite.rotation = sprite._anglev * (Math.PI / 180);

    const velocity = sprite.getVelocity(Constants.Weather.StormSpeed);
    sprite.ax += velocity[0];
    sprite.ay += velocity[1];
  }

  /**
   * @method _updateSnowSprite
   * @param {Sprite} sprite
   * @private
   */
  _updateSnowSprite(sprite) {
    sprite.bitmap = this._snowBitmap;
    sprite.ax += sprite._anglev;
    sprite.ay += sprite._particleSpeed;
  }

  _updateLeafSprite(sprite) {
    sprite.bitmap = this._leafBitmap;
    sprite.rotation += 0.025 * sprite._anglev;
    sprite.ax += 2 * Math.sin(0.0078 * sprite._randomSeed);
    sprite.ay += sprite._particleSpeed;
  }

  _updateHeatSprite(sprite) {
    // sprite.bitmap = this._heatBitmap;
    // sprite.ax += 2 * Math.sin(0.0139 * sprite._randomSeed);
    // sprite.ay -= sprite._particleSpeed;
  }

  _updateShineSprite(sprite) {
    // sprite.bitmap = this._shineBitmap;
  }

  /**
   * @method _rebornSprite
   * @param {Sprite} sprite
   * @private
   */
  _rebornSprite(sprite) {
    sprite.ax = Math.randomInt(Graphics.width + 500) - 200 + this.origin.x;
    sprite.ay = Math.randomInt(Graphics.height + 600) - 400 + this.origin.y;

    switch (this.type) {
      case 'rain':
        this._setupRain(sprite);
        break;
      case 'storm':
        this._setupStorm(sprite);
        break;
      case 'snow':
        this._setupSnow(sprite);
        break;
      case 'leaves':
        this._setupLeaves(sprite);
        break;
      case 'embers':
        this._setupEmbers(sprite);
        break;
      case 'shine':
        this._setupShine(sprite);
        break;
    }    
  }
}

window.Weather = Weather;
module.exports = Weather;