const Window_Base = require('engine/windows/Window_Base');
const { Input } = require('engine/core/Input');

//-----------------------------------------------------------------------------
// Window_Selectable
//
// The window class with cursor movement and scroll functions.

class Window_Selectable extends Window_Base {
  initialize(x, y, width, height) {
    super.initialize(x, y, width, height);
    this._index = -1;
    this._cursorFixed = false;
    this._cursorAll = false;
    this._stayCount = 0;
    this._helpWindow = null;
    this._infoWindow = null;
    this._handlers = {};
    this._touching = false;
    this._scrollX = 0;
    this._scrollY = 0;
    this._checkInput = true;
    this._allowTouchWhenInactive = false;
    this._selectCallback = false;
    this.deactivate();
  }

  index() {
    return this._index;
  }

  cursorFixed() {
    return this._cursorFixed;
  }

  setCursorFixed(cursorFixed) {
    this._cursorFixed = cursorFixed;
  }

  cursorAll() {
    return this._cursorAll;
  }

  setCursorAll(cursorAll) {
    this._cursorAll = cursorAll;
  }

  maxCols() {
    return 1;
  }

  maxItems() {
    return 0;
  }

  spacing() {
    return 12;
  }

  horizontalMargin() {
    return 0;
  }

  verticalMargin() {
    return 0;
  }

  topMargin() {
    return this.verticalMargin();
  }

  bottomMargin() {
    return this.verticalMargin();
  }

  itemWidth() {
    const horizontalMargin = this.horizontalMargin();
    const totalWidth = this.width - (this.paddingH * 2 ) - (horizontalMargin * 2) + this.spacing();
    return Math.floor(totalWidth / this.maxCols() - this.spacing());
  }

  itemHeight() {
    return this.lineHeight();
  }

  maxRows() {
    return Math.max(Math.ceil(this.maxItems() / this.maxCols()), 1);
  }

  activate() {
    super.activate();
    this.reselect();
  }

  deactivate() {
    super.deactivate();
    this.reselect();
  }

  select(index) {
    const oldIndex = this._index;
    this._index = index;
    this._stayCount = 0;
    this.ensureCursorVisible();

    if (index != oldIndex) {
      this.updateCursor();
      this.callUpdateHelp();
    }

    if (this._selectCallback) {
      this._selectCallback(index);
    }
  }

  deselect() {
    this.select(-1);
  }

  reselect() {
    this.select(this._index);
  }

  row() {
    return Math.floor(this.index() / this.maxCols());
  }

  col() {
    const index = this.index();

    if (index >= 0) {
      return index % this.maxCols();
    }

    return -1;
  }

  selectAt(row, col) {
    if (col >= this.maxCols()) {
      this.select(0);
      return;
    }

    const index = row * this.maxCols() + col;

    if (index < this.maxItems()) {
      this.select(index);
    } else {
      this.select(0);
    }
  }

  itemFullHeight() {
    return this.itemHeight() + this.verticalItemSpacing();
  }

  topRow() {
    return Math.floor(this._scrollY / this.itemFullHeight());
  }

  maxTopRow() {
    return Math.max(0, this.maxRows() - this.maxPageRows());
  }

  setTopRow(row) {
    const scrollY = row.clamp(0, this.maxTopRow()) * this.itemFullHeight();
    if (this._scrollY !== scrollY) {
      this._scrollY = scrollY;
      this.refresh();
      this.updateCursor();
    }
  }

  resetScroll() {
    this.setTopRow(0);
  }

  maxPageRows() {
    const pageHeight = this.height - this.paddingV * 2 - (this.topMargin() + this.bottomMargin());
    return Math.floor(pageHeight / this.itemFullHeight());
  }

  maxPageItems() {
    return this.maxPageRows() * this.maxCols();
  }

  isHorizontal() {
    return this.maxPageRows() === 1;
  }

  bottomRow() {
    return Math.max(0, this.topRow() + this.maxPageRows() - 1);
  }

  setBottomRow(row) {
    this.setTopRow(row - (this.maxPageRows() - 1));
  }

  topIndex() {
    return this.topRow() * this.maxCols();
  }

  verticalItemSpacing() {
    return 0;
  }

  itemRect(index) {
    const rect = new Rectangle();
    const maxCols = this.maxCols();
    const vSpacing = this.verticalItemSpacing();

    rect.width = this.itemWidth();
    rect.height = this.itemHeight();
    rect.x = index % maxCols * (rect.width + this.spacing()) - this._scrollX + this.horizontalMargin();
    rect.y = Math.floor(index / maxCols) * (rect.height + vSpacing) - this._scrollY + this.topMargin();
    return rect;
  }

  itemRectForText(index) {
    const rect = this.itemRect(index);
    rect.x += this.textPadding();
    rect.width -= this.textPadding() * 2;
    return rect;
  }

  setHelpWindow(helpWindow) {
    this._helpWindow = helpWindow;
    this.callUpdateHelp();
  }

  setInfoWindow(infoWindow) {
    this._infoWindow = infoWindow;
    this.callUpdateHelp();
  }

  showHelpWindow() {
    if (this._helpWindow) {
      this._helpWindow.show();
    }
    if (this._infoWindow) {
      this._infoWindow.show();
    }
  }

  hideHelpWindow() {
    if (this._helpWindow) {
      this._helpWindow.hide();
    }
    if (this._infoWindow) {
      this._infoWindow.hide();
    }
  }

  setHandler(symbol, method) {
    this._handlers[symbol] = method;
  }

  isHandled(symbol) {
    return !!this._handlers[symbol];
  }

  callHandler(symbol) {
    if (this.isHandled(symbol)) {
      this._handlers[symbol]();
    }
  }

  isOpenAndActive() {
    return this.isOpen() && this.active;
  }

  isOpenAndVisible() {
    return this.isOpen() && this.visible;
  }

  isCursorMovable() {
    return (this.isOpenAndVisible() && !this._cursorFixed && !this._cursorAll && this.maxItems() > 0);
  }

  cursorDown(wrap) {
    const index = this.index();
    const maxItems = this.maxItems();
    const maxCols = this.maxCols();
    if (index < maxItems - maxCols || (wrap && maxCols === 1)) {
      this.select((index + maxCols) % maxItems);
    }
  }

  cursorUp(wrap) {
    let index = this.index();
    const maxItems = this.maxItems();
    const maxCols = this.maxCols();
    if (index >= maxCols || (wrap && maxCols === 1)) {
      if (index < 0) {
        index = 0;
      }

      this.select((index - maxCols + maxItems) % maxItems);
    }
  }

  cursorRight(wrap) {
    const index = this.index();
    const maxItems = this.maxItems();
    const maxCols = this.maxCols();
    if (maxCols >= 2 && (index < maxItems - 1 || (wrap && this.isHorizontal()))) {
      this.select((index + 1) % maxItems);
    }
  }

  cursorLeft(wrap) {
    const index = this.index();
    const maxItems = this.maxItems();
    const maxCols = this.maxCols();
    if (maxCols >= 2 && (index > 0 || (wrap && this.isHorizontal()))) {
      this.select((index - 1 + maxItems) % maxItems);
    }
  }

  cursorPagedown() {
    const index = this.index();
    const maxItems = this.maxItems();
    if (this.topRow() + this.maxPageRows() < this.maxRows()) {
      this.setTopRow(this.topRow() + this.maxPageRows());
      this.select(Math.min(index + this.maxPageItems(), maxItems - 1));
    } else {
      this.setTopRow(0);
      this.select(0);
    }
  }

  cursorPageup() {
    const index = this.index();
    if (this.topRow() > 0) {
      this.setTopRow(this.topRow() - this.maxPageRows());
      this.select(Math.max(index - this.maxPageItems(), 0));
    } else {
      this.setTopRow(this.maxTopRow());
      this.select(this.maxItems() - 1);
    }
  }

  scrollDown() {
    if (this.topRow() + 1 < this.maxRows()) {
      this.setTopRow(this.topRow() + 1);
    }
  }

  scrollUp() {
    if (this.topRow() > 0) {
      this.setTopRow(this.topRow() - 1);
    }
  }

  update() {
    super.update();
    this.updateArrows();
    this.processCursorMove();
    this.processHandling();
    this.processWheel();
    this.processTouch();
    this._stayCount++;
  }

  updateArrows() {
    const topRow = this.topRow();
    const maxTopRow = this.maxTopRow();
    this.downArrowVisible = maxTopRow > 0 && topRow < maxTopRow;
    this.upArrowVisible = topRow > 0;
  }

  processCursorMove() {
    if (!this._checkInput) return;

    if (this.isCursorMovable() && this.active) {
      const lastIndex = this.index();
      if (Input.isRepeated('down')) {
        this.cursorDown(Input.isTriggered('down'));
      }
      if (Input.isRepeated('up')) {
        this.cursorUp(Input.isTriggered('up'));
      }
      if (Input.isRepeated('right')) {
        this.cursorRight(Input.isTriggered('right'));
      }
      if (Input.isRepeated('left')) {
        this.cursorLeft(Input.isTriggered('left'));
      }
      if (!this.isHandled('pagedown')) {
        if (Input.isTriggered('pagedown') || Input.isTriggered('right_trigger')) {
          this.cursorPagedown();
        }
      }
      if (!this.isHandled('pageup')) {
        if (Input.isTriggered('pageup') || Input.isTriggered('left_trigger')) {
          this.cursorPageup();
        }
      }
      if (this.index() !== lastIndex) {
        Managers.Sound.playCursor();
      }
    }
  }

  processHandling() {
    if (!this._checkInput) return;
    if ($gameTemp.shouldSkipSelectableWindowHandling()) return;

    if (this.isOpenAndActive()) {
      if (this.isOkEnabled() && this.isOkTriggered()) {
        this.processOk();
      } else if (this.isCancelEnabled() && this.isCancelTriggered()) {
        this.processCancel();
      } else if (this.isHandled('pagedown') && (Input.isTriggered('pagedown') || Input.isTriggered('right_trigger') )) {
        this.processPagedown();
      } else if (this.isHandled('pageup') && (Input.isTriggered('pageup') || Input.isTriggered('left_trigger'))) {
        this.processPageup();
      }
    }
  }

  processWheel() {
    if (!this._checkInput) return;

    if (this.isOpenAndVisible()) {
      const threshold = 20;
      if (TouchInput.wheelY >= threshold) {
        this.scrollDown();
      }
      if (TouchInput.wheelY <= -threshold) {
        this.scrollUp();
      }
    }
  }

  processTouch() {
    if (this.isOpenAndVisible()) {
      if (TouchInput.isTriggered() && this.isTouchedInsideFrame()) {
        this._touching = true;
        this.onTouch(true);
      } else if (TouchInput.isCancelled()) {
        if (this.isCancelEnabled()) {
          this.processCancel();
        }
      } else if (this.active || this._allowTouchWhenInactive) {
        this.onTouch(false);
      }

      // if (this._touching) {
      //   if (TouchInput.isPressed()) {
      //     this.onTouch(false);
      //   } else {
      //     this._touching = false;
      //   }
      // }
    } else {
      this._touching = false;
    }
  }

  isTouchedInsideFrame() {
    const x = this.canvasToLocalX(TouchInput.x);
    const y = this.canvasToLocalY(TouchInput.y);
    return x >= 0 && y >= 0 && x < this.width && y < this.height;
  }

  isTouchedInsideItem() {
    const x = this.canvasToLocalX(TouchInput.x);
    const y = this.canvasToLocalY(TouchInput.y);

    if (x >= 0 && y >= 0 && x < this.width && y < this.height) {
      return this.hitTest(x, y) >= 0;
    }

    return false;
  }

  isPositionInsideFrame(x, y) {
    return x >= 0 && y >= 0 && x < this.width && y < this.height;
  }

  isMouseInsideFrame() {
    const x = this.canvasToLocalX(TouchInput.mouseX);
    const y = this.canvasToLocalY(TouchInput.mouseY);

    return this.isPositionInsideFrame(x, y);
  }

  onMove(x, y) {
    const lastIndex = this.index();

    const hitIndex = this.hitTest(x, y);
    if (hitIndex >= 0) {
      if (this.isCursorMovable()) {
        this.select(hitIndex);
      }
    } else if (this._stayCount >= 10) {
      if (y < this.paddingV) {
        this.cursorUp();
      } else if (y >= this.height - this.paddingV) {
        this.cursorDown();
      }
    }
    if (this.index() !== lastIndex) {
      Managers.Sound.playCursor();
    }
  }

  getPositionLabel(x, y) {
    x = this.canvasToLocalX(x);
    y = this.canvasToLocalY(y);

    if (!this.isPositionInsideFrame(x, y)) return '';

    const hitIndex = this.hitTest(x, y);
    if (hitIndex >= 0) {
      return this.getIndexLabel(hitIndex);
    } else {
      return '';
    }
  }

  getIndexLabel(index) {
    return '';
  }

  onTouchOutside() {
  }

  selectByTouch(index) {
    this.select(index);
  }

  allowScrollOnHover() {
    return true;
  }

  onTouch(triggered) {
    if (!Managers.Config.enableMouse) return;
    if (this._allowTouchWhenInactive === false && !this.active) return;

    const lastIndex = this.index();
    const x = this.canvasToLocalX(TouchInput.mouseX);
    const y = this.canvasToLocalY(TouchInput.mouseY);

    if (x == this._oldX && y == this._oldY && !triggered) {
      return;
    }

    if (!this.isPositionInsideFrame(x, y)) {
      this.onTouchOutside();
      return;
    }

    this._oldX = x;
    this._oldY = y;

    const hitIndex = this.hitTest(x, y);
    if (hitIndex >= 0) {
      if (hitIndex === this.index()) {
        if (triggered && this.isTouchOkEnabled()) {
          this.processOk();
        }
      } else if (this.isCursorMovable()) {
        if (this.active || this._allowTouchWhenInactive) {
          this.selectByTouch(hitIndex);
        }
      }
    } else if (this._stayCount >= 10 && (this.active || this._allowTouchWhenInactive)) {
      if (this.allowScrollOnHover()) {
        if (y < this.paddingV) {
          this.cursorUp();
        } else if (y >= this.height - this.paddingV) {
          this.cursorDown();
        }
      }
    }

    if (this.index() !== lastIndex) {
      Managers.Sound.playCursor();
    }
  }

  hitTest(x, y) {
    if (this.isContentsArea(x, y)) {
      let cx = x - this.paddingH;
      let cy = y - this.paddingV;

      cy -= this.extraSpacing();
      cx -= this.extraSpacing();

      const topIndex = this.topIndex();
      for (let i = 0; i < this.maxPageItems(); i++) {
        const index = topIndex + i;
        if (index < this.maxItems()) {
          const rect = this.itemRect(index);
          const right = rect.x + rect.width;
          const bottom = rect.y + rect.height;
          if (cx >= rect.x && cy >= rect.y && cx < right && cy < bottom) {
            return index;
          }
        }
      }
    }
    return -1;
  }

  isContentsArea(x, y) {
    const left = this.paddingH;
    const top = this.paddingV;
    const right = this.width - this.paddingH;
    const bottom = this.height - this.paddingV;
    return (x >= left && y >= top && x < right && y < bottom);
  }

  isTouchOkEnabled() {
    return this.isOkEnabled();
  }

  isOkEnabled() {
    return this.isHandled('ok');
  }

  isCancelEnabled() {
    return this.isHandled('cancel');
  }

  isOkTriggered() {
    return Input.isRepeated('ok') || Input.isRepeated('use');
  }

  processOk() {
    if (this.isCurrentItemEnabled()) {
      this.playOkSound();
      this.updateInputData();
      this.deactivate();
      this.callOkHandler();
    } else {
      this.playBuzzerSound();
    }
  }

  playOkSound() {
    Managers.Sound.playOk();
  }

  playBuzzerSound() {
    Managers.Sound.playBuzzer();
  }

  callOkHandler() {
    this.callHandler('ok');
  }

  processCancel() {
    Managers.Sound.playCancel();
    this.updateInputData();
    this.deactivate();
    this.callCancelHandler();
  }

  callCancelHandler() {
    this.callHandler('cancel');
  }

  processPageup() {
    Managers.Sound.playCursor();
    this.updateInputData();
    this.deactivate();
    this.callHandler('pageup');
  }

  processPagedown() {
    Managers.Sound.playCursor();
    this.updateInputData();
    this.deactivate();
    this.callHandler('pagedown');
  }

  updateInputData() {
    // Kadokawa put this workaround here to prevent multiple windows from triggering their ok or cancel events at the same time
    // this causes some properties, such as triggered and released, to be unreliable between here and the rest of the frame
    // Input.update();
    // TouchInput.update();

    // I've removed kadokawa's workaround and added a temp flag to make all selectable windows stop checking input until the end of the frame
    $gameTemp.skipSelectableWindowHandling(true);
  }

  configureCursorAll() {
    const allRowsHeight = this.maxRows() * this.itemFullHeight();
    this.setCursorRect(0, 0, this.contents.width, allRowsHeight);
    this.setTopRow(0);
  }

  configureCursorRect(rect) {
    this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
  }

  configureEmptyCursor() {
    this.setCursorRect(0, 0, 0, 0);
  }

  updateCursor() {
    if (this._cursorAll) {
      this.configureCursorAll();
    } else if (this.isCursorVisible()) {
      const rect = this.itemRect(this.index());
      this.configureCursorRect(rect);
    } else {
      this.configureEmptyCursor();
    }
  }

  isCursorVisible() {
    const row = this.row();
    return row >= this.topRow() && row <= this.bottomRow();
  }

  ensureCursorVisible() {
    const row = this.row();
    if (row < this.topRow()) {
      this.setTopRow(row);
    } else if (row > this.bottomRow()) {
      this.setBottomRow(row);
    }
  }

  callUpdateHelp() {
    if (this.active && (this._helpWindow || this._infoWindow)) {
      this.updateHelp();
    }
  }

  updateHelp() {
    if (this._helpWindow && this._helpWindow.clear) {
      this._helpWindow.clear();
    }
    if (this._infoWindow && this._infoWindow.clear) {
      this._infoWindow.clear();
    }

    this.callHandler('updateHelp');
  }

  setHelpWindowItem(item) {
    if (this._helpWindow) {
      this._helpWindow.setItem(item);
    }
    if (this._infoWindow) {
      this._infoWindow.setItem(item);
    }
  }

  isCurrentItemEnabled() {
    return true;
  }

  drawAllItems() {
    const topIndex = this.topIndex();
    for (let i = 0; i < this.maxPageItems(); i++) {
      const index = topIndex + i;
      if (index < this.maxItems()) {
        this.drawItem(index);
      }
    }
  }

  drawItem(index) {
  }

  clearItem(index) {
    const rect = this.itemRect(index);
    this.contents.clearRect(rect.x, rect.y, rect.width, rect.height);
  }

  redrawItem(index) {
    if (index >= 0) {
      this.clearItem(index);
      this.drawItem(index);
    }
  }

  redrawCurrentItem() {
    this.redrawItem(this.index());
  }

  refresh() {
    if (this.contents) {
      this.contents.clear();
      this.updateCursor();
      this.drawAllItems();
    }
  }

  extraSpacing() {
    return 0;
  }

  fittingHeight(numLines) {
    return numLines * this.lineHeight() + this.verticalPadding() * 2 + this.topMargin() + this.bottomMargin();
  }
}

module.exports = Window_Selectable;
