const Window_Base = require('engine/windows/Window_Base');
const Window_ChoiceList = require('./Window_ChoiceList');
const Window_NumberInput = require('./Window_NumberInput');
const Window_Portrait = require('./Portrait');

const { Input } = require('engine/core/Input');

//-----------------------------------------------------------------------------
// Window_Message
//
// The window for displaying text messages.

class Window_Message extends Window_Base {
  initialize() {
    const width = this.windowWidth();
    const height = this.windowHeight();
    const x = (Graphics.width - width) / 2;
    super.initialize(x, 0, width, height);
    this.openness = 0;
    this.initMembers();

    this.backOpacity = 255;
  }

  initMembers() {
    this._imageReservationId = Utils.generateRuntimeId();
    this._background = 0;
    this._positionType = 2;
    this._waitCount = 0;
    this._textState = null;
    this.clearFlags();
  }

  subWindows() {
    return [this._choiceWindow, this._numberWindow];
  }

  _createBackSpriteBitmap() {
    this._windowBackSprite = new Sprite();
    this._windowBackSprite.bitmap = Managers.Images.loadPicture('window_1');
    this._windowBackSprite.scale.x = Graphics.windowZoomLevel;
    this._windowBackSprite.scale.y = Graphics.windowZoomLevel;
    this._windowSpriteContainer.addChild(this._windowBackSprite);
  }

  _createWindowFrameSprite() {
  }

  _refreshBack() {
  }

  resetFontSettings() {
    super.resetFontSettings();
    this.contents.fontSize = Utils.convertFontSize(Constants.MESSAGE_WINDOW_FONT_SIZE, Graphics.windowZoomLevel);
  }

  standardOutlineWidth() {
    return 0;
  }

  createSubWindows() {
    if (!this._choiceWindow) {
      this._choiceWindow = new Window_ChoiceList(this);
    }
    if (!this._numberWindow) {
      this._numberWindow = new Window_NumberInput(this);
    }
    if (!this._portraitWindow) {
      this._portraitWindow = new Window_Portrait(this);
    }
  }

  updateSubWindows() {
    this.subWindows().forEach((windowObj) => {
      this.parent.addChild(windowObj);
    }, this);

    this.parent.addChild(this._portraitWindow);
  }

  numVisibleRows() {
    return $gameSystem.messageRows();
  }

  windowWidth() {
    return 351 * Graphics.windowZoomLevel;
  }

  windowHeight() {
    return 85 * Graphics.windowZoomLevel;
  }

  adjustWindowSettings() {
    this.width = this.windowWidth();
    this.height = Math.min(this.windowHeight(), Graphics.height);

    if (Math.abs(Graphics.height - this.height) < this.lineHeight()) {
      this.height = Graphics.height;
    }
    this.createContents();
    // this.x = (Graphics.width - this.width) / 2;
  }

  clearFlags() {
    this._showFast = false;
    this._lineShowFast = false;
    this._pauseSkip = false;
  }

  update() {
    this.checkToNotClose();
    super.update();

    if (Managers.CommonEvent._skipped) {
      this.close();
    }

    while (!this.isOpening() && !this.isClosing()) {
      if (this.updateWait()) {
        return;
      } else if (this.updateLoading()) {
        return;
      } else if (this.updateInput()) {
        return;
      } else if (this.updateMessage()) {
        return;
      } else if (this.canStart()) {
        this.startMessage();
      } else {
        this.startInput();
        return;
      }
    }
  }

  checkToNotClose() {
    if (this.isClosing() && this.isOpen()) {
      if (this.doesContinue()) {
        this.open();
      }
    }
  }

  canStart() {
    return $gameMessage.hasText() && !$gameMessage.scrollMode();
  }

  startMessage() {
    if (this._portraitWindow) {
      this._portraitWindow.deactivate();
    }
    this._textState = {};
    this._textState.index = 0;
    this._textState.text = this.convertEscapeCharacters($gameMessage.allText());
    this.newPage(this._textState);
    this.updatePlacement();
    this.updateBackground();
    this.open();

    if (this._background == 2) {
      this.changeTextColor(this.blueColor());
    }

    this.changeWindowDimensions();
    this.setPopSettings();
  }

  updatePlacement() {
    if (this._lastPortraitName) {
      this._windowBackSprite.bitmap = Managers.Images.loadPicture('window_1');
    } else {
      this._windowBackSprite.bitmap = Managers.Images.loadPicture('window_3');
    }

    this.changeWindowDimensions();
    this._positionType = $gameMessage.positionType();

    this.y = this._positionType * (Graphics.height - this.height) / 2;

    if ($gameTemp.shouldShowCutsceneEffect()) {
      switch (this._positionType) {
        case 0:
          this.y += 50;
          break;
        case 2:
          this.y -= 50;
          break;
      }
    }

    this._portraitWindow.dockToWindowMessage();
    this._portraitWindow.updateVisiblity(undefined, true);
    this._portraitWindow.refresh();
  }

  updateBackground() {
    this._background = $gameMessage.background();
    this.setBackgroundType(this._background);
  }

  terminateMessage() {
    if (this._portraitWindow) {
      this._portraitWindow.deactivate();
    }

    this._lastPortraitName = null;
    this.close();

    $gameMessage.clear();

    if ($gameMap) {
      $gameMap.addMouseDelay(30);
    }
  }

  updateWait() {
    if (this.isFastForward()) return false;

    if (this._waitCount > 0) {
      this._waitCount--;
      return true;
    } else {
      return false;
    }
  }

  updateLoading() {
    return false;
  }

  updateInput() {
    if (this.pause && this.isFastForward()) {
      if (!this._textState) {
        this.pause = false;
        this.terminateMessage();
      }
    }

    if (this.isAnySubWindowActive()) {
      return true;
    }

    if (this.pause) {
      if (this.isTriggered()) {
        Input.update();
        this.pause = false;
        if (!this._textState) {
          this.terminateMessage();
        }
      }

      return true;
    }

    return false;
  }

  isAnySubWindowActive() {
    if (this._choiceWindow && this._choiceWindow.active) return true;
    if (this._numberWindow && this._numberWindow.active) return true;

    return false;
  }

  updateMessage() {
    if (this._textState) {
      while (!this.isEndOfText(this._textState)) {
        if (Managers.CommonEvent._skipped) return true;

        if (this.needsNewPage(this._textState)) {
          this.newPage(this._textState);
        }
        this.updateShowFast();
        this.processCharacter(this._textState);
        if (!this._showFast && !this._lineShowFast) {
          break;
        }
        if (this.pause || this._waitCount > 0) {
          break;
        }
      }
      if (this.isEndOfText(this._textState)) {
        this.onEndOfText();
      }

      this._soundCount = this._soundCount || 0;
      this._soundCount--;

      if (this._soundCount <= 0) {
        this._soundCount = $gameSystem.messageSoundInterval();
        if ($gameSystem.isMessageSoundEnabled()) {
          Managers.Sound.playMessageSound();
        }
      }

      return true;
    } else {
      return false;
    }
  }

  onEndOfText() {
    if (!this.startInput()) {
      if (!this._pauseSkip) {
        this.startPause();
      } else {
        this.terminateMessage();
      }
    }
    this._textState = null;
  }

  startInput() {
    if ($gameMessage.isChoice()) {
      this._choiceWindow.start();
      return true;
    } else if ($gameMessage.isNumberInput()) {
      this._numberWindow.start();
      return true;
    } else {
      return false;
    }
  }

  isTriggered() {
    return (Input.isRepeated('ok') || Input.isRepeated('cancel') || TouchInput.isRepeated());
  }

  canTriggerMoreOptions() {
    return !$gameSystem.isPlayingCutscene() || Switches.talkingOnHouseDoor;
  }

  isMoreOptionsTriggered() {
    return (Input.isPressed('use') || Input.isPressed('map') || Input.isPressed('extra'));
  }

  doesContinue() {
    return ($gameMessage.hasText() && !$gameMessage.scrollMode() && !this.areSettingsChanged());
  }

  areSettingsChanged() {
    return (this._background !== $gameMessage.background() || this._positionType !== $gameMessage.positionType());
  }

  updateShowFast() {
    if (Managers.Config.quickText) {
      this._showFast = true;
      return;
    }

    if (this.isFastForward()) {
      this._showFast = true;
    }

    if (this.isTriggered()) {
      this._showFast = true;
    }
  }

  newPage(textState) {
    this.adjustWindowSettings();
    this._soundCount = 0;
    this.contents.clear();
    this.resetFontSettings();
    this.clearFlags();
    textState.x = this.newLineX();
    textState.y = 18 * Graphics.windowZoomLevel;
    textState.left = this.newLineX();
    textState.height = this.calcTextHeight(textState, false);
  }

  standardFontFace() {
    return 'GameFont';
  }

  standardPadding() {
    return 0;
  }

  newLineX() {
    return 14 * Graphics.windowZoomLevel;
  }

  textMargin() {
    return 14 * Graphics.windowZoomLevel;
  }

  isFastForward() {
    if (!$gameSystem.isFastFowardEnabled()) return false;
    return Input.isPressed('pagedown');
  }

  processNewLine(textState) {
    this._lineShowFast = false;
    super.processNewLine(textState);
    if (this.needsNewPage(textState)) {
      this.startPause();
    }
  }

  processNewPage(textState) {
    super.processNewPage(textState);
    if (textState.text[textState.index] === '\n') {
      textState.index++;
    }
    textState.y = this.contents.height;
    this.startPause();
  }

  isEndOfText(textState) {
    return textState.index >= textState.text.length;
  }

  needsNewPage(textState) {
    return (!this.isEndOfText(textState) && textState.y + textState.height > this.contents.height);
  }

  processEscapeCharacter(code, textState) {
    if (this._testMode && Constants.MESSAGE_WINDOW_TEST_IGNORE_CHARS.contains(code)) {
      code = '.';
    }

    let value;
    switch (code) {
      case 'EMP':
        switch(this.obtainEscapeParam(textState)) {
          case 0:
            $gameSystem._msgSoundEnable = true;
            break;
          case 1:
            $gameSystem._msgSoundEnable = false;
            break;
          case 2:
            $gameSystem.initMessageSounds();
            break;
        }
        break;
      case 'LSN':
        $gameSystem._msgSoundName = this.obtainEscapeString(textState);
        break;
      case 'LSV':
        value = this.obtainEscapeParam(textState);
        $gameSystem._msgSoundVol = value;
        break;
      case 'LSPIV':
        value = this.obtainEscapeParam(textState);
        $gameSystem._msgSoundPitchVar = value;
        break;
      case 'LSPI':
        value = this.obtainEscapeParam(textState);
        $gameSystem._msgSoundPitch = value;
        break;
      case 'LSPAV':
        value = this.obtainEscapeParam(textState);
        $gameSystem._msgSoundPanVar = value;
        break;
      case 'LSPA':
        value = this.obtainEscapeParam(textState);
        $gameSystem._msgSoundPan = value;
        break;
      case 'LSI':
        value = this.obtainEscapeParam(textState);
        $gameSystem._msgSoundInterval = value;
        break;
      case 'LSRESET':
        $gameSystem.initMessageSounds();
        break;
      case 'W':
        this.startWait(this.obtainEscapeParam(textState));
        break;
      case '.':
        this.startWait(15);
        break;
      case '|':
        this.startWait(60);
        break;
      case '!':
        if (!this.isFastForward()) this.startPause();
        break;
      case '>':
        this._lineShowFast = true;
        break;
      case '<':
        this._lineShowFast = false;
        break;
      case '^':
        this._pauseSkip = true;
        break;
      default:
        super.processEscapeCharacter(code, textState);
        break;
    }
  }

  startWait(count) {
    if (this._testMode) {
      return;
    }
    this._waitCount = Utils.getFrameCount(count);
    if (this.isFastForward()) {
      this._waitCount = 0;
    }
    this._soundCount = 0;
  }

  startPause() {
    if (this._testMode) return;

    this.startWait(10);
    this.pause = true;
  }

  endPause() {
    this.pause = false;
    if (!this._textState) {
      this.terminateMessage();
    }
  }

  convertEscapeCharacters(text) {
    text = super.convertEscapeCharacters(text);
    text = this.convertNameBox(text);
    text = this.convertLetterSounds(text);

    text = this.convertPortrait(text);
    return text;
  }

  convertLetterSounds(text) {
    text = text.replace(/\x1bLSON/gi, '\x1bEMP[0]');
    text = text.replace(/\x1bLSOFF/gi, '\x1bEMP[1]');
    text = text.replace(/\x1bLSR/gi, '\x1bEMP[2]');
    return text;
  }

  convertPortrait(text) {
    const windowMessage = this;
    text = text.replace(/\x1bNP<(.*?)>/gi, function() {
      windowMessage.loadPortrait(arguments[1]);
      return '';
    }, this);

    return text;
  }

  convertNameBox(text) {
    const windowMessage = this;
    text = text.replace(/\x1bN<(.*?)>/gi, (line, name) => {
      windowMessage._portraitWindow.name = name;
      windowMessage._portraitWindow.updateVisiblity(undefined, true);
      return '';
    }, this);

    text = text.replace(/\x1bNR<(.*?)>/gi, (line, name) => {
      //#ToDo: move the portrait to the right side
      windowMessage._portraitWindow.name = name;
      windowMessage._portraitWindow.updateVisiblity(undefined, true);
      return '';
    }, this);

    // #ToDo: Dimmed windows
    text = text.replace(/\x1bND<(.*?)>/gi, (line, name) => {
      windowMessage._portraitWindow.name = name;
      windowMessage._portraitWindow.updateVisiblity(undefined, true);
      return '';
    }, this);

    text = text.replace(/\x1bNDR<(.*?)>/gi, (line, name) => {
      windowMessage._portraitWindow.name = name;
      windowMessage._portraitWindow.updateVisiblity(undefined, true);
      return '';
    }, this);

    return text;
  }

  loadPortrait(characterName) {
    if (this._lastPortraitName === characterName) {
      return;
    }

    this._lastPortraitName = characterName;

    if (!characterName) {
      return;
    }
    if (!Managers.Villagers.shouldLoadPortrait(characterName)) {
      return;
    }

    const basePortraitName = characterName == 'Player' ? Managers.Player.generatePlayerSpriteName() : characterName.toLowerCase();
    const fileName = `default/${ basePortraitName }`;

    const bitmap = Managers.Images.loadPortrait(fileName);
    if (!bitmap) {
      this._lastPortraitName = null;
      return;
    }

    this._portraitWindow.portrait = bitmap;
    this._portraitWindow.updateVisiblity(undefined, true);
  }

  setPopSettings() {
    switch ($gameMessage._positionType) {
      case 1: // MIDDLE
        this.yOffset = -(this.height / 2);
        break;
      case 2: // BOTTOM
        this.yOffset = 6;
        break;
    }
  }

  changeWindowDimensions() {
    this.yOffset = 0;
    const totalWidth = this.windowWidth();

    this.width = totalWidth;
    this.height = this.windowHeight();

    let x = (Graphics.width - totalWidth) / 2;
    if (this._portraitWindow.shouldBeVisible()) {
      x += 67 * Graphics.windowZoomLevel / 2;
    }

    this.x = x;
  }

  _refreshPauseSign() {
    super._refreshPauseSign();

    const oX = 0;
    const oY = -12;

    this._windowPauseSignSprite.anchor.y = 0.5;
    this._windowPauseSignSprite.anchor.x = 0.5;

    const x = this._width / 2 + oX;
    const y = this._height + oY;

    this._windowPauseSignSprite.move(x, y);
  }

  _createAllParts() {
    super._createAllParts();

    this._windowPauseSignSprite.scale.x = Constants.MESSAGE_WINDOW_CURSOR_ZOOM;
    this._windowPauseSignSprite.scale.y = Constants.MESSAGE_WINDOW_CURSOR_ZOOM;
    this._downArrowSprite.scale.x = Constants.MESSAGE_WINDOW_CURSOR_ZOOM;
    this._downArrowSprite.scale.y = Constants.MESSAGE_WINDOW_CURSOR_ZOOM;
    this._upArrowSprite.scale.x = Constants.MESSAGE_WINDOW_CURSOR_ZOOM;
    this._upArrowSprite.scale.y = Constants.MESSAGE_WINDOW_CURSOR_ZOOM;
  }

  normalColor() {
    return this.blackColor();
  }
}

module.exports = Window_Message;
