const BasicWindow = require('../core/BasicWindow');
const { Input } = require('engine/core/Input');

//-----------------------------------------------------------------------------
// Window_Base
//
// The superclass of all windows within the game.

class Window_Base extends BasicWindow {
  initialize(x, y, width, height) {
    this._windowSpriteContainer = null;
    this._windowBackSprite = null;
    this._windowCursorSprite = null;
    this._windowFrameSprite = null;
    this._windowContentsSprite = null;
    this._windowPauseSignSprite = null;
    this._cursorRect = new Rectangle();
    this._openness = 255;
    this._colorTone = [0, 0, 0];


    /**
     * The active state for the window.
     *
     * @property active
     * @type Boolean
     */
    this.active = true;

    /**
     * The visibility of the down scroll arrow.
     *
     * @property downArrowVisible
     * @type Boolean
     */
    this.downArrowVisible = false;

    /**
     * The visibility of the up scroll arrow.
     *
     * @property upArrowVisible
     * @type Boolean
     */
    this.upArrowVisible = false;

    /**
     * The visibility of the pause sign.
     *
     * @property pause
     * @type Boolean
     */
    this.pause = false;

    Core.Window.prototype.initialize.call(this);

    this.loadWindowskin();
    this.move(x, y, width, height);
    this.updatePadding();
    this.updateBackOpacity();
    this.updateTone();
    this.createContents();
    this._opening = false;
    this._closing = false;
    this._dimmerSprite = null;
    this._heightCache = [];
    this._widthCache = [];
  }

  /**
   * The opacity of the window without contents (0 to 255).
   *
   * @property opacity
   * @type Number
   */
  get opacity() {
    if (this.enableWindowFrame()) {
      return this._windowSpriteContainer.alpha * 255;
    }

    return 0;
  }

  set opacity(value) {
    if (!this.enableWindowFrame()) return;
    this._windowSpriteContainer.alpha = value.clamp(0, 255) / 255;
  }

  /**
   * The opacity of the window background (0 to 255).
   *
   * @property backOpacity
   * @type Number
   */
  get backOpacity() {
    if (!this.enableWindowFrame()) return 0;

    return this._windowBackSprite.alpha * 255;
  }

  set backOpacity(value) {
    if (!this.enableWindowFrame()) return;
    this._windowBackSprite.alpha = value.clamp(0, 255) / 255;
  }

  /**
   * The opacity of the window contents (0 to 255).
   *
   * @property contentsOpacity
   * @type Number
   */
  get contentsOpacity() {
    return this._windowContentsSprite.alpha * 255;
  }

  set contentsOpacity(value) {
    this._windowContentsSprite.alpha = value.clamp(0, 255) / 255;
  }

  /**
   * The openness of the window (0 to 255).
   *
   * @property openness
   * @type Number
   */
  get openness() {
    return this._openness;
  }

  set openness(value) {
    if (this._openness !== value) {
      this._openness = value.clamp(0, 255);

      if (this.enableWindowFrame()) {
        this._windowSpriteContainer.scale.y = this._openness / 255;
        this._windowSpriteContainer.y = this.height / 2 * (1 - this._openness / 255);
      }
    }
  }

  /**
   * Returns true if the window is completely closed (openness == 0).
   *
   * @method isClosed
   */
  isClosed() {
    return this._openness <= 0;
  }

  /**
   * Sets the position of the command cursor.
   *
   * @method setCursorRect
   * @param {Number} x The x coordinate of the cursor
   * @param {Number} y The y coordinate of the cursor
   * @param {Number} width The width of the cursor
   * @param {Number} height The height of the cursor
   */
  setCursorRect(x, y, width, height) {
    const cx = Math.floor(x || 0);
    const cy = Math.floor(y || 0);
    const cw = Math.floor(width || 0);
    const ch = Math.floor(height || 0);
    const rect = this._cursorRect;
    if (rect.x !== cx || rect.y !== cy || rect.width !== cw || rect.height !== ch) {
      this._cursorRect.x = cx;
      this._cursorRect.y = cy;
      this._cursorRect.width = cw;
      this._cursorRect.height = ch;
      this._refreshCursor();
    }
  }

  /**
   * Changes the color of the background.
   *
   * @method setTone
   * @param {Number} r The red value in the range (-255, 255)
   * @param {Number} g The green value in the range (-255, 255)
   * @param {Number} b The blue value in the range (-255, 255)
   */
  setTone(r, g, b) {
    const tone = this._colorTone;
    if (r !== tone[0] || g !== tone[1] || b !== tone[2]) {
      this._colorTone = [r, g, b];
      this._refreshBack();
    }
  }

  /**
   * Adds a child between the background and contents.
   *
   * @method addChildToBack
   * @param {Object} child The child to add
   * @return {Object} The child that was added
   */
  addChildToBack(child) {
    if (this.enableWindowFrame()) {
      const containerIndex = this.children.indexOf(this._windowSpriteContainer);
      return this.addChildAt(child, containerIndex + 1);
    } else {
      this.addChildAt(child, 0);
    }
  }

  standardPadding() {
    return 18;
  }

  standardBackOpacity() {
    return 255;
  }

  updateBackOpacity() {
    this.backOpacity = this.standardBackOpacity();
  }

  updateTone() {
    const tone = $gameSystem.windowTone();
    this.setTone(tone[0], tone[1], tone[2]);
  }

  update() {
    super.update();
    this.updateTone();
    this.updateOpen();
    this.updateClose();
    this.updateBackgroundDimmer();
  }

  updateOpen() {
    if (this._opening) {
      this.openness += Math.floor(32 * Managers.Scenes._gameSpeed);
      if (this.isOpen()) {
        this._opening = false;
      }
    }
  }

  updateClose() {
    if (this._closing) {
      this.openness -= Math.floor(32 * Managers.Scenes._gameSpeed);
      if (this.isClosed()) {
        this._closing = false;
      }
    }
  }

  open() {
    if (!this.isOpen()) {
      this._opening = true;
    }
    this._closing = false;
  }

  close() {
    if (!this.isClosed()) {
      this._closing = true;
    }
    this._opening = false;
  }

  isOpening() {
    return this._opening;
  }

  isClosing() {
    return this._closing;
  }

  activate() {
    this.active = true;
  }

  deactivate() {
    this.active = false;
  }

  setBackgroundType(type) {
    if (type === 0) {
      this.opacity = 255;
    } else {
      this.opacity = 0;
    }
    if (type === 1) {
      this.showBackgroundDimmer();
    } else {
      this.hideBackgroundDimmer();
    }
  }

  showBackgroundDimmer() {
    if (!this._dimmerSprite) {
      this._dimmerSprite = new Sprite();
      this._dimmerSprite.bitmap = new Bitmap(0, 0);
      this.addChildToBack(this._dimmerSprite);
    }
    const bitmap = this._dimmerSprite.bitmap;
    if (bitmap.width !== this.width || bitmap.height !== this.height) {
      this.refreshDimmerBitmap();
    }
    this._dimmerSprite.visible = true;
    this.updateBackgroundDimmer();
  }

  hideBackgroundDimmer() {
    if (this._dimmerSprite) {
      this._dimmerSprite.visible = false;
    }
  }

  updateBackgroundDimmer() {
    if (this._dimmerSprite) {
      this._dimmerSprite.opacity = this.openness;
    }
  }

  refreshDimmerBitmap() {
    if (this._dimmerSprite) {
      const bitmap = this._dimmerSprite.bitmap;
      const w = this.width;
      const h = this.height;
      const vPadding = this.paddingV;
      const c1 = this.dimColor1();
      const c2 = this.dimColor2();
      bitmap.resize(w, h);
      bitmap.gradientFillRect(0, 0, w, vPadding, c2, c1, true);
      bitmap.fillRect(0, vPadding, w, h - vPadding * 2, c1);
      bitmap.gradientFillRect(0, h - vPadding, w, vPadding, c1, c2, true);
      this._dimmerSprite.setFrame(0, 0, w, h);
    }
  }

  dimColor1() {
    return 'rgba(0, 0, 0, 0.6)';
  }

  dimColor2() {
    return 'rgba(0, 0, 0, 0)';
  }

  /**
   * Returns true if the window is completely open (openness == 255).
   *
   * @method isOpen
   */
  isOpen() {
    return this._openness >= 255 && this.isOnCurrentScene();
  }

  /**
   * @method updateTransform
   * @private
   */
  updateTransform() {
    this._updateCursor();
    this._updateArrows();
    this._updatePauseSign();
    Core.Window.prototype.updateTransform.call(this);
  }

  enableCursorSprite() {
    return true;
  }

  enableWindowFrame() {
    return true;
  }

  _createBackSpriteBitmap() {
    this._windowBackSprite = new Sprite();
    this._windowBackSprite.bitmap = new Bitmap(1, 1);
    this._windowBackSprite.alpha = 192 / 255;
    this._windowSpriteContainer.addChild(this._windowBackSprite);
  }

  _createWindowFrameSprite() {
    this._windowFrameSprite = new Sprite();
    this._windowSpriteContainer.addChild(this._windowFrameSprite);
  }

  /**
   * @method _createAllParts
   * @private
   */
  _createAllParts() {
    const enableWindowFrame = this.enableWindowFrame();
    const enableCursorSprite = this.enableCursorSprite();

    if (enableWindowFrame) {
      this._windowSpriteContainer = new PIXI.Container();
      this._downArrowSprite = new Sprite();
      this._upArrowSprite = new Sprite();
      this._windowPauseSignSprite = new Sprite();

      this.addChild(this._windowSpriteContainer);
      this._createBackSpriteBitmap();
      this._createWindowFrameSprite();

    }

    if (enableCursorSprite) {
      this._windowCursorSprite = new Sprite();
      this.addChild(this._windowCursorSprite);
    }

    Core.Window.prototype._createAllParts.call(this);

    if (enableWindowFrame) {
      this.addChild(this._downArrowSprite);
      this.addChild(this._upArrowSprite);
      this.addChild(this._windowPauseSignSprite);
    }
  }

  _refreshAllParts() {
    this._refreshBack();
    this._refreshFrame();
    this._refreshCursor();
    Core.Window.prototype._refreshAllParts.call(this);
    this._refreshArrows();
    this._refreshPauseSign();
  }

  /**
   * @method _refreshBack
   * @private
   */
  _refreshBack() {
    if (!this.enableWindowFrame()) return;
    if (!this._windowBackSprite) return;

    const m = this._margin;
    const w = this._width - m * 2;
    const h = this._height - m * 2;

    let bitmap = this._windowBackSprite.bitmap;
    if (!bitmap) {
      bitmap = new Bitmap(w, h);
    } else if (bitmap.width !== w || bitmap.height !== h) {
      bitmap.resize(w, h);
    }

    this._windowBackSprite.bitmap = bitmap;
    this._windowBackSprite.setFrame(0, 0, w, h);
    this._windowBackSprite.move(m, m);

    if (w > 0 && h > 0 && this._windowskin) {
      const p = 96;
      bitmap.bltImage(this._windowskin, 0, 0, p, p, 0, 0, w, h);
      for (let y = 0; y < h; y += p) {
        for (let x = 0; x < w; x += p) {
          bitmap.bltImage(this._windowskin, 0, p, p, p, x, y, p, p);
        }
      }
      const tone = this._colorTone;
      bitmap.adjustTone(tone[0], tone[1], tone[2]);
    }
  }

  /**
   * @method _refreshFrame
   * @private
   */
  _refreshFrame() {
    if (!this.enableWindowFrame()) return;
    if (!this._windowFrameSprite) return;

    const w = this._width;
    const h = this._height;
    const m = 24;

    let bitmap = this._windowFrameSprite.bitmap;
    if (!bitmap) {
      bitmap = new Bitmap(w, h);
    } else if (bitmap.width != w || bitmap.height != h) {
      bitmap.resize(w, h);
    }

    this._windowFrameSprite.bitmap = bitmap;
    this._windowFrameSprite.setFrame(0, 0, w, h);

    if (w > 0 && h > 0 && this._windowskin) {
      const skin = this._windowskin;
      const p = 96;
      const q = 96;
      bitmap.bltImage(skin, p+m, 0+0, p-m*2, m, m, 0, w-m*2, m);
      bitmap.bltImage(skin, p+m, 0+q-m, p-m*2, m, m, h-m, w-m*2, m);
      bitmap.bltImage(skin, p+0, 0+m, m, p-m*2, 0, m, m, h-m*2);
      bitmap.bltImage(skin, p+q-m, 0+m, m, p-m*2, w-m, m, m, h-m*2);
      bitmap.bltImage(skin, p+0, 0+0, m, m, 0, 0, m, m);
      bitmap.bltImage(skin, p+q-m, 0+0, m, m, w-m, 0, m, m);
      bitmap.bltImage(skin, p+0, 0+q-m, m, m, 0, h-m, m, m);
      bitmap.bltImage(skin, p+q-m, 0+q-m, m, m, w-m, h-m, m, m);
    }
  }

  /**
   * @method _refreshCursor
   * @private
   */
  _refreshCursor() {
    if (!this._windowCursorSprite) return;

    const padV = this._paddingV;
    const padH = this._paddingH;

    const x = this._cursorRect.x + padH - this.origin.x;
    const y = this._cursorRect.y + padV - this.origin.y;
    const w = this._cursorRect.width;
    const h = this._cursorRect.height;
    const m = 4;
    const x2 = Math.max(x, padH);
    const y2 = Math.max(y, padV);
    const ox = x - x2;
    const oy = y - y2;
    const w2 = Math.min(w, this._width - padH - x2);
    const h2 = Math.min(h, this._height - padV - y2);

    let bitmap = this._windowCursorSprite.bitmap;

    if (!bitmap) {
      bitmap = new Bitmap(w2, h2);
    } else if (bitmap.width !== w2 || bitmap.height !== h2) {
      bitmap.resize(w2, h2);
    }

    this._windowCursorSprite.bitmap = bitmap;
    this._windowCursorSprite.setFrame(0, 0, w2, h2);
    this._windowCursorSprite.move(x2, y2);

    if (w > 0 && h > 0 && this._windowskin) {
      const skin = this._windowskin;
      const p = 96;
      const q = 48;
      bitmap.bltImage(skin, p+m, p+m, q-m*2, q-m*2, ox+m, oy+m, w-m*2, h-m*2);
      bitmap.bltImage(skin, p+m, p+0, q-m*2, m, ox+m, oy+0, w-m*2, m);
      bitmap.bltImage(skin, p+m, p+q-m, q-m*2, m, ox+m, oy+h-m, w-m*2, m);
      bitmap.bltImage(skin, p+0, p+m, m, q-m*2, ox+0, oy+m, m, h-m*2);
      bitmap.bltImage(skin, p+q-m, p+m, m, q-m*2, ox+w-m, oy+m, m, h-m*2);
      bitmap.bltImage(skin, p+0, p+0, m, m, ox+0, oy+0, m, m);
      bitmap.bltImage(skin, p+q-m, p+0, m, m, ox+w-m, oy+0, m, m);
      bitmap.bltImage(skin, p+0, p+q-m, m, m, ox+0, oy+h-m, m, m);
      bitmap.bltImage(skin, p+q-m, p+q-m, m, m, ox+w-m, oy+h-m, m, m);
    }
  }

  /**
   * @method _refreshArrows
   * @private
   */
  _refreshArrows() {
    if (!this.enableWindowFrame()) return;

    const w = this._width;
    const h = this._height;
    const p = 24;
    const q = p/2;
    const sx = 96+p;
    const sy = 0+p;
    this._downArrowSprite.bitmap = this._windowskin;
    this._downArrowSprite.anchor.x = 0.5;
    this._downArrowSprite.anchor.y = 0.5;
    this._downArrowSprite.setFrame(sx+q, sy+q+p, p, q);
    this._downArrowSprite.move(w/2, h-q);
    this._upArrowSprite.bitmap = this._windowskin;
    this._upArrowSprite.anchor.x = 0.5;
    this._upArrowSprite.anchor.y = 0.5;
    this._upArrowSprite.setFrame(sx+q, sy, p, q);
    this._upArrowSprite.move(w/2, q);
  }

  /**
   * @method _refreshPauseSign
   * @private
   */
  _refreshPauseSign() {
    if (!this.enableWindowFrame()) return;

    const sx = 144;
    const sy = 96;
    const p = 24;
    this._windowPauseSignSprite.bitmap = this._windowskin;
    this._windowPauseSignSprite.anchor.x = 0.5;
    this._windowPauseSignSprite.anchor.y = 1;
    this._windowPauseSignSprite.move(this._width / 2, this._height);
    this._windowPauseSignSprite.setFrame(sx, sy, p, p);
    this._windowPauseSignSprite.alpha = 0;
  }

  /**
   * @method _updateCursor
   * @private
   */
  _updateCursor() {
    if (!this._windowCursorSprite) return;

    const blinkCount = this._animationCount % 40;
    let cursorOpacity = this.contentsOpacity;
    if (this.active) {
      if (blinkCount < 20) {
        cursorOpacity -= blinkCount * 8;
      } else {
        cursorOpacity -= (40 - blinkCount) * 8;
      }
    }
    this._windowCursorSprite.alpha = cursorOpacity / 255;
    this._windowCursorSprite.visible = this.isOpen();
  }

  /**
   * @method _updateArrows
   * @private
   */
  _updateArrows() {
    if (!this.enableWindowFrame()) return;

    this._downArrowSprite.visible = this.isOpen() && this.downArrowVisible;
    this._upArrowSprite.visible = this.isOpen() && this.upArrowVisible;
  }

  /**
   * @method _updatePauseSign
   * @private
   */
  _updatePauseSign() {
    if (!this.enableWindowFrame()) return;

    const sprite = this._windowPauseSignSprite;
    const x = Math.floor(this._animationCount / 16) % 2;
    const y = Math.floor(this._animationCount / 16 / 2) % 2;
    const sx = 144;
    const sy = 96;
    const p = 24;
    if (!this.pause) {
      sprite.alpha = 0;
    } else if (sprite.alpha < 1) {
      sprite.alpha = Math.min(sprite.alpha + 0.1, 1);
    }
    sprite.setFrame(sx+x*p, sy+y*p, p, p);
    sprite.visible = this.isOpen();
  }

  isCancelTriggered() {
    if ($gameTemp.shouldSkipSelectableWindowHandling()) return false;

    if (Input.isRepeated('cancel')) return true;

    if (Input.isRepeated('alternate')) {
      return true;
    }

    return false;
  }

  requireBitmap(bitmap) {
    if (!bitmap) return;

    if (!bitmap.isReady()) {
      this._refreshExpected = true;
      bitmap.addLoadListener(() => {
        this._refreshExpected = false;
        this.refresh();
      });
    }
  }

  refresh() {
    this._refreshExpected = false;
    
  }

  drawImagePiece(fileName, x, y, sx, sy, sw, sh, zoomLevel, zoomAffectsPosition = true) {
    if (x === undefined) x = 0;
    if (y === undefined) y = 0;
    zoomLevel = zoomLevel || 1;

    const bitmap = Managers.Images.loadPicture(fileName);

    const width = sw;
    const height = sh;
    let drawWidth = width * zoomLevel;
    const drawHeight = height * zoomLevel;

    if (zoomAffectsPosition !== false) {
      x *= zoomLevel;
      y *= zoomLevel;
    }

    this.contents.bltBitmap(bitmap, sx, sy, sw, sh, x, y, drawWidth, drawHeight);
  }

  drawDayNumber(x, y, zoomLevel) {
    const numberH = 43;

    const day = Managers.Time.day;
    const digits = day.toString().split('');
    const positions = [];
    const shadowSizes = [];
    const sizes = [];

    positions.push(2);shadowSizes.push(34);sizes.push(31);
    positions.push(41);shadowSizes.push(29);sizes.push(26);
    positions.push(76);shadowSizes.push(33);sizes.push(30);
    positions.push(111);shadowSizes.push(37);sizes.push(34);
    positions.push(150);shadowSizes.push(33);sizes.push(30);

    positions.push(2);shadowSizes.push(33);sizes.push(30);
    positions.push(39);shadowSizes.push(33);sizes.push(30);
    positions.push(77);shadowSizes.push(31);sizes.push(28);
    positions.push(115);shadowSizes.push(31);sizes.push(28);
    positions.push(151);shadowSizes.push(32);sizes.push(30);

    let bgX = x;
    if (digits.length == 1) {
      digits.splice(0, 0, '0');
    }

    for (const number of digits) {
      let numberY = 0;
      if (number >= 5) {
        numberY += numberH;
      }

      this.drawImagePiece('timeHud/calendar_font2', bgX, y, positions[number], numberY, shadowSizes[number], numberH, zoomLevel);

      bgX += sizes[number] - 3;
    }

    let fgX = x;
    if (digits.length == 1) {
      fgX += 16;
    }

    for (const numberFG of digits) {
      let numberYFG = 0;
      if (numberFG >= 5) {
        numberYFG += numberH;
      }

      this.drawImagePiece('timeHud/calendar_font', fgX, y, positions[numberFG], numberYFG, sizes[numberFG], numberH, zoomLevel);
      fgX += sizes[numberFG] - 3;
    }
  }

  drawZoomedBitmap(bitmap, x, y, zoomLevel) {
    this.requireBitmap(bitmap);
    if (!bitmap.isReady()) return;

    const width = bitmap._canvas.width;
    const height = bitmap._canvas.height;
    const drawWidth = width * zoomLevel;
    const drawHeight = height * zoomLevel;

    this.contents.bltBitmap(bitmap, 0, 0, width, height, x, y, drawWidth, drawHeight);
  }

  drawBitmap(bitmap, originalX, originalY, zoomLevel) {
    this.drawZoomedBitmap(bitmap, originalX * zoomLevel, originalY * zoomLevel, zoomLevel);
  }

  drawPercentValue(x, y, value, maxWidth, align = 'center', includeSymbol = true) {
    const charData = {};
    const charH = 23;
    const charW = 20;
    const charMinW = 17;
    const zoomLevel = this._zoomLevel ? this._zoomLevel : Graphics.windowZoomLevel;

    for (let i = 0; i <= 4; i++) {
      charData[i.toString()] = {
        x : i * charW,
        y : 0
      };
      charData[((i + 5)).toString()] = {
        x : i * charW,
        y : charH
      };
    }

    charData['%'] = {
      x : 0,
      y : charH * 2
    };

    charData['O'] = {
      x : charW,
      y : charH * 2
    };
    charData['l'] = {
      x : charW * 2,
      y : charH * 2
    };
    charData['#'] = {
      x : charW * 3,
      y : charH * 2
    };
    charData['+'] = {
      x : charW * 4,
      y : charH * 2
    };

    // When showing "100%", we use lowercase L and uppercase o to represent the number in the bitmap table
    // this is because the coloring of the numbers is a bit different
    if (typeof value == 'number') {
      if (value === 100) {
        value = 'lOO';
        if (includeSymbol) {
          value += '#';
        }
      } else {
        value = value.toString();
        if (includeSymbol) {
          value += '%';
        }
      }
    } else if (value === '100') {
      value = 'lOO';
      if (includeSymbol) {
        value += '#';
      }
    } else if (value === '100%') {
      value = 'lOO#';
    }

    value = value.toString().split('');

    let charX = x;
    const charY = y;
    const totalWidth = value.length * charMinW * zoomLevel;

    if (align == 'right') {
      if (totalWidth < maxWidth) {
        charX += (maxWidth - totalWidth);
      }
    } else if (align == 'center') {
      if (totalWidth < maxWidth) {
        charX += (maxWidth - totalWidth) / 2;
      }
    }

    for (let charIdx = 0; charIdx < value.length; charIdx++) {
      const data = charData[value[charIdx]];
      if (data) {
        this.drawImagePiece(`menu/stamp_font`, charX, charY, data.x, data.y, charW, charH, zoomLevel, false);
        charX += (charMinW - 1) * zoomLevel;
      }
    }
  }
}

module.exports = Window_Base;
