const Window_Base = require('engine/windows/Window_Base');

class Window_Portrait extends Window_Base {
  initialize(messageWindow) {
    const width = 200 * Graphics.windowZoomLevel;
    const height = 120 * Graphics.windowZoomLevel;

    super.initialize(0, 0, width, height);

    this._messageWindow = messageWindow;
    this._bgBitmap = Managers.Images.loadPicture('portrait_dock');
    this._nameBitmap = Managers.Images.loadPicture('name_box');

    this._zoomLevel = Graphics.windowZoomLevel;
  }

  set portrait(value) {
    this._portrait = value;
    this.refresh();
  }

  get portrait() {
    return this._portrait;
  }

  set name(value) {
    this._name = value;
  }

  get name() {
    return this._name;
  }

  enableWindowFrame() {
    return false;
  }

  refresh() {
    if (this.contents) {
      this.contents.clear();
      this.drawContent();
    }
  }

  drawBitmap(bitmap, originalX, originalY) {
    super.drawBitmap(bitmap, originalX, originalY, this._zoomLevel);
  }

  drawContent() {
    this.contents.fontSize = Utils.convertFontSize(22, this._zoomLevel);

    const originalWidth = this.width / this._zoomLevel;
    const originalHeight = this.height / this._zoomLevel;

    const centerX = originalWidth / 2;

    this.drawBitmap(this._bgBitmap, centerX - 37, originalHeight - 85);

    if (this._portrait) {
      const pX = (centerX - (this._portrait.width / 2)) + 0.5 * this._zoomLevel;
      const pY = (originalHeight - this._portrait.height) - 3 * this._zoomLevel;
      this.drawBitmap(this._portrait, pX, pY);
    }

    this.drawBitmap(this._nameBitmap, centerX - 28, originalHeight - 17);

    if (this._name) {
      this.drawText(this._name, (centerX - 28) * this._zoomLevel, (originalHeight - 19) * this._zoomLevel, 60 * this._zoomLevel, 'center');
    }
  }

  dockToWindowMessage() {
    if (!this._messageWindow) return;

    const originalWidth = this.width / this._zoomLevel;
    const desiredRightX = this._messageWindow.x + 9 * this._zoomLevel;
    const rightMargin = (originalWidth / 2 - 37) * this._zoomLevel;

    this.x = desiredRightX + rightMargin - this.width;
    this.y = this._messageWindow.y + this._messageWindow.height - this.height;
  }

  shouldBeVisible() {
    return !!this._portrait || !!this._name;
  }

  standardPadding() {
    return 0;
  }

  deactivate() {
    super.deactivate();
    this._portrait = null;
    this._name = null;
    this.hide();
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  updateVisiblity(newValue, instant) {
    newValue = newValue !== undefined ? newValue : this.shouldBeVisible();
    if (this.hideTimeout) {
      clearTimeout(this.hideTimeout);
      this.hideTimeout = null;
    }

    if (newValue == this.visible) {
      return;
    }

    if (this.visible) {
      if (instant) {
        this.visible = false;
      } else {
        this.hideTimeout = setTimeout(() => {
          this.visible = false;
          this.hideTimeout = null;
        }, 100);
      }
    } else {
      this.visible = true;
    }
  }

  show() {
    this.updateVisiblity(true);
  }

  hide() {
    this.updateVisiblity(false);
  }
}

module.exports = Window_Portrait;
