//-----------------------------------------------------------------------------
// Spriteset_Map
//
// The set of sprites on the map screen.

function Spriteset_Map() {
  this.initialize.apply(this, arguments);
}

Spriteset_Map.prototype = Object.create(Spriteset_Base.prototype);
Spriteset_Map.prototype.constructor = Spriteset_Map;

Spriteset_Map.prototype.initialize = function() {
  Spriteset_Base.prototype.initialize.call(this);
};

Spriteset_Map.prototype.terminate = function() {
  this._tilemap.removeChildren();
  if (this._characterSprites) {
    this._characterSprites = [];
  }
  this._tvChannel = undefined;
  this.removeChildren();
};

Spriteset_Map.prototype.createLowerLayer = function() {
  Spriteset_Base.prototype.createLowerLayer.call(this);

  this.createParallax();
  this.createTilemap();
  this.createObjectInformation();
  this.createFarmObjects();
  this.createCharacters();
  this.createAffectedTiles();
  this.createFilters();
  this.createTvChannel();
  this.createLightmask();
  this.createWeather();
};

Spriteset_Map.prototype.update = function() {
  if (Managers.Scenes.isSkippedFrame) return;

  Spriteset_Base.prototype.update.call(this);
  this.updateTileset();
  this.updateParallax();
  this.updateTilemap();
  this.updateWeather();

  $gameTemp.markFarmObjectsAsUpdated();
};

Spriteset_Map.prototype.hideCharacters = function() {
  for (var i = 0; i < this._characterSprites.length; i++) {
    var sprite = this._characterSprites[i];
    if (!sprite.isTile || !sprite.isTile()) {
      sprite.hide();
    }
  }
};

Spriteset_Map.prototype.createParallax = function() {
  this._parallax = new TilingSprite();
  this._parallax.move(0, 0, Graphics.width, Graphics.height);
  this._baseSprite.addChild(this._parallax);
};

var uniqueLightmask = false;
Spriteset_Map.prototype.createLightmask = function() {
  if (uniqueLightmask) {
    this._lightmask = uniqueLightmask;
  } else {
    this._lightmask = new Lightmask();
    uniqueLightmask = this._lightmask;
  }

  this._lightmask.setPlayerSprite(this._playerSprite);
  this.addChild(this._lightmask);
};

Spriteset_Map.prototype.createTilemap = function() {
  if (Graphics.isWebGL()) {
    this._tilemap = new ShaderTilemap();
  } else {
    this._tilemap = new Tilemap();
  }
  this._tilemap.tileWidth = $gameMap.tileWidth();
  this._tilemap.tileHeight = $gameMap.tileHeight();
  this._tilemap.setData($gameMap.width(), $gameMap.height(), $gameMap.data());
  this._tilemap.horizontalWrap = $gameMap.isLoopHorizontal();
  this._tilemap.verticalWrap = $gameMap.isLoopVertical();
  this._tilemap.tilesetId = $gameMap.tilesetId();
  this.loadTileset();
  this._baseSprite.addChild(this._tilemap);

  this._displayLayer = new PIXI.display.Layer();
  this._displaySprite = new Sprite();
  // this._higherLayer = new PIXI.display.Layer();
  // this._higherSprite = new Sprites.HigherLayer();

  if (this._tilemap._lowerLayer) {
    this._tilemap._lowerLayer.parentLayer = this._displayLayer;
  }
  if (this._tilemap.lowerZLayer) {
    this._tilemap.lowerZLayer.parentLayer = this._displayLayer;
  }
  if (this._tilemap._upperLayer) {
    this._tilemap._upperLayer.parentLayer = this._displayLayer;
  }
  if (this._tilemap.upperZLayer) {
    this._tilemap.upperZLayer.parentLayer = this._displayLayer;
  }

  this._tilemap.addChild(this._displaySprite);
  this._displaySprite.addChild(this._displayLayer);

  this._tilemap.displayLayer = this._displayLayer;
  this._displayLayer.group.enableSort = true;

  // this._tilemap.addChild(this._higherSprite);
  // this._higherSprite.setHigherLayer(this._higherLayer);
  // this._higherLayer.group.enableSort = true;
};

Spriteset_Map.prototype.loadTileset = function() {
  this._tileset = $gameMap.tileset();
  if (this._tileset) {
    var tilesetNames = this._tileset.tilesetNames;
    for (var i = 0; i < tilesetNames.length; i++) {
      this._tilemap.bitmaps[i] = Managers.Images.loadTileset(tilesetNames[i]);
    }
    var newTilesetFlags = $gameMap.tilesetFlags();
    this._tilemap.refreshTileset();
    if (!this._tilemap.flags.equals(newTilesetFlags)) {
      this._tilemap.refresh();
    }
    this._tilemap.flags = newTilesetFlags;
  }
};

Spriteset_Map.prototype.createFarmObjects = function() {
  this._farmObjectSprites = [];
  this._soilSprites = [];

  this._spritesetCropSoils = new Sprites.Spriteset_CropSoils();
  this._spritesetCropSoils.parentLayer = this._displayLayer;
  this._spritesetCropSoils.zIndex = 0;
  this._spritesetCropSoils.zOrder = -99999;
  this._tilemap.addChild(this._spritesetCropSoils);

  $gameMap.farmObjectEvents().forEach(function(farmObjectEvent){
    if (farmObjectEvent.usesNoGraphic()) return;

    const objectSprites = farmObjectEvent.createSprites();
    this._farmObjectSprites.push(...objectSprites);
  }, this);

  var len = this._farmObjectSprites.length;
  for (var i = 0; i < len; i++) {
    this._displaySprite.addChild(this._farmObjectSprites[i]);
    this._farmObjectSprites[i].parentLayer = this._displayLayer;
  }
};

Spriteset_Map.prototype.addEventSprite = function(gameEvent) {
  var spriteClass = gameEvent.getSpriteClass();
  var sprite = new spriteClass(gameEvent);

  sprite.parentLayer = this._displayLayer;
  this._characterSprites.push(sprite);
  this._tilemap.addChild(sprite);
};

Spriteset_Map.prototype.addObjectSprite = function(farmObjectEvent) {
  this._soilSprites.push(null);

  const objectSprites = farmObjectEvent.createSprites();
  for (const objSprite of objectSprites) {
    if (!objSprite) {
      continue;
    }

    this._farmObjectSprites.push(objSprite);
    this._displaySprite.addChild(objSprite);
    objSprite.parentLayer = this._displayLayer;
    objSprite.updatePosition(true);
  }
};

Spriteset_Map.prototype.findAllObjectSprites = function(farmObject) {
  const list = [];

  for (var i = 0; i < this._farmObjectSprites.length; i++) {
    if (!this._farmObjectSprites[i]) continue;
    if (!this._farmObjectSprites[i]._farmObject) continue;
    if (this._farmObjectSprites[i]._farmObject != farmObject) continue;

    list.push(this._farmObjectSprites[i]);
  }

  return list;
};

Spriteset_Map.prototype.removeSpriteReferences = function(sprite) {
  if (!sprite) {
    return;
  }

  let index = this._characterSprites.indexOf(sprite);
  if (index >= 0) {
    this._characterSprites.splice(index, 1);
  }

  index = this._farmObjectSprites.indexOf(sprite);
  if (index >= 0) {
    this._farmObjectSprites.splice(index, 1);

    const soilSprite = this._soilSprites.find((soilSprite) => {
      return soilSprite && soilSprite._farmObject == sprite._farmObject;
    });

    if (soilSprite) {
      if (soilSprite.removeItself) {
        soilSprite.removeItself();
      }

      const idx = this._soilSprites.indexOf(soilSprite);
      if (idx >= 0) {
        this._soilSprites.splice(idx, 1);
      }
    }
  }
};

Spriteset_Map.prototype.addSoilSprite = function(farmObjectEvent) {
  const objectSprite = farmObjectEvent._sprite;
  let sprite = this._soilSprites.find((sprite) => {
    return sprite && sprite._farmObject == farmObjectEvent;
  });

  if (!sprite || sprite._destroyed) {
    const eventSoilSprite = (farmObjectEvent._soilSprite  && !farmObjectEvent._soilSprite._destroyed) ? farmObjectEvent._soilSprite : false;
    const objectSoilSprite = (objectSprite._soilSprite && !objectSprite._soilSprite._destroyed) ? objectSprite._soilSprite : false;

    sprite = eventSoilSprite || objectSoilSprite || new Sprites.Soil(farmObjectEvent, objectSprite);
   
    this._soilSprites.push(sprite);
    this._spritesetCropSoils.addChild(sprite); 
  }
};

Spriteset_Map.prototype.createTvChannel = function() {
  this._tvChannel = new Sprite_Channel();
  this._tilemap.addChild(this._tvChannel);
  this._tvChannel.parentLayer = this._displayLayer;
};

Spriteset_Map.prototype.createCharacters = function() {
  this._characterSprites = [];
  $gameMap.events().forEach(function(event) {
    if (event.usesNoGraphic()) return;

    var classType = event.getSpriteClass();

    this._characterSprites.push(new classType(event));
  }, this);

  $gameMap.villagerEvents().forEach(function(event) {
    this._characterSprites.push(new Sprite_Villager(event));
  }, this);

  $gameMap.creatureEvents().forEach(function(event) {
    this._characterSprites.push(new Sprite_Creature(event));
  }, this);

  $gamePlayer.followers().reverseEach(function(follower) {
    this._characterSprites.push(new Sprite_Follower(follower));
  }, this);

  var playerSprite = new Sprite_Player($gamePlayer);
  this._playerSprite = playerSprite;
  this._characterSprites.push(playerSprite);
  
  for (var i = 0; i < this._characterSprites.length; i++) {
    this._characterSprites[i].parentLayer = this._displayLayer;
    this._displaySprite.addChild(this._characterSprites[i]);
  }
};

Spriteset_Map.prototype.createFilters = function() {
  if (!Graphics.outlineFilter) {
    Graphics.outlineFilter = new PIXI.filters.OutlineFilter(2, 0xffffff);
    Graphics.winterOutlineFilter = new PIXI.filters.OutlineFilter(4, 0x66ff66);
  }

  if (!Graphics.invalidOutlineFilter) {
    Graphics.invalidOutlineFilter = new PIXI.filters.OutlineFilter(2, 0xff9999);
  }

  // if (!Graphics.glowFilter) {
  //   Graphics.glowFilter = new PIXI.filters.GlowFilter(15, 3, 0, 0xffffff, 0.5);
  // }

  if (!Graphics.namingOutlineFilter) {
    Graphics.namingOutlineFilter = new PIXI.filters.OutlineFilter(2, 0x3D2B25);
  }

  if (!Graphics.shadowFilter) {
    Graphics.shadowFilter = new PIXI.filters.DropShadowFilter({ distance: 10});
  }
};

Spriteset_Map.prototype.createAffectedTiles = function() {
  this._affectedTilesSprite = new Sprite_AffectedTile();
  this._affectedTilesSprite.z = 9;

  this._affectedTilesSprite.parentLayer = this._displayLayer;
  this._displaySprite.addChild(this._affectedTilesSprite);
};

Spriteset_Map.prototype.createObjectInformation = function() {
  this._objectInformationSprite = new Sprites.ObjectInformation();
  this._objectInformationSprite.z = 9;
  this._baseSprite.addChild(this._objectInformationSprite);
};

var uniqueWeather = false;
Spriteset_Map.prototype.createWeather = function() {
  if (uniqueWeather) {
    this._weather = uniqueWeather;
    
    this._weather.power = 0;
    this._weather.type = 'none';
    this._weather.origin = new Point();
  } else {
    this._weather = new Weather();
    uniqueWeather = this._weather;
  }

  this.addChild(this._weather);
};

Spriteset_Map.prototype.updateTileset = function() {
  if (this._tileset !== $gameMap.tileset()) {
    this.loadTileset();
  }
};

Spriteset_Map.prototype.updateParallax = function() {
  if (this._parallaxName !== $gameMap.parallaxName()) {
    this._parallaxName = $gameMap.parallaxName();
    this._parallax.bitmap = Managers.Images.loadParallax(this._parallaxName);
  }
  if (this._parallax.bitmap) {
    this._parallax.origin.x = $gameMap.parallaxOx();
    this._parallax.origin.y = $gameMap.parallaxOy();
  }
};

Spriteset_Map.prototype.updateTilemap = function() {
  this._tilemap.origin.x = $gameMap.displayX() * $gameMap.tileWidth();
  this._tilemap.origin.y = $gameMap.displayY() * $gameMap.tileHeight();

  // this.checkSpritesAbovePlayerRadius();
};

Spriteset_Map.prototype.checkSpritesAbovePlayerRadius = function() {
  const len = this._farmObjectSprites.length;
  for (let i = 0; i < len; i++) {
    const sprite = this._farmObjectSprites[i];
    if (!sprite) continue;

    if (sprite.isAbovePlayerRadius(this._playerSprite) && !Switches.hidePlayer) {
      sprite.parentLayer = this._higherLayer;
    } else {
      sprite.parentLayer = this._displayLayer;
    }
  }
};

Spriteset_Map.prototype.updateWeather = function() {
  this._weather.type = $gameScreen.weatherType();
  this._weather.power = $gameScreen.weatherPower();
  this._weather.origin.x = $gameMap.displayX() * $gameMap.tileWidth();
  this._weather.origin.y = $gameMap.displayY() * $gameMap.tileHeight();
};

Spriteset_Map.prototype.updatePosition = function() {
  var scale = $gameMap.zoom;
  var screen = $gameScreen;
  this.x = Math.fix(-$gameMap.zoom.x * (scale.x - 1));
  this.y = Math.fix(-$gameMap.zoom.y * (scale.x - 1));
  this.x = Math.fix(this.x + screen.shake());
  
  if (this.scale.x !== scale.x || this.scale.y !== scale.y) {
    var sw = Graphics.width / scale.x + this._tilemap._margin * 2;
    var sh = Graphics.height / scale.y + this._tilemap._margin * 2;

    if (sw > this._tilemap.width || sh > this._tilemap.height) {
      this._tilemap.width = sw;
      this._tilemap.height = sh;
      this._tilemap.refresh();
    }
    
    this.scale = new PIXI.Point(scale.x, scale.y);
    this._weather.scale = new PIXI.Point(1.0 / scale.x,  1.0 / scale.y);
    this._parallax.move(this._parallax.x, this._parallax.y, Graphics.width / scale.x, Graphics.height / scale.y);
  }
};

Spriteset_Map.prototype.refreshTilemap = function() {
  this._tilemap.refresh();
};