require('./Sprite_Base');

class Sprite_StaticEvent extends Sprite_Base {
  initialize(character) {
    super.initialize(character);
    this.initMembers();
    this.setCharacter(character);

    this._randomCacheIndex = Math.randomInt(10);
  }

  initMembers() {
    this.anchor.x = 0.5;
    this.anchor.y = 1;
    this._character = null;

    this._frameWidth = undefined;
    this._frameHeight = undefined;
  }

  setCharacter(character) {
    this._character = character;
  }

  isNearTheScreen() {
    return $gameScreen.isPositionNearTheScreen(this.x, this.y);
  }

  shouldSkipUpdate() {
    if (Graphics.frameCount % 10 !== this._randomCacheIndex) {
      if (this.y > Graphics.height * 2 || this.y < (0 - Graphics.height)) {
        return true;
      }
      if (this.x > Graphics.width * 2 || this.x < (0 - Graphics.width)) {
        return true;
      }
    }

    return false;
  }


  update() {
    super.update();

    if (this._character && this._character._erased) {

      if (this.parent) {
        this.removeItself();
        this.destroy();
      }
      return;
    }

    if (this.shouldSkipUpdate()) {
      return;
    }

    this.updateBitmap();
    this.updateFrame();

    let alreadyUpdatedPosition = false;
    if (this._character && this._character._playerImpersonator) {
      if (Managers.Scenes._scene instanceof GameScenes.Map) {
        if (Managers.Scenes._scene._spriteset && Managers.Scenes._scene._spriteset._playerSprite) {
          var playerSprite = Managers.Scenes._scene._spriteset._playerSprite;
          this.x = playerSprite.x;
          this.y = playerSprite.y;
          this.z = playerSprite.z + (this._character._zVariation || 0);
          this.zIndex = playerSprite.zIndex + (this._character._zVariation || 0);
          this.zOrder = playerSprite.zOrder;
          alreadyUpdatedPosition = true;
        }
      }
    }

    if (!alreadyUpdatedPosition) {
      this.updatePosition();
    }
  }

  onImageChanged() {
    if (!this.bitmap) {
      return;
    }

    this.bitmap.addLoadListener(() => {
      this._frameWidth = this.patternWidth();
      this._frameHeight = this.patternHeight();
    });
  }

  updateBitmap() {
    if (!this.isImageChanged()) {
      return;
    }

    this._characterName = this._character.characterName();
    this._characterIndex = this._character.characterIndex();

    this.setCharacterBitmap();
    this.onImageChanged();
  }

  isImageChanged() {
    return (this._characterName !== this._character.characterName() || this._characterIndex !== this._character.characterIndex());
  }

  updateFrameCount() {
    if (!this._characterName) {
      this._characterFrameCount = undefined;
      return;
    }

    const regex = /\[(\d*)\]/;
    const match = this._characterName.match(regex);
    if (match && match.length >= 2) {
      this._characterFrameCount = parseInt(match[1]);
    } else {
      this._characterFrameCount = undefined;
    }
  }

  setCharacterBitmap() {
    this.bitmap = Managers.Images.loadCharacter(this._characterName);
    this._isBigCharacter = Managers.Images.isBigCharacter(this._characterName);
    this._isCharacterRow = Managers.Images.isCharacterRow(this._characterName);

    if (Managers.Images.isSingleImage(this._characterName)) {
      this._isSingleImage = true;
    } else if (this.character instanceof Objects.Event) {
      this._isSingleImage = this.character.singleImage;
    } else {
      this._isSingleImage = false;
    }

    this._isToolBitmap = false;
    this.updateFrameCount();
  }

  updateFrame() {
    this._mirror = false;
    this.updateCharacterFrame();
  }

  updateCharacterFrame() {
    if ((!this._frameWidth || !this._frameHeight) && this.bitmap && this.bitmap.isReady()) {
      this._frameWidth = this.patternWidth();
      this._frameHeight = this.patternHeight();
    }

    const pw = this._frameWidth || 0;
    const ph = this._frameHeight || 0;
    const sx = (this.characterBlockX() + this.characterPatternX()) * pw;
    const sy = (this.characterBlockY() + this.characterPatternY()) * ph;
    this.setFrame(sx, sy, pw, ph);
  }

  characterBlockX() {
    if (this._isBigCharacter || this._isSingleImage || this._isCharacterRow) {
      return 0;
    }

    const frameCount = this._characterFrameCount || 3;
    const index = this._character.characterIndex();
    return index % 4 * frameCount;
  }

  characterBlockY() {
    if (this._isBigCharacter || this._isSingleImage || this._isCharacterRow) {
      return 0;
    }

    const index = this._character.characterIndex();
    return Math.floor(index / 4) * 4;
  }

  pattern() {
    return this._character.pattern();
  }

  direction() {
    return this._character.direction();
  }

  characterPatternX() {
    return this.pattern();
  }

  characterPatternY() {
    const direction = this.direction();

    return (direction - 2) / 2;
  }

  patternWidth() {
    if (this._isSingleImage) {
      return this.bitmap.width;
    }

    if (this._characterFrameCount) {
      if (this._isBigCharacter || this._isCharacterRow) {
        return this.bitmap.width / this._characterFrameCount;
      }

      return this.bitmap.width / (this._characterFrameCount * 4);
    }

    if (this._isBigCharacter || this._isCharacterRow) {
      return this.bitmap.width / 3;
    }

    return this.bitmap.width / 12;
  }

  patternHeight() {
    if (this._isSingleImage || this._isCharacterRow) {
      return this.bitmap.height;
    }

    if (this._isBigCharacter) {
      return this.bitmap.height / 4;
    }

    return this.bitmap.height / 8;
  }

  updatePosition() {
    const oldX = this.x;
    const oldY = this.y;

    const newX = this._character.screenX();
    const newY = this._character.screenY();

    this.x = newX;
    this.y = newY;

    if (newX !== oldX || newY !== oldY) {
      $gameTemp.requestFarmObjectsUpdate();
    }

    this.z = this._character.screenZ();
    this.zIndex = this.z;
    this.zOrder = -newY;
  }

  _refresh() {
    if (this._character && this._character._erased) {
      this.visible = false;
      return;
    }

    super._refresh();
  }
}

Sprites.StaticEvent = Sprite_StaticEvent;
