require('./Sprite_Base');
//-----------------------------------------------------------------------------
// Sprite_Object
//
// The sprite for displaying a farmObject.

function Sprite_Object() {
  this.initialize.apply(this, arguments);
}

Sprite_Object.prototype = Object.create(Sprite_Base.prototype);
Sprite_Object.prototype.constructor = Sprite_Object;

Sprite_Object.prototype.initialize = function(farmObject) {
  Sprite_Base.prototype.initialize.call(this);
  this.initMembers();
  this.setFarmObject(farmObject);
  this._isSmall = farmObject &&  farmObject.farmObjectData && !farmObject.farmObjectData.isBigFarmObject();

  this._report = false;
  this._balloonDuration = 0;
  this._balloonOffset = 0;

  this.assignSpriteToObject(farmObject);
  this.cache = {};
  this.longCache = {};

  this._randomCacheIndex = Math.randomInt(10);
  this._refreshDelay = 0;
};

Sprite_Object.prototype.assignSpriteToObject = function(object) {
  object._sprite = this;
};

Sprite_Object.prototype.initMembers = function() {
  this.cache = {};
  this.longCache = {};
  this.anchor.x = 0.5;
  this.anchor.y = 1;
  this._farmObject = null;
  this._tilesetId = 0;
};

Sprite_Object.prototype.setFarmObject = function(farmObject) {
  this._farmObject = farmObject;
};

Sprite_Object.prototype.isNearTheScreen = function() {
  var sx = this.x;
  var sy = this.y;

  return $gameScreen.isPositionNearTheScreen(sx, sy);
};

Sprite_Object.prototype.shouldSkipUpdate = function() {
  if (Graphics.frameCount % 10 !== this._randomCacheIndex) {
    if (this.y > Graphics.height * 2 || this.y < (0 - Graphics.height)) {
      return true;
    }
    if (this.x > Graphics.width * 2 || this.x < (0 - Graphics.width)) {
      return true;
    }
  }

  return false;
};

Sprite_Object.prototype.update = function() {
  this.cache = {};
  this._changedBitmap = false;

  if (this._farmObject._erased) {
    this.endBalloon();
    if (this.parent) {
      this.removeItself();
      this.destroy();
    }
    return;
  }

  let shouldRefresh = $gameTemp.shouldRefreshFarmObjects();
  if (!shouldRefresh) {
    if (this.shouldSkipUpdate()) {
      return;
    }

    // force a refresh every 5 frames
    if (this._refreshDelay > 0) {
      this._refreshDelay--;
    } else {
      shouldRefresh = true;
    }
  }

  if (shouldRefresh) {
    this._refreshDelay = 5; //Utils.getFrameCount(5);
  }

  const imageChanged = this.isImageChanged();
  const detailsChanged = this.isImageDetailsChanged();

  if (detailsChanged) {
    this._oldDirection = this.farmObjectDirection();
    this._oldPattern = this.farmObjectPattern();
  }

  if (!this._isSmall || shouldRefresh) {
    if (imageChanged || shouldRefresh ) {
      Sprite_Base.prototype.update.call(this);

      this.updateBitmap();
    }
  }

  this.updateShader();
  if (imageChanged || detailsChanged || !this._hasFrame) {
    this.updateFrame();
  }

  this.updatePosition();

  this.updateAnimation();
  // this.updateBalloon();

  // const newOpacity = this._farmObject.opacity();
  // if (this.opacity != newOpacity) {
  //   this.opacity = newOpacity;
  // }
};

Sprite_Object.prototype.isAbovePlayerRadius = function(playerSprite) {
  if (this._farmObject._priorityType === 0) {
    return false;
  }

  const halfWidth = this.width / 2;
  const thisRight = this.x + halfWidth;
  const thisLeft = this.x - halfWidth;
  const thisBottom = this.y;
  const thisTop = this.y - this.height;

  const halfPlayerWidth = playerSprite.width / 2;
  const playerRight = playerSprite.x + halfPlayerWidth;
  const playerLeft = playerSprite.x - halfPlayerWidth;
  const playerTop = playerSprite.y - playerSprite.height;
  const playerBottom = playerSprite.y;

  if (thisBottom < playerBottom) return false;
  if (thisRight < playerLeft) return false;
  if (playerTop < thisTop) return false;
  if (playerRight < thisLeft) return false;

  return true;
};

Sprite_Object.prototype.isTile = function() {
  return this._farmObject.tileId > 0;
};

Sprite_Object.prototype.tilesetBitmap = function(tileId) {
  return Managers.Images.loadTileId(tileId, $gameMap.tileset());
};

Sprite_Object.prototype.updateBitmap = function() {
  if (this.isImageChanged()) {
    this._tilesetId = $gameMap.tilesetId();
    this._tileId = this._farmObject.tileId();
    this._iconIndex = this._farmObject.iconIndex();
    this._characterName = this._farmObject.characterName();
    this._characterIndex = this._farmObject.characterIndex();

    if (this._iconIndex > 0) {
      this.setIconBitmap();
    } else if (this._tileId > 0) {
      this.setTileBitmap();
    } else {
      this.setFarmObjectBitmap();
    }

    this._hasFrame = false;

    if (this._changedBitmap) {
      this.longCache = {};
    }
  }
};

Sprite_Object.prototype.isImageChanged = function() {
  if (this._characterIndex !== this._farmObject.characterIndex()) return true;
  if (this._iconIndex !== this._farmObject.iconIndex()) return true;
  if (this._characterName !== this._farmObject.characterName()) return true;
  if (this._tileId !== this._farmObject.tileId()) return true;
  if (this._tilesetId !== $gameMap.tilesetId()) return true;

  return false;
};

Sprite_Object.prototype.isImageDetailsChanged = function() {
  if (this._oldDirection !== this.farmObjectDirection()) return true;
  if (this._oldPattern !== this.farmObjectPattern()) return true;

  return false;
};

Sprite_Object.prototype.setIconBitmap = function() {
  this.bitmap = Managers.Images.loadIconWithShadow(this._iconIndex);
  this._changedBitmap = true;
};

Sprite_Object.prototype.setTileBitmap = function() {
  this.bitmap = this.tilesetBitmap(this._tileId);
  this._changedBitmap = true;
};

Sprite_Object.prototype.setFarmObjectBitmap = function() {
  if (this._oldCharacterName != this._characterName) {
    this.bitmap = Managers.Images.loadCharacter(this._characterName);
    this._changedBitmap = true;
  }

  this._isBigCharacter = Managers.Images.isBigCharacter(this._characterName);
  this._isCharacterRow = Managers.Images.isCharacterRow(this._characterName);
  this._oldCharacterName = this._characterName;

  if (Managers.Images.isSingleImage(this._characterName)) {
    this._isSingleImage = true;
  } else if (this.farmObject) {
    this._isSingleImage = this.farmObject.singleImage;
  } else {
    this._isSingleImage = false;
  }
};

Sprite_Object.prototype.updateFrame = function() {
  if (!this._bitmap || !this._bitmap.isReady()) {
    this._hasFrame = false;
    return;
  }

  this._mirror = false;

  if (this._iconIndex > 0) {
    this.updateIconFrame();
  } else if (this._tileId > 0) {
    this.updateTileFrame();
  } else {
    this.updateCharacterFrame();
  }

  this._hasFrame = true;
};

Sprite_Object.prototype.updateIconFrame = function() {
};

Sprite_Object.prototype.updateTileFrame = function() {
  const size = Constants.TILE_SIZE;
  this.setFrame(0, 0, size, size);
};

Sprite_Object.prototype.updateCharacterFrame = function() {
  var pw = this.patternWidth();
  var ph = this.patternHeight();
  var sx = (this.farmObjectBlockX() + this.farmObjectPatternX()) * pw;
  var sy = (this.farmObjectBlockY() + this.farmObjectPatternY()) * ph;

  this.setFrame(sx, sy, pw, ph);
};

Sprite_Object.prototype.farmObjectBlockX = function() {
  if (this._isBigCharacter || this._isSingleImage || this._isCharacterRow) {
    return 0;
  } else {
    var index;

    index = this._farmObject.characterIndex();
    return index % 4 * 3;
  }
};

Sprite_Object.prototype.farmObjectBlockY = function() {
  if (this._isBigCharacter || this._isSingleImage || this._isCharacterRow) {
    return 0;
  } else {
    var index;

    index = this._farmObject.characterIndex();
    return Math.floor(index / 4) * 4;
  }
};

Sprite_Object.prototype.farmObjectPattern = function() {
  return this._farmObject.pattern();
};

Sprite_Object.prototype.farmObjectDirection = function() {
  return this._farmObject.direction();
};

Sprite_Object.prototype.farmObjectPatternX = function() {
  return this.farmObjectPattern();
};

Sprite_Object.prototype.direction = function() {
  if (this.cache.direction !== undefined) {
    return this.cache.direction;
  }

  this.cache.direction = this.farmObjectDirection();
  return this.cache.direction;
};

Sprite_Object.prototype.farmObjectPatternY = function() {
  if (this.cache.patternY !== undefined) {
    return this.cache.patternY;
  }

  var direction = this.direction();
  this.cache.patternY = (direction - 2) / 2;

  return this.cache.patternY;
};

Sprite_Object.prototype.getBitmapWidth = function() {
  if (!this.longCache.bitmapWidth) {
    this.longCache.bitmapWidth = this.bitmap.width;
  }

  return this.longCache.bitmapWidth;
};

Sprite_Object.prototype.getBitmapHeight = function() {
  if (!this.longCache.bitmapHeight) {
    this.longCache.bitmapHeight = this.bitmap.height;
  }

  return this.longCache.bitmapHeight;
};

Sprite_Object.prototype.patternWidth = function() {
  if (this.longCache.patternWidth !== undefined) {
    return this.longCache.patternWidth;
  }

  this.longCache.patternWidth = (function(){
    if (this._isSingleImage) {
      return this.getBitmapWidth();
    }

    if (this._tileId > 0) {
      return Constants.TILE_SIZE;
    }

    if (this._isBigCharacter || this._isCharacterRow) {
      return this.getBitmapWidth() / 3;
    }

    return this.getBitmapWidth() / 12;
  }).bind(this)();

  return this.longCache.patternWidth;
};

Sprite_Object.prototype.patternHeight = function() {
  if (this.longCache.patternHeight !== undefined) {
    return this.longCache.patternHeight;
  }

  this.longCache.patternHeight = (function(){
    if (this._isSingleImage || this._isCharacterRow) {
      return this.getBitmapHeight();
    }

    if (this._tileId > 0) {
      return $gameMap.tileHeight();
    } else if (this._isBigCharacter) {
      return this.getBitmapHeight() / 4;
    } else {
      return this.getBitmapHeight() / 8;
    }
  }).bind(this)();

  return this.longCache.patternHeight;
};

Sprite_Object.prototype.getFarmObjectPriorityType = function() {
  return this._farmObject._priorityType;
};

Sprite_Object.prototype.getAdditionalOffsetX = function() {
  return 0;
};

Sprite_Object.prototype.getAdditionalOffsetY = function() {
  return 0;
};

Sprite_Object.prototype.getFarmObjectScreenZ = function() {
  return this._farmObject.screenZ();
};

Sprite_Object.prototype.updatePosition = function(ignoreOptimizations) {
  var shouldUpdatePosition = true;
  var shouldUseRelativePosition = false;

  if (!ignoreOptimizations) {
    if (this._oldY == this._farmObject._y && this._oldX == this._farmObject._x) {
      if (this._oldScreenX == $gameMap._displayX && this._oldScreenY == $gameMap._displayY) {
        shouldUpdatePosition = false;
      } else {
        // shouldUseRelativePosition = true;
      }
    }
  }

  if (shouldUpdatePosition) {
    this._oldScreenX = $gameMap._displayX;
    this._oldScreenY = $gameMap._displayY;

    if (shouldUseRelativePosition && this._relativeX !== undefined && this._relativeY !== undefined) {
      this.x = this._relativeX - this._oldScreenX;
      this.y = this._relativeY - this._oldScreenY;
    } else {
      this.x = this._farmObject.screenX() + this.getAdditionalOffsetX();
      this.y = this._farmObject.screenY() + this.getAdditionalOffsetY();

      this._oldY = this._farmObject._y;
      this._oldX = this._farmObject._x;
    }

    this._relativeX = this._oldX + this._oldScreenX;
    this._relativeY = this._oldY + this._oldScreenY;
  }

  if (this._oldPriority != this.getFarmObjectPriorityType()) {
    var newZ = this.getFarmObjectScreenZ();
    this.z = newZ;

    this._oldPriority = this.getFarmObjectPriorityType();
  }

  if (this.zIndex != this.z) {
    this.zIndex = this.z;
  }
  if (this.zOrder != -this.y) {
    this.zOrder = -this.y;
  }
};

Sprite_Object.prototype.updateAnimation = function() {
  this.setupAnimation();
  if (!this.isAnimationPlaying()) {
    this._farmObject.endAnimation();
  }
  if (!this.isBalloonPlaying()) {
    this._farmObject.endBalloon();
  }
};

Sprite_Object.prototype.setupAnimation = function() {
  if (!this._farmObject) return;

  if (this._farmObject.animationId() > 0) {
    var animation = $dataAnimations[this._farmObject.animationId()];
    this.startAnimation(animation, false, 0);
    this._farmObject.startAnimation();
  }
};

Sprite_Object.prototype.canShowOutline = function(outlineData) {
  return true;
};

Sprite_Object.prototype.addOutlineFilter = function(outlineData) {
  const filter = this.pickDesiredOutlineFilter(outlineData);
  if (filter) {
    this.filters = [filter];
  }
};

Sprite_Object.prototype.pickDesiredOutlineFilter = function(outlineData) {
  if (!outlineData) return null;
  if (!this.canShowOutline(outlineData)) return null;

  var mapMonth = Managers.Map.mapMonth($gameMap._mapId);

  var validFilter = mapMonth == Seasons.WINTER ? Graphics.winterOutlineFilter : Graphics.outlineFilter;
  return outlineData.valid ? validFilter : Graphics.invalidOutlineFilter;
};

Sprite_Object.prototype.updateShader = function() {
  if (!Graphics.isWebGL()) {
    return;
  }

  if (this._farmObject._erased) {
    this.filters = [];
    return;
  }

  const outlineData = this._farmObject.getOutlineData();
  const outlineFilter = this.pickDesiredOutlineFilter(outlineData);

  const filters = [];
  if (outlineFilter) {
    filters.push(outlineFilter);
  }

  // If it's not set or the length is different, set the new list
  if (!this.filters || (!filters.length && this.filters.length) || (!this.filters.length && filters.length)) {
    this.filters = filters;
    return;
  }

  // Nothing to change
  if (!filters.length) {
    return;
  }
  
  if (this.filters[0] != filters[0]) {
    this.filters = filters;
  }
};

Sprite_Object.prototype.onImageChanged = function() {
  if (this._report) {
    console.log('Clearing cache');
  }
  this.longCache = {};
  this.cache = {};

  if (this._bitmap && !this._bitmap.isReady()) {
    this._bitmap.addLoadListener(function(){
      this.longCache = {};
      this.cache = {};
    }.bind(this));
  }
};

Sprite_Object.prototype.setupBalloon = function() {
  if (this._farmObject.balloonId() > 0) {
    this.startBalloon();
    this._farmObject.startBalloon();
  }
};

Sprite_Object.prototype.startBalloon = function() {
  if (!this._balloonSprite) {
    this._balloonSprite = new Sprite_Balloon();
  }

  this._balloonOffset = this._farmObject.getBalloonOffset();
  this._balloonSprite.setup(this._farmObject.balloonId());
  this.parent.addChild(this._balloonSprite);
};

Sprite_Object.prototype.updateBalloon = function() {
  this.setupBalloon();

  if (this._balloonSprite) {
    this._balloonSprite.x = this.x;
    this._balloonSprite.y = this.y - this.height + this._balloonOffset;
    if (!this._balloonSprite.isPlaying()) {
      this.endBalloon();
    }
  }
};

Sprite_Object.prototype.endBalloon = function() {
  if (this._balloonSprite) {
    this.parent.removeChild(this._balloonSprite);
    this._balloonSprite = null;
  }
};

Sprite_Object.prototype.isBalloonPlaying = function() {
  return !!this._balloonSprite;
};

Sprite_Object.prototype.isErased = function() {
  if (this._farmObject) {
    return this._farmObject._erased;
  } else {
    return true;
  }
};
