require('./Sprite_Base');
//-----------------------------------------------------------------------------
// Sprite_Character
//
// The sprite for displaying a character.

function Sprite_Character() {
  this.initialize.apply(this, arguments);
}

Sprite_Character.prototype = Object.create(Sprite_Base.prototype);
Sprite_Character.prototype.constructor = Sprite_Character;

Sprite_Character.prototype.initialize = function(character) {
  Sprite_Base.prototype.initialize.call(this);
  this.initMembers();
  this.setCharacter(character);

  this._randomCacheIndex = Math.randomInt(10);
};

Sprite_Character.prototype.initMembers = function() {
  this.anchor.x = 0.5;
  this.anchor.y = 1;
  this._character = null;
  this._balloonDuration = 0;
  this._tilesetId = 0;

  this._frameWidth = undefined;
  this._frameHeight = undefined;
};

Sprite_Character.prototype.setCharacter = function(character) {
  this._character = character;
};

Sprite_Character.prototype.isNearTheScreen = function() {
  var sx = this.x;
  var sy = this.y;

  return $gameScreen.isPositionNearTheScreen(sx, sy);
};

Sprite_Character.prototype.shouldSkipUpdate = function() {
  if (Graphics.frameCount % 10 !== this._randomCacheIndex) {
    if (this.y > Graphics.height * 2 || this.y < (0 - Graphics.height)) {
      return true;
    }
    if (this.x > Graphics.width * 2 || this.x < (0 - Graphics.width)) {
      return true;
    }
  }

  return false;
};

Sprite_Character.prototype.update = function() {
  Sprite_Base.prototype.update.call(this);

  if (this._character && this._character._erased) {
    this.endBalloon();
    this.endItemBalloon();

    if (this.parent) {
      this.removeItself();
      this.destroy();
    }
    return;
  }

  if (this.shouldSkipUpdate()) {
    return;
  }

  this.updateBitmap();
  this.updateFrame();

  var alreadyUpdatedPosition = false;

  if (this._character && this._character._playerImpersonator) {
    if (Managers.Scenes._scene instanceof GameScenes.Map) {
      if (Managers.Scenes._scene._spriteset && Managers.Scenes._scene._spriteset._playerSprite) {
        var playerSprite = Managers.Scenes._scene._spriteset._playerSprite;
        this.x = playerSprite.x;
        this.y = playerSprite.y;
        this.z = playerSprite.z + (this._character._zVariation || 0);
        this.zIndex = playerSprite.zIndex + (this._character._zVariation || 0);
        this.zOrder = playerSprite.zOrder;
        alreadyUpdatedPosition = true;
      }
    }
  }

  if (!alreadyUpdatedPosition) {
    this.updatePosition();
  }

  this.updateAnimation();
  this.updateBalloon();
  this.updateItemBalloon();
  this.updateOther();
};

Sprite_Character.prototype.isTile = function() {
  return this._character.tileId > 0;
};

Sprite_Character.prototype.tilesetBitmap = function(tileId) {
  var tileset = $gameMap.tileset();
  var setNumber = 5 + Math.floor(tileId / 256);
  return Managers.Images.loadTileset(tileset.tilesetNames[setNumber]);
};

Sprite_Character.prototype.onImageChanged = function() {
  if (!this.bitmap) {
    return;
  }

  this.bitmap.addLoadListener(() => {
    this._frameWidth = this.patternWidth();
    this._frameHeight = this.patternHeight();
  });
};

Sprite_Character.prototype.updateBitmap = function() {
  if (this.isImageChanged()) {
    this._tilesetId = $gameMap.tilesetId();
    this._tileId = this._character.tileId();
    this._iconIndex = this._character.iconIndex();
    this._characterName = this._character.characterName();
    this._characterIndex = this._character.characterIndex();
    this._toolId = this._character.toolId();
    this._toolState = this._character.toolState();

    if (this._iconIndex > 0) {
      this.setIconBitmap();
    } else if (this._toolId !== undefined && this._toolId !== 'none' && this._characterName !== '') {
      this.setToolBitmap();
    } else if (this._tileId > 0) {
      this.setTileBitmap();
    } else {
      this.setCharacterBitmap();
    }

    this.onImageChanged();
  }
};

Sprite_Character.prototype.isImageChanged = function() {
  return (this._tilesetId !== $gameMap.tilesetId() ||
    this._tileId !== this._character.tileId() ||
    this._characterName !== this._character.characterName() ||
    this._characterIndex !== this._character.characterIndex() ||
    this._toolId !== this._character.toolId() ||
    this._toolState !== this._character.toolState() ||
    this._iconIndex !== this._character.iconIndex());
};

Sprite_Character.prototype.setIconBitmap = function() {
  this.bitmap = Managers.Images.loadIcon(this._iconIndex);
};

Sprite_Character.prototype.setTileBitmap = function() {
  this.bitmap = this.tilesetBitmap(this._tileId);
};

Sprite_Character.prototype.updateFrameCount = function() {
  if (!this._characterName) {
    this._characterFrameCount = undefined;
    return;
  }

  var regex = /\[(\d*)\]/;
  var match = this._characterName.match(regex);
  if (match && match.length >= 2) {
    this._characterFrameCount = parseInt(match[1]);
  } else {
    this._characterFrameCount = undefined;
  }
};

Sprite_Character.prototype.setCharacterBitmap = function() {
  this.bitmap = Managers.Images.loadCharacter(this._characterName);
  this._isBigCharacter = Managers.Images.isBigCharacter(this._characterName);
  this._isCharacterRow = Managers.Images.isCharacterRow(this._characterName);

  if (Managers.Images.isSingleImage(this._characterName)) {
    this._isSingleImage = true;
  } else if (this.character instanceof Objects.Event) {
    this._isSingleImage = this.character.singleImage;
  } else {
    this._isSingleImage = false;
  }

  this._isToolBitmap = false;
  this.updateFrameCount();
};

Sprite_Character.prototype.setToolBitmap = function() {
  var toolData = Managers.Tools.getTool(this._toolId);
  if (toolData === undefined || toolData.spriteName === '') {
    this.setCharacterBitmap();
    return;
  }

  var toolName = toolData.getSpriteName(this._toolState);
  var toolIndex = toolData.getSpriteIndex(this._toolState);

  var fileName = this._characterName + '_' + toolName;
  this.bitmap = Managers.Images.loadCharacter(fileName);
  this._toolIndex = toolIndex;
  this._isBigCharacter = Managers.Images.isBigCharacter(fileName);
  this._isCharacterRow = false;
  this._isSingleImage = false;
  this._isToolBitmap = true;
};

Sprite_Character.prototype.updateFrame = function() {
  this._mirror = false;

  if (this._iconIndex > 0) {
    this.updateIconFrame();
  } else if ((this._toolId === undefined || this._toolId === 'none' ) && this._tileId > 0) {
    this.updateTileFrame();
  } else {
    this.updateCharacterFrame();
  }
};

Sprite_Character.prototype.updateIconFrame = function() {
};

Sprite_Character.prototype.updateTileFrame = function() {
  var pw = this._frameWidth || 0;
  var ph = this._frameHeight || 0;
  var sx = (Math.floor(this._tileId / 128) % 2 * 8 + this._tileId % 8) * pw;
  var sy = Math.floor(this._tileId % 256 / 8) % 16 * ph;
  this.setFrame(sx, sy, pw, ph);
};

Sprite_Character.prototype.updateCharacterFrame = function() {
  if ((!this._frameWidth || !this._frameHeight) && this.bitmap && this.bitmap.isReady()) {
    this._frameWidth = this.patternWidth();
    this._frameHeight = this.patternHeight();
  }

  var pw = this._frameWidth || 0;
  var ph = this._frameHeight || 0;
  var sx = (this.characterBlockX() + this.characterPatternX()) * pw;
  var sy = (this.characterBlockY() + this.characterPatternY()) * ph;
  this.setFrame(sx, sy, pw, ph);
};

Sprite_Character.prototype.characterBlockX = function() {
  if (this._isBigCharacter || this._isSingleImage || this._isCharacterRow) {
    return 0;
  } else {
    var index;

    if (this._toolId !== undefined && this._toolId !== 'none' && this._isToolBitmap) {
      index = this._toolIndex;
      return index % 4 * 4;
    } else {
      var frameCount = this._characterFrameCount || 3;
      index = this._character.characterIndex();
      return index % 4 * frameCount;
    }
  }
};

Sprite_Character.prototype.characterBlockY = function() {
  if (this._isBigCharacter || this._isSingleImage || this._isCharacterRow) {
    return 0;
  } else {
    var index;

    if (this._toolId !== undefined && this._toolId !== 'none' && this._isToolBitmap) {
      index = this._toolIndex;
    } else {
      index = this._character.characterIndex();
    }

    return Math.floor(index / 4) * 4;
  }
};

Sprite_Character.prototype.pattern = function() {
  return this._character.pattern();
};

Sprite_Character.prototype.direction = function() {
  return this._character.direction();
};

Sprite_Character.prototype.characterPatternX = function() {
  if (this._toolId !== undefined && this._toolId !== 'none' && this._isToolBitmap) {
    switch (this.direction()) {
      case Direction.DOWN :
        return 0;
      case Direction.UP :
        return 1;
      case Direction.LEFT :
        return 2;
      case Direction.RIGHT :
        return 3;
      default :
        return 0;
    }
  }

  return this.pattern();
};

Sprite_Character.prototype.characterPatternY = function() {
  var direction = this.direction();
  if (this._toolId !== undefined && this._toolId !== 'none') {
    direction = this._character.toolDirection();
  }

  return (direction - 2) / 2;
};

Sprite_Character.prototype.patternWidth = function() {
  if (this._isSingleImage) {
    return this.bitmap.width;
  }

  if (this._characterFrameCount) {
    if (this._isBigCharacter || this._isCharacterRow) {
      return this.bitmap.width / this._characterFrameCount;
    }

    return this.bitmap.width / (this._characterFrameCount * 4);
  }

  if (this._toolId !== undefined && this._toolId !== 'none' && this._isToolBitmap) {
    if (this._isBigCharacter) {
      return this.bitmap.width / 4;
    }

    return this.bitmap.width / 16;
  }

  if (this._tileId > 0) {
    return $gameMap.tileWidth();
  }

  if (this._isBigCharacter || this._isCharacterRow) {
    return this.bitmap.width / 3;
  }

  return this.bitmap.width / 12;
};

Sprite_Character.prototype.patternHeight = function() {
  if (this._isSingleImage || this._isCharacterRow) {
    return this.bitmap.height;
  }

  if (this._toolId === undefined && this._toolId !== 'none' && this._tileId > 0) {
    return $gameMap.tileHeight();
  } else if (this._isBigCharacter) {
    return this.bitmap.height / 4;
  } else {
    return this.bitmap.height / 8;
  }
};

Sprite_Character.prototype.toolYOffset = function() {
  return 0;
};

Sprite_Character.prototype.updatePosition = function() {
  var oldX = this.x;
  var oldY = this.y;

  var newX = this._character.screenX();
  var newY = this._character.screenY();
  var extraY = 0;

  if (this._iconIndex <= 0) {
    if (this._toolId !== undefined && this._toolId !== 'none') {
      var toolData = Managers.Tools.getTool(this._toolId);
      var toolName = toolData.getSpriteName(this._toolState);
      if (toolName) {
        extraY += this.toolYOffset();
      }
    }
  }

  this.x = newX;
  this.y = newY + extraY;

  if (this.x !== oldX || this.y !== oldY) {
    $gameTemp.requestFarmObjectsUpdate();
  }

  this.z = this._character.screenZ();

  this.zIndex = this.z;
  this.zOrder = -newY;
};

Sprite_Character.prototype.updateAnimation = function() {
  this.setupAnimation();
  if (!this.isAnimationPlaying()) {
    this._character.endAnimation();
  }

  if (!this.isBalloonPlaying()) {
    this._character.endBalloon();
  }

  if (!this._character._itemIconBalloon) {
    this.endItemBalloon();
  }
};

Sprite_Character.prototype.setupAnimation = function() {
  if (this._character.animationId() > 0) {
    var animation = $dataAnimations[this._character.animationId()];
    this.startAnimation(animation, false, 0);
    this._character.startAnimation();
  }
};

Sprite_Character.prototype.updateOther = function() {
  if (this._character._opacity != this.opacity) {
    this.opacity = this._character._opacity;
  }

  if (this._character._blendMode != this.blendMode) {
    this.blendMode = this._character._blendMode;
  }
};

Sprite_Character.prototype.setupBalloon = function() {
  if (this._character.asleep && !this._character._sleepDelay) {
    if (!this._balloonSprite) {
      this.startBalloon();
    }
    return;
  }

  if (this._character.balloonId() > 0) {
    this.startBalloon();
    this._character.startBalloon();
    return;
  }
};

Sprite_Character.prototype.setupItemBalloon = function() {
  if (!this._character) {
    return;
  }

  if (this._character._itemIconBalloon && !this.isItemBalloonPlaying()) {
    this.startItemBalloon();
    return;
  }
};

Sprite_Character.prototype.startBalloon = function() {
  if (!this._balloonSprite) {
    this._balloonSprite = new Sprite_Balloon();
  }

  // if (this._character.asleep && !this._character._sleepDelay) {
  //   // this._balloonSprite.setup(Balloons.SLEEP, true);
  // } else {
  this._balloonSprite.setup(this._character.balloonId());
  // }

  this.parent.addChild(this._balloonSprite);
};

Sprite_Character.prototype.startItemBalloon = function() {
  if (!this._itemBalloonSprite) {
    this._itemBalloonSprite = new Sprites.ItemBalloon();
  }

  this._itemBalloonSprite.setup(this._character._itemIconBalloon);

  this.parent.addChild(this._itemBalloonSprite);
};

Sprite_Character.prototype.updateBalloonPosition = function() {
  let height = this.height;

  if (height < 1) {
    height = 48;
  }

  this._balloonSprite.x = this.x;
  this._balloonSprite.y = this.y - height;

  if (this._balloonSprite._balloonId == Balloons.DROP) {
    this._balloonSprite.y += 25;
    this._balloonSprite.x -= 2;
  }
};

Sprite_Character.prototype.updateBalloon = function() {
  this.setupBalloon();

  if (this._balloonSprite) {
    if (this._balloonSprite.loop && !this._character.asleep) {
      this.endBalloon();
      return;
    }

    this.updateBalloonPosition();

    if (!this._balloonSprite.isPlaying()) {
      this.endBalloon();
    }
  }
};

Sprite_Character.prototype.updateItemBalloon = function() {
  this.setupItemBalloon();

  if (this._itemBalloonSprite) {
    if (this._character.erased || !this._character._itemIconBalloon) {
      this.endItemBalloon();
      return;
    }

    this._itemBalloonSprite.x = this.x;
    this._itemBalloonSprite.y = this.y - this.height;
  }
};

Sprite_Character.prototype.endBalloon = function() {
  if (this._balloonSprite) {
    this.parent.removeChild(this._balloonSprite);
    this._balloonSprite = null;
  }
};

Sprite_Character.prototype.endItemBalloon = function() {
  if (this._itemBalloonSprite) {
    this.parent.removeChild(this._itemBalloonSprite);
    this._itemBalloonSprite = null;
  }
};

Sprite_Character.prototype.isBalloonPlaying = function() {
  return !!this._balloonSprite;
};

Sprite_Character.prototype.isItemBalloonPlaying = function() {
  return !!this._itemBalloonSprite;
};

Sprite_Character.prototype.isErased = function() {
  if (this._character) {
    return this._character._erased;
  } else {
    return true;
  }
};

Sprite_Character.prototype._refresh = function() {
  if (this._character && this._character._erased) {
    this.visible = false;
    return;
  }
  Sprite_Base.prototype._refresh.call(this);
};