const Scene_Base = require('./Scene_Base');
const { Input } = require('engine/core/Input');
const { Inventory } = require('engine/managers/Inventory');
const { Data } = require('engine/managers/Data');

//-----------------------------------------------------------------------------
// Scene_Map
//
// The scene class of the map screen.

let uniqueTimeHudWindow = false;
let uniqueToolHudWindow = false;
let uniqueMessageWindow = false;
let uniqueScrollWindow = false;
let uniqueCutsceneEffect = false;

// #ToDo: move this to an inherited class 
const Window_DateChange = require('game/windows/Window_DateChange');
const Window_TimeHud = require('game/windows/Window_TimeHud');
const Window_ToolHud = require('game/windows/Window_ToolHud');
const Window_Message = require('../windows/Window_Message');
const Window_ScrollText = require('game/windows/Window_ScrollText');
const Window_Letter = require('game/windows/Window_Letter');
const FreeCanvas = require('../windows/FreeCanvas');

class Scene_Map extends Scene_Base {
  initialize() {
    super.initialize();
    this._waitCount = 0;
    this._touchCount = 0;
    this._goldWidth = 330;
    this._delayFrames = Utils.getFrameCount(4);
    this._mouseMoveDelay = 0;
    this._needsFarmObjectRefresh = false;
    this._startTime = new Date();
    this._playingDiaryAnimation = false;
    this._openingInventory = false;
    this._diaryAnimationStage = 0;
  }

  create() {
    super.create();
    this._transfer = $gamePlayer.isTransferring();
    this.menuCalling = false;
    var mapId = this._transfer ? $gamePlayer.newMapId() : $gameMap.mapId();
    if (!Managers.Map.mapExists(mapId)) {
      mapId = $gameSystem.currentHome();
      $gamePlayer.reserveTransfer(mapId, 15, 10, Direction.UP, 0, false, true);
    }
    Data.loadMapData(mapId);
    if ($gameSystem.isPlayerSleeping) {
      this._sleepCallback = false;
      this._sleepOldDate = Managers.Time.getDateTime();
      this._sleepNewDate = Managers.Time.getDateTime();
    }
    Managers.Scenes.clearStack();
    //If the gameMap object is fully loaded at this point, this means that the scene is starting with a preloaded map.
    //In this case, we need to refresh the farmObject events, to recreate the customTileId collection.
    this._needsFarmObjectRefresh = $gameMap.isMapFullyLoaded() && mapId == $gameMap.mapId();
    if (this._toolHudWindow) {
      this._toolHudWindow.requestUpdate();
    }
  }

  isReady() {
    if (!this._fullyLoaded && Data.isMapLoaded()) {
      this.continueLoading();
    }
    return this._fullyLoaded && super.isReady();
  }

  releaseImages() {
    const releasePictureFn = Managers.Images.releasePicture.bind(Managers.Images);

    for (let i = 2; i <= 13; i++) {
      releasePictureFn(`diary-animation/${ i }`);
    }
  }

  loadInASingleFrame() {
    return false;
  }

  addLoadOperation(name, method, delay) {
    if (!this._loadingOperations) {
      this._loadingOperations = {};
    }
    if (!this._loadingOperations[name]) {
      method.call(this);
      this.finishLoadOperation(name, delay);
      return true;
    }
    return false;
  }

  isLoadOperationFinished(name) {
    if (!this._loadingOperations) {
      this._loadingOperations = {};
    }
    return !!this._loadingOperations[name];
  }

  finishLoadOperation(name, delay) {
    if (!this._loadingOperations) {
      this._loadingOperations = {};
    }
    this._loadingOperations[name] = true;
    if (delay !== undefined) {
      this._loadDelay = Utils.getFrameCount(delay);
    }
  }

  loadOperationTransferStart() {
    Switches.waitingForTransfer = false;
    $gameTemp.clearDestination();
    if (Switches.finishedIntroEvent) {
      $gameSystem.abortCutscene();
    }
    $gamePlayer.setDirection($gamePlayer._newDirection);
  }

  loadOperationTransferCheckCutscenes() {
    $gameMap.checkCutscenes($gamePlayer._newMapId);
  }

  loadOperationTransferMapSetup() {
    if ($gamePlayer._newMapId !== $gameMap.mapId() || $gamePlayer._needsMapReload) {
      $gameSystem.registerMapChange();
      $gameMap.setup($gamePlayer._newMapId);
      $gamePlayer._needsMapReload = false;
    }
  }

  loadOperationTransferPostSetup() {
    $gamePlayer.locate($gamePlayer._newX, $gamePlayer._newY);
    $gamePlayer.refresh();
    $gamePlayer.clearTransferInfo();
  }

  loadOperationTransferCheckAutoSave() {
    if (!$gameTemp.isCutsceneReserved()) {
      $gamePlayer.checkAutoSave();
    }
  }

  loadOperationTransferPerformAutoSave() {
    if (!this.isLoadOperationFinished('autosave-start')) {
      if (!$gameTemp.isAutoSaveRequested()) {
        this.finishLoadOperation('autosave');
        return;
      }
    }
    function prepareToSave() {
      Data.prepareToSave();
    }
    function makeSaveContents() {
      this.autosave_contents = Data.makeSaveContents();
    }
    function stringify() {
      this.autosave_json = JsonEx.stringify(this.autosave_contents);
    }
    function performAutoSave() {
      const result = Managers.Storage.saveFileToGameFolder('auto.ffsave', this.autosave_json, false, (err) => {
        if (err) {
          this.finishLoadOperation('autosave');
          return;
        }

        this.finishLoadOperation('autosave-waiting');

        const folderPath = Managers.Storage.directoryPath(true);
        const saveFilePath = Wrapper.joinFileNames(folderPath, 'auto.ffsave');
        Data.cacheSaveData(saveFilePath, this.autosave_contents);
      });

      if (result && result.error) {
        this.finishLoadOperation('autosave');
      }
    }
    function updateGlobalInfo() {
      Data.updateGlobalInfo('auto');
    }
    if (this.addLoadOperation('autosave-start', prepareToSave, 30))
      return;
    try {
      if (this.addLoadOperation('autosave-make-contents', makeSaveContents, 30))
        return;
      if (this.addLoadOperation('autosave-stringify', stringify, 30))
        return;
      if (this.addLoadOperation('autosave-perform', performAutoSave, 30))
        return;
      if (!this.isLoadOperationFinished('autosave-waiting')) {
        return;
      }
      if (this.addLoadOperation('autosave-update-global-info', updateGlobalInfo, 30))
        return;
    }
    catch (e) {
      console.log(...log(e, 'autoSave'));
    }
    if (this.autosave_contents)
      delete this.autosave_contents;
    if (this.autosave_json)
      delete this.autosave_json;
    $gameTemp.clearAutoSaveRequest();
    this.finishLoadOperation('autosave');
  }

  loadOperationTransferUpdateWeather() {
    Managers.Weather.updateWeather(1);
  }

  loadOperationTransferCheckDoor() {
    $gamePlayer.checkDoor();
  }

  loadOperationTransfer() {
    // if (this._loadDelay && this._loadDelay > 0) {
    //   this._loadDelay--;
    //   return;
    // }
    if (!this.isLoadOperationFinished('transfer-start')) {
      if (!this._transfer || !$gamePlayer.isTransferring()) {
        this.finishLoadOperation('transfer');
        return;
      }
    }

    if (this.addLoadOperation('transfer-start', this.loadOperationTransferStart)) {
      return;
    }

    if (this.addLoadOperation('transfer-check-cutscenes', this.loadOperationTransferCheckCutscenes)) {
      return;
    }

    if (this.addLoadOperation('transfer-map-setup', this.loadOperationTransferMapSetup)) {
      return;
    }

    if (this.addLoadOperation('transfer-post-setup', this.loadOperationTransferPostSetup)) {
      return;
    }

    if (this.addLoadOperation('transfer-check-autosave', this.loadOperationTransferCheckAutoSave)) {
      return;
    }

    if (!this.isLoadOperationFinished('autosave')) {
      this.loadOperationTransferPerformAutoSave();
      return;
    }

    if (this.addLoadOperation('transfer-update-weather', this.loadOperationTransferUpdateWeather)) {
      return;
    }

    if (this.addLoadOperation('transfer-check-door', this.loadOperationTransferCheckDoor)) {
      return;
    }
    this.finishLoadOperation('transfer');
  }

  continueLoading() {
    Managers.Scenes._forceNewFrame = true;
    if (!this.isLoadOperationFinished('transfer')) {
      this.loadOperationTransfer();
      return;
    }

    if (this.addLoadOperation('create-spriteset', this.createSpriteset)) {
      return;
    }

    if (this.addLoadOperation('create-mapname-window', this.createMapNameWindow)) {
      return;
    }

    if (this.addLoadOperation('create-window-layer', this.createWindowLayer)) {
      return;
    }

    if (this.addLoadOperation('create-toolhud', this.createToolHudWindow)) {
      return;
    }

    if (this.addLoadOperation('create-timehud', this.createTimeHudWindow)) {
      return;
    }

    if (this.addLoadOperation('create-message-window', this.createMessageWindow)) {
      return;
    }

    if (this.addLoadOperation('create-message-window-subwindows', this.createMessageWindowSubWindows)) {
      return;
    }

    if (this.addLoadOperation('update-message-window-subwindows', this.updateMessageWindowSubWindows)) {
      return;
    }

    if (this.addLoadOperation('create-scrolltext-window', this.createScrollTextWindow)) {
      return;
    }

    if (this.addLoadOperation('create-cutscene-effect', this.createCutsceneEffect)) {
      return;
    }

    if (this.addLoadOperation('create-extra-windows', this.createExtraWindows)) {
      return;
    }

    if (this.addLoadOperation('check-transfer', this.loadingProcessCheckTransfer)) {
      return;
    }

    if (this.addLoadOperation('reserve-player-sprites', function () { Managers.Player.preloadSprites(); })) {
      return;
    }

    if (this.addLoadOperation('reload-tileset', function () { this._spriteset.loadTileset(); })) {
      return;
    }

    if (this.addLoadOperation('request-farm-objects-refresh', function () { $gameTemp.requestFarmObjectsRefresh(); })) {
      return;
    }

    $gamePlayer.clearTransferInfo();
    this._fullyLoaded = true;
  }

  loadingProcessCheckTransfer() {
    if (this._transfer) {
      this.fadeInForTransfer();
      // this._mapNameWindow.open();
      $gameMap.autoplay();
    }
    else if (this.needsFadeIn()) {
      $gamePlayer.clearTransferInfo();
      this.startFadeIn(this.fadeSpeed(), false);
    }
    else {
      $gamePlayer.clearTransferInfo();
    }
  }

  start() {
    super.start();
    if (this._needsFarmObjectRefresh) {
      Managers.FarmObjects.refreshMapEvents();
      this._needsFarmObjectRefresh = false;
    }
    if (this._toolHudWindow) {
      this._toolHudWindow.requestUpdate();
    }
  }

  update() {
    this.updateDestination();
    this.updateMainMultiply();
    if (this.isSceneChangeOk()) {
      this.updateScene();
    }
    this.updateWaitCount();
    
    if (this._timeHudWindow !== undefined) {
      this._timeHudWindow.visible = Switches.forceTimeHud || (Switches.timeHudEnabled && !$gameSystem.isPlayingCutscene() && !$gameSystem.isPlayerSleeping && !Switches.isInsideFestival && !this._playingDiaryAnimation && !this.menuCalling);
    }

    super.update();
    
    if (this._toolHudWindow !== undefined) {
      this._toolHudWindow.visible = this._toolHudWindow.shouldBeVisible();
      this._toolHudWindow.update();
    }
    if (this._cutsceneEffect) {
      if (this.shouldShowCutsceneEffect()) {
        this._cutsceneEffect.show();
      }
      else {
        this._cutsceneEffect.hide();
      }
    }
  }

  shouldShowCutsceneEffect() {
    return $gameTemp.shouldShowCutsceneEffect();
  }

  updateMainMultiply() {
    if ($gameSystem.isPlayerSleepingStanding) {
      for (var i = 0; i < Constants.MAGIC_SLEEP_FACTOR; i++) {
        this.updateMain();
      }
      return;
    }

    $gameTemp.clearFastForwardFrame();
    this.updateMain();

    if (this.isFastForward()) {
      $gameTemp.setFastFowardFrame();
      this.updateMain();
      $gameTemp.clearFastForwardFrame();
    }
  }

  updateMain() {
    const active = this.isActive();
    if (this.menuCalling) {
      this.updateMapAndPlayer(active && !this.menuCalling);
      $gameScreen.update();
      return;
    }

    Managers.Time.update();

    if (this.hasLetter()) {
      if (this._letterDelay === undefined || this._letterDelay <= 0) {
        if (Input.isPressed('ok') || Input.isPressed('escape') || Input.isPressed('alternate') || TouchInput.isTriggered()) {
          this.clearLetter();
          Managers.Sound.playOk();
        }
      } else {
        this._letterDelay--;
      }
    } else {
      this.updateMapAndPlayer(active);
    }

    $gameScreen.update();
  }

  updateMapAndPlayer(active) {
    if ($gameSystem.isPlayerSleeping) {
      this.updateSleep();
    } else {
      if (Switches.isInsideFestival) {
        Managers.Festivals.update();
      }

      $gameMap.update(active);
      $gamePlayer.update(active);
      $gameTimer.update(active);
    }
  }

  startSleep(callback, oldDate, newDate, passedOut) {
    $gameScreen.startFadeOut(24);
    $gameSystem.isPlayerSleeping = true;
    $gameSystem.isPlayerPassedOut = !!passedOut;
    this._sleepCallback = callback;
    this._sleepOldDate = oldDate;
    this._sleepNewDate = newDate;
  }

  updateSleep() {
    if ($gameScreen._fadeOutDuration > 0)
      return;
    if ($gameScreen._fadeInDuration > 0)
      return;
    if (this._dateChangeWindow === undefined) {
      this._dateChangeWindow = new Window_DateChange();
      this.addWindow(this._dateChangeWindow);
      this._dateChangeWindow.hide();
    }
    if (!this._dateChangeWindow.visible) {
      var oldDateStr = Managers.Time.getDateDescription(this._sleepOldDate);
      var newDateStr = Managers.Time.getDateDescription(this._sleepNewDate);
      this._dateChangeWindow.setDates(oldDateStr, newDateStr, this._sleepCallback);
      this._dateChangeWindow.show();
    }
    this._dateChangeWindow.update();
    if (this._dateChangeWindow._isDone) {
      if ($gameTemp.canCloseDateChangeWindow()) {
        $gameScreen.startFadeIn(24);
        $gameSystem.isPlayerSleeping = false;
        $gameSystem.isPlayerPassedOut = false;
        this._dateChangeWindow.hide();
        this._timeHudWindow.update();
      }
    }
  }

  isFastForward() {
    if (!$gameMap.isEventRunning()) {
      return false;
    }

    if (Managers.Scenes.isSceneChanging()) {
      return false;
    }

    if (Input.isLongPressed('ok') || TouchInput.isLongPressed()) {
      return true;
    }

    return false;
  }

  stop() {
    super.stop();
    $gamePlayer.straighten();

    if (this.needsSlowFadeOut()) {
      this.startFadeOut(this.slowFadeSpeed(), false);
    }
    else if (Managers.Scenes.isNextScene(GameScenes.Map) || Managers.Scenes.isNextScene(GameScenes.GameMap)) {
      this.fadeOutForTransfer();
    }
  }

  isBusy() {
    return ((this._messageWindow && this._messageWindow.isClosing()) || this._waitCount > 0 || this._playingDiaryAnimation || super.isBusy());
  }

  terminate() {
    $gameTemp.clearTvChannel();
    super.terminate();
    Managers.Images.clearRequest();
    $gameScreen.clearZoom();

    if (this._fadeSprite) {
      this.removeChild(this._fadeSprite);
    }

    if (this._windowLayer) {
      this.removeChild(this._windowLayer);
    }

    if (this._spriteset) {
      this.removeChild(this._spriteset);
      this._spriteset.terminate();
    }

    this.releaseImages();
  }

  needsFadeIn() {
    return Managers.Scenes.isPreviousScene(GameScenes.LoadGame);
  }
  
  needsSlowFadeOut() {
    return (Managers.Scenes.isNextScene(GameScenes.Title));
  }
  
  updateWaitCount() {
    if (this._waitCount > 0) {
      this._waitCount--;
      return true;
    }
    return false;
  }
  
  updateDestination() {
    this.processMouseMove();
    if (this.isMapTouchOk()) {
      this.processMapTouch();
    } else {
      this._touchCount = 0;
    }
  }
  
  isMapTouchOk() {
    return this.isActive() && ($gamePlayer.canMove() || $gamePlayer._toolId !== undefined);
  }

  processHudTouch(touchX, touchY, button) {
    const canChangeItems = $gamePlayer.canChangeItems();
    
    let x;
    let y;
    const toolIconSize = this._toolHudWindow.getIconSize();
    const zoomLevel = Graphics.hudZoomLevel;
    y = this._toolHudWindow.toolbarTop() + 7;

    //If an icon from the toolbar was touched
    if (touchY >= y && touchY < y + toolIconSize) {
      x = this._toolHudWindow.toolbarLeft();
      x += 8 * zoomLevel;
      if (touchX >= x) {
        var tools = this._toolHudWindow.toolsToDraw();
        var toolIndex = Math.floor((touchX - x) / toolIconSize);
        if (this._toolHudWindow._includeEmptyTool) {
          toolIndex--;
        }

        if (toolIndex < 0) {
          if (canChangeItems) {
            Managers.Tools.storeTool();
          }
        } else if (toolIndex < tools.length) {
          var tool = tools[toolIndex];
          if (tool) {
            if (Inventory._specialItem) {
              return false;
            }

            if (TouchInput.isRightTriggered()) {
              if (Managers.Tools.useToolId(tool.id)) {
                return true;
              }
            }

            if (canChangeItems) {
              Inventory._itemIndex = -1;
              Managers.Tools.setTool(tool.id);
            }
          } else {
            if (canChangeItems) {
              Managers.Tools.storeTool();
            }
          }
        }
        return true;
      }
    }

    y = this._toolHudWindow.itembarTop();
    const iconsY = y + 7 * zoomLevel;
    //If an icon from the itembar was touched
    if (touchY >= y && touchY < iconsY + toolIconSize) {
      x = this._toolHudWindow.firstItemLeft();
      //Just a small correction
      x += 1 * zoomLevel;
      if (touchX >= x) {
        if (!canChangeItems) {
          return true;
        }

        const items = this._toolHudWindow.itemsToDraw();
        const itemIndex = Math.floor((touchX - x) / toolIconSize);

        if (itemIndex == Inventory._itemIndex || itemIndex == Inventory._itemIndex + 1) {
          const deselectY = iconsY - Core.BasicWindow._iconHeight * zoomLevel / 2;
          const deselectIconSize = Core.BasicWindow._iconWidth * zoomLevel;
          // account for the margin inside the icon itself
          const realDeselectY = deselectY + (Core.BasicWindow._iconHeight  / 5) * zoomLevel;
          const realDeselectSize = deselectIconSize - (Core.BasicWindow._iconHeight  / 5) * zoomLevel;

          if (touchY >= realDeselectY && touchY < realDeselectY + realDeselectSize) {
            const deselectX = x + (toolIconSize * (Inventory._itemIndex + 1)) - deselectIconSize / 2;

            if (touchX >= deselectX && touchX < deselectX + realDeselectSize) {
              Inventory._itemIndex = -1;
              Managers.Sound.playCursor();
              return true;
            }
          }
        }

        if (touchY >= iconsY) {
          if (itemIndex < items.length) {
            if (button == 2) {
              if (!$gamePlayer.isRiding()) {
                const item = items[itemIndex];
                if (item) {
                  Inventory.consumeItem(item);
                  $gamePlayer.genericDelay(20);
                  return true;
                }
              }
              return false;
            }
          }

          if (itemIndex < 10) {
            if (!Inventory._specialItem) {
              Inventory._itemIndex = itemIndex;
              if (Inventory.isHoldingItem()) {
                Managers.Sound.playPickItem();
              } else {
                Managers.Sound.playCursor();
              }
              return true;
            }
            return false;
          }

          //Check if the player clicked on the backpack index
          if (itemIndex == 10) {
            var minX = (toolIconSize * 10) + x;
            var maxX = minX + 8 * zoomLevel;

            if (touchX <= maxX && touchX >= minX) {
              var backpackY = touchY - iconsY;
              var backpackIndex = Math.floor(backpackY / (9 * zoomLevel));

              if (backpackIndex <= Inventory.maxItemBarIndex()) {
                Inventory._itemBarIndex = backpackIndex;
                Managers.Sound.playCursor();
                return true;
              }
            }
          }

        }
        return true;
      }
    }
    return false;
  }

  processMouseMove() {
    if (!Managers.Config.enableMouse) {
      return;
    }

    if (this._mouseMoveDelay > 0) {
      this._mouseMoveDelay--;
      return;
    }

    let className = '';
    let isDistant = false;

    if (!TouchInput.wasMouseUsed()) {
      return;
    }

    const mouseX = TouchInput.mouseX;
    const mouseY = TouchInput.mouseY;

    if (mouseY >= this._toolHudWindow.y + this._toolHudWindow.itembarTop()) {
      if (mouseX < this._toolHudWindow.itembarWidth() || mouseX >= this._toolHudWindow.toolbarLeft()) {
        document.body.className = className;
        this._mouseMoveDelay = Utils.getFrameCount(12);
        return;
      }
    }

    if (mouseY < this._timeHudWindow.height) {
      if (mouseY < this._timeHudWindow.staminaHeight()) {
        if (mouseX < this._timeHudWindow.staminaWidth()) {
          if (!this._timeHudWindow.isPlayerUnderHealth()) {
            document.body.className = className;
            this._mouseMoveDelay = Utils.getFrameCount(12);
            return;
          }
        }
      }

      if (mouseY < this._timeHudWindow.calendarHeight()) {
        if (mouseX > (Graphics.width - this._timeHudWindow.calendarWidth())) {
          if (!this._timeHudWindow.isPlayerUnderCalendar()) {
            document.body.className = className;
            this._mouseMoveDelay = Utils.getFrameCount(12);
            return;
          }
        }
      }
    }

    const x = $gameMap.canvasToMapX(mouseX, true);
    const y = $gameMap.canvasToMapY(mouseY, true);
    const events = $gameMap.anythingTouchableXy(x, y);
    let event;
    let lowPriorityEvent;

    for (let i = 0; i < events.length; i++) {
      if (!events[i]) {
        continue;
      }

      if (events[i]._erased) {
        continue;
      }

      if (events[i].hasAnythingToRun(true)) {
        event = events[i];
        break;
      }

      if (events[i].hasAnythingToRunOnLowPriority(true)) {
        lowPriorityEvent = events[i];
        continue;
      }
    }

    if (!event) {
      event = lowPriorityEvent;
    }

    if (event && event.shouldShowMouseIcon()) {
      className = 'look';
      const sprites = this._spriteset._characterSprites;

      for (let j = 0; j < sprites.length; j++) {
        if (!sprites[j])
          continue;
        if (sprites[j]._character !== event)
          continue;
        if (event instanceof Objects.Villager) {
          className = 'talk';
        }
        break;
      }

      const distance = event.distanceToPlayer();
      if (distance > event.maxActivationDistance()) {
        isDistant = true;
      }
    }

    if (isDistant && className) {
      className += '-invalid';
    }

    document.body.className = className;
    this._mouseMoveDelay = Utils.getFrameCount(12);
  }

  processMapTouch() {
    if (Managers.Config.enableMouse && TouchInput.wasMouseUsed()) {
      const mouseTileX = $gameMap.canvasToMapX(TouchInput.mouseX);
      const mouseTileY = $gameMap.canvasToMapY(TouchInput.mouseY);

      // If the mouse was moved to a different tile, refresh the tool hud
      if (mouseTileX != this._oldMouseTileX || mouseTileY !== this._oldMouseTileY) {
        this._oldMouseTileX = mouseTileX;
        this._oldMouseTileY = mouseTileY;

        if (this._toolHudWindow) {
          this._toolHudWindow.requestUpdate();
        }
      }
    }

    if (!TouchInput.isPressed() && !TouchInput.isRightPressed()) {
      this._touchCount = 0;
      return;
    }

    if (!Switches.isInsideFestival) {
      if (TouchInput.isTriggered()) {
        if (this.processHudTouch(TouchInput.x, TouchInput.y, 0)) {
          return;
        }
      }
      else if (TouchInput.isRightTriggered()) {
        if (this.processHudTouch(TouchInput.rightX, TouchInput.rightY, 2)) {
          return;
        }

        if (Inventory.isHoldingItem()) {
          if (Inventory.selectedItemCanBeUsedWithTheMouse()) {
            if (Inventory.useSelectedItem()) {
              $gamePlayer.genericDelay(5);
            }
          }

          return;
        }

        if (Managers.Tools.toolCanBeUsedAnywhere()) {
          if (Managers.Tools.useTool()) {
            return;
          }
        }

        return;
      }
    }
    $gameMap.processMapTouch();
    this._touchCount++;
  }
  
  isSceneChangeOk() {
    return this.isActive() && !$gameMessage.isBusy();
  }

  updateScene() {
    if (!Managers.Scenes.isSceneChanging()) {
      this.updateTransferPlayer();
    }

    if (!Managers.Scenes.isSceneChanging()) {
      this.updateCallMenu();
    }

    if ($gameTemp.isRequestingActionButton()) {
      Input._currentState.ok = true;
      $gameTemp.clearActionButtonRequest();
    } else if ($gameTemp.isRequestingActionButtonCancel()) {
      Input._currentState.ok = false;
      $gameTemp.clearActionButtonCancelRequest();
    }
  }

  createSpriteset() {
    this._spriteset = new Spriteset_Map();
    this.addChild(this._spriteset);
  }

  createExtraWindows() {
  }

  createMapNameWindow() {
  }

  createTimeHudWindow() {
    if (uniqueTimeHudWindow) {
      this._timeHudWindow = uniqueTimeHudWindow;
      this._timeHudWindow._dirty = true;
    }
    else {
      this._timeHudWindow = new Window_TimeHud();
      uniqueTimeHudWindow = this._timeHudWindow;
    }
    this._timeHudWindow.x = 0;
    this._timeHudWindow.y = 0;
    // this._timeHudWindow.opacity = 0;
    this._timeHudWindow.padding = 0;
    this._timeHudWindow.margin = 0;
    this._timeHudWindow.visible = Switches.forceTimeHud || (Switches.timeHudEnabled && !$gameSystem.isPlayingCutscene());
    this._timeHudWindow._drawnGoldValue = Inventory.gold;
    this.addWindow(this._timeHudWindow);
  }

  createToolHudWindow() {
    if (uniqueToolHudWindow) {
      this._toolHudWindow = uniqueToolHudWindow;
    }
    else {
      this._toolHudWindow = new Window_ToolHud();
      uniqueToolHudWindow = this._toolHudWindow;
    }
    this._toolHudWindow.x = 0;
    this._toolHudWindow.y = 0;
    this._toolHudWindow.opacity = 0;
    this._toolHudWindow.padding = 0;
    this._toolHudWindow.margin = 0;
    this._toolHudWindow.visible = this._toolHudWindow.shouldBeVisible();
    this.addWindow(this._toolHudWindow);
  }

  createMessageWindow() {
    if (uniqueMessageWindow) {
      this._messageWindow = uniqueMessageWindow;
    } else {
      this._messageWindow = new Window_Message();
      uniqueMessageWindow = this._messageWindow;
    }

    this.addWindow(this._messageWindow);
  }

  createMessageWindowSubWindows() {
    this._messageWindow.createSubWindows();
  }

  updateMessageWindowSubWindows() {
    this._messageWindow.updatePlacement();
    this._messageWindow.updateSubWindows();
  }
  
  createScrollTextWindow() {
    if (uniqueScrollWindow) {
      this._scrollTextWindow = uniqueScrollWindow;
    }
    else {
      this._scrollTextWindow = new Window_ScrollText();
      uniqueScrollWindow = this._scrollTextWindow;
    }
    this.addWindow(this._scrollTextWindow);
  }

  updateTransferPlayer() {
    if ($gamePlayer.isTransferring()) {
      Managers.Scenes.goToScene(Scene_Map);
    }
  }

  updateDiaryAnimation() {
    const frameCount = Utils.getFrameCount(6);

    const zoom = Graphics.windowZoomLevel;

    if (this._openingInventory) {
      if (this._diaryAnimationStage >= frameCount * 2) {
        this._diaryAnimationStage = frameCount * 14;
      }
    }

    switch(this._diaryAnimationStage) {
      case 0:
        if (!this._diaryAnimationSprite && !this._openingInventory) {
          this._diaryAnimationSprite = new Sprite();
        }

        Managers.Sound.playOk();
        Managers.Scenes.snapForBackground();
        $gameTemp.clearDestination();

        if (!this._openingInventory) {
          for (let i = 2; i <= 13; i++) {
            Managers.Images.loadPicture(`diary-animation/${ i.toString() }`);
          }
        }
        break;
      case frameCount:
        break;
      case frameCount * 2:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/2`);
        this._diaryAnimationSprite.scale.x = zoom;
        this._diaryAnimationSprite.scale.y = zoom;
        this.addChild(this._diaryAnimationSprite);
        break;
      case frameCount * 3:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/3`);
        break;
      case frameCount * 4:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/4`);
        break;
      case frameCount * 5:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/5`);
        break;
      case frameCount * 6:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/6`);
        break;
      case frameCount * 7:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/7`);
        break;
      case frameCount * 8:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/8`);
        break;
      case frameCount * 9:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/9`);
        break;
      case frameCount * 10:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/10`);
        break;
      case frameCount * 11:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/11`);
        break;
      case frameCount * 12:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/12`);
        break;
      case frameCount * 13:
        this._diaryAnimationSprite.bitmap = Managers.Images.loadPicture(`diary-animation/13`);
        break;
      case frameCount * 14:
        this._playingDiaryAnimation = false;
        Managers.Scenes.push(GameScenes.Diary);
        if (this._openingInventory) {
          Managers.Scenes.prepareNextScene('items');
          this._openingInventory = false;
        }
        this._waitCount = Utils.getFrameCount(2);
        break;
    }

    this._diaryAnimationStage++;
  }

  updateCallMenu() {
    if (this._playingDiaryAnimation) {
      this.updateDiaryAnimation();
      return;
    }

    if (!!this._menuCallDelay && this._menuCallDelay > 0) {
      this._menuCallDelay--;
      return;
    }

    if (this.isMenuEnabled()) {
      if (this.isMenuCalled()) {
        this.menuCalling = true;
      }

      if (this.menuCalling && !$gamePlayer.isMoving()) {
        this.callMenu();
        return;
      }

      if (Input.isPressed('map') && !Input.isPressed('shift')) {
        this.callMap();
        return;
      }
    } else {
      this.menuCalling = false;
    }
  }

  isMenuEnabled() {
    if (!$gameSystem.isMenuEnabled()) return false;
    if ($gameMap.isEventRunning()) return false;
    if (this.isBusy()) return false;
    if (this.hasLetter()) return false;

    if ($gamePlayer.isTransferring()) return false;
    if ($gameTemp.isCommonEventReserved()) return false;
    if ($gameTemp.hasAnyLockTimeout()) return false;

    if (Switches.isInsideFestivalCompetition) return false;

    return true;
  }

  isMenuCalled() {
    if (this.isBusy()) {
      return false;
    }

    if (this.hasLetter()) {
      return false;
    }

    if (!Managers.Content.canCheckInput()) {
      return false;
    }

    if ($gameSystem.isPlayerSleeping) {
      return false;
    }

    if (Input.isTriggered('menu')) {
      return true;
    }

    // Inventory button calls the menu as well, but without animations and straight to the inventory tab.
    if (Input.isTriggered('inventory')) {
      return true;
    }

    if (TouchInput.isCancelled()) {
      return true;
    }

    if (Managers.Steam.showingOvelay) {
      return true;
    }

    return false;
  }

  callMenu() {
    this._playingDiaryAnimation = true;
    this._diaryAnimationStage = 0;
    this._openingInventory = Input.isTriggered('inventory');
  }

  callMap() {
    Managers.Sound.playOk();
    Managers.Scenes.push(GameScenes.MapDisplay);
    $gameTemp.clearDestination();
    this._waitCount = Utils.getFrameCount(2);
  }

  fadeInForTransfer() {
    var fadeType = $gamePlayer.fadeType();
    switch (fadeType) {
      case 0:
      case 1:
        this.startFadeIn(this.fadeSpeed(), fadeType === 1);
        break;
      default:
        break;
    }
  }

  fadeOutForTransfer() {
    var fadeType = $gamePlayer.fadeType();

    switch (fadeType) {
      case 0:
      case 1:
        this.startFadeOut(this.fadeSpeed(), fadeType === 1);
        break;
      default:
        break;
    }
  }

  setupLetter(text) {
    if (this._letterWindow === undefined) {
      this._letterWindow = new Window_Letter();
      this.addWindow(this._letterWindow);
    }
    this._letterWindow.setup(text);
    this._letterDelay = Utils.getFrameCount(30);
  }

  clearLetter() {
    if (this._letterWindow !== undefined) {
      this._letterWindow.parent.removeChild(this._letterWindow);
      this._letterWindow = undefined;
    }
    this._waitCount = Utils.getFrameCount(20);
  }

  hasLetter() {
    return this._letterWindow !== undefined;
  }

  createCutsceneEffect() {
    var wx = 0;
    var wy = 0;
    var ww = Graphics.width;
    var wh = Graphics.height;
    if (uniqueCutsceneEffect) {
      this._cutsceneEffect = uniqueCutsceneEffect;
    }
    else {
      this._cutsceneEffect = new FreeCanvas(wx, wy, ww, wh);
      uniqueCutsceneEffect = this._cutsceneEffect;
    }
    this._cutsceneEffect.contents.fillRect(0, 0, ww, 50, 'black');
    this._cutsceneEffect.contents.fillRect(0, wh - 50, ww, 50, 'black');
    this.addChild(this._cutsceneEffect);
    if (this.shouldShowCutsceneEffect()) {
      this._cutsceneEffect.show();
    }
    else {
      this._cutsceneEffect.hide();
    }
  }

  refreshTilemap() {
    if (this._spriteset) {
      this._spriteset.refreshTilemap();
    }
  }
}

GameScenes.Map = Scene_Map;
module.exports = Scene_Map;