const Scene_Base = require('./Scene_Base');
const { Input } = require('engine/core/Input');
const { Utils } = require('engine/core/Utils');
const { Data } = require('engine/managers/Data');

const Window_GameBootTransition = require('../../game/windows/transitions/GameBoot');
const Window_EarlyAccessTransition = require('../../game/windows/transitions/EarlyAccess');

class Scene_Boot extends Scene_Base {
  initialize() {
    super.initialize();
    this._startDate = Date.now();
    this._animationDone = false;
    this._secondAnimationDone = false;
    this._modList = [];
    this._modIndex = 0;

    if (Utils._isAutoTestMode) {
      this._animationDone = true;
      this._secondAnimationDone = true;
      this._loadedLanguagePlugin = true;
      this._startedLoadingMods = true;
      this._finishedLoadingMods = true;
      this._preloadedSaves = true;
      this._registeredModImages = true;
      this._startedSyncingCloudSaves = true;

      require('nw.gui').Window.get().showDevTools();
    }
  }

  create() {
    super.create(...arguments);
    this.createForeground();

    window.bootLoadTime = new Date().getTime();
    Data.loadDatabase();
  }

  loadSystemWindowImage() {
    Managers.Images.requestSystem('DiarySkin');
  }

  static loadSystemImages() {
    Managers.Images.requestSystem('ItemIcons');
  }

  isGameFontLoaded() {
    if (Graphics.isFontLoaded('GameFont') && Graphics.isFontLoaded('ThinFont')) {
      return true;
    } else if (!Graphics.canUseCssFontLoading()){
      var elapsed = Date.now() - this._startDate;
      if (elapsed >= 60000) {
        throw new Error('Failed to load GameFont');
      }
    }
  }

  start() {
    super.start();
  }

  updateAnimation() {
    if (!this._transition) {
      return;
    }

    if (!this._transition.isDone()) {
      this._transition.update();
      return;
    }

    this._animationDone = true;
  }

  isAnimationDone() {
    return this._animationDone;
  }

  updateSecondAnimation() {
    if (!this._earlyAccessTransition) {
      return;
    }

    if (Utils.isOptionValid('test') && Input.isPressed('shift')) {
      this._secondAnimationDone = true;
      return;
    }

    if (Input.isTriggered('ok') || TouchInput.isTriggered()) {
      this._secondAnimationDone = true;
      return;
    }

    this._transition.hide();

    if (!this._earlyAccessTransition.isDone()) {
      this._earlyAccessTransition.update();
      return;
    }

    this._secondAnimationDone = true;
  }

  isSecondAnimationDone() {
    return this._secondAnimationDone;
  }

  changeBootStatus(newStatus) {
    this._bootStatus = newStatus;
    if (Utils._isAutoTestMode) {
      return;
    }

    if (newStatus) {
      console.log(...log(`[Boot] ${ newStatus }`));
    }

    this._transition.changeBootStatus(newStatus);
  }

  update() {
    if (!this.isAnimationDone()) {
      this.updateAnimation();
      return;
    }

    if (!Data.isDatabaseLoaded()) {
      return;
    }

    if (!this.isGameFontLoaded()) {
      return;
    }

    if (!this._checkedPlayerLocation) {
      this.checkPlayerLocation();
      this._checkedPlayerLocation = true;
      return;
    }

    if (!this._initializedSteam) {
      this.changeBootStatus('Detecting Steam');
      Managers.Steam.initialize();
      this._initializedSteam = true;
      return;
    }

    var me = this;

    // If dsv mode and holding control, skip loading language
    if (Utils.isOptionValid('test') && Input.isPressed('control')) {
      this._loadedLanguagePlugin = true;
    }

    if (!this._loadedLanguagePlugin) {
      this.changeBootStatus('Loading Language Plugin');

      setTimeout(() => {
        const language = Managers.Steam.getGameLanguage();

        if (language) {
          Managers.Text.changeLanguage(language);
        }
      }, 1);

      this._loadedLanguagePlugin = true;
      return;
    }

    if (!this._generatedEvents) {
      this.changeBootStatus('Loading Characters');
      this._generatedEvents = true;

      setTimeout(() => {
        Data.loadDataFile('$dataActors', 'Characters.json', () => {
          this.changeBootStatus('Compiling Events');

          Managers.Villagers.onLoadCharacters();
          Managers.EventScript.generateCommonEvents();
          me._finishedGeneratingEvents = true;
        });
      }, 1);
      return;
    }

    if (!this._finishedGeneratingEvents) {
      return;
    }

    if (!this._loadedPackages) {
      if (!Utils._isAutoTestMode) {
        console.log(...log('[ContentManager] Loading Internal Mods'));
      }

      setTimeout(() => {
        Managers.Packages.autoLoadPackageFiles();

        Managers.Content.onLoadModManager();
      }, 1);

      this._loadedPackages = true;
      return;
    }

    if (window.isDemo) {
      if (!this._loadedDemoPlugin) {
        OrangeSeason.Plugins.Demo.activate();
        this._loadedDemoPlugin = true;
        return;
      }

      this._finishedLoadingMods = true;
    } else {
      if (!this._startedLoadingMods) {
        this.changeBootStatus('Loading User Mod List');

        this._startedLoadingMods = true;
        setTimeout(() => {
          this._modList = Managers.Mods.getModList();
          if (this._modList.length === 0) {
            this._finishedLoadingMods = true;
          }
        }, 1);
        return;
      }
    }

    if (!this._finishedLoadingMods) {
      this.changeBootStatus('Loading User Mods');
      setTimeout(() => {
        this.loadNextMod();
      }, 1);
      return;
    }

    // if (!this._registeredModImages) {
    //   this.changeBootStatus('Registering Mod Images');
    //   setTimeout(() => {
    //     Managers.Mods.registerAllImages();
    //   }, 1);
    //   this._registeredModImages = true;
    //   return;
    // }

    if (!this._preloadedFiles) {
      this.changeBootStatus('Preloading Files');

      this._preloadedFiles = true;
      setTimeout(() => {
        Managers.Preload.preloadFiles(function(){
          me._finishedPreloadingFiles = true;
        }, 'boot');
      }, 1);

      return;
    }

    if (!this._loadedSystemWindowImage) {
      this.changeBootStatus('Loading System Images');

      setTimeout(() => {
        this.loadSystemWindowImage();
      }, 1);

      this._loadedSystemWindowImage = true;
      return;
    }

    if (!this._preloadedSounds) {
      this.changeBootStatus('Preloading Sounds');

      setTimeout(() => {
        Managers.Sound.preloadImportantSounds();
      }, 1);

      this._preloadedSounds = true;
      return;
    }

    if (!this._finishedPreloadingFiles) {
      return;
    }

    if (!this._madeConditionTable) {
      this.changeBootStatus('Compiling Condition Table');

      setTimeout(() => {
        Managers.CommonEvent.makeConditionTable();
      }, 1);

      this._madeConditionTable = true;
      return;
    }

    if (!this._newGameReady) {
      this.changeBootStatus('Preparing for New Game');

      setTimeout(() => {
        Data.setupNewGame();
      }, 1);
      this._newGameReady = true;
      return;
    }

    if (!this._updatedTitle) {
      this.changeBootStatus('Update Window Title');

      this.updateDocumentTitle();
      this._updatedTitle = true;
      return;
    }

    if (!this._preloadedSaves) {
      this.changeBootStatus('Preloading Save Files');

      setTimeout(() => {
        Data.maybeBuildSaveFileCache();
      }, 1);

      this._preloadedSaves = true;
      return;
    }

    if (!this._startedSyncingCloudSaves) {
      this.changeBootStatus('Syncing Cloud Saves');

      setTimeout(() => {
        Data.syncCloudSaves();
      }, 1);

      this._startedSyncingCloudSaves = true;
      return;
    }

    if (!this._registeredModels) {
      Managers.FarmObjects.autoRegisterAllModels();
      this._registeredModels = true;
    }

    if (!Data.finishedSyncingCloudSaves()) {
      return;
    }

    if (this._bootStatus) {
      this.changeBootStatus('Completed Boot Process');
      this.changeBootStatus(false);
    }

    if (!this.isSecondAnimationDone()) {
      this.updateSecondAnimation();
      return;
    }

    if (this._waitCount > 0) {
      this._waitCount--;
      return;
    }

    window.titleLoadTime = new Date().getTime();
    Graphics.adjustScreenSize(true);
    this._waitCount = Utils.getFrameCount(200);

    // If dsv and holding shift, auto load the latest save
    if (Utils.isOptionValid('test') && Input.isPressed('shift')) {
      Data.autoLoad('<latest>');
      return; 
    }

    Managers.Scenes.goToScene(Utils._isAutoTestMode ? GameScenes.AutoTests : GameScenes.Title);
  }

  loadNextMod() {
    if (this._modIndex >= this._modList.length) {
      this._finishedLoadingMods = true;
      return;
    }

    const modToLoad = this._modList[this._modIndex];
    this._modIndex++;

    this.changeBootStatus(`Loading Mod: ${ modToLoad }`);

    Managers.Mods.loadMod(modToLoad);
  }

  updateDocumentTitle() {
    document.title = $dataSystem.gameTitle;
  }

  checkPlayerLocation() {
    if ($dataSystem.startMapId === 0) {
      throw new Error('Player\'s starting position is not set');
    }
  }

  createForeground() {
    if (Utils._isAutoTestMode) {
      return;
    }

    this._transition = new Window_GameBootTransition(0, 0, Graphics.width, Graphics.height);
    this.addChild(this._transition);

    this._earlyAccessTransition = new Window_EarlyAccessTransition(0, 0, Graphics.width, Graphics.height);
    this.addChild(this._earlyAccessTransition);
  }
}

GameScenes.Boot = Scene_Boot;
module.exports = Scene_Boot;
