const Stage = require('../core/Stage');
const { Audio } = require('engine/managers/Audio');

//-----------------------------------------------------------------------------
// Scene_Base
//
// The superclass of all scenes within the game.
class Scene_Base extends Stage {
  initialize() {
    super.initialize();
    this._fullyLoaded = false;
    this._active = false;
    this._fadeSign = 0;
    this._fadeDuration = 0;
    this._fadeSprite = null;
  }

  create() {
  }

  prepare() {
  }

  isFullyLoaded() {
    return this._fullyLoaded;
  }

  loadInASingleFrame() {
    return true;
  }

  isActive() {
    return this._active;
  }

  isReady() {
    return Managers.Images.isReady();
  }

  start() {
    this._active = true;
    if (this.loadInASingleFrame()) {
      this._fullyLoaded = true;
    }
  }

  continueLoading() {
  }

  update() {
    this.updateFade();
    this.updateChildren();
  }

  stop() {
    this._active = false;
  }

  isBusy() {
    return this._fadeDuration > 0;
  }

  terminate() {
    if (this._windowLayer) {
      this._windowLayer.removeChildren();
    }
    this.removeChildren();
  }

  createWindowLayer() {
    var width = Graphics.width;
    var height = Graphics.height;
    var x = (Graphics.width - width) / 2;
    var y = (Graphics.height - height) / 2;
    this._windowLayer = new WindowLayer();
    this._windowLayer.move(x, y, width, height);
    this.addChild(this._windowLayer);
  }

  addWindow(childWindow) {
    this._windowLayer.addChild(childWindow);
  }

  startFadeIn(duration, white) {
    this.createFadeSprite(white);
    this._fadeSign = 1;
    this._fadeDuration = duration || 30;
    this._fadeSprite.opacity = 255;
  }

  startFadeOut(duration, white) {
    this.createFadeSprite(white);
    this._fadeSign = -1;
    this._fadeDuration = duration || 30;
    this._fadeSprite.opacity = 0;
  }

  createFadeSprite(white) {
    if (!this._fadeSprite) {
      this._fadeSprite = new ScreenSprite();
      this.addChild(this._fadeSprite);
    }
    if (white) {
      this._fadeSprite.setWhite();
    }
    else {
      this._fadeSprite.setBlack();
    }
  }

  updateFade() {
    if (this._fadeDuration > 0) {
      var d = this._fadeDuration;
      if (this._fadeSign > 0) {
        this._fadeSprite.opacity -= this._fadeSprite.opacity / d;
      }
      else {
        this._fadeSprite.opacity += (255 - this._fadeSprite.opacity) / d;
      }
      this._fadeDuration--;
    }
  }

  updateChildren() {
    this.children.forEach(function (child) {
      if (child.update) {
        child.update();
      }
    });
  }

  popScene() {
    Managers.Scenes.pop();
  }

  mustFadeOutBgm() {
    return true;
  }

  fadeOutAll() {
    var time = this.slowFadeSpeed() / 60;
    if (this.mustFadeOutBgm()) {
      Audio.fadeOutBgm(time);
    }
    Audio.fadeOutBgs(time);
    Audio.fadeOutMe(time);
    this.startFadeOut(this.slowFadeSpeed());
  }

  fadeSpeed() {
    return 24;
  }

  slowFadeSpeed() {
    return this.fadeSpeed() * 2;
  }
}

window.Scene_Base = Scene_Base;
module.exports = Scene_Base;
