const { ItemContainer } = require('engine/core/ItemContainer');

Objects.Temp = class Temp {
  constructor() {
    this.initialize(...arguments);
  }

  initialize() {
    this._commonEventList = [];
    this._asyncCommonEventList = [];
    this._reservedCutscene = undefined;
    this._dialogueList = null;
    this._destinationX = null;
    this._destinationY = null;
    this._shouldUpdateFarmObjects = true;
    this._shouldRefreshFarmObjects = true;
    this._shouldRefreshCropSoils = true;
    this._shouldUpdateCropSoils = true;
    this._playerDirectionPressed = null;
    this._requestingActionButton = false;
    this._currentTvChannel = false;
    this._quickStart = false;
    this._refreshAffectedTiles = false;
    this._isGameRunning = false;
    this._requestedAutoSave = false;
    this._autoSaveLoadedSlot = false;
    this._customTiles = {};
    this._skipSelectableWindowHandling = false;
    this._updateNotices = [];
    this._timeouts = [];
    this._loadedGamePath = false;
    this._realLoadedGamePath = false;
    this._namedObjectData = false;
    this._lastResponse = '';

    this._loadedTitleOnce = false;
    this._fishContainer = false;
  }

  isPlaytest() {
    return Utils._isTestMode;
  }

  isDsv() {
    return Utils.isOptionValid('test');
  }

  setFastFowardFrame() {
    this._isFastForwardFrame = true;
  }

  clearFastForwardFrame() {
    this._isFastForwardFrame = false;
  }

  isFastForwardFrame() {
    return Boolean(this._isFastForwardFrame);
  }

  reserveCommonEvent(commonEventId) {
    if (this.isCutsceneReserved()) return;

    if (commonEventId > 0) {
      this._commonEventList.push(commonEventId);
    }
  }

  reserveAsyncCommonEvent(commonEventId) {
    if (commonEventId > 0 && !this.isAsyncCommonEventReserved(commonEventId)) {
      this._asyncCommonEventList.push(commonEventId);

      if ($gameMap) {
        $gameMap.markCommonEventAsParallel(commonEventId);
        $gameMap.requestRefresh();
      }
    }
  }

  reserveCutscene(cutsceneName) {
    this._reservedCutscene = cutsceneName;
    // Clear all reserved common events when reserving a cutscene
    this._commonEventList = [];
  }

  reserveDialogue(dialogueList) {
    this._dialogueList = dialogueList;
  }

  clearCutscene() {
    this._reservedCutscene = undefined;
  }

  clearCustomTiles() {
    this._customTiles = {};
  }

  setCustomTile(x, y, tileId) {
    if (!this._customTiles[x]) {
      this._customTiles[x] = {};
    }

    this._customTiles[x][y] = tileId;
  }

  getCustomTile(x, y) {
    if (!this._customTiles[x]) return 0;
    if (!this._customTiles[x][y]) return 0;

    return this._customTiles[x][y];
  }

  clearCustomTile(x, y) {
    if (!this._customTiles[x]) return;
    delete this._customTiles[x][y];

    $gameMap.refreshTilemap();
  }

  clearCommonEvent() {
    this._commonEventList = [];
  }

  clearAsyncCommonEvent() {
    this._asyncCommonEventList = [];
  }

  clearDialogue() {
    this._dialogueList = null;
  }

  isCutsceneReserved() {
    return this._reservedCutscene !== undefined;
  }

  isCommonEventReserved() {
    return this._commonEventList.length > 0;
  }

  enableCutSceneEffect() {
    this._cutsceneEffect = true;
  }

  disableCutsceneEffect() {
    this._cutsceneEffect = false;
  }

  shouldShowCutsceneEffect() {
    return ($gameSystem.isPlayingCutscene() || $gameTemp.isCutsceneReserved()) && Boolean(this._cutsceneEffect);
  }

  isThisCommonEventReserved(eventId) {
    return this._commonEventList.includes(eventId);
  }

  isDialogueReserved() {
    return !!this._dialogueList;
  }

  reservedCutscene() {
    return this._reservedCutscene;
  }

  reservedCommonEvent() {
    const commonEventId = this._commonEventList.shift();

    return $dataCommonEvents[commonEventId];
  }

  isAsyncCommonEventReserved(eventId) {
    return this._asyncCommonEventList.length > 0 && this._asyncCommonEventList.includes(eventId);
  }

  removeAsyncCommonEventFromReservedList(eventId) {
    const index = this._asyncCommonEventList.indexOf(eventId);
    if (index >= 0) {
      this._asyncCommonEventList.splice(index, 1);
    }
  }

  clearReservedAsyncCommonEvents() {
    this._asyncCommonEventList = [];
  }

  reservedDialogue() {
    return this._dialogueList;
  }

  setDestination(x, y) {
    if ($gamePlayer._destinationCharacter === undefined) {
      this._destinationX = x;
      this._destinationY = y;
    }
  }

  clearDestination() {
    this._destinationX = null;
    this._destinationY = null;
  }

  isDestinationValid() {
    return this._destinationX !== null;
  }

  destinationX() {
    return this._destinationX;
  }

  destinationY() {
    return this._destinationY;
  }

  shouldRefreshCropSoils() {
    return this._shouldRefreshCropSoils;
  }

  shouldUpdateCropSoils() {
    return this._shouldRefreshCropSoils || this._shouldUpdateCropSoils;
  }

  shouldRefreshFarmObjects() {
    return this._shouldRefreshFarmObjects;
  }

  shouldUpdateFarmObjects() {
    return this._shouldUpdateFarmObjects || this._shouldRefreshFarmObjects;
  }

  markFarmObjectsAsUpdated() {
    this._shouldUpdateFarmObjects = false;
    this._shouldRefreshFarmObjects = false;
  }

  markCropSoilsAsUpdated() {
    this._shouldRefreshCropSoils = false;
    this._shouldUpdateCropSoils = false;
  }

  requestFarmObjectsRefresh() {
    this._shouldRefreshCropSoils = true;
    this._shouldRefreshFarmObjects = true;
  }

  requestFarmObjectsUpdate() {
    this._shouldUpdateFarmObjects = true;
    this._shouldUpdateCropSoils = true;
  }

  setPlayerDirectionPressed(direction) {
    this._playerDirectionPressed = direction;
  }

  getPlayerDirectionPressed() {
    return this._playerDirectionPressed;
  }

  clearPlayerDirectionPressed() {
    this._playerDirectionPressed = null;
  }

  requestActionButton() {
    this._requestingActionButton = true;
  }

  requestAutoSave() {
    this._requestedAutoSave = true;
  }

  clearAutoSaveRequest() {
    this._requestedAutoSave = false;
  }

  isAutoSaveRequested() {
    return this._requestedAutoSave;
  }

  isRequestingActionButton() {
    return this._requestingActionButton;
  }

  clearActionButtonRequest() {
    this._requestingActionButton = false;
    this._requestingActionButtonCancel = true;
  }

  isRequestingActionButtonCancel() {
    return this._requestingActionButtonCancel;
  }

  clearActionButtonCancelRequest() {
    this._requestingActionButtonCancel = false;
  }

  getCurrentTvChannel() {
    return this._currentTvChannel;
  }

  setCurrentTvChannel(tvChannel, tvEvent) {
    this._currentTvChannel = tvChannel;
    this._tvEvent = tvEvent;
  }

  clearTvChannel() {
    this.setCurrentTvChannel(false, false);
  }

  isPlayingTv() {
    return !!this._currentTvChannel;
  }

  enableQuickStart() {
    this._quickStart = true;
  }

  disableQuickStart() {
    this._quickStart = false;
  }

  isQuickStart() {
    return this._quickStart;
  }

  canCloseDateChangeWindow() {
    return true;
  }

  requestAffectedTilesRefresh() {
    this._refreshAffectedTiles = true;
  }

  shouldRefreshAffectedTiles() {
    return this._refreshAffectedTiles;
  }

  markAffectedTilesAsRefreshed() {
    this._refreshAffectedTiles = false;
  }

  requestProfile() {
    this._requestedProfile = true;
  }

  isProfileRequested() {
    return !!this._requestedProfile;
  }

  clearProfileRequest() {
    this._requestedProfile = false;
  }

  skipSelectableWindowHandling(value) {
    this._skipSelectableWindowHandling = value !== false;
  }

  shouldSkipSelectableWindowHandling() {
    return this._skipSelectableWindowHandling;
  }

  addUpdateNotice(text) {
    this._updateNotices.push(text);
  }

  hasUpdateNotice() {
    return this._updateNotices.length > 0;
  }

  getNextUpdateNotice() {
    return this._updateNotices.shift();
  }

  setTimeout(callback, frames, avoidLock) {
    if (!callback) return;
    if (!frames) {
      callback();
      return;
    }

    frames = Utils.getFrameCount(frames);

    this._timeouts.push({
      callback,
      frames,
      lock: !avoidLock
    });
  }

  updateTimeouts() {
    for (let i = 0; i < this._timeouts.length; i++) {
      const timeout = this._timeouts[i];

      if (timeout.frames > 0) {
        timeout.frames--;
      } else {
        const { callback } = timeout;
        this._timeouts.splice(i, 1);

        callback();
      }
    }
  }

  hasAnyLockTimeout() {
    for (let timeout of this._timeouts) {
      if (!timeout) continue;
      if (timeout.lock) return true;
    }

    return false;
  }

  hasAnyTimeout() {
    return this._timeouts.length > 0;
  }

  setLoadedGamePath(gamePath) {
    this._realLoadedGamePath = gamePath;

    if (gamePath) {
      const fileName = Utils.getFileNameFromPath(gamePath);
      if (['quick.ffsave', 'auto.ffsave'].indexOf(fileName) >= 0) {
        gamePath = null;
      }
    }

    this._loadedGamePath = gamePath;
  }

  getLoadedGamePath() {
    return this._loadedGamePath;
  }

  getRealLoadedGamePath() {
    return this._realLoadedGamePath;
  }

  requestAutoSaveLoadedSlot() {
    this._autoSaveLoadedSlot = true;
  }

  clearAutoSaveLoadedSlotRequest() {
    this._autoSaveLoadedSlot = false;
  }

  isAutoSaveLoadedSlotRequested() {
    return this._autoSaveLoadedSlot;
  }

  setNamedObject(event) {
    this._namedObjectData = {
      x: $gamePlayer.x,
      y: $gamePlayer.y,
      direction: $gamePlayer._direction,

      mouseUsed: TouchInput.wasMouseUsed(),
      mouseTileX: $gameMap.canvasToMapX(TouchInput.mouseX),
      mouseTileY: $gameMap.canvasToMapY(TouchInput.mouseY),

      displayX: event.screenX(),
      displayY: event.screenY() - $gameMap.tileWidth()
    };
  }

  isNamedObjectUpToDate() {
    const data = this._namedObjectData;
    if (!data) {
      return false;
    }

    if (Managers.Config.enableMouse) {
      if (TouchInput.wasMouseUsed() !== data.mouseUsed) {
        this.clearNamedObject();
        return false;
      }

      if (data.mouseUsed) {
        if (data.mouseTileX !== $gameMap.canvasToMapX(TouchInput.mouseX) || data.mouseTileY !== $gameMap.canvasToMapY(TouchInput.mouseY)) {
          this.clearNamedObject();
          return false;
        }
        
        return true;
      }
    }

    if (data.x !== $gamePlayer.x || data.y !== $gamePlayer.y || data.direction !== $gamePlayer._direction) {
      this.clearNamedObject();
      return false;
    }

    return true;
  }

  getNamedObject() {
    return this._namedObjectData;
  }

  clearNamedObject() {
    this._namedObjectData = false;
  }

  playAnimationStep(animation, event, index) {
    const data = animation[index];
    $gameTemp.setTimeout(() => {
      if ('pattern' in data) {
        event._pattern = data.pattern;
      }

      if ('direction' in data) {
        event._direction = data.direction;
      }

      if ('method' in data) {
        data.method();
      }

      if (index < animation.length -1) {
        $gameTemp.playAnimationStep(animation, event, index + 1);
      }
    }, data.frames);
  }

  setResponse(response) {
    this._lastResponse = response;
  }

  getLastResponse()  {
    return this._lastResponse;
  }

  startNewFishContainer() {
    this._fishContainer = new ItemContainer();
  }

  getFishContainer() {
    if (!this._fishContainer) {
      this.startNewFishContainer();
    }

    return this._fishContainer;
  }

  clearFishContainer() {
    this._fishContainer = false;
  }
};
