const { Inventory } = require('engine/managers/Inventory');
const { Audio } = require('engine/managers/Audio');

//-----------------------------------------------------------------------------
// Game_System
//
// The game object class for the system data.

Objects.System = class System {
  constructor() {
    this.initialize(...arguments);
  }

  get isPlayerSleeping() {
    return this._isPlayerSleeping;
  }
  set isPlayerSleeping(value) {
    this._isPlayerSleeping = value;
  }
  get isPlayerSleepingStanding() {
    return this._isPlayerSleepingStanding;
  }
  set isPlayerSleepingStanding(value) {
    this._isPlayerSleepingStanding = value;
  }
  get isPlayerPassedOut() {
    return this._isPlayerPassedOut;
  }
  set isPlayerPassedOut(value) {
    this._isPlayerPassedOut = value;
  }
  get isPlayingDialogue() {
    return this._isPlayingDialogue;
  }
  set isPlayingDialogue(value) {
    this._isPlayingDialogue = value;
  }
  get currentEvent() {
    return this._currentEvent;
  }
  set currentEvent(value) {
    this._currentEvent = value;
  }

  initialize() {
    this._saveEnabled = true;
    this._saveLocked = false;
    this._menuEnabled = true;
    this._saveCount = 0;
    this._versionId = 0;
    this._framesOnSave = 0;
    this._bgmOnSave = null;
    this._bgsOnSave = null;
    this._windowTone = null;
    this._savedBgm = null;
    this._walkingBgm = null;
    this._lastDialogues = {};
    this._dailyDialogueHistory = {};
    this._wordWrap = true;
    this._fastForward = false;
    this._playingCutscene = false;
    this._retainCutscene = false;
    this._letPlayerLeaveCutscene = false;
    this._isPlayerSleeping = false;
    this._isPlayerSleepingStanding = false;
    this._isPlayingDialogue = false;
    this._initializedMaps = {};
    this._dailyItemsGenerated = {};
    this._currentEvent = 0;
    this._extraData = {};
    this._fixedRoosterAchievement = false;
    this._usedMods = !!window.usedMods;
    // this._activeAchievements = {};

    this.initMessageSounds();
  }

  getDataForSaveFile() {
    return Object.assign({}, this);
  }

  initMessageSounds() {
    this._msgSoundEnable = true;
    this._msgSoundName = 'Message4';
    this._msgSoundVol = 5;
    this._msgSoundPitch = 100;
    this._msgSoundPitchVar = 10;
    this._msgSoundPan = 0;
    this._msgSoundPanVar = 10;
    this._msgSoundInterval = 1;
  }

  isJapanese() {
    return $dataSystem.locale.match(/^ja/);
  }

  isChinese() {
    return $dataSystem.locale.match(/^zh/);
  }

  isKorean() {
    return $dataSystem.locale.match(/^ko/);
  }

  isCJK() {
    return $dataSystem.locale.match(/^(ja|zh|ko)/);
  }

  isRussian() {
    return $dataSystem.locale.match(/^ru/);
  }

  isSideView() {
    return $dataSystem.optSideView;
  }

  //Save lock is used internally to lock the save functionality for a few frames.
  lockSave() {
    this._saveLocked = true;
  }

  unlockSave() {
    this._saveLocked = false;
  }

  //Disable save is used by the game to disable saving on certain areas or situations
  disableSave() {
    this._saveEnabled = false;
  }

  enableSave() {
    this._saveEnabled = true;
  }

  isSaveEnabled() {
    if (Switches.isInsideFestival) return false;
    if (this._playingCutscene) return false;
    if (this._saveLocked) return false;
    if (!this._saveEnabled) return false;

    return true;
  }

  isMenuEnabled() {
    return this._menuEnabled && !this._playingCutscene;
  }

  disableMenu() {
    this._menuEnabled = false;
  }

  enableMenu() {
    this._menuEnabled = true;
  }

  isFormationEnabled() {
    return this._formationEnabled;
  }

  saveCount() {
    return this._saveCount;
  }

  versionId() {
    return this._versionId;
  }

  windowTone() {
    return this._windowTone || $dataSystem.windowTone;
  }

  setWindowTone(value) {
    this._windowTone = value;
  }

  messageRows() {
    return 4;
  }

  messageWidth() {
    return Graphics.width / 2;
  }

  wordWrap() {
    return this._wordWrap;
  }

  setWordWrap(state) {
    this._wordWrap = state;
  }

  isFastFowardEnabled() {
    return this._fastForward;
  }

  setFastFoward(state) {
    this._fastForward = state;
  }

  clearCustomDataBeforeSave() {
    Variables.creatureEvent = null;
    Variables.creatureData = null;

    if (window.$gameMap && $gameMap._interpreter) {
      if ($gameMap._interpreter._waitingFor) {
        $gameMap._interpreter._waitingFor = null;
      }
    }
  }

  onBeforeSave() {
    this._saveCount++;
    this._versionId = $dataSystem.versionId;
    this._framesOnSave = Graphics.frameCount;
    this._bgmOnSave = Audio.saveBgm();
    this._bgsOnSave = Audio.saveBgs();

    this.clearCustomDataBeforeSave();
  }

  onAfterLoad() {
    this._lastDialogues = this._lastDialogues || {};
    this._dailyDialogueHistory = this._dailyDialogueHistory || {};
    if (window.usedMods) {
      this._usedMods = true;
    }
    Graphics.frameCount = this._framesOnSave;
    Audio.playBgm(this._bgmOnSave);
    Audio.playBgs(this._bgsOnSave);
    Switches.alwaysOn = true;

    Managers.Player.changeActorData();

    $gameTemp.requestFarmObjectsRefresh();
    Managers.Images.requestCurrentTilesets();
    Inventory.requestHudRefresh();
    Inventory.displayItemId = undefined;

    Managers.Festivals.clearSwitches();
    Managers.Festivals.updateSwitches();

    Managers.Achievements.checkAchievements();
  }

  playtime() {
    return Math.floor(Graphics.frameCount / 60);
  }

  playtimeText() {
    const hour = Math.floor(this.playtime() / 60 / 60);
    const min = Math.floor(this.playtime() / 60) % 60;
    const sec = this.playtime() % 60;
    return `${hour.padZero(2)}:${min.padZero(2)}:${sec.padZero(2)}`;
  }

  saveBgm() {
    this._savedBgm = Audio.saveBgm();
  }

  replayBgm() {
    if (this._savedBgm) {
      Audio.replayBgm(this._savedBgm);
    }
  }

  saveWalkingBgm() {
    this._walkingBgm = Audio.saveBgm();
  }

  replayWalkingBgm() {
    if (this._walkingBgm) {
      Audio.playBgm(this._walkingBgm);
    }
  }

  saveWalkingBgm2() {
    this._walkingBgm = $dataMap.bgm;
  }

  clearSelfSwitches(mapId, eventId) {
    const switches = ["A", "B", "C", "D"];

    switches.forEach(switchId => {
      const key = [mapId, eventId, switchId];
      $gameSelfSwitches.setValue(key, false);
    });
  }

  initCustomEvents(mapId) {
    if (this._customEvents === undefined) {
      this._customEvents = {};
    }

    if (this._customEvents[mapId] === undefined) {
      this._customEvents[mapId] = {};
    }
  }

  addCustomEvent(mapId, event) {
    this.initCustomEvents(mapId);
    this.clearSelfSwitches(mapId, event.id);
    this._customEvents[mapId][event.id] = event;

    return event;
  }

  removeCustomEvent(mapId, eventId) {
    this.initCustomEvents(mapId);
    this.clearSelfSwitches(mapId, eventId);
    delete this._customEvents[mapId][eventId];
  }

  clearCustomEvents(mapId) {
    this.initCustomEvents();
    this._customEvents[mapId] = {};
  }

  getCustomEvents(mapId) {
    this.initCustomEvents();
    return this._customEvents[mapId];
  }

  currentHome() {
    if (Switches.builtSecondFloor) {
      return Maps.HOME_3;
    }

    if (Switches.expandedHouse) {
      return Maps.HOME_2;
    }

    return Maps.HOME;
  }

  isCharacterKnown(characterName) {
    return Managers.Relationship.isCharacterKnown(characterName);
  }

  markCharacterAsKnown(characterName) {
    Managers.Relationship.markCharacterAsKnown(characterName);
  }

  getLastDialogue(characterName) {
    return this._lastDialogues[characterName];
  }

  setLastDialogue(characterName, dialogueName) {
    //Set last dialogue
    this._lastDialogues[characterName] = dialogueName;

    //Adds to history
    const history = this._dailyDialogueHistory[characterName] || [];
    const idx = history.indexOf(dialogueName);
    if (idx >= 0) {
      history.splice(idx, 1);
    }
    history.push(dialogueName);

    this._dailyDialogueHistory[characterName] = history;
  }

  getCharacterDialogueHistory(characterName) {
    return this._dailyDialogueHistory[characterName] || [];
  }

  overrideCharacterDialogueHistory(characterName, newHistory) {
    this._dailyDialogueHistory[characterName] = newHistory;
  }

  getCharacterFriendPoints(characterName) {
    return Managers.Relationship.getFriendship(characterName);
  }

  getCharacterHearts(characterName) {
    return Managers.Relationship.getHeartsNum(characterName);
  }

  //Remove the whole history of every character
  eraseDialogueHistories() {
    this._dailyDialogueHistory = {};

    for (const characterName in this._lastDialogues) {
      if (!this._lastDialogues.hasOwnProperty(characterName)) continue;

      // const lastDialogueName = this.getLastDialogue(characterName);
      // this._dailyDialogueHistory[characterName] = [lastDialogueName];

      this._dailyDialogueHistory[characterName] = [];
    }

    this._lastDialogues = {};
  }

  registerMapChange() {
    this._lastDialogues = [];
  }

  isPlayerDating() /*characterName*/{
    return false;
  }

  isPlayerMarriedTo() /*characterName*/{
    return false;
  }

  getMessageSound() {
    const obj = {
      name: this._msgSoundName,
      volume: this._msgSoundVol,
      pitch: this._msgSoundPitch,
      pan: this._msgSoundPan
    };

    let max = this._msgSoundPitch + this._msgSoundPitchVar;
    let min = this._msgSoundPitch - this._msgSoundPitchVar;
    obj.pitch += Math.floor(Math.random() * ( max - min + 1) + min);

    max = this._msgSoundPan + this._msgSoundPanVar;
    min = this._msgSoundPan - this._msgSoundPanVar;
    obj.pan += Math.floor(Math.random() * ( max - min + 1) + min);
    return obj;
  }

  isMessageSoundEnabled() {
    return this._msgSoundEnable;
  }

  messageSoundInterval() {
    return this._msgSoundInterval;
  }

  startGame() {
    this._currentEvent = 1;
  }

  startCutscene(killVillagers, hideCreatures) {
    if (killVillagers !== false) {
      Managers.Villagers.killAllVillagerEvents();
    }

    if (hideCreatures !== false) {
      $gameMap.hideCreatures();
    }

    this._playingCutscene = true;
    Switches.playedCutscene = true;
  }

  retainCutscene() {
    this._retainCutscene = true;
  }

  letPlayerLeaveCutscene() {
    this._letPlayerLeaveCutscene = true;
  }

  endCutscene() {
    if (this._retainCutscene) {
      return;
    }

    this._playingCutscene = false;

    Managers.Villagers.clearTimeFrozenFlag();
    Managers.Villagers.refreshMapVillagers();
    $gameMap.showHiddenCreatures();

    Managers.Achievements.checkAchievements();
    $gamePlayer.checkAutoSave();

    $gameMap.autoplay();
  }

  abortCutscene() {
    this._playingCutscene = false;
    this._retainCutscene = false;
    this._letPlayerLeaveCutscene = false;
    $gameTemp.disableCutsceneEffect();
    Switches.playedCutscene = false;
  }

  canTeleportNow() {
    if (this._playingCutscene) {
      if (!this._letPlayerLeaveCutscene) {
        return false;
      }
    }

    return true;
  }

  isPlayingCutscene() {
    return !!this._playingCutscene;
  }

  startSleep(callback, oldDate, newDate, passedOut) {
    if (Managers.Scenes._scene instanceof GameScenes.Map) {
      Managers.Scenes._scene.startSleep(callback, oldDate, newDate, passedOut);
    }
  }

  setupLetter(text) {
    if (Managers.Scenes._scene instanceof GameScenes.Map) {
      Managers.Scenes._scene.setupLetter(text);
    }
  }

  clearLetter() {
    if (Managers.Scenes._scene instanceof GameScenes.Map) {
      Managers.Scenes._scene.clearLetter();
    }
  }

  hasLetter() {
    if (Managers.Scenes._scene instanceof GameScenes.Map) {
      return Managers.Scenes._scene.hasLetter();
    }

    return false;
  }

  initializeMap(mapId) {
    this._initializedMaps[mapId] = true;
  }

  generatedDailyMapEvents(mapId) {
    if (this._dailyItemsGenerated === undefined) this._dailyItemsGenerated = {};

    this._dailyItemsGenerated[mapId] = true;
  }

  isMapInitialized(mapId) {
    return !!this._initializedMaps[mapId];
  }

  hasMapGeneratedDailyEvents(mapId) {
    if (this._dailyItemsGenerated === undefined) this._dailyItemsGenerated = {};
    return !!this._dailyItemsGenerated[mapId];
  }

  checkData() {
    if (!this._initializedMaps) {
      this._initializedMaps = {};
    }

    if (this._isPlayerSleepingStanding) {
      this.wakeUp();
    }
  }

  clearDailyStuff() {
    this._dailyItemsGenerated = {};
  }

  getCurrentSubjects() {
    const subjects = [];
    subjects.push(this._currentEvent);

    return subjects;
  }

  getExtraData(dataName) {
    if (!this._extraData) this._extraData = {};

    return this._extraData[dataName];
  }

  setExtraData(dataName, dataValue) {
    if (!this._extraData) this._extraData = {};
    this._extraData[dataName] = dataValue;
  }

  resetMap(mapId, forceErasure) {
    $gameSystem._initializedMaps[mapId] = false;
    $gameSystem._dailyItemsGenerated[mapId] = false;
    Managers.FarmObjects.clearMapObjects(mapId, forceErasure);
  }

  sleepStanding() {
    $gamePlayer.setSleepDelay(60);
    $gameSystem._isPlayerSleepingStanding = true;
    Managers.Time.refreshTimeSystem();
  }

  wakeUp() {
    $gamePlayer.clearToolData();
    $gameSystem._isPlayerSleepingStanding = false;
    Managers.Time.refreshTimeSystem();
  }

  runWhenAvailable(callback) {
    const intervalHandler = setInterval(() => {
      if (Managers.Time.paused) return;
      if (Managers.Time.isInternallyPaused()) return;
      if (!(Managers.Scenes._scene instanceof GameScenes.GameMap)) return;

      clearInterval(intervalHandler);
      callback();
    }, 50);
  }
};
