//-----------------------------------------------------------------------------
// Game_Screen
//
// The game object class for screen effect data, such as changes in color tone
// and flashes.

Objects.Screen = class Screen {
  constructor() {
    this.initialize(...arguments);
  }

  initialize() {
    this.clear();
  }

  clear() {
    this.clearFade();
    this.clearTone();
    this.clearFlash();
    this.clearShake();
    this.clearZoom();
    this.clearWeather();
    this.clearPictures();
  }

  getDataForSaveFile() {
    return Object.assign({}, this);
  }

  brightness() {
    return this._brightness;
  }

  tone() {
    return this._tone;
  }

  flashColor() {
    return this._flashColor;
  }

  shake() {
    return this._shake;
  }

  zoomX() {
    return this._zoomX;
  }

  zoomY() {
    return this._zoomY;
  }

  zoomScale() {
    return this._zoomScale;
  }

  weatherType() {
    return this._weatherType;
  }

  weatherPower() {
    return this._weatherPower;
  }

  picture(pictureId) {
    const realPictureId = this.realPictureId(pictureId);
    return this._pictures[realPictureId];
  }

  realPictureId(pictureId) {
    return pictureId;
  }

  clearFade() {
    this._brightness = 255;
    this._fadeOutDuration = 0;
    this._fadeInDuration = 0;
  }

  isFading() {
    return this._fadeOutDuration > 0 || this._fadeInDuration > 0 || this._brightness < 255;
  }

  clearTone() {
    this._tone = [0, 0, 0, 0];
    this._toneTarget = [0, 0, 0, 0];
    this._toneDuration = 0;
  }

  clearFlash() {
    this._flashColor = [0, 0, 0, 0];
    this._flashDuration = 0;
  }

  clearShake() {
    this._shakePower = 0;
    this._shakeSpeed = 0;
    this._shakeDuration = 0;
    this._shakeDirection = 1;
    this._shake = 0;
  }

  clearZoom() {
    this._zoomX = 0;
    this._zoomY = 0;
    this._zoomScale = 1;
    this._zoomScaleTarget = 1;
    this._zoomDuration = 0;
  }

  clearWeather() {
    this._weatherType = 'none';
    this._weatherPower = 0;
    this._weatherPowerTarget = 0;
    this._weatherDuration = 0;
    this._weatherWind = [0, 0];
    this._stormThunderSound = '';
    this._stormThunderFlash = 0;
    this._stormThunderCount = 4 * (Constants.Weather.ThunderWait + Constants.Weather.ThunderWaitVariation);
    this._stormThunderFrequency = 0;
    this._snowBloom = 0;
  }

  clearPictures() {
    this._pictures = [];
  }

  maxPictures() {
    return 100;
  }

  startFadeOut(duration) {
    this._fadeOutDuration = duration;
    this._fadeInDuration = 0;
  }

  startFadeIn(duration) {
    this._fadeInDuration = duration;
    this._fadeOutDuration = 0;
  }

  startTint(tone, duration) {
    this._toneTarget = tone.clone();
    this._toneDuration = duration;
    if (this._toneDuration === 0) {
      this._tone = this._toneTarget.clone();
    }
  }

  startFlash(color, duration) {
    this._flashColor = color.clone();
    this._flashDuration = duration;
  }

  startShake(power, speed, duration) {
    this._shakePower = power;
    this._shakeSpeed = speed;
    this._shakeDuration = duration;
  }

  startZoom(x, y, scale, duration) {
    this._zoomX = x;
    this._zoomY = y;
    this._zoomScaleTarget = scale;
    this._zoomDuration = duration;
  }

  setZoom(x, y, scale) {
    this._zoomX = x;
    this._zoomY = y;
    this._zoomScale = scale;
  }

  changeWeather(type, power, duration) {
    if (type !== 'none' || duration === 0) {
      this._weatherType = type;
    }
    this._weatherPowerTarget = type === 'none' ? 0 : power;
    this._weatherDuration = duration;
    if (duration === 0) {
      this._weatherPower = this._weatherPowerTarget;
    }
  }

  update() {
    this.updateFadeOut();
    this.updateFadeIn();
    this.updateTone();
    this.updateFlash();
    this.updateShake();
    this.updateZoom();
    this.updateWeather();
    this.updatePictures();
  }

  updateFadeOut() {
    if (this._fadeOutDuration > 0) {
      const d = this._fadeOutDuration;
      this._brightness = (this._brightness * (d - 1)) / d;
      this._fadeOutDuration--;
    }
  }

  updateFadeIn() {
    if (this._fadeInDuration > 0) {
      const d = this._fadeInDuration;
      this._brightness = (this._brightness * (d - 1) + 255) / d;
      this._fadeInDuration--;
    }
  }

  updateTone() {
    if (this._toneDuration > 0) {
      const d = this._toneDuration;
      for (let i = 0; i < 4; i++) {
        this._tone[i] = (this._tone[i] * (d - 1) + this._toneTarget[i]) / d;
      }
      this._toneDuration--;
    }
  }

  updateFlash() {
    if (this._flashDuration > 0) {
      const d = this._flashDuration;
      this._flashColor[3] *= (d - 1) / d;
      this._flashDuration--;
    }
  }

  updateShake() {
    if (this._shakeDuration > 0 || this._shake !== 0) {
      const delta = (this._shakePower * this._shakeSpeed * this._shakeDirection) / 10;
      if (this._shakeDuration <= 1 && this._shake * (this._shake + delta) < 0) {
        this._shake = 0;
      } else {
        this._shake += delta;
      }
      if (this._shake > this._shakePower * 2) {
        this._shakeDirection = -1;
      }
      if (this._shake < - this._shakePower * 2) {
        this._shakeDirection = 1;
      }
      this._shakeDuration--;
    }
  }

  updateZoom() {
    if (this._zoomDuration > 0) {
      const d = this._zoomDuration;
      const t = this._zoomScaleTarget;
      this._zoomScale = (this._zoomScale * (d - 1) + t) / d;
      this._zoomDuration--;
    }
  }

  updateWeather() {
    if (this._weatherDuration > 0) {
      const d = this._weatherDuration;
      const t = this._weatherPowerTarget;

      this._weatherPower = (this._weatherPower * (d - 1) + t) / d;
      this._weatherDuration--;

      if (this._weatherDuration === 0 && this._weatherPowerTarget === 0) {
        this._weatherType = 'none';
      }
    }

    if (this._weatherType === 'storm' && Constants.Weather.StormThunder) {
      this.updateStorm();
    }
  }

  updateStorm() {
    this._stormThunderCount -= (1 + (0.5 * Number(this._weatherPower)));

    if (this._stormThunderCount <= 0) {
      const weather = Constants.Weather;

      const alpha = Math.min((weather.ThunderFlash - weather.ThunderFlashVariation) + Math.randomInt(1 + weather.ThunderFlashVariation * 2), 255);
      this.startFlash([255, 255, 255, alpha], 4);

      this._stormThunderCount = 4 * ((weather.ThunderWait - weather.ThunderWaitVariation) + Math.randomInt(1 + weather.ThunderWaitVariation * 2));
      Managers.Sound.playThunderSound();
    }
  }

  updatePictures() {
    this._pictures.forEach(picture => {
      if (picture) {
        picture.update();
      }
    });
  }

  showPicture(pictureId, name, origin, x, y, scaleX, scaleY, opacity, blendMode) {
    const realPictureId = this.realPictureId(pictureId);
    const picture = new Objects.Picture();
    picture.show(name, origin, x, y, scaleX, scaleY, opacity, blendMode);
    this._pictures[realPictureId] = picture;
  }

  movePicture(pictureId, origin, x, y, scaleX, scaleY, opacity, blendMode, duration) {
    const picture = this.picture(pictureId);
    if (picture) {
      picture.move(origin, x, y, scaleX, scaleY, opacity, blendMode, duration);
    }
  }

  rotatePicture(pictureId, speed) {
    const picture = this.picture(pictureId);
    if (picture) {
      picture.rotate(speed);
    }
  }

  tintPicture(pictureId, tone, duration) {
    const picture = this.picture(pictureId);
    if (picture) {
      picture.tint(tone, duration);
    }
  }

  erasePicture(pictureId) {
    const realPictureId = this.realPictureId(pictureId);
    this._pictures[realPictureId] = null;
  }

  isPositionNearTheScreen(x, y) {
    if (x < -500) return false;
    if (x > Graphics.width + 500) return false;
    if (y < -500) return false;
    if (y > Graphics.height + 500) return false;

    return true;
  }
};