const { Input } = require('engine/core/Input');
const { Inventory } = require('engine/managers/Inventory');

//-----------------------------------------------------------------------------
// Game_Player
//
// The game object class for the player. It contains event starting
// determinants and map scrolling functions.

Objects.Player = class Player extends Objects.ActorCharacter {
  get hitboxX() {
    return this.getHitboxX();
  }
  set hitboxX(value) {
    this.setHitboxX(value);
  }
  get hitboxY() {
    return this.getHitboxY();
  }
  set hitboxY(value) {
    this.setHitboxY(value);
  }
  get hitboxWidth() {
    return this.getHitboxWidth();
  }
  set hitboxWidth(value) {
    this.setHitboxWidth(value);
  }
  get hitboxHeight() {
    return this.getHitboxHeight();
  }
  set hitboxHeight(value) {
    this.setHitboxHeight(value);
  }

  initialize() {
    super.initialize();

    this._triedAlternateMovement = false;
    this._alternateMovementDelay = 0;
    this._postTransferWaitCount = 0;
    this._itemWaitCount = 0;
    this._genericWaitCount = 0;
    this._moveSpeed = 3;
    this._wheelDelay = 0;
    this._tiredAnimationDelay = 0;
    this._footstepsDelay = 0;

    this._animalId = undefined;
    this._animalSpriteName = undefined;
    this._animalRunningSpriteName = undefined;
    this._animalIndex = undefined;
    this._animalType = undefined;
  }

  restoreData(data) {
    const propertiesToSkip = ['_followers'];
    Utils.assignProperties(this, data, propertiesToSkip);

    this._followers = new Objects.FollowerHub();
    if (data._followers) {
      this._followers.restoreData(data._followers);
    }
  }

  getDataForSaveFile() {
    return Object.assign({}, this);
  }

  restoreSpeed() {
    this._moveSpeed = 3;
    // this._moveSpeed = 3.8;
    // this._moveSpeed = 4;
  }

  setItemWaitCount(count) {
    this._itemWaitCount = Utils.getFrameCount(count);
  }

  clearTempData() {
    this._tiredAnimationDelay = 0;
    this._wheelDelay = 0;
    this._itemWaitCount = 0;
    this._genericWaitCount = 0;
    this._postTransferWaitCount = 0;
    this._alternateMovementDelay = 0;
    this._triedAlternateMovement = false;
  }

  delayItemSprite() {
    this.setItemWaitCount(12);
  }

  genericDelay(frames) {
    this._genericWaitCount = Math.max(Utils.getFrameCount(frames), this._genericWaitCount);
  }

  canJumpLeft() {
    return false;
    // var regions = [Region.JUMP_LEFT, Region.JUMP_UP_LEFT, Region.JUMP_LEFT_RIGHT, Region.JUMP_DOWN_LEFT];
    // return this.isTouchingOnlyRegion(regions);
  }

  canJumpRight() {
    return false;
    // var regions = [Region.JUMP_RIGHT, Region.JUMP_UP_RIGHT, Region.JUMP_LEFT_RIGHT, Region.JUMP_DOWN_RIGHT];
    // return this.isTouchingOnlyRegion(regions);
  }

  canJumpDown() {
    return false;
    // var regions = [Region.JUMP_DOWN, Region.JUMP_DOWN_RIGHT, Region.JUMP_DOWN_LEFT, Region.JUMP_UP_DOWN];
    // return this.isTouchingOnlyRegion(regions);
  }

  canJumpUp() {
    return false;
    // var regions = [Region.JUMP_UP, Region.JUMP_UP_RIGHT, Region.JUMP_UP_LEFT, Region.JUMP_UP_DOWN];
    // return this.isTouchingOnlyRegion(regions);
  }

  canJumpTo() /*d*/{
    return false;

    // switch(d) {
    //   case Direction.LEFT :
    //     return this.canJumpLeft();
    //   case Direction.RIGHT :
    //     return this.canJumpRight();
    //   case Direction.UP :
    //     return this.canJumpUp();
    //   case Direction.DOWN :
    //     return this.canJumpDown();
    //   default :
    //     return false;
    // }
  }

  initMembers() {
    super.initMembers();
    this._needsMapReload = false;
    this._transferring = false;
    this._performedTransfer = false;
    this._newMapId = 0;
    this._newX = 0;
    this._newY = 0;
    this._oldX = 0;
    this._oldY = 0;
    this._newDirection = 0;
    this._fadeType = 0;
    this._followers = new Objects.FollowerHub();
  }

  clearTransferInfo() {
    //Wait this amount of frames before unlocking player control
    if (this._performedTransfer) {
      this._postTransferWaitCount = Utils.getFrameCount(60);
    } else {
      this._postTransferWaitCount = Utils.getFrameCount(30);
    }
    this._transferring = false;
    this._newMapId = 0;
    this._newX = 0;
    this._newY = 0;
    this._newDirection = 0;

    this.clearDestination();
    this.restoreSpeed();
  }

  clearPerformedTransferInfo() {
    this._performedTransfer = false;
  }

  followers() {
    return this._followers;
  }

  actor() {
    if ($gameParty._actors.length > 0) {
      return $gameActors.actor($gameParty._actors[0]);
    } else {
      return undefined;
    }
  }

  follower() {
    return this._followers.follower(0);
  }

  refresh() {
    const actor = $gameParty.leader();
    const characterName = actor ? actor.characterName() : '';
    const characterIndex = actor ? actor.characterIndex() : 0;
    this.setImage(characterName, characterIndex);
    this._followers.refresh();
  }

  quickTransfer(mapId, x, y, d, fadeType, autoSave, allowFestival) {
    this.clearTransferInfo();

    if (!this.reserveTransfer(mapId, x, y, d, fadeType, autoSave, allowFestival)) return;

    Managers.Scenes.goToScene(GameScenes.Transfer);
  }

  backupTransferPosition() {
    this._storedNewX = this._newX;
    this._storedNewY = this._newY;
    this._storedNewDirection = this._newDirection;
  }

  transferToStoredPosition() {
    return this.reserveTransfer($gameMap._mapId, this._storedNewX, this._storedNewY, this._storedNewDirection, 0, false);
  }

  reserveTransfer(mapId, x, y, d, fadeType, autoSave, allowFestival) {
    if (allowFestival !== true) {
      if (Switches.isInsideFestival) return false;
    }

    if (autoSave !== false) {
      this._autoSave = true;
    } else {
      this._autoSave = false;
    }

    this._transferring = true;
    this._performedTransfer = true;
    this._newMapId = mapId;
    this._newX = x;
    this._newY = y;
    this._newDirection = d;
    this._fadeType = fadeType;

    return true;
  }

  requestMapReload() {
    this._needsMapReload = true;
  }

  isTransferring() {
    return this._transferring;
  }

  isWaitingAfterTransfer() {
    return this._postTransferWaitCount > 0;
  }

  newMapId() {
    return this._newMapId;
  }

  fadeType() {
    return this._fadeType;
  }

  performQuickTransfer() {
    if (!this.isTransferring()) return;

    Switches.waitingForTransfer = false;
    $gameTemp.clearDestination();

    $gameSystem.abortCutscene();
    this.setDirection(this._newDirection);

    if (this._newMapId !== $gameMap.mapId() || this._needsMapReload) {
      $gameSystem.registerMapChange();
      $gameMap.setup(this._newMapId);
      this._needsMapReload = false;
    }
    this.locate(this._newX, this._newY);
    this.refresh();
    this.clearTransferInfo();

    if (!$gameTemp.isCutsceneReserved()) {
      this.checkAutoSave();
    }

    Managers.Weather.updateWeather(1);
    this.checkDoor();
  }

  performTransfer() {
    console.log(...log('[OBSOLETE] Invalid call to Game_Player.performTransfer.'));
  }

  checkDoor() {
    if (this._direction !== Direction.DOWN) return;

    const events = $gameMap.eventsXyNt(this._x, this._y - 1);

    if (events.length > 0) {
      const doors = events.filter(event => event.event().name == 'door');

      if (doors.length == 1) {
        $gameTemp.setTimeout(() => {
          doors[0]._direction = Direction.UP;
          doors[0].doCloseDoorAnimation(true);
        }, 12);
      }
    }
  }

  cancelAutoSave() {
    this._autoSave = false;
  }

  checkAutoSave() {
    if (Switches.isInsideFestival) return;

    if (!!this._autoSave && $gameSystem.isSaveEnabled()) {
      this._autoSave = false;
      $gameTemp.requestAutoSave();
    }
  }

  isNormal() {
    return !this.isMoveRouteForcing();
  }

  isDebugThrough() {
    return Input.isPressed('control') && $gameTemp.isPlaytest();
  }

  isCollided(x, y) {
    if (this.isThrough()) {
      return false;
    } else {
      return this.pos(x, y) || this._followers.isSomeoneCollided(x, y);
    }
  }

  centerX() {
    const baseX = (Graphics.width / $gameMap.tileWidth() - 1) / 2.0;
    if ($gameMap.zoom) {
      return baseX / $gameMap.zoom.x;
    }
    return baseX / Graphics.zoomLevel;
  }

  centerY() {
    const baseY = (Graphics.height / $gameMap.tileHeight() - 1) / 2.0;
    if ($gameMap.zoom) {
      return baseY / $gameMap.zoom.y;
    }

    return baseY / Graphics.zoomLevel;
  }

  center(x, y) {
    return $gameMap.setDisplayPos(x - this.centerX(), y - this.centerY());
  }

  locate(x, y) {
    super.locate(x, y);
    this.center(x, y);
    if ($gameParty.hasCompanion()) {
      this._followers.synchronize(x, y, this.direction());
    }
  }

  increaseSteps() {
    super.increaseSteps();
    if (this.isNormal()) {
      $gameParty.increaseSteps();
    }
  }

  startMapVillager(x, y) {
    if ($gameMap.isEventRunning()) return true;

    let ranAny = false;
    this.runForAllPositions(x, y, (blockX, blockY) => {
      if (ranAny) return;
      if ($gameMap.isEventRunning()) return;

      const lowPriority = [];

      $gameMap.villagersXy(blockX, blockY).forEach(event => {
        if ($gameMap.isEventRunning() || !!ranAny) return;

        if (event._erased) return;
        if (event._trigger !== 0) return;
        if (!event.hasAnythingToRun(false)) {
          if (event.hasAnythingToRunOnLowPriority(false)) {
            lowPriority.push(event);
            return;
          }
        }

        if (event.start()) {
          ranAny = true;
          return;
        }
      });

      for (let i = 0; i < lowPriority.length; i++) {
        if (lowPriority[i].start()) {
          ranAny = true;
          return;
        }
      }
    });

    return ranAny;
  }

  startMapEvent(x, y, normal) {
    const playerUsedMouse = false;

    if ($gameMap.isEventRunning()) {
      return true;
    }

    let minX = x;
    let minY = y;
    let maxX = x;
    let maxY = y;

    if (this._direction != Direction.UP) {
      maxY = Math.ceil(maxY + 1);
    }
    if (this._direction !== Direction.DOWN) {
      minY = Math.floor(minY - 1);
    }
    if (this._direction != Direction.RIGHT) {
      minX = Math.floor(minX - 1);
    }
    if (this._direction != Direction.LEFT) {
      maxX = Math.ceil(maxX + 1);
    }

    const player = this;
    const lowPriority = [];
    const highPriority = [];
    const tiles = [];

    this.runForAllPositions(x, y, (block_x, block_y) => {
      if (block_x < minX || block_x > maxX || block_y < minY || block_y > maxY) return;

      $gameMap.anythingXy(block_x, block_y).forEach(event => {
        if (highPriority.includes(event) || lowPriority.includes(event)) {
          return;
        }

        if (event._erased) return;
        if (event._trigger !== 0 ) return;
        if (!event.canBeActivatedFrom(player._direction, player.left, player.top, playerUsedMouse)) return;

        if (!event.hasAnythingToRun(playerUsedMouse)) {
          if (event.hasAnythingToRunOnLowPriority(playerUsedMouse)) {
            lowPriority.push(event);
            return;
          }
        }

        highPriority.push(event);
      });

      tiles.push({x: block_x, y: block_y});
    });

    const sortedHigh = highPriority.sort((event1, event2) => {
      return event1.distanceToPlayer() - event2.distanceToPlayer();
    });

    for (let event of sortedHigh) {
      if (event.start() || $gameMap.isEventRunning()) {
        return true;
      }
    }

    const sortedLow = lowPriority.sort((event1, event2) => {
      return event1.distanceToPlayer() - event2.distanceToPlayer();
    });

    for (let event of sortedLow) {
      if (event.start() || $gameMap.isEventRunning()) {
        return true;
      }
    }

    for (let tile of tiles) {
      if (Managers.FarmObjects.triggerProtectedTile(tile.x, tile.y)) {
        return true;
      }
    }

    if ($gameParty.hasCompanion()) {
      const follower = $gamePlayer.follower();
      if (follower && follower.canBeActivatedByPlayer()) {
        follower.start();
        return true;
      }
    }

    return false;
  }

  isDirectionFixed() {
    return this._directionFix;
  }

  moveByInput() {
    if (!this.canCheckInput()) return;

    if (this.isMoving() && !this.isAboutToStop()) {
      return;
    }

    if (!this.canMove()) {
      return;
    }

    let direction = Input.dir4;
    let diagonal_d = Input.dir8;
    let alternative_d = direction;

    // If there's a directional button pressed, clear the mouse destination
    if (direction > 0) {
      $gameTemp.clearDestination();
    }
    //If no direction key is pressed, but the gamepad alternate button is, then move forward automatically
    else if (Input.isPressed('alternate')) {
      direction = this._direction;
      diagonal_d = direction;
      $gameTemp.clearDestination();
    }
    // If there's a valid mouse destination, pick the direction from it
    else if ($gameTemp.isDestinationValid()) {
      diagonal_d = this.determineDirectionToDestination();
      if (DirectionHelper.isDiagonal(diagonal_d)) {
        direction = this.getAlternativeDirection(0, diagonal_d);
      } else {
        direction = diagonal_d;
      }
    }
    else {
      diagonal_d = $gameTemp.getPlayerDirectionPressed();
      if (isNaN(diagonal_d) || diagonal_d === null) {
        diagonal_d = 0;
        direction = 0;
      } else {
        direction = this.getAlternativeDirection(0, diagonal_d);
        if (direction === 0) direction = diagonal_d;
      }
    }

    $gameTemp.clearPlayerDirectionPressed();

    if (direction < 0) {
      this.setDirection(Math.abs(direction));
      direction = 0;
      diagonal_d = 0;
    }

    // If the player is pressing two direction buttons and the direction picked by dir4 is unavailable, try the other non-diagonal direction
    alternative_d = this.getAlternativeDirection(direction, diagonal_d);

    if (direction === 0) {
      return;
    }

    let canPass = this.canPass(this._x, this._y, direction);
    let skipDiagonal = direction === alternative_d;
    if (!canPass && !skipDiagonal) {
      canPass = this.canPass(this._x, this._y, alternative_d);
      skipDiagonal = true;
    }

    if (canPass) {
      this.onBeforeMove();

      // Try moving diagonally first
      if (!skipDiagonal) {
        this.executeMove(diagonal_d);
        if (this.isMovementSucceeded()) {
          return;
        }

        this.executeMove(direction);
      }

      if (skipDiagonal || !this.isMovementSucceeded()) {
        this.setMovementSuccess(true);
        this._triedAlternateMovement = false;
        this.moveStraightNoTest(alternative_d);

        // if (Input.dir4 > 0) {
        //   if (this.checkPushableObjects(direction)) {
        //     return;
        //   }
        // }
      }
    } else {
      if (Input.dir4 > 0) {
        if (this.checkPushableObjects(direction)) {
          return;
        }
      }

      // If the movement failed, call to try a different movement option
      if (this.trySavingFailedMovement(direction)) {
        return;
      }

      if (this._direction != direction) {
        this.setDirection(direction);
        this._triedAlternateMovement = false;
      }
    }
  }

  tryPushingEvent(event, direction) {
    if (event.isMoving()) return;

    // If the player is just barely touching this object, then don't push it.
    switch(direction) {
      case Direction.UP:
      case Direction.DOWN:
        if (event.left > this.right - 0.5) {
          return;
        }

        if (event.right < this.left + 0.5) {
          return;
        }

        break;
      case Direction.LEFT:
      case Direction.RIGHT:
        if (event.top > this.bottom - 0.25) {
          return;
        }
        if (event.bottom < this.top + 0.25) {
          return;
        }
        break;
    }

    if (!event._through) {
      //Set the pushable object as through and check if the player would then be able to pass
      //this is needed because otherwise the character could get locked on an impassable tile

      event._through = true;
      const canPass = this.canPass(this._x, this._y, direction);
      event._through = false;
      if (!canPass) {
        return;
      }
    }

    this.setDirection(direction);

    if (event.pushCount < 20) {
      event.pushCount++;
      return true;
    }

    event.start();
    return true;
  }

  tryPushingCreature(creature, direction) {
    if (creature.isMoving()) {
      return;
    }

    this.setDirection(direction);

    if (creature.pushCount < 20) {
      creature.pushCount++;
      return true;
    }

    creature.push();
    return true;
  }

  checkPushableObjects(direction) {
    if (this.isRiding()) return false;

    const tileList = TileHelper.getTilesAtDirection(1, direction);
    if (!tileList.length) return false;

    for (const tile of tileList) {
      if (!tile) continue;

      const events = $gameMap.eventsXy(tile.x, tile.y);
      let len = events.length;
      
      for (let i = 0; i < len; i++) {
        if (events[i] instanceof Objects.PushableObject) {
          if (this.tryPushingEvent(events[i], direction)) {
            return true;
          }
        }

        if (Managers.Content.tryPushingEvent(events[i], direction)) {
          return true;
        }
      }

      const creatures = $gameMap.creaturesXy(tile.x, tile.y);
      len = creatures.length;
      for (let j = 0; j < len; j++) {
        if (creatures[j] instanceof Objects.PushableAnimal) {
          if (this.tryPushingCreature(creatures[j], direction)) {
            return true;
          }
        }
      }
    }

    return false;
  }

  canMove() {
    if (Switches.holdPlayer || Switches.holdPlayerAsync) {
      return false;
    }

    if (this._itemWaitCount > 0) {
      return false;
    }

    if (this._toolId !== undefined) {
      return false;
    }

    if ($gameMap.isEventRunning() || $gameMessage.isBusy()) {
      return false;
    }

    if (this.isMoveRouteForcing() || this.areFollowersGathering()) {
      return false;
    }

    if (this._genericWaitCount > 0) {
      return false;
    }

    return true;
  }

  getInputDirection() {
    return Input.dir4;
  }

  executeMove(direction, maxStepSize) {
    this._triedAlternateMovement = false;

    switch (direction) {
      case Direction.UP:
      case Direction.DOWN:
      case Direction.LEFT:
      case Direction.RIGHT:
        this.moveStraight(direction, maxStepSize);
        break;

      case Direction.UP_LEFT:
        this.moveDiagonally(Direction.LEFT, Direction.UP);
        break;
      case Direction.UP_RIGHT:
        this.moveDiagonally(Direction.RIGHT, Direction.UP);
        break;
      case Direction.DOWN_LEFT:
        this.moveDiagonally(Direction.LEFT, Direction.DOWN);
        break;
      case Direction.DOWN_RIGHT:
        this.moveDiagonally(Direction.RIGHT, Direction.DOWN);
        break;

      default:
        break;
    }
  }

  moveStraightNoTest(d, maxStepSize) {
    this.beforeMove(d, maxStepSize);
    this.setDiagonalDirection(0, 0);
    this.setMovementSuccess(true);
    const stepSize = this.myStepSize(maxStepSize);

    if (this.isMovementSucceeded()) {
      this.beforeSuccessfulMoveStraight(d, maxStepSize);

      this.setDirection(d);
      this._x = this.roundFractionXWithDirection(this._x, d, stepSize);
      this._y = this.roundFractionYWithDirection(this._y, d, stepSize);

      if ((this._x > this._realX && this._lostX > 0) || (this._x < this._realX && this._lostX < 0)) {
        this._realX += this._lostX;
      }

      if ((this._y > this._realY && this._lostY > 0) || (this._y < this._realY && this._lostY < 0)) {
        this._realY += this._lostY;
      }

      this._lostX = 0;
      this._lostY = 0;
      this.afterMove();
    } else {
      this.setDirection(d);
    }
  }

  setSleepDelay(newDelay) {
    this._sleepDelay = Utils.getFrameCount(newDelay);
  }

  updateMagicSleep() {
    this._asleep = true;

    if (!!this._sleepDelay && this._sleepDelay > 0) {
      this._sleepDelay--;
      return;
    }

    if (Input.isTriggered('ok') || Input.isTriggered('use') || TouchInput.isTriggered() || TouchInput.isRightTriggered()) {
      $gameSystem.wakeUp();
      $gamePlayer.doToolJumpAnimation('none');
      this.setItemWaitCount(30);
    }

    super.update();
  }

  update(sceneActive) {
    if (this._itemWaitCount > 0) {
      this._itemWaitCount--;
    }
    if (this._footstepsDelay > 0) {
      this._footstepsDelay--;
    }

    if ($gameSystem.isPlayingCutscene()) {
      this._genericWaitCount = 0;
      this._postTransferWaitCount = 0;
    }

    if (this._genericWaitCount > 0) {
      this._genericWaitCount--;
    }

    if (this._postTransferWaitCount > 0) {
      if (!Managers.Scenes._scene.isFullyLoaded()) return;

      this._postTransferWaitCount--;
      return;
    }

    if (Switches.forceGoHome) {
      if ($gamePlayer.isRiding()) {
        const farmMap = Managers.Map.getClass(Maps.FARM);
        farmMap.enterFromFarmDoor();
      } else {
        this.goHome();
      }
      Switches.forceGoHome = false;
      return;
    }

    const lastScrolledX = this.scrolledX();
    const lastScrolledY = this.scrolledY();
    const wasMoving = this.isMoving();

    this._oldX = this._realX;
    this._oldY = this._realY;
    this._oldD = this._direction;

    if ($gameSystem.isPlayerSleepingStanding) {
      this.updateMagicSleep();
      return;
    }
    this._asleep = false;

    this.updateDashing();

    if (sceneActive) {
      this.moveByInput();
    }
    super.update();
    this.updateScroll(lastScrolledX, lastScrolledY);
    if (this.isMoving()) {
      if (!wasMoving) {
        this.updatePattern();
        // Update hud to hide crop info when the player starts moving
        Inventory.requestHudRefresh();
      }
      this.triggerToolsWhileMoving();
    } else {
      if (this._oldD !== this._direction) {
        // Update hud to show/hide crop info when the player changes direction
        Inventory.requestHudRefresh();
      }
      this.updateNonmoving(wasMoving);
    }
    this.triggerExtra();
    this.triggerRiding();

    if ($gameParty.hasCompanion()) {
      this._followers.update();
    }
  }

  isMoving() {
    if (this.isTransferring()) return false;

    return this._realX !== this._oldX || this._realY !== this._oldY || super.isMoving();
  }

  canDash() {
    if (this.isRiding()) return true;

    if (Managers.Health.staminaLevel <= 10) return false;
    if ($gameSystem.isPlayingCutscene()) return false;

    return true;
  }

  updateDashing() {
    if (this.isMoving()) {
      return;
    }

    if (this.canMove() && this.canDash()) {
      this._dashing = true;
    } else {
      this._dashing = false;
    }
  }

  updateScroll(lastScrolledX, lastScrolledY) {
    if ($gameMap.isScrolling()) return;
    if (Managers.Content.updateScroll(lastScrolledX, lastScrolledY)) return;

    const x1 = lastScrolledX;
    const y1 = lastScrolledY;
    const x2 = this.scrolledX();
    const y2 = this.scrolledY();
    if (y2 > y1 && y2 > this.centerY()) {
      $gameMap.scrollDown(y2 - y1);
    }
    if (x2 < x1 && x2 < this.centerX()) {
      $gameMap.scrollLeft(x1 - x2);
    }
    if (x2 > x1 && x2 > this.centerX()) {
      $gameMap.scrollRight(x2 - x1);
    }
    if (y2 < y1 && y2 < this.centerY()) {
      $gameMap.scrollUp(y1 - y2);
    }
  }

  triggerToolsWhileMoving() {
    if (this.triggerTool(true)) {
      return;
    }
  }

  updateNonmoving(wasMoving) {
    if (!this.canCheckInput()) return;

    if (!$gameMap.isEventRunning()) {
      if (wasMoving) {
        $gameParty.onPlayerWalk();
        if ($gameMap.setupStartingEvent()) {
          return;
        }

        // Update hud to display crop info when the player stops moving
        Inventory.requestHudRefresh();
      }

      if (this.triggerAction()) {
        return;
      }

      if (this.triggerItem()) {
        return;
      }

      if (this.triggerTool()) {
        return;
      }

      if (this.triggerHotkey()) {
        return;
      }

      if (this.triggerCheat()) {
        return;
      }

      if (!wasMoving) {
        $gameTemp.clearDestination();
      }
    }
  }

  triggerRiding() {
    if (!this.isRiding()) return false;
    if ($gameMap.isEventRunning()) return false;

    if (Input.isTriggered('use') || TouchInput.isRightTriggered()) {
      if (!$gameTemp.isCommonEventReserved()) {
        if (this.getOffRide()) {
          this.addToolDelay(30);
          return true;
        }
      }
    }
  }

  triggerExtraButtonsHoldingShift() {
    if (Input.isTriggered('pageup')) {
      return Inventory.changeTool(false);
    }

    if (Input.isTriggered('pagedown')) {
      return Inventory.changeItemIndex(true);
    }

    if (this._wheelDelay === 0) {
      if (TouchInput.wheelY >= Constants.WHEELTHRESHOLD) {
        this._wheelDelay = Utils.getFrameCount(30);
        return Inventory.changeItemIndex(true);
      } else if (TouchInput.wheelY <= -Constants.WHEELTHRESHOLD) {
        this._wheelDelay = Utils.getFrameCount(30);
        return Inventory.changeItemIndex(false);
      }
    }
  }

  triggerExtraButtons() {
    // if (Input.isTriggered('left_backbutton') && Input.isTriggered('right_backbutton')) {
    //   Inventory.switchBackpack();
    //   return true;
    // }

    if (!Inventory.specialItem) {
      if (Input.isPressed('left_trigger') && Input.isPressed('right_trigger')) {
        Inventory._itemIndex = -1;
        return true;
      }

      if (Input.isTriggered('pageup') || Input.isTriggered('right_backbutton')) {
        return Inventory.changeTool(true);
      }
      if (Input.isTriggered('left_backbutton')) {
        return Inventory.changeTool(false);
      }
    }

    if (this._wheelDelay === 0) {
      if (TouchInput.wheelY >= Constants.WHEELTHRESHOLD) {
        this._wheelDelay = Utils.getFrameCount(5);
        return Inventory.changeItemIndex(false);
      } else if (TouchInput.wheelY <= -Constants.WHEELTHRESHOLD) {
        this._wheelDelay = Utils.getFrameCount(5);
        return Inventory.changeItemIndex(true);
      }
    }

    if (Input.isTriggered('pagedown') || Input.isTriggered('right_trigger')) {
      if (Input.isPressed('control')) {
        Inventory._itemIndex = -1;
        return true;
      }

      return Inventory.changeItemIndex(false);
    }

    if (Input.isTriggered('left_trigger')) {
      return Inventory.changeItemIndex(true);
    }

  }

  triggerExtra() {
    if (!this.canMove()) return false;
    if (!this.canChangeItems()) return false;

    if (!this._wheelDelay) {
      this._wheelDelay = 0;
    }
    if (this._wheelDelay > 0) {
      this._wheelDelay--;
    }

    if (!Inventory.waiting) {
      if (Input.isPressed('shift')) {
        if (Input.isTriggered('map')) {
          Inventory.switchBackpack();
          return true;
        }

        const shiftResult = this.triggerExtraButtonsHoldingShift();
        if (shiftResult !== undefined) return shiftResult;
      } else {
        const regularResult = this.triggerExtraButtons();
        if (regularResult !== undefined) return regularResult;
      }
    }

    if (Input.isTriggered('extra')) {
      if (Inventory.isHoldingItem() && !this.isRiding()) {
        return Inventory.explainSelectedItem() !== false;
      }
    }

    if (Input.isTriggered('change_backpack')) {
      Inventory.switchBackpack();
      return true;
    }
  }

  triggerItem() {
    if (!this.canChangeItems()) return false;
    if (this.isRiding()) return false;
    if (!this.canMove()) return false;
    if (!Inventory.isHoldingItem()) return false;

    if (Input.isTriggered('use')) {
      return Inventory.useSelectedItem();
    }

    return false;
  }

  triggerCheat() {
    if (!this.canMove()) return false;

    if (Managers.Content.triggerCheat()) {
      return true;
    }

    return false;
  }

  triggerHotkey() {
    if (!this.canMove()) return false;
    if (this.isRiding()) return false;

    let anyPressed = false;
    for (let i = 0; i <= 10; i++) {
      if (Input.isTriggered(i.toString())) {
        anyPressed = true;
        break;
      }
    }

    if (!anyPressed) return false;

    const config = Managers.Tools.getHotkeyConfig();

    for (const key in config) {
      if (!config.hasOwnProperty(key)) continue;
      if (!Input.isTriggered(key)) continue;

      Managers.Tools.setTool(config[key]);

      //This is the only situation in which unequipping a tool triggers the sound
      if (config[key] == 'empty') {
        Managers.Sound.playEquip();
      }
      return;
    }
  }

  clearToolData() {
    this._usingTool = false;
    this._doingToolAnimation = false;
    this._doingCircleAnimation = false;
    this._doingDelayAnimation = false;
    this._toolId = undefined;
  }

  addToolDelay(delay) {
    if (!this._toolDelay || this._toolDelay < 0) {
      this._toolDelay = Utils.getFrameCount(delay);
    } else {
      this._toolDelay += Utils.getFrameCount(delay);
    }
  }

  maybeUseCurrentTool() {
    const tool = Managers.Tools.effectiveTool;
    if (!tool) {
      return false;
    }

    const x = $gameMap.canvasToMapX(TouchInput.mouseX);
    const y = $gameMap.canvasToMapY(TouchInput.mouseY);

    if (!Managers.Tools.isTileBetterThanDefault(x, y, true)) {
      return false;
    }

    const tile = TileHelper.newTile(x, y);
    if (!tool.isTileValidForMouseDown(tile)) {
      return false;
    }

    return this.useCurrentTool(true, null, true, x, y);
  }

  useCurrentTool(usingMouse = false, targetEvent = null, acceptItems = false, fixedX = null, fixedY = null) {
    if (!this.canMove()) return false;
    const toolId = Managers.Tools.effectiveToolId;

    if (toolId === undefined) return false;
    if (!Managers.Tools.isToolValid(toolId)) return false;
    if (!acceptItems) {
      if (Inventory.selectedItem !== undefined) return false;
    }

    Managers.Tools.clearTargetTile();
    if (targetEvent) {
      $gamePlayer.turnTowardPosition(targetEvent.x, targetEvent.y);
      Managers.Tools.setTargetTile(targetEvent.x, targetEvent.y);
    } else if (fixedX !== null && fixedY !== null) {
      $gamePlayer.turnTowardPosition(fixedX, fixedY);
      Managers.Tools.setTargetTile(fixedX, fixedY);
    }

    if (usingMouse && !targetEvent && !fixedX && !fixedY) {
      //If the mouse was used, check if the tile that the mouse is over is a better target for the tool
      const x = $gameMap.canvasToMapX(TouchInput.mouseX);
      const y = $gameMap.canvasToMapY(TouchInput.mouseY);

      if (Managers.Tools.isTileBetterThanDefault(x, y)) {
        $gamePlayer.turnTowardPosition(x, y);
        Managers.Tools.setTargetTile(x, y);
      }
    }

    Managers.Tools.useTool(Managers.Tools.targetTile);
    return true;
  }

  triggerToolOnFestival(checkMouseOnly = false) {
    if (this._toolDelay && this._toolDelay > 0) {
      this._toolDelay--;
      return;
    }

    return Managers.Content.triggerToolOnFestival(checkMouseOnly) === true;
  }

  triggerTool(checkMouseOnly = false) {
    if (Switches.isInsideFestival) {
      return this.triggerToolOnFestival(checkMouseOnly);
    }

    if (!this.canUseTools()) return false;

    if (this._toolDelay && this._toolDelay > 0) {
      this._toolDelay--;
      return false;
    }

    if (!checkMouseOnly) {
      //If the use button was just pressed
      if (Input.isTriggered('use')) {
        return this.useCurrentTool(false);
      }
    }

    //If the mouse button is pressed, check if the tool should be triggered by the clicked tile
    if (TouchInput.isPressed()) {
      if (this.maybeUseCurrentTool()) {
        return true;
      }
    }

    return false;
  }

  triggerAction() {
    if (this.canMove()) {
      if (this.triggerButtonAction()) {
        return true;
      }
    }
    return false;
  }

  triggerButtonAction(forceCheck) {
    if (forceCheck || Input.isTriggered('ok')) {
      if (Managers.Content.checkEventTriggerHere()) return true;

      if (this.checkEventTriggerHere()) return true;
      if ($gameMap.setupStartingEvent()) return true;

      if (Managers.Map.checkEventTriggerHere()) return true;

      if (Managers.Content.checkEventTriggerThere()) return true;
      if (this.checkEventTriggerThere()) return true;
      if ($gameMap.setupStartingEvent()) return true;
      if (Managers.Map.checkEventTriggerThere()) return true;

      if (Managers.Content.checkEventTriggerFurtherOverThere()) return true;
      if (this.checkEventTriggerFurtherOverThere()) return true;
      if ($gameMap.setupStartingEvent()) return true;

      if (this.canChangeItems() && !this.isRiding()) {
        if (Inventory.dropItem(true)) return true;
      }
    }
    return false;
  }

  checkEventTriggerHere() {
    if (this.canStartLocalEvents()) {
      return this.startMapEvent(this.x, this.y, false);
    }

    return false;
  }

  checkEventTriggerThere() {
    if (this.canStartLocalEvents()) {
      const frontTile = TileHelper.getSmartFrontTile(1, undefined, undefined, true);
      if (frontTile) {
        const result = this.startMapEvent(frontTile.x, frontTile.y, true);

        if (result) {
          return result;
        }


        if ($gameMap.isCounter(frontTile.x, frontTile.y)) {
          const nextTile = TileHelper.getSmartFrontTile(2);

          if (nextTile) {
            return this.startMapEvent(nextTile.x, nextTile.y, true);
          }
        }

        return result;
      } else {
        const direction = this.direction();
        const x1 = this._x;
        const y1 = this._y;
        const x2 = $gameMap.roundXWithDirection(x1, direction);
        const y2 = $gameMap.roundYWithDirection(y1, direction);

        if (this.startMapEvent(x2, y2, true)) return true;

        if ($gameMap.isCounter(x2, y2) && !$gameMap.isAnyEventStarting()) {
          const x3 = $gameMap.roundXWithDirection(x2, direction);
          const y3 = $gameMap.roundYWithDirection(y2, direction);
          return this.startMapEvent(x3, y3, true);
        }

        return false;
      }
    }
  }

  checkEventTriggerFurtherOverThere() {
    if (this.canStartLocalEvents()) {
      const frontTile = TileHelper.getSmartFrontTile(2);
      if (frontTile) {
        const result = this.startMapVillager(frontTile.x, frontTile.y, true);

        if (result) {
          return result;
        }

        if ($gameMap.isCounter(frontTile.x, frontTile.y)) {
          const nextTile = TileHelper.getSmartFrontTile(3);

          if (nextTile) {
            return this.startMapVillager(nextTile.x, nextTile.y, true);
          }
        }

        return result;
      }
    }
  }

  canStartLocalEvents() {
    return true;
  }

  getTilesNearPlayer(radius) {
    return TileHelper.getCircularTiles(Math.floor(this.x), Math.floor(this.y), radius);
  }

  beforeSuccessfulMoveStraight(d) {
    super.beforeSuccessfulMoveStraight(d);
    if ($gameParty.hasCompanion()) {
      this._followers.updateMove();
    }
  }

  beforeSuccessfulMoveDiagonally(horz, vert) {
    super.beforeSuccessfulMoveDiagonally(horz, vert);
    if ($gameParty.hasCompanion()) {
      this._followers.updateMove();
    }
  }

  jump(xPlus, yPlus, lockDirection) {
    super.jump(xPlus, yPlus, lockDirection);
    if ($gameParty.hasCompanion()) {
      this._followers.jumpAll();
    }
  }

  showFollowers() {
    this._followers.show();
  }

  hideFollowers() {
    this._followers.hide();
  }

  gatherFollowers() {
    this._followers.gather();
  }

  areFollowersGathering() {
    return this._followers.areGathering();
  }

  areFollowersGathered() {
    return this._followers.areGathered();
  }

  onBeforeMove() {

  }

  afterMove() {
    super.afterMove();

    this._alternateMovementDelay = Utils.getFrameCount(4);
    Managers.Player.walkingExp += 0.002;

    if (!this._footstepsDelay) {
      const x = Math.round(this.x);
      const y = Math.round(this.y);

      this.playFootsteps(x, y);

      this._footstepsDelay = Utils.getFrameCount(14);
    }
  }

  playFootsteps(x, y) {
    if (this.isRiding()) return;
    Managers.Sound.playFootsteps(Managers.Time.month % 4);
  }

  maxOffset() {
    return 0.8;
  }

  maxEventOffset() {
    return 0.4;
  }

  setMovementBlockingReason({ reason, x, y, blockX, blockY, event, left, top, right, bottom }) {
    if (this._lastCollisionCheck !== 'characters') {
      return;
    }

    if (reason == 'object' && event) {
      if (event._farmObjectData && event._farmObjectData.modelName == 'fence') {
        this._lastCollisionCheck = 'fence';
        return;
      }

      if (event._farmObjectData && event._farmObjectData.itemName) {
        this._lastCollisionCheck = 'item';
        return;
      }

      this._lastCollisionCheck = 'object';
    }
  }

  _desiredMaxOffset() {
    if (this._lastCollisionCheck == 'characters' || this._lastCollisionCheck == 'object') {
      return this.maxEventOffset();
    }

    if (this._lastCollisionCheck == 'fence') {
      return this.maxOffset();
    }

    if (this._lastCollisionCheck == 'item') {
      return 0;
    }

    return this.maxOffset();
  }

  trySavingFailedMovement(direction) {
    if (this._triedAlternateMovement) {
      return false;
    }

    if (this._alternateMovementDelay > 0) {
      this._alternateMovementDelay--;
      return false;
    }

    this._triedAlternateMovement = true;
    const maxOffset = this._desiredMaxOffset();
    const avoided = this.tryToAvoid(direction, maxOffset);

    if (avoided) {
      // Try moving once more to make movement smoother
      this.executeMove(direction);

      this._alternateMovementDelay = 0;
      return avoided;
    }

    if (this.canJumpTo(direction)) {
      this.jumpToDirection(direction, 2);
      return true;
    }

    if (this.tryJumpingOverFencesAndObjects(direction)) {
      this._triedAlternateMovement = false;
      return true;
    }

    return false;
  }

  tryJumpingOverFencesAndObjects(direction) {
    let distance = 1;
    if (this._lastCollisionCheck !== 'fence') return false;

    if (direction != Direction.LEFT && direction != Direction.UP) {
      distance = 0.1;
    }

    const fenceTiles = TileHelper.getTilesAtDirection(distance, direction, this);

    for (const tile of fenceTiles) {
      if (!Managers.FarmObjects.positionCanBeJumpedOver($gameMap._mapId, Math.floor(tile.x), Math.floor(tile.y))) {
        return false;
      }
    }

    distance = Math.floor(distance) + 1;

    const freeTiles = TileHelper.getTilesAtDirection(distance, direction, this);
    const collisions = [];

    for (const freeTile of freeTiles) {
      const { x, y } = freeTile;

      if (!$gameMap.isTilePassable(x, y)) return false;
      if (Managers.FarmObjects.isTileProtected($gameMap._mapId, x, y)) return false;

      const events = $gameMap.eventsXyNt(x, y);
      if (events.length > 0) return false;

      const farmObjects = $gameMap.farmObjectsXy(x, y);

      for (let i = 0; i < farmObjects.length; i++) {
        if (farmObjects[i]._farmObjectData.isItem()) continue;
        if (!farmObjects[i]._farmObjectData.hasCollision()) continue;

        if (farmObjects[i]._priorityType == EventPriorities.NORMAL) {
          if (collisions.indexOf(farmObjects[i]) < 0) {
            collisions.push(farmObjects[i]);
          }
          continue;
        }

        if (farmObjects[i]._farmObjectData.hasCollision()) {
          if (collisions.indexOf(farmObjects[i]) < 0) {
            collisions.push(farmObjects[i]);
          }
          continue;
        }
      }
    }

    let finalDistance = 1;
    if (this.isRiding()) {
      finalDistance = 1.5;
    }

    let { left, right, top, bottom } = $gamePlayer;
    switch(direction) {
      case Direction.LEFT:
        left -= finalDistance;
        right -= finalDistance;
        break;
      case Direction.RIGHT:
        left += finalDistance;
        right += finalDistance;
        break;
      case Direction.UP:
        top -= finalDistance;
        bottom -= finalDistance;
        break;
      case Direction.DOWN:
        top += finalDistance;
        bottom += finalDistance;
        break;
    }

    for (const object of collisions) {
      if (object.realPosIn(left, top, right, bottom)) {
        return false;
      }

      if (object._farmObjectData && object._farmObjectData.modelName == 'fence') {
        const state = object._farmObjectData.getState(true);

        if (['fence-horizontal-down', 'fence-horizontal-up', 'fence-vertical-right'].includes(state)) {
          return false;
        }

        if (state == 'fence-vertical-left') {
          if (direction == Direction.UP || direction == Direction.LEFT) {
            return false;
          }
        }

        switch(direction) {
          case Direction.UP:
            if (['fence-horizontal', 'fence-top-right', 'fence-top-left'].includes(state)) {
              return false;
            }
            break;
          case Direction.DOWN:
            if (['fence-horizontal', 'fence-bottom-right', 'fence-bottom-left'].includes(state)) {
              return false;
            }
            break;
          case Direction.LEFT:
            if (['fence-vertical', 'fence-top-left', 'fence-bottom-left'].includes(state)) {
              return false;
            }
            break;
          case Direction.RIGHT:
            if (['fence-vertical', 'fence-top-right', 'fence-bottom-right'].includes(state)) {
              return false;
            }
            break;
        }
      }
    }

    this.jumpToDirection(direction, finalDistance);
    return true;
  }

  jumpToDirection(direction, distance) {
    switch(direction) {
      case Direction.LEFT :
        this.jump(-distance, 0);
        break;
      case Direction.RIGHT :
        this.jump(distance, 0);
        break;
      case Direction.UP :
        this.jump(0, -distance);
        break;
      case Direction.DOWN :
        this.jump(0, distance);
        break;
      default :
        break;
    }
  }

  tryToAvoid(direction, maxOffset) {
    let previousOffset = 0;
    let offset = this.myStepSize(this.myDefaultStepSize());

    const tryDirection = function(xOffset, yOffset, movementDirection, faceDirection) {
      // Test if the player would be able to move on the faceDirection if they were at the offset position. If they would, then move towards that position for now.
      if (this.canPass(Math.fix(this._x + xOffset), Math.fix(this._y + yOffset), faceDirection)) {
        switch(movementDirection) {
          case Direction.LEFT :
          case Direction.RIGHT :
            this.executeMove(movementDirection, Math.abs(xOffset));
            break;
          case Direction.DOWN :
          case Direction.UP :
            this.executeMove(movementDirection, Math.abs(yOffset));
            break;
        }

        this.setDirection(faceDirection);

        return true;
      }

      return false;
    };

    if (direction == Direction.LEFT || direction == Direction.RIGHT) {
      // If the player can't walk horizontally on the current position, but would be able to walk if he were a little higher or lower then move vertically instead
      // on the next iterations it will keep trying to move horizontaly again and it will eventually work before the offset is reached
      let downEnabled = true;
      let upEnabled = true;

      while (offset <= maxOffset) {
        if (downEnabled) {
          if (!this.canPass(this._x, Math.fix(this._y + previousOffset), Direction.DOWN)) {
            downEnabled = false;
          }
        }

        if (upEnabled) {
          if (!this.canPass(this._x, Math.fix(this._y - previousOffset), Direction.UP)) {
            upEnabled = false;
          }
        }

        if (downEnabled === true && tryDirection.call(this, 0, offset, Direction.DOWN, direction)) {
          return true;
        }

        if (upEnabled === true && tryDirection.call(this, 0, -offset, Direction.UP, direction)) {
          return true;
        }

        previousOffset = offset;
        offset += this.myStepSize(this.myDefaultStepSize());
      }
    }
    else if (direction == Direction.UP || direction == Direction.DOWN) {
      // If the player can't walk vertically on the current position, but would be able to walk if he were a little left or right then move horizontally instead
      // on the next iterations it will keep trying to move vertically again and it will eventually work before the offset is reached
      let leftEnabled = true;
      let rightEnabled = true;

      while (offset <= maxOffset) {
        if (leftEnabled) {
          if (!this.canPass(Math.fix(this._x - previousOffset), this._y, Direction.LEFT)) {
            leftEnabled = false;
          }
        }
        if (rightEnabled) {
          if (!this.canPass(Math.fix(this._x + previousOffset), this._y, Direction.RIGHT)) {
            rightEnabled = false;
          }
        }

        if (rightEnabled === true && tryDirection.call(this, offset, 0,  Direction.RIGHT, direction)) {
          return true;
        }

        if (leftEnabled === true && tryDirection.call(this, -offset, 0, Direction.LEFT, direction)) {
          return true;
        }

        previousOffset = offset;
        offset += this.myStepSize(this.myDefaultStepSize());
      }
    }

    return false;
  }

  determineDirectionToDestination() {
    const x = $gameTemp.destinationX();
    const y = $gameTemp.destinationY();

    let horzDirection;
    let vertDirection;

    const currX = this._x;
    const currY = this._y;

    if (x < currX) {
      horzDirection = Direction.LEFT;
    } else if (x > currX) {
      horzDirection = Direction.RIGHT;
    }

    if (y < currY) {
      vertDirection = Direction.UP;
    } else if (y > currY) {
      vertDirection = Direction.DOWN;
    }

    if (horzDirection !== undefined && vertDirection !== undefined) {
      return DirectionHelper.joinDirections(horzDirection, vertDirection);
    }

    if (horzDirection !== undefined) {
      return horzDirection;
    }

    if (vertDirection !== undefined) {
      return vertDirection;
    }

    $gameTemp.clearDestination();
    return 0;
  }

  // If the player is holding two direction buttons, Input.dir4 will give you one of them and this method will give you the other one
  getAlternativeDirection(direction, diagonal_d) {
    if (direction != diagonal_d) {
      switch (diagonal_d) {
        case Direction.UP_LEFT:
          return direction == Direction.UP ? Direction.LEFT : Direction.UP;
        case Direction.UP_RIGHT:
          return direction == Direction.UP ? Direction.RIGHT : Direction.UP;
        case Direction.DOWN_LEFT:
          return direction == Direction.DOWN ? Direction.LEFT : Direction.DOWN;
        case Direction.DOWN_RIGHT:
          return direction == Direction.DOWN ? Direction.RIGHT : Direction.DOWN;
        default:
          break;
      }
    }

    return direction;
  }

  isDashing() {
    return this._dashing;
  }

  holdingFlashlight() {
    return Managers.Tools.toolId == 'flashlight';
  }

  holdingLantern() {
    return Managers.Tools.toolId == 'lantern';
  }

  isHealthLow() {
    return Managers.Health.staminaLevel < 15;
  }

  isHealthFull() {
    return Managers.Health.staminaLevel == 100;
  }

  canCheckInput() {
    if (Switches.lockInputForFestival) return false;
    if (Switches.waitingForTransfer) return false;
    if (this.isTransferring()) return false;
    if (Inventory.waiting) return false;

    if (!Managers.Scenes._scene.isFullyLoaded()) return false;
    if ($gameScreen.isFading()) return false;
    if (!Managers.Content.canCheckInput()) return false;

    return true;
  }

  canUseTools() {
    if (Switches.isInsideFestival) return false;
    if (this.isRiding()) {
      if (Managers.Tools.toolId != 'teleport') return false;
    }

    return true;
  }

  canChangeItems() {
    if (Switches.isInsideFestival) return false;

    return true;
  }

  getRidingSprite() {
    if (this._animalType) {
      return Managers.Player.generatePlayerRidingSpriteName(this._animalType);
    }

    return '';
  }

  characterName() {
    if (Switches.hidePlayer || Switches.hideParty) {
      return '';
    }

    if (this._animalType) {
      return '';
    }

    return this._characterName;
  }

  characterIndex() {
    let base = super.characterIndex();

    if (Switches.hidePlayer) return base;
    if (Switches.hideParty) return base;

    if (Switches.playerWearingSwimsuit) {
      return 5;
    }

    if (this._animalType) return base;
    if (this._toolId) return base;

    if (this._itemWaitCount <= 0) {
      if (Inventory.displayItemId != 'none') {
        if (Inventory.displayItemId || (Inventory.isHoldingItem() && !Inventory.waiting)) {
          base = 2;
        }
      }
    }

    if (this.isDashing()) {
      return base + 4;
    } else {
      return base;
    }
  }

  getFeetTile() {
    const leftX = this.realLeft;
    const rightX = this.realRight;
    const leftDiff = 1 - (leftX - leftX.floor());
    const rightDiff = rightX - rightX.floor();
    let x;

    if (leftDiff > rightDiff) {
      x = leftX.floor();
    } else {
      x = rightX.floor();
    }

    const y = this.realBottom.floor();

    return new TileHelper.Tile(x, y);
  }

  isFeetTouchingTile(x, y) {
    if (x < this.realLeft.floor() || x > this.realRight.floor()) {
      return false;
    }

    if (y !== Math.floor(this.realBottom)) {
      return false;
    }

    return true;
  }

  isFeetTouchingBush() {
    const minX = this.realLeft.floor();
    const maxX = this.realRight.floor();
    const y = this.realBottom.floor();

    for (let x = minX; x <= maxX; x++) {
      if (Managers.FarmObjects.isBushObject($gameMap._mapId, x, y)) {
        return true;
      }
    }

    return false;
  }

  leaveTile(x, y) {
    if (!this.isTouchingTile(x, y)) return;

    switch(this._direction) {
      case Direction.LEFT :
        if (Math.floor(this._x) == x) {
          this._x = x + 1;
        }
        break;
      case Direction.RIGHT :
        if (Math.ceil(this._x) == x) {
          this._x = x - 1;
        }
        break;
      case Direction.DOWN :
        if (Math.ceil(this._y) == y) {
          this._y = y - 1;
        }
        break;
      case Direction.UP :
        if (Math.floor(this._y) == y) {
          this._y = y + 1;
        }
        break;
    }
  }

  stepBack() {
    switch(this._direction) {
      case Direction.LEFT :
        this._x += 1;
        break;
      case Direction.RIGHT :
        this._x -= 1;
        break;
      case Direction.DOWN :
        this._y -= 1;
        break;
      case Direction.UP :
        this._y += 1;
        break;
    }
  }

  goHome() {
    this.reserveTransfer($gameSystem.currentHome(), 15, 10, Direction.UP, 0, true, true);
  }

  characterMoveSpeed() {
    let speed = this._moveSpeed;

    if (this.isRiding()) {
      switch(this._animalType) {
        case 'horse':
          speed += 0.5;
          break;
      }
    }

    return speed;
  }

  dashSpeedBoost() {
    return 1.3;
  }

  checkTiredAnimations() {
    // if (this._tiredAnimationDelay) {
    //   this._tiredAnimationDelay--;
    // }

    if (!this.isRiding()) {
      const staminaLevel = Managers.Health.staminaLevel;
      const oldStaminaLevel = Managers.Health.oldStaminaLevel;
      let animationName = false;

      if (staminaLevel <= 1) {
        animationName = 'player_pass_out';
      } else if (staminaLevel < 10) {
        if (oldStaminaLevel >= 10) {
          animationName = 'player_very_tired';
        }
      } else if (staminaLevel <= 50) {
        if (oldStaminaLevel > 50) {
          animationName = 'player_tired';
        }
      }

      if (animationName) {
        Managers.CommonEvent.playEvent(animationName);
      }
    }

    Managers.Health.syncStaminaLevel();
  }

  setDirectionAfterMoving(d) {
  }

  getRectPosition() {
    // Let's take into account the player's position offset, because without it the end result feels weird

    let left = this.left;
    let top = this.top;

    let width = this.right - this.left;
    let height = this.bottom - this.top;

    const offsetHeight = (8 / $gameMap.tileHeight());
    const extraWidth = (8 / $gameMap.tileWidth());

    left -= extraWidth;
    top -= offsetHeight;
    width += extraWidth * 2;
    height += offsetHeight * 2;


    return new PIXI.Rectangle(left, top, width, height);
  }

  getCenterPosition() {
    const left = this.left;
    const top = this.top;

    const width = this.right - this.left;
    const height = this.bottom - this.top;

    return new PIXI.Point(left + width / 2, top + height / 2);
  }

  mountOnAnimal(animalData) {
    const typeData = animalData.getCreatureType();
    if (!typeData) return false;

    if (typeData.isHorse) {
      this._animalType = 'horse';
    } else {
      return;
    }

    this._animalSpriteName = typeData._spriteName;
    this._animalRunningSpriteName = typeData._runningSpriteName || typeData._spriteName;
    this._animalRunningSpriteFrames = typeData._runningSpriteFrames || 3;

    const regex = /\[(\d*)\]/;
    if (this._animalSpriteName) {
      const match = this._animalSpriteName.match(regex);
      if (match && match.length >= 2) {
        this._animalRunningSpriteFrames = parseInt(match[1]);
      } else {
        this._animalRunningSpriteFrames = 3;
      }
    } else {
      this._animalRunningSpriteFrames = undefined;
    }

    this._animalIndex = typeData._spriteIndex;
    this._animalId = animalData.id;
  }

  getCurrentAnimalSpriteName() {
    if (this.isRunning()) {
      return this._animalRunningSpriteName || this._animalSpriteName;
    } else {
      return this._animalSpriteName;
    }
  }

  getCurrentAnimalSpriteFrameCount() {
    if (this.isRunning()) {
      return this._animalRunningSpriteFrames || 3;
    } else {
      return 3;
    }
  }

  maxPattern() {
    return this._animalRunningSpriteFrames || 3;
  }

  mountOnAnimalEvent(event) {
    if (!event) return;
    if (!event._creatureData) return;

    const typeData = event._creatureData.getCreatureType();
    if (!typeData) return;

    event._moveType = 0;

    const baseSprite = typeData._spriteName;
    const runningSprite = typeData._runningSpriteName;

    Managers.Images.loadCharacter(baseSprite);
    if (runningSprite) {
      Managers.Images.loadCharacter(runningSprite);
    }

    const { x, y } = this;

    this.genericDelay(20);
    this.jumpTo(event.x, event.y);

    $gameTemp.setTimeout(() => {
      $gamePlayer._direction = event._direction;
      event.erase();
      $gamePlayer.mountOnAnimal(event._creatureData);

      // If the player got stuck, then move them back to the position they were before jumping
      if (this.isMovementBlocked()) {
        $gameTemp.setTimeout(() => {
          this._x = x;
          this._y = y;
        }, 10);
      }
    }, 10);
  }

  isRiding() {
    return Boolean(this._animalId);
  }

  isRidingAnimal(animalId) {
    if (!this.isRiding()) return false;

    if (this._animalId != animalId) return false;

    return true;
  }

  clearRideData() {
    this._animalType = undefined;
    this._animalSpriteName = undefined;
    this._animalRunningSpriteName = undefined;
    this._animalRunningSpriteFrames = undefined;
    this._animalIndex = 0;
    this._animalId = '';
  }

  getOffRide() {
    const animalData = Managers.Creatures.getCreatureById(this._animalId);

    if (!animalData) {
      return;
    }

    const { x, y } = $gamePlayer;

    if (!TileHelper.isTileEmpty(x, y, true, false, false, true)) {
      $gamePlayer.requestBalloon(Balloons.FAILURE);
      Managers.Sound.playBuzzer();
      return false;
    }

    this.clearRideData();

    const event = Managers.Creatures.dropAnimalAt($gameMap._mapId, Math.round($gamePlayer.x), Math.round($gamePlayer.y), animalData.type, animalData);
    if (event && event.setPosition) {
      event.setPosition($gamePlayer.x, $gamePlayer.y);
    }

    return true;
  }

  canRideAnimal(animalData) {
    if (Inventory.isHoldingItem()) return false;
    if (this.isRiding()) return false;
    if ($gameParty.hasCompanion()) return false;
    if (Managers.Content.canRideAnimal(animalData) === false) return false;

    // if ($gameTemp.isPlaytest()) {
    //   return true;
    // }

    const typeData = animalData.getCreatureType();
    return Boolean(typeData && typeData.isHorse);
  }

  checkMapPassable(x, y, d) {
    const regionId = $gameMap.regionId(x, y);
    if (this.isRiding()) {
      if (regionId == Region.BLOCK_ANIMALS) return false;
      if (regionId == Region.STAIR_TILES) {
        if (this._animalType != 'horse') {
          return false;
        }
      }
    }

    return Objects.CustomEvent.prototype.checkMapPassable.apply(this, arguments);
  }

  endToolAnimation() {
    Objects.Character.prototype.endToolAnimation.call(this);
    this.genericDelay(10);
  }

  getHitboxX() {
    return 3;
  }

  getHitboxY() {
    return 8;
  }

  getHitboxWidth() {
    return 25;
  }

  getHitboxHeight() {
    // if (this.isRiding()) {
    //   return 25;
    // } else {
    return 17;
    // }
  }

  setFollowerPosition(x, y, d) {
    const follower = this.follower();
    if (!follower) return;

    follower._x = x;
    follower._y = y;
    follower._realX = x;
    follower._realY = y;
    follower._direction = d;
    follower._lostX = 0;
    follower._lostY = 0;
  }

  itemToDisplay() {
    return Inventory.itemToDisplay;
  }

  animateSpaJump() {
    this._y -= 0.75;

    const spriteName = Managers.Player.generatePlayerSpriteName() + '_animations';
    const relaxingSpriteName = Managers.Player.generatePlayerSpriteName() + '_extra';
    Managers.Images.loadCharacter(spriteName);

    const speed = 6;

    const makeSplash = (character) => {
      const splash = $gameMap.createPlayerImpersonatorEvent('villagers/$splash', 0, 2, false, false, 0, true);
      splash.animate([
        { direction: 2, pattern: 0 },
        { pattern: 1 },
        { pattern: 2 },
        { direction: 4, pattern: 0 },
        { erase: true },
      ], { defaultStepLength: speed });
    };

    const doAnimation = () => {
      if (!Managers.Images.loadCharacter(spriteName)) {
        $gameTemp.setTimeout(() => {
          doAnimation();
        }, 12);
        return;
      }

      Switches.hidePlayer = true;
      $gamePlayer.offsetX = 1;
      const impersonator = $gameMap.createPlayerImpersonatorEvent(spriteName, 0, Direction.DOWN, false, false, 0, true);
      impersonator.animate([
        { direction: 2, pattern: 0, y: 0 },
        { pattern: 1, y: 0 },
        { pattern: 2, y: 0 },
        { direction: 4, pattern: 0, y: -0.25, offsetX: 0 },
        { pattern: 1, y: -0.25 },
        { pattern: 2, y: -0.1 },
        { direction: 6, pattern: 0, y: 0.25, callback: makeSplash },
        { pattern: 1, y: 0 },
        { fileName: 'villagers/$underwater', direction: 2, pattern: 0, y: -0.125, x: -0.125 },
        { pattern: 1, y: -0.125, x: -0.25 },
        { pattern: 2, y: -0.125, x: -0.25 },
        { pattern: 0, y: -0.125, x: -0.25 },
        { pattern: 1, x: -0.25 },
        { pattern: 2, x: -0.25 },
        { fileName: spriteName, direction: 6, pattern: 2 },
        { fileName: relaxingSpriteName, index: 1, direction: 2, pattern: 0, stepAnime: true, lockPattern: false, offsetY: 4, frameLength: 120 },
        { eventName: 'fade_out_spa', frameLength: 20 },
        { erase: true}
      ], { defaultStepLength: speed });
    };

    $gameTemp.setTimeout(() => {
      doAnimation();
    }, 30);
  }
};
